/**
 * @file ntmcmd_fill.c
 * @author Shinichiro Nakamura
 * @brief fillコマンドの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmcmd_fill.h"
#include "ntmuart.h"
#include "ntmlibc.h"
#include "ntmrand.h"
#include "ntmio.h"
#include "ntmconf.h"
#include "ntmcmd_common.h"

#if (NTMCONF_ENABLE_FILL == 1)

static const char *DISPTEXT_USAGE_FB = "FB [ADDR] [LEN] [DATA|PATTERN]   (PATTERN: inc, dec, rand, bitwalk)\r\n";
static const char *DISPTEXT_USAGE_FW = "FW [ADDR] [LEN] [DATA|PATTERN]   (PATTERN: inc, dec, rand, bitwalk)\r\n";
static const char *DISPTEXT_USAGE_FL = "FL [ADDR] [LEN] [DATA|PATTERN]   (PATTERN: inc, dec, rand, bitwalk)\r\n";

static const char *FILLTYPE_INC     = "inc";
static const char *FILLTYPE_DEC     = "dec";
static const char *FILLTYPE_RAND    = "rand";
static const char *FILLTYPE_BITWALK = "bitwalk";

typedef enum {
    FillTypeValue,
    FillTypeInc,
    FillTypeDec,
    FillTypeRand,
    FillTypeBitwalk
} FillType;

static ntmrand_t ntmrand;
static int ntmrand_initialized = 0;

/**
 * @brief FILLコマンド。
 *
 * @retval 0 与えられたパラメータはFILLコマンドでなかった。
 * @retval 1 与えられたパラメータをFILLコマンドとして処理した。
 */
int ntmcmd_fill(int argc, char **argv, void *extobj)
{
    NtmAccessMode am = NtmAccessModeDefault;
    uint32_t adr;
    uint32_t len;
    uint32_t dat;
    NtmStrType adr_type;
    NtmStrType len_type;
    NtmStrType dat_type;
    FillType filltype = FillTypeValue;
    uint32_t i;

    /*
     * コマンドからアクセスモードを決定する。
     */
    if ((ntmlibc_strcmp(argv[0], "fb") == 0) || (ntmlibc_strcmp(argv[0], "FB") == 0)) {
        am = NtmAccessModeByte;
    } else if ((ntmlibc_strcmp(argv[0], "fw") == 0) || (ntmlibc_strcmp(argv[0], "FW") == 0)) {
        am = NtmAccessModeWord;
    } else if ((ntmlibc_strcmp(argv[0], "fl") == 0) || (ntmlibc_strcmp(argv[0], "FL") == 0)) {
        am = NtmAccessModeLong;
    } else {
        /*
         * 与えられたコマンドは、このモジュールのコマンドに該当しなかった。
         */
        return 0;
    }

    /*
     * パラメータ数を検証する。
     */
    if (argc != 4) {
        switch (am) {
            case NtmAccessModeByte:
                ntmuart_write(DISPTEXT_USAGE_FB, ntmlibc_strlen(DISPTEXT_USAGE_FB));
                return 1;
            case NtmAccessModeWord:
                ntmuart_write(DISPTEXT_USAGE_FW, ntmlibc_strlen(DISPTEXT_USAGE_FW));
                return 1;
            case NtmAccessModeLong:
                ntmuart_write(DISPTEXT_USAGE_FL, ntmlibc_strlen(DISPTEXT_USAGE_FL));
                return 1;
            case NtmAccessModeDefault:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
            default:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
        }
    }

    /*
     * ユーザからの入力を取得する。
     */
    ntmstr_string_to_number(argv[1], &adr, &adr_type, NTMCONF_DEFAULT_STRTYPE);
    ntmstr_string_to_number(argv[2], &len, &len_type, NTMCONF_DEFAULT_STRTYPE);
    ntmstr_string_to_number(argv[3], &dat, &dat_type, NTMCONF_DEFAULT_STRTYPE);

    /*
     * ユーザからの入力を検証する。
     * 不正な指定が有った場合、エラーメッセージを出力して終了する。
     */
    if (adr_type == NtmStrTypeInvalid) {
        /*
         * 認識できない型を発見したということは、不正なアドレス指定。
         */
        ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
        return 1;
    }
    if (len_type == NtmStrTypeInvalid) {
        /*
         * 認識できない型を発見したということは、不正な長さ指定。
         */
        ntmuart_write(NTMCMD_COMMON_TEXT_INVLEN, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVLEN));
        return 1;
    }
    if (dat_type == NtmStrTypeInvalid) {
        if (ntmlibc_strcmp(FILLTYPE_INC, argv[3]) == 0) {
            filltype = FillTypeInc;
        }
        if (ntmlibc_strcmp(FILLTYPE_DEC, argv[3]) == 0) {
            filltype = FillTypeDec;
        }
        if (ntmlibc_strcmp(FILLTYPE_RAND, argv[3]) == 0) {
            filltype = FillTypeRand;
        }
        if (ntmlibc_strcmp(FILLTYPE_BITWALK, argv[3]) == 0) {
            filltype = FillTypeBitwalk;
        }
        if (filltype == FillTypeValue) {
            ntmuart_write(NTMCMD_COMMON_TEXT_INVDAT, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVDAT));
            return 1;
        }
    } else {
        filltype = FillTypeValue;
    }

    /*
     * 乱数器を初期化する。
     */
    if (ntmrand_initialized == 0) {
        ntmrand_init(&ntmrand, 0x1B3DA2C4);
        ntmrand_initialized = 1;
    }

    /*
     * 乱数器の再初期化を予約する。
     */
    if (ntmlibc_strcmp(argv[0], "seed") == 0) {
        ntmrand_initialized = 0;
        return 1;
    }

    /*
     * アクセスモードに応じて処理を実行する。
     */
    switch (am) {
        case NtmAccessModeByte:
            if (dat & 0xFFFFFF00) {
                /*
                 * バイト書き込みにも関わらず0xFF以外のビットが立っているのはおかしい。
                 */
                ntmuart_write(NTMCMD_COMMON_TEXT_INVDAT, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVDAT));
                return 1;
            }
            for (i = 0; i < len / 1; i++) {
                /*
                 * データを準備する。
                 */
                switch (filltype) {
                    case FillTypeValue:
                        // datは既に格納された値を使う。
                        break;
                    case FillTypeInc:
                        if (i == 0) {
                            dat = 0x00;
                        } else {
                            dat = (dat == 0xFF) ? 0x00 : (dat + 1);
                        }
                        break;
                    case FillTypeDec:
                        if (i == 0) {
                            dat = 0xFF;
                        } else {
                            dat = (dat == 0x00) ? 0xFF : (dat - 1);
                        }
                        break;
                    case FillTypeRand:
                        ntmrand_pop(&ntmrand, 8, &dat);
                        break;
                    case FillTypeBitwalk:
                        dat = (1 << ((i * 1) % 8));
                        break;
                }

                /*
                 * 指定アドレスにデータを書き込む。
                 */
                ntmio_write_byte(adr + (i * 1), dat);
            }
            break;
        case NtmAccessModeWord:
            if (dat & 0xFFFF0000) {
                /*
                 * ワード書き込みにも関わらず0xFFFF以外のビットが立っているのはおかしい。
                 */
                ntmuart_write(NTMCMD_COMMON_TEXT_INVDAT, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVDAT));
                return 1;
            }
            for (i = 0; i < len / 2; i++) {
                /*
                 * データを準備する。
                 */
                switch (filltype) {
                    case FillTypeValue:
                        // datは既に格納された値を使う。
                        break;
                    case FillTypeInc:
                        if (i == 0) {
                            dat = 0x0000;
                        } else {
                            dat = (dat == 0xFFFF) ? 0x0000 : (dat + 1);
                        }
                        break;
                    case FillTypeDec:
                        if (i == 0) {
                            dat = 0xFFFF;
                        } else {
                            dat = (dat == 0x0000) ? 0xFFFF : (dat - 1);
                        }
                        break;
                    case FillTypeRand:
                        ntmrand_pop(&ntmrand, 16, &dat);
                        break;
                    case FillTypeBitwalk:
                        dat = (1 << (i % 16));
                        break;
                }

                /*
                 * 指定アドレスにデータを書き込む。
                 */
                ntmio_write_word(adr + (i * 2), dat);
            }
            break;
        case NtmAccessModeLong:
            for (i = 0; i < len / 4; i++) {
                /*
                 * データを準備する。
                 */
                switch (filltype) {
                    case FillTypeValue:
                        // datは既に格納された値を使う。
                        break;
                    case FillTypeInc:
                        if (i == 0) {
                            dat = 0x00000000;
                        } else {
                            dat = (dat == 0xFFFFFFFF) ? 0x00000000 : (dat + 1);
                        }
                        break;
                    case FillTypeDec:
                        if (i == 0) {
                            dat = 0xFFFFFFFF;
                        } else {
                            dat = (dat == 0x00000000) ? 0xFFFFFFFF : (dat - 1);
                        }
                        break;
                    case FillTypeRand:
                        ntmrand_pop(&ntmrand, 32, &dat);
                        break;
                    case FillTypeBitwalk:
                        dat = (1 << (i % 32));
                        break;
                }

                /*
                 * 指定アドレスにデータを書き込む。
                 */
                ntmio_write_long(adr + (i * 4), dat);
            }
            break;
        default:
            ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
            break;
    }
    return 1;
}

#endif

