/**
 * @file ntmcmd_dump.c
 * @author Shinichiro Nakamura
 * @brief dumpコマンドの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmcmd_dump.h"
#include "ntmuart.h"
#include "ntmlibc.h"
#include "ntmio.h"
#include "ntmconf.h"
#include "ntmcmd_common.h"

#if (NTMCONF_ENABLE_DUMP == 1)

static const char *DISPTEXT_USAGE_DB = "DB [ADDR] (LEN)\r\n";
static const char *DISPTEXT_USAGE_DW = "DW [ADDR] (LEN)\r\n";
static const char *DISPTEXT_USAGE_DL = "DL [ADDR] (LEN)\r\n";

static const char *DISPTEXT_DUMP_HEAD1_DB = "--------+---------------------------------------------------+----------------\r\n";
static const char *DISPTEXT_DUMP_HEAD2_DB = "  ADDR  | +0 +1 +2 +3 +4 +5 +6 +7   +8 +9 +A +B +C +D +E +F |   ASCII TEXT   \r\n";
static const char *DISPTEXT_DUMP_HEAD3_DB = "--------+---------------------------------------------------+----------------\r\n";

static const char *DISPTEXT_DUMP_HEAD1_DW = "--------+-------------------------------------------+----------------\r\n";
static const char *DISPTEXT_DUMP_HEAD2_DW = "  ADDR  |  +0   +1   +2   +3     +4   +5   +6   +7  |   ASCII TEXT   \r\n";
static const char *DISPTEXT_DUMP_HEAD3_DW = "--------+-------------------------------------------+----------------\r\n";

static const char *DISPTEXT_DUMP_HEAD1_DL = "--------+---------------------------------------+----------------\r\n";
static const char *DISPTEXT_DUMP_HEAD2_DL = "  ADDR  |    +0       +4         +8       +C    |   ASCII TEXT   \r\n";
static const char *DISPTEXT_DUMP_HEAD3_DL = "--------+---------------------------------------+----------------\r\n";

/**
 * @brief DUMPコマンド。
 *
 * @retval 0 与えられたパラメータはDUMPコマンドでなかった。
 * @retval 1 与えられたパラメータをDUMPコマンドとして処理した。
 */
int ntmcmd_dump(int argc, char **argv, void *extobj)
{
    NtmAccessMode am = NtmAccessModeDefault;
    uint32_t adr;
    uint32_t len;
    NtmStrType adr_type;
    NtmStrType len_type;
    char string[16];
    char ascii[24];
    uint32_t i;

    /*
     * コマンドからアクセスモードを決定する。
     */
    if ((ntmlibc_strcmp(argv[0], "db") == 0) || (ntmlibc_strcmp(argv[0], "DB") == 0)) {
        am = NtmAccessModeByte;
    } else if ((ntmlibc_strcmp(argv[0], "dw") == 0) || (ntmlibc_strcmp(argv[0], "DW") == 0)) {
        am = NtmAccessModeWord;
    } else if ((ntmlibc_strcmp(argv[0], "dl") == 0) || (ntmlibc_strcmp(argv[0], "DL") == 0)) {
        am = NtmAccessModeLong;
    } else {
        /*
         * 与えられたコマンドは、このモジュールのコマンドに該当しなかった。
         */
        return 0;
    }

    /*
     * パラメータ数を検証する。
     */
    if ((argc != 2) && (argc != 3)) {
        switch (am) {
            case NtmAccessModeByte:
                ntmuart_write(DISPTEXT_USAGE_DB, ntmlibc_strlen(DISPTEXT_USAGE_DB));
                return 1;
            case NtmAccessModeWord:
                ntmuart_write(DISPTEXT_USAGE_DW, ntmlibc_strlen(DISPTEXT_USAGE_DW));
                return 1;
            case NtmAccessModeLong:
                ntmuart_write(DISPTEXT_USAGE_DL, ntmlibc_strlen(DISPTEXT_USAGE_DL));
                return 1;
            case NtmAccessModeDefault:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
            default:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
        }
    }

    /*
     * ユーザからの入力を取得する。
     */
    ntmstr_string_to_number(argv[1], &adr, &adr_type, NTMCONF_DEFAULT_STRTYPE);
    if (argc == 3) {
        ntmstr_string_to_number(argv[2], &len, &len_type, NTMCONF_DEFAULT_STRTYPE);
    } else {
        len_type = NtmStrTypeHex;
        len = 0x100;
    }

    /*
     * ユーザからの入力を検証する。
     * 不正な指定が有った場合、エラーメッセージを出力して終了する。
     */
    if (adr_type == NtmStrTypeInvalid) {
        ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
        return 1;
    }
    if (len_type == NtmStrTypeInvalid) {
        ntmuart_write(NTMCMD_COMMON_TEXT_INVLEN, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVLEN));
        return 1;
    }

    /*
     * ユーザからの入力を補正する。
     * ユーザが見たいデータを含むように補正するために、
     * アドレスは切り捨て方向に、長さは切り上げ方向に補正する。
     */
    if ((adr % 0x10) != 0) {
        adr = ((adr + 0x00) / 0x10) * 0x10;
    }
    if ((len % 0x10) != 0) {
        len = ((len + 0x0F) / 0x10) * 0x10;
    }

    /*
     * アクセスモードに応じて処理を実行する。
     */
    switch (am) {
        case NtmAccessModeByte:
            {
                /*
                 * ヘッドを出力する。
                 */
                ntmuart_write(DISPTEXT_DUMP_HEAD1_DB, ntmlibc_strlen(DISPTEXT_DUMP_HEAD1_DB));
                ntmuart_write(DISPTEXT_DUMP_HEAD2_DB, ntmlibc_strlen(DISPTEXT_DUMP_HEAD2_DB));
                ntmuart_write(DISPTEXT_DUMP_HEAD3_DB, ntmlibc_strlen(DISPTEXT_DUMP_HEAD3_DB));

                /*
                 * 指定された長さに応じて処理を実行する。
                 */
                for (i = 0; i < len / 1; i++) {
                    uint8_t data;
                    /*
                     * 16バイト境界毎にヘッダを表示する。
                     */
                    if (((i * 1) % 0x10) == 0x0) {
                        ntmstr_number_to_string(adr + (i * 1), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("| ", 2);
                    }

                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_byte(adr + (i * 1), &data);

                    /*
                     * ASCII表示用にデータを用意する。
                     */
                    ascii[(i * 1) % 16] = ntmstr_isprint(data) ? data : '.';

                    /*
                     * 値を表示する。
                     */
                    ntmstr_number_to_string(data, &string[0], 2, NtmStrTypeHex);
                    ntmuart_write(string, ntmlibc_strlen(string));

                    /*
                     * 値と値の間に空白を作る。
                     */
                    if (((i * 1) % 0x10) == 0x7) {
                        ntmuart_write(" - ", 3);
                    } else {
                        ntmuart_write(" ", 1);
                    }

                    /*
                     * 行末ならばASCII表示を加えて改行する。
                     */
                    if (((i * 1) % 0x10) == 0xF) {
                        ntmuart_write("|", 1);
                        ascii[16] = '\0';
                        ntmuart_write(ascii, ntmlibc_strlen(ascii));
                        ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));
                    }
                }
                ntmuart_write(DISPTEXT_DUMP_HEAD1_DB, ntmlibc_strlen(DISPTEXT_DUMP_HEAD1_DB));
            }
            break;
        case NtmAccessModeWord:
            {
                /*
                 * ヘッドを出力する。
                 */
                ntmuart_write(DISPTEXT_DUMP_HEAD1_DW, ntmlibc_strlen(DISPTEXT_DUMP_HEAD1_DW));
                ntmuart_write(DISPTEXT_DUMP_HEAD2_DW, ntmlibc_strlen(DISPTEXT_DUMP_HEAD2_DW));
                ntmuart_write(DISPTEXT_DUMP_HEAD3_DW, ntmlibc_strlen(DISPTEXT_DUMP_HEAD3_DW));

                /*
                 * 指定された長さに応じて処理を実行する。
                 */
                for (i = 0; i < len / 2; i++) {
                    uint16_t data;
                    /*
                     * 16バイト境界毎にヘッダを表示する。
                     */
                    if (((i * 2) % 0x10) == 0x0) {
                        ntmstr_number_to_string(adr + (i * 2), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("| ", 2);
                    }

                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_word(adr + (i * 2), &data);

                    /*
                     * ASCII表示用にデータを用意する。
                     */
                    ascii[((i * 2) % 16) + 0] = ntmstr_isprint((uint8_t)(data >> 0)) ? ((uint8_t)(data >> 0)) : '.';
                    ascii[((i * 2) % 16) + 1] = ntmstr_isprint((uint8_t)(data >> 8)) ? ((uint8_t)(data >> 8)) : '.';

                    /*
                     * 値を表示する。
                     */
                    ntmstr_number_to_string(data, &string[0], 4, NtmStrTypeHex);
                    ntmuart_write(string, ntmlibc_strlen(string));

                    /*
                     * 値と値の間に空白を作る。
                     */
                    if (((i * 2) % 0x10) == 0x6) {
                        ntmuart_write(" - ", 3);
                    } else {
                        ntmuart_write(" ", 1);
                    }

                    /*
                     * 行末ならばASCII表示を加えて改行する。
                     */
                    if (((i * 2) % 0x10) == 0xE) {
                        ntmuart_write("|", 1);
                        ascii[16] = '\0';
                        ntmuart_write(ascii, ntmlibc_strlen(ascii));
                        ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));
                    }
                }
                ntmuart_write(DISPTEXT_DUMP_HEAD1_DW, ntmlibc_strlen(DISPTEXT_DUMP_HEAD1_DW));
            }
            break;
        case NtmAccessModeLong:
            {
                /*
                 * ヘッドを出力する。
                 */
                ntmuart_write(DISPTEXT_DUMP_HEAD1_DL, ntmlibc_strlen(DISPTEXT_DUMP_HEAD1_DL));
                ntmuart_write(DISPTEXT_DUMP_HEAD2_DL, ntmlibc_strlen(DISPTEXT_DUMP_HEAD2_DL));
                ntmuart_write(DISPTEXT_DUMP_HEAD3_DL, ntmlibc_strlen(DISPTEXT_DUMP_HEAD3_DL));

                /*
                 * 指定された長さに応じて処理を実行する。
                 */
                for (i = 0; i < len / 4; i++) {
                    uint32_t data;
                    /*
                     * 16バイト境界毎にヘッダを表示する。
                     */
                    if (((i * 4) % 0x10) == 0x0) {
                        ntmstr_number_to_string(adr + (i * 4), &string[0], 8, NtmStrTypeHex);
                        ntmuart_write(string, ntmlibc_strlen(string));
                        ntmuart_write("| ", 2);
                    }

                    /*
                     * 指定アドレスのデータを読み込む。
                     */
                    ntmio_read_long(adr + (i * 4), &data);

                    /*
                     * ASCII表示用にデータを用意する。
                     */
                    ascii[((i * 4) % 16) + 0] = ntmstr_isprint((uint8_t)(data >>  0)) ? ((uint8_t)(data >>  0)) : '.';
                    ascii[((i * 4) % 16) + 1] = ntmstr_isprint((uint8_t)(data >>  8)) ? ((uint8_t)(data >>  8)) : '.';
                    ascii[((i * 4) % 16) + 2] = ntmstr_isprint((uint8_t)(data >> 16)) ? ((uint8_t)(data >> 16)) : '.';
                    ascii[((i * 4) % 16) + 3] = ntmstr_isprint((uint8_t)(data >> 24)) ? ((uint8_t)(data >> 24)) : '.';

                    /*
                     * 値を表示する。
                     */
                    ntmstr_number_to_string(data, &string[0], 8, NtmStrTypeHex);
                    ntmuart_write(string, ntmlibc_strlen(string));

                    /*
                     * 値と値の間に空白を作る。
                     */
                    if (((i * 4) % 0x10) == 0x4) {
                        ntmuart_write(" - ", 3);
                    } else {
                        ntmuart_write(" ", 1);
                    }

                    /*
                     * 行末ならばASCII表示を加えて改行する。
                     */
                    if (((i * 4) % 0x10) == 0xC) {
                        ntmuart_write("|", 1);
                        ascii[16] = '\0';
                        ntmuart_write(ascii, ntmlibc_strlen(ascii));
                        ntmuart_write(NTMCMD_COMMON_TEXT_CRLF, ntmlibc_strlen(NTMCMD_COMMON_TEXT_CRLF));
                    }
                }
                ntmuart_write(DISPTEXT_DUMP_HEAD1_DL, ntmlibc_strlen(DISPTEXT_DUMP_HEAD1_DL));
            }
            break;
        default:
            ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
            break;
    }
    return 1;
}

#endif

