/**
 * @file ntmcmd_copy.c
 * @author Shinichiro Nakamura
 * @brief copyコマンドの実装。
 */

/*
 * ===============================================================
 *  Natural Tiny Monitor (NT-Monitor)
 * ===============================================================
 * Copyright (c) 2011-2012 Shinichiro Nakamura
 * Inspired by M, Murakami
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntmcmd_copy.h"
#include "ntmuart.h"
#include "ntmlibc.h"
#include "ntmio.h"
#include "ntmconf.h"
#include "ntmcmd_common.h"

#if (NTMCONF_ENABLE_COPY == 1)

static const char *DISPTEXT_USAGE_CB = "CB [ADDR-SRC] [ADDR-DES] [LEN]\r\n";
static const char *DISPTEXT_USAGE_CW = "CW [ADDR-SRC] [ADDR-DES] [LEN]\r\n";
static const char *DISPTEXT_USAGE_CL = "CL [ADDR-SRC] [ADDR-DES] [LEN]\r\n";

/**
 * @brief コピーモード。
 * @details
 * メモリをコピーする場合、指定する二つのアドレスの位置関係によって、
 * コピーを前方から行なうか、後方から行なうかを適切に選択しなければならない。
 *
 * 指定される二つのアドレスの関係は、
 * 「重ならない」
 * 一部が重なり、コピー元アドレスよりもコピー先アドレスのほうが大きい」
 * 「一部が重なり、コピー元アドレスよりもコピー先アドレスのほうが小さい」
 * の３つに分類することができる。
 *
 * このうち、「一部が重なり、コピー元アドレスよりもコピー先アドレスの方が大きい」場合、
 * コピーを後方から行なう必要がある。
 */
typedef enum {
    CopyModeForward,    /**< 前方からコピーする。 */
    CopyModeBackward    /**< 後方からコピーする。 */
} CopyMode;

/**
 * @brief COPYコマンド。
 *
 * @retval 0 与えられたパラメータはCOPYコマンドでなかった。
 * @retval 1 与えられたパラメータをCOPYコマンドとして処理した。
 */
int ntmcmd_copy(int argc, char **argv, void *extobj)
{
    CopyMode copy_mode = CopyModeForward;
    NtmAccessMode am = NtmAccessModeDefault;
    uint32_t adr_src;
    uint32_t adr_des;
    uint32_t len;
    NtmStrType adr_src_type;
    NtmStrType adr_des_type;
    NtmStrType len_type;
    uint32_t i;

    /*
     * コマンドからアクセスモードを決定する。
     */
    if ((ntmlibc_strcmp(argv[0], "cb") == 0) || (ntmlibc_strcmp(argv[0], "CB") == 0)) {
        am = NtmAccessModeByte;
    } else if ((ntmlibc_strcmp(argv[0], "cw") == 0) || (ntmlibc_strcmp(argv[0], "CW") == 0)) {
        am = NtmAccessModeWord;
    } else if ((ntmlibc_strcmp(argv[0], "cl") == 0) || (ntmlibc_strcmp(argv[0], "CL") == 0)) {
        am = NtmAccessModeLong;
    } else {
        /*
         * 与えられたコマンドは、このモジュールのコマンドに該当しなかった。
         */
        return 0;
    }

    /*
     * パラメータ数を検証する。
     */
    if (argc != 4) {
        switch (am) {
            case NtmAccessModeByte:
                ntmuart_write(DISPTEXT_USAGE_CB, ntmlibc_strlen(DISPTEXT_USAGE_CB));
                return 1;
            case NtmAccessModeWord:
                ntmuart_write(DISPTEXT_USAGE_CW, ntmlibc_strlen(DISPTEXT_USAGE_CW));
                return 1;
            case NtmAccessModeLong:
                ntmuart_write(DISPTEXT_USAGE_CL, ntmlibc_strlen(DISPTEXT_USAGE_CL));
                return 1;
            case NtmAccessModeDefault:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
            default:
                ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
                return 1;
        }
    }

    /*
     * ユーザの入力を取得する。
     */
    ntmstr_string_to_number(argv[1], &adr_src, &adr_src_type, NTMCONF_DEFAULT_STRTYPE);
    ntmstr_string_to_number(argv[2], &adr_des, &adr_des_type, NTMCONF_DEFAULT_STRTYPE);
    ntmstr_string_to_number(argv[3], &len, &len_type, NTMCONF_DEFAULT_STRTYPE);

    /*
     * ユーザからの入力を検証する。
     * 不正な指定が有った場合、エラーメッセージを出力して終了する。
     */
    if (adr_src_type == NtmStrTypeInvalid) {
        ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
        return 1;
    }
    if (adr_des_type == NtmStrTypeInvalid) {
        ntmuart_write(NTMCMD_COMMON_TEXT_INVADR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVADR));
        return 1;
    }
    if (len_type == NtmStrTypeInvalid) {
        ntmuart_write(NTMCMD_COMMON_TEXT_INVLEN, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INVLEN));
        return 1;
    }

    /*
     * 二つのアドレスの関係を見てコピーモードを決定する。
     */
    if ((adr_src < adr_des) && (adr_des < (adr_src + len))) {
        copy_mode = CopyModeBackward;
    }

    /*
     * アクセスモードに応じて処理を実行する。
     */
    switch (am) {
        case NtmAccessModeByte:
            for (i = 0; i < len / 1; i++) {
                uint8_t data;
                if (copy_mode == CopyModeForward) {
                    ntmio_read_byte(adr_src + (i * 1), &data);
                    ntmio_write_byte(adr_des + (i * 1), data);
                } else {
                    ntmio_read_byte(adr_src + (len - 1) - (i * 1), &data);
                    ntmio_write_byte(adr_des + (len - 1) - (i * 1), data);
                }
            }
            break;
        case NtmAccessModeWord:
            for (i = 0; i < len / 2; i++) {
                uint16_t data;
                if (copy_mode == CopyModeForward) {
                    ntmio_read_word(adr_src + (i * 2), &data);
                    ntmio_write_word(adr_des + (i * 2), data);
                } else {
                    ntmio_read_word(adr_src + (len - 2) - (i * 2), &data);
                    ntmio_write_word(adr_des + (len - 2) - (i * 2), data);
                }
            }
            break;
        case NtmAccessModeLong:
            for (i = 0; i < len / 4; i++) {
                uint32_t data;
                if (copy_mode == CopyModeForward) {
                    ntmio_read_long(adr_src + (i * 4), &data);
                    ntmio_write_long(adr_des + (i * 4), data);
                } else {
                    ntmio_read_long(adr_src + (len - 4) - (i * 4), &data);
                    ntmio_write_long(adr_des + (len - 4) - (i * 4), data);
                }
            }
            break;
        default:
            ntmuart_write(NTMCMD_COMMON_TEXT_INTERR, ntmlibc_strlen(NTMCMD_COMMON_TEXT_INTERR));
            break;
    }
    return 1;
}

#endif

