﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using NotepadNeueExtension;

namespace ParseExtension
{
    class PhpParser : ParserBase
    {
        public override TokenizerBase Tokenizer
        {
            get { return new PhpTokenizer(); }
        }

        public PhpParser(IExtensionHost host)
            : base(host)
        {
        }

        #region IParser メンバー

        protected override int ParseImpl(int i, ParseToken[] tokens, BasicTree tree, ObjectInfo objectInfo,
            NotepadNeueExtension.ParseEventArgs e)
        {
            List<ParseToken> attributes = new List<ParseToken>();
            VariableInfo[] variablesInBrace = null;
            for (; i < tokens.Length; i++)
            {
                switch (tokens[i].TokenType)
                {
                    case TokenType.Comment:
                    case TokenType.Literal:
                    case TokenType.NewLine:
                    case TokenType.WhiteSpace:
                        continue;
                    case TokenType.LetterOrDigit:
                        switch (tokens[i].Token)
                        {
                            case "class":
                                int next = SkipTo(TokenType.LetterOrDigit, i + 1, tokens);
                                if (next >= tokens.Length)
                                {
                                    return next;
                                }
                                ObjectInfo newObjectInfo = new ObjectInfo()
                                {
                                    Name = tokens[next].Token,
                                    DefineLocation = new DefineLocation()
                                    {
                                        DefineIndex = tokens[next].Index,
                                        Editor = e.AssistData.Editor,
                                    },
                                };
                                e.AssistData.GlobalClass.Add(newObjectInfo);
                                next = SkipTo("{", TokenType.Symbol, i + 2, tokens);
                                if (next >= tokens.Length)
                                {
                                    return next;
                                }
                                BasicTree newTree = new BasicTree()
                                {
                                    Start = tokens[next].Index,
                                };
                                newTree.AddData(new EitherVariableOrTree()
                                {
                                    VariableInfo = new VariableInfo()
                                    {
                                        Name = "$this",
                                        DefineLocation = new DefineLocation()
                                        {
                                            DefineIndex = tokens[next].Index + tokens[next].Token.Length,
                                            Editor = e.AssistData.Editor,
                                        },
                                        HintText = "",
                                        Type = newObjectInfo.Name,
                                    },
                                }, tokens[next].Index + tokens[next].Token.Length);
                                tree.AddData(new EitherVariableOrTree()
                                {
                                    BasicTree = newTree,
                                }, tokens[next].Index);
                                i = ParseImpl(next + 1, tokens, newTree, newObjectInfo, e);
                                if (newTree.End <= 0)
                                {
                                    newTree.End = e.ParseText.Length;
                                }
                                break;
                            case "if":
                            case "for":
                            case "foreach":
                            case "while":
                            case "switch":
                            case "catch":
                                int braceOpenIndex = SkipTo("(", TokenType.Symbol, i + 1, tokens);
                                if (braceOpenIndex >= tokens.Length)
                                {
                                    return braceOpenIndex;
                                }
                                int braceEndIndex = SkipToMatchedBrace(braceOpenIndex, tokens);
                                if (braceEndIndex >= tokens.Length)
                                {
                                    return braceEndIndex;
                                }
                                variablesInBrace = tokens[i].Token == "foreach" ?
                                    ParseVariableInForeach(braceOpenIndex + 1, braceEndIndex, tokens, e) :
                                    ParseVariableInBrace(braceOpenIndex + 1, braceEndIndex, tokens, e);
                                i = braceEndIndex;
                                break;
                            default:
                                if (tokens[i].Token.StartsWith("$"))
                                {
                                    next = SkipTo(TokenType.Symbol, i + 1, tokens);
                                    if (next < tokens.Length &&
                                        (tokens[next].Token == "=" || tokens[next].Token == ";") && attributes.FirstOrDefault(p => p.Token == "return") == null)
                                    {
                                        string type = "";
                                        if (tokens[next].Token == "=")
                                        {
                                            next = SkipTo(TokenType.LetterOrDigit, next + 1, tokens);
                                            if (next < tokens.Length && tokens[next].Token == "new")
                                            {
                                                next = SkipTo(TokenType.LetterOrDigit, next + 1, tokens);
                                                if (next < tokens.Length)
                                                {
                                                    type = tokens[next].Token;
                                                }
                                            }
                                        }
                                        VariableInfo variableInfo = new VariableInfo()
                                        {
                                            Name = tokens[i].Token,
                                            DefineLocation = new DefineLocation()
                                            {
                                                DefineIndex = tokens[i].Index,
                                                Editor = e.AssistData.Editor,
                                            },
                                            HintText = "",
                                            Type = type,
                                        };
                                        if (objectInfo == null)
                                        {
                                            tree.AddData(new EitherVariableOrTree()
                                            {
                                                VariableInfo = variableInfo
                                            }, tokens[i].Index);
                                        }
                                        else
                                        {
                                            objectInfo.AddVariable(new VariableInfo()
                                            {
                                                AccessType = GetAccessType(attributes),
                                                Name = tokens[i].Token.Substring(1),
                                                DefineLocation = new DefineLocation()
                                                {
                                                    DefineIndex = tokens[i].Index + 1,
                                                    Editor = e.AssistData.Editor,
                                                },
                                                HintText = "",
                                            }, tokens[i].Token);
                                        }
                                    }
                                    i = SkipTo(";", TokenType.Symbol, i, tokens);
                                    attributes.Clear();
                                }
                                else if (tokens[i].Token == "function")
                                {
                                    next = SkipTo(TokenType.LetterOrDigit, i + 1, tokens);
                                    if (next >= tokens.Length)
                                    {
                                        return next;
                                    }
                                    FunctionInfo functionInfo = new FunctionInfo()
                                    {
                                        AccessType = GetAccessType(attributes),
                                        Name = tokens[next].Token,
                                        DefineLocation = new DefineLocation()
                                        {
                                            DefineIndex = tokens[next].Index,
                                            Editor = e.AssistData.Editor,
                                        },
                                    };
                                    int braceStart = SkipTo("(", TokenType.Symbol, next + 1, tokens);
                                    int braceEnd = SkipTo(")", TokenType.Symbol, braceStart + 1, tokens);
                                    if (braceStart >= tokens.Length || braceEnd >= tokens.Length)
                                    {
                                        return tokens.Length;
                                    }
                                    List<ParseToken> letterOrDigits = new List<ParseToken>();
                                    List<VariableInfo> variables = new List<VariableInfo>();
                                    for (int j = braceStart + 1; j < braceEnd; j++)
                                    {
                                        switch (tokens[j].TokenType)
                                        {
                                            case TokenType.Comment:
                                            case TokenType.Literal:
                                            case TokenType.NewLine:
                                                break;
                                            case TokenType.LetterOrDigit:
                                                if (tokens[j].Token.StartsWith("$"))
                                                {
                                                    variables.Add(new VariableInfo()
                                                    {
                                                        Name = tokens[j].Token,
                                                        DefineLocation = new DefineLocation()
                                                        {
                                                            DefineIndex = tokens[j].Index,
                                                            Editor = e.AssistData.Editor,
                                                        },
                                                        Type = letterOrDigits.Aggregate("", (s, p) =>
                                                            String.IsNullOrEmpty(s) ? p.Token : String.Format("{0} {1}", s, p.Token)),
                                                    });
                                                }
                                                else
                                                {
                                                    letterOrDigits.Add(tokens[j]);
                                                }
                                                break;
                                            case TokenType.Symbol:
                                                letterOrDigits.Clear();
                                                break;
                                        }
                                    }
                                    string joinedText = tokens.Skip(braceStart + 1).
                                        Take(braceEnd - braceStart - 1).Aggregate("", (prev, p2) =>
                                        {
                                            switch (p2.TokenType)
                                            {
                                                case TokenType.Symbol:
                                                case TokenType.LetterOrDigit:
                                                    if (String.IsNullOrEmpty(prev))
                                                    {
                                                        return p2.Token;
                                                    }
                                                    else if (p2.Token == ",")
                                                    {
                                                        return String.Format("{0}{1}", prev, p2.Token);
                                                    }
                                                    return String.Format("{0} {1}", prev, p2.Token);
                                                default:
                                                    return prev;
                                            }
                                        });
                                    string[] arguments = joinedText.Split(new string[] { "," }, StringSplitOptions.RemoveEmptyEntries);
                                    if (arguments.Length > 0)
                                    {
                                        functionInfo.AddArgument(arguments);
                                    }
                                    functionInfo.AddContent(String.Format("function {0}({1})", functionInfo.Name, joinedText));
                                    int semiBraceStart = SkipTo("{", TokenType.Symbol, braceEnd + 1, tokens);
                                    if (semiBraceStart >= tokens.Length)
                                    {
                                        return semiBraceStart;
                                    }
                                    tree.AddFunctionData(functionInfo, tokens[next].Index);
                                    if (objectInfo != null)
                                    {
                                        objectInfo.AddMethod(functionInfo, tokens[next].Token);
                                    }
                                    newTree = new BasicTree()
                                    {
                                        Start = tokens[semiBraceStart].Index,
                                    };
                                    int iter = 0;
                                    foreach (VariableInfo variable in variables)
                                    {
                                        newTree.AddData(new EitherVariableOrTree()
                                        {
                                            VariableInfo = variable,
                                        }, tokens[semiBraceStart].Index + ++iter);
                                    }
                                    tree.AddData(new EitherVariableOrTree()
                                    {
                                        BasicTree = newTree,
                                    }, tokens[semiBraceStart].Index);
                                    i = ParseImpl(semiBraceStart + 1, tokens, newTree, null, e);
                                    if (newTree.End <= 0)
                                    {
                                        newTree.End = e.ParseText.Length;
                                    }
                                }
                                else
                                {
                                    attributes.Add(tokens[i]);
                                }
                                break;
                        }
                        break;
                    case TokenType.Symbol:
                        switch (tokens[i].Token)
                        {
                            case "}":
                                tree.End = tokens[i].Index;
                                return i;
                            case "{":
                                BasicTree newTree = new BasicTree()
                                {
                                    Start = tokens[i].Index,
                                };
                                if (variablesInBrace != null && variablesInBrace.Length > 0)
                                {
                                    foreach (VariableInfo variable in variablesInBrace)
                                    {
                                        newTree.AddData(new EitherVariableOrTree()
                                        {
                                            VariableInfo = variable,
                                        }, tokens[i].Index + 1);
                                    }
                                    variablesInBrace = null;
                                }
                                tree.AddData(new EitherVariableOrTree()
                                {
                                    BasicTree = newTree
                                }, tokens[i].Index);
                                i = ParseImpl(i + 1, tokens, newTree, null, e);
                                if (newTree.End <= 0)
                                {
                                    newTree.End = e.ParseText.Length;
                                }
                                break;
                            case ";":
                                attributes.Clear();
                                break;
                        }
                        break;
                }
            }

            return i;
        }

        private AccessType GetAccessType(List<ParseToken> tokens)
        {
            if (tokens == null || tokens.Count == 0)
            {
                return AccessType.Public;
            }

            foreach (ParseToken token in tokens)
            {
                switch (token.TokenType)
                {
                    case TokenType.LetterOrDigit:
                        switch (token.Token)
                        {
                            case "public":
                                return AccessType.Public;
                            case "protected":
                                return AccessType.Protected;
                            case "private":
                                return AccessType.Private;
                        }
                        break;
                }
            }

            return AccessType.Public;
        }

        private VariableInfo[] ParseVariableInBrace(int startIndex, int endIndex, ParseToken[] tokens, ParseEventArgs e)
        {
            List<VariableInfo> ret = new List<VariableInfo>();
            for (int i = startIndex; i < endIndex; i++)
            {
                switch (tokens[i].TokenType)
                {
                    case TokenType.LetterOrDigit:
                        if (tokens[i].Token.StartsWith("$"))
                        {
                            int next = SkipTo("=", TokenType.Symbol, i + 1, endIndex, tokens);
                            if (next >= endIndex)
                            {
                                i = endIndex;
                                break;
                            }
                            ret.Add(CreateVariableInfo(i, tokens, e));
                        }
                        break;
                    case TokenType.Symbol:
                        break;
                }
            }

            return ret.ToArray();
        }

        private VariableInfo CreateVariableInfo(int index, ParseToken[] tokens, ParseEventArgs e)
        {
            return new VariableInfo()
            {
                Name = tokens[index].Token,
                DefineLocation = new DefineLocation()
                {
                    DefineIndex = tokens[index].Index,
                    Editor = e.AssistData.Editor,
                },
            };
        }

        private VariableInfo[] ParseVariableInForeach(int startIndex, int endIndex, ParseToken[] tokens, ParseEventArgs e)
        {
            List<VariableInfo> ret = new List<VariableInfo>();
            for (int i = startIndex; i < endIndex; i++)
            {
                switch (tokens[i].TokenType)
                {
                    case TokenType.LetterOrDigit:
                        if (tokens[i].Token == "as")
                        {
                            int next = SkipTo(TokenType.LetterOrDigit, i + 1, tokens);
                            if (next >= endIndex)
                            {
                                i = endIndex;
                                break;
                            }
                            if (tokens[next].Token.StartsWith("$"))
                            {
                                ret.Add(CreateVariableInfo(next, tokens, e));
                            }
                            next = SkipTo("=>", TokenType.NewLine, next + 1, tokens);
                            if (next >= endIndex)
                            {
                                i = endIndex;
                                break;
                            }
                            next = SkipTo(TokenType.LetterOrDigit, next + 1, tokens);
                            if (next >= endIndex)
                            {
                                i = endIndex;
                                break;
                            }
                            if (tokens[next].Token.StartsWith("$"))
                            {
                                ret.Add(CreateVariableInfo(next, tokens, e));
                            }
                            i = endIndex;
                        }
                        break;
                    case TokenType.Symbol:
                        break;
                }
            }

            return ret.ToArray();
        }

        #endregion
    }
}
