﻿using System;
using System.Collections.Generic;
using System.Text;
using System.Windows.Forms;
using Sgry.Azuki;
using System.Text.RegularExpressions;

namespace NotepadNeueExtension
{
    /// <summary>
    /// パースイベントハンドラー
    /// </summary>
    /// <param name="e">イベント引数</param>
    public delegate void ParseEventHandler(ParseEventArgs e);

    /// <summary>
    /// 補完ワードイベントハンドラー
    /// </summary>
    /// <param name="e">イベント引数</param>
    public delegate void CheckWordEventHandleer(CheckWordEventArgs e);
    /// <summary>
    /// プラグインホスト、つまりNotepadNeueが継承するインターフェース
    /// 基本的にこのクラスが持っているメソッドを通じてしかプラグインは動作しません。
    /// （何か文章を参照したりする場合）
    /// </summary>
    public interface IExtensionHost
    {
        /// <summary>
        /// メインフォームが閉じるときに呼び出されるイベントです
        /// 通常のForm_Closedイベントをフックしても意味がありませんので
        /// このイベントをフックしてください。
        /// </summary>
        event EventHandler MainFormClosed;

        /// <summary>
        /// 補完情報取得開始時に呼ばれます。
        /// マルチスレッド内で呼ばれるのでクロススレッドに注意してください。
        /// </summary>
        event ParseEventHandler ParseStarted;

        /// <summary>
        /// 表示する補完単語を決定するときに呼ばれます。
        /// メインスレッド上で呼ばれるので処理時間は極力すくなるなるように心がけてください。
        /// </summary>
        event CheckWordEventHandleer CheckWordStarted;

        /// <summary>
        /// スキンの設定。
        /// 気にする必要はないです。
        /// </summary>
        NotepadNeueSkin Skin
        {
            get;
        }

        /// <summary>
        /// NotepadNeue.exeのあるディレクトリ
        /// </summary>
        string CurrentDirectory
        {
            get;
        }

        /// <summary>
        /// 現在のドキュメントが存在するかどうかを返す真偽値です。
        /// </summary>
        bool IsDocumentEmpty
        {
            get;
        }

        /// <summary>
        /// 現在のドキュメントの選択された文字列を返します。
        /// 現在のドキュメントがない場合はNullを返します。
        /// </summary>
        string SelectedTextInDocument
        {
            get;
        }

        /// <summary>
        /// 現在のドキュメントの選択範囲を返すか、設定します。
        /// 現在のドキュメントがない場合は無効なRangeを返します。
        /// </summary>
        Range SelectionInDocument
        {
            get;
            set;
        }

        /// <summary>
        /// 現在のドキュメントのテキスト全てを取得します。
        /// 現在のドキュメントがない場合はNullを返します。
        /// </summary>
        string DocumentText
        {
            get;
        }

        /// <summary>
        /// 現在のドキュメントのテキストを改行コードで分割した配列で返します
        /// 終了行を含みません。
        /// </summary>
        /// <param name="startlineindex">開始行</param>
        /// <param name="endlineindex">終了行</param>
        /// <returns>行ごとのテキスト</returns>
        string[] DocumentTextAsArray(int startlineindex, int endlineindex);

        /// <summary>
        /// 現在のドキュメントのテキストの長さを取得します。
        /// 現在のドキュメントがない場合は-1を返します。
        /// </summary>
        int DocumentTextLength
        {
            get;
        }

        /// <summary>
        /// 現在のドキュメントの行の数を取得します。
        /// 現在のドキュメントがない場合は-1を返します。
        /// </summary>
        int DocumentLineNum
        {
            get;
        }

        /// <summary>
        /// 現在のドキュメントが保存されているかを取得します。
        /// 現在のドキュメントがない場合もfalseを返すので注意してください。
        /// </summary>
        bool DocumentSaved
        {
            get;
        }

        /// <summary>
        /// 現在のドキュメントのファイルパスを返します。
        /// 現在のドキュメントがない場合はNullを返します。
        /// </summary>
        string DocumentPath
        {
            get;
        }

        /// <summary>
        /// 全てのドキュメントのファイルパスを返します。
        /// </summary>
        string[] AllDocumentPath
        {
            get;
        }

        /// <summary>
        /// 現在のドキュメントに新しくファイルを開く際に使います。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool OpenFile(string path);

        /// <summary>
        /// 新しいドキュメントに新しいファイルを開く際に使います。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool OpenFileToNewDocument(string path);

        /// <summary>
        /// 既に開いている場合にそのファイルに含まれるファイル名を変更するかどうかを調べます
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <param name="filename">新しいファイル名</param>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool QueryAlreadyOpenFile(string path, string filename);

        /// <summary>
        /// キャレットの位置まですクルールする際に使用します。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool ScrollToCaret();

        /// <summary>
        /// 現在のドキュメントからtextに一致する部分列を探します。
        /// </summary>
        /// <param name="text">テキストのパターン</param>
        /// <param name="startindex">検索を開始する文字のインデックス</param>
        /// <param name="ignorecase">大文字小文字を区別しないか</param>
        /// <returns>一致するRange。ない場合は無効なRange</returns>
        Range FindNext(string text, int startindex, bool ignorecase);

        /// <summary>
        /// 現在のドキュメントから指定された正規表現にマッチする文字列を検索します。
        /// ない場合は無効なRangeを返します。
        /// </summary>
        /// <param name="regex">正規表現</param>
        /// <param name="startindex">検索を開始する文字のインデックス</param>
        /// <returns></returns>
        Range FindNext(System.Text.RegularExpressions.Regex regex, int startindex);

        /// <summary>
        /// 指定した範囲に含まれる文字列を取得します。
        /// </summary>
        /// <param name="range">範囲</param>
        /// <returns>現在のドキュメントがない場合はNullを返します。</returns>
        string GetTextInRange(Range range);

        /// <summary>
        /// 現在のドキュメントの選択範囲を置き換えます。
        /// </summary>
        /// <param name="text">置き換える内容</param>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool Replace(string text);

        /// <summary>
        /// 現在のドキュメントのRangeに含まれる文字列を置き換えます。
        /// </summary>
        /// <param name="text"></param>
        /// <param name="range"></param>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool Replace(string text, Range range);

        /// <summary>
        /// ドキュメントのイベント(文字列変更、キャレット移動)を無効にします。
        /// 何度もドキュメントの文字列を変更したり、キャレットを移動する場合に使用してください。
        /// また、変更、移動が終了した場合にはMakeValidChange()を使用してイベントを有効にしてください。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool MakeInvalidChange();

        /// <summary>
        /// ドキュメントのイベント(文字列変更、キャレット移動)を有効にします。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool MakeValidChange();

        /// <summary>
        /// 次のドキュメントに切り替えます。
        /// </summary>
        /// <returns>切り替えた後のオブジェクトの参照。ドキュメントが一つもない場合はNullを返します。</returns>
        object NextDocument();

        /// <summary>
        /// 前のドキュメントに切り替えます。
        /// </summary>
        /// <returns>切り替えた後のオブジェクトの参照。ドキュメントが一つもない場合はNullを返します。</returns>
        object PreviousDocument();

        /// <summary>
        /// 現在のドキュメントを取得します。
        /// </summary>
        /// <returns>現在のドキュメントの参照。ドキュメントが一つもない場合はNullを返します。</returns>
        object PresentDocument();

        /// <summary>
        /// ファイルからPythonを実行します。
        /// editorという名前のドキュメントオブジェクトがデフォルトで追加されます。
        /// またitemsに含まれているペアも全て追加します。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <param name="items">ストリング、オブジェクトのペアの配列</param>
        void RunPythonFromFile(string path, KeyValuePair<string, object>[] items);

        /// <summary>
        /// ファイルからPythonを実行します。
        /// editorという名前のドキュメントオブジェクトがデフォルトで追加されます。
        /// またitemsに含まれているペアも全て追加します。
        /// returnsにある変数名をスクリプト終了後に取得し返します。
        /// </summary>
        /// <param name="path">ファイルパス</param>
        /// <param name="items">ストリング、オブジェクトのペアの配列</param>
        /// <param name="returns">欲しい変数名</param>
        KeyValuePair<string, object>[] RunPythonFromFile(string path, KeyValuePair<string, object>[] items, string[] returns);

        /// <summary>
        /// 文字列からPythonを実行します。
        /// editorという名前のドキュメントオブジェクトがデフォルトで追加されます。
        /// またitemsに含まれているペアも全て追加します。
        /// </summary>
        /// <param name="expression">スクリプトコード</param>
        /// <param name="items">ストリング、オブジェクトのペアの配列</param>
        void RunPythonFromString(string expression, KeyValuePair<string, object>[] items);

        /// <summary>
        /// 文字列からPythonを実行します。
        /// editorという名前のドキュメントオブジェクトがデフォルトで追加されます。
        /// またitemsに含まれているペアも全て追加します。
        /// returnsにある変数名をスクリプト終了後に取得し返します。
        /// </summary>
        /// <param name="expression">スクリプトコード</param>
        /// <param name="items">ストリング、オブジェクトのペアの配列</param>
        /// <param name="returns">欲しい変数名</param>
        KeyValuePair<string, object>[] RunPythonFromString(string expression, KeyValuePair<string, object>[] items, string[] returns);

        /// <summary>
        /// バルーンチップをタスクトレイに表示します。
        /// </summary>
        /// <param name="title">タイトル</param>
        /// <param name="text">内容</param>
        /// <param name="iconkind">アイコンの種類</param>
        /// <param name="displayinterval">表示する時間(ミリ秒)</param>
        void ShowBallonTip(string title, string text, ToolTipIcon iconkind, int displayinterval);

        /// <summary>
        /// 現在のドキュメントのモードを変更します。
        /// </summary>
        /// <param name="extension">拡張子の文字列。.(ドット)ありでもなしでもどちらでも構わない</param>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool ChangeDocumentMode(string extension);

        /// <summary>
        /// ドキュメントのモードを変更します。
        /// </summary>
        /// <param name="extension">拡張子の文字列。.(ドット)ありでもなしでもどちらでも構わない</param>
        /// <param name="filename">モードを変更するドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <returns>指定したファイル名がない場合はfalse</returns>
        bool ChangeDocumentMode(string extension, string filename);

        /// <summary>
        /// 新しいドキュメントを開きます。
        /// </summary>
        void CreateNewDocument();

        /// <summary>
        /// ドキュメントを閉じます。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool CloseDocument();

        /// <summary>
        /// 指定したドキュメントを閉じます。
        /// </summary>
        /// <param name="filename">閉じるドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <returns>指定したドキュメントがない場合はfalse</returns>
        bool CloseDocument(string filename);

        /// <summary>
        /// 現在のドキュメントを垂直分割します。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool VerticalSplitDocument();

        /// <summary>
        /// 指定したドキュメントを垂直分割します。
        /// </summary>
        /// <param name="filename">分割するドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <returns>指定したファイル名が開かれていない場合はfalse</returns>
        bool VerticalSplitDocument(string filename);

        /// <summary>
        /// 現在のドキュメントを水平分割します。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool HorizontalSplitDocument();

        /// <summary>
        /// 指定したドキュメントを水平分割します。
        /// </summary>
        /// <param name="filename">分割するドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <returns>指定したファイル名が開かれていない場合はfalse</returns>
        bool HorizontalSplitDocument(string filename);

        /// <summary>
        /// 現在のドキュメントを保存します。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool SaveDocument();

        /// <summary>
        /// 指定したドキュメントを保存します。
        /// </summary>
        /// <param name="filename">保存するドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <returns>指定したファイル名が開かれていない場合はfalse</returns>
        bool SaveDocument(string filename);

        /// <summary>
        /// 現在のドキュメントを再読み込みします。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool ReloadDocument();

        /// <summary>
        /// 指定したドキュメントを再読み込みします。
        /// </summary>
        /// <param name="filename">再読み込みするドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <returns>指定したファイル名が開かれていない場合はfalse</returns>
        bool ReloadDocument(string filename);

        /// <summary>
        /// NotepadNeueを終了します。
        /// </summary>
        void Exit();

        /// <summary>
        /// 現在のドキュメントをコンパイルをします。
        /// </summary>
        void Compile();

        /// <summary>
        /// 現在のドキュメントをコンパイルをします。
        /// </summary>
        /// <param name="compileEndAction">コンパイル終了時のアクション。</param>
        void Compile(Action<bool> compileEndAction);

        /// <summary>
        /// 指定したドキュメントをコンパイルします。
        /// </summary>
        /// <param name="filename">コンパイルするドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        void Compile(string filename);

        /// <summary>
        /// 指定したドキュメントをコンパイルします。
        /// </summary>
        /// <param name="filename">コンパイルするドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <param name="compileEndAction">コンパイル終了時のアクション。</param>
        void Compile(string filename, Action<bool> compileEndAction);

        /// <summary>
        /// 現在のドキュメントのコンパイルしたものを実行します。
        /// </summary>
        void Execute();

        /// <summary>
        /// 指定したドキュメントのファイルを実行します。
        /// </summary>
        /// <param name="filename">実行するドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        void Execute(string filename);

        /// <summary>
        /// 現在のドキュメントをコンパイルしユーザー設定が満たされている場合のみコンパイルしたものを実行します。
        /// </summary>
        void CompileAndExecute();

        /// <summary>
        /// 現在のドキュメントをコンパイルしユーザー設定が満たされている場合のみコンパイルしたものを実行します。
        /// </summary>
        /// <param name="compileEndAction">コンパイル終了時のアクション。</param>
        void CompileAndExecute(Action<bool> compileEndAction);

        /// <summary>
        /// 指定したドキュメントをコンパイルしユーザー設定が満たされている場合のみコンパイルしたものを実行します。
        /// </summary>
        /// <param name="filename">コンパイルし実行するドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        void CompileAndExecute(string filename);

        /// <summary>
        /// 指定したドキュメントをコンパイルしユーザー設定が満たされている場合のみコンパイルしたものを実行します。
        /// </summary>
        /// <param name="filename">コンパイルし実行するドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <param name="compileEndAction">コンパイル終了時のアクション。</param>
        void CompileAndExecute(string filename, Action<bool> compileEndAction);

        /// <summary>
        /// 現在のドキュメントのバックアップを取ります。
        /// </summary>
        /// <returns>現在のドキュメントがない場合はfalse</returns>
        bool BackUpFile();

        /// <summary>
        /// 指定したのドキュメントのバックアップを取ります。
        /// </summary>
        /// <param name="filename">バックアップを取るドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <returns>指定したファイル名が開かれていない場合はfalse</returns>
        bool BackUpFile(string filename);

        /// <summary>
        /// エラーを追加します。
        /// </summary>
        /// <param name="errortype">エラーのタイプ</param>
        /// <param name="text">エラーテキスト</param>
        /// <param name="charIndex">何文字目か</param>
        /// <param name="filename">エラー元ファイル名</param>
        void AddError(ErrorType errortype, string text, int charIndex, string filename);

        /// <summary>
        /// エラーを追加します。
        /// </summary>
        /// <param name="errortype">エラーのタイプ</param>
        /// <param name="text">エラーテキスト</param>
        /// <param name="row">エラー列</param>
        /// <param name="column">エラー行</param>
        /// <param name="filename">エラー元ファイル名</param>
        void AddError(ErrorType errortype, string text, int row, int column, string filename);

        /// <summary>
        /// エラーリストを削除します。
        /// </summary>
        void ClearError(string filename);

        /// <summary>
        /// バッファデータを読み取ります。
        /// </summary>
        /// <param name="filename">バッファファイル名</param>
        /// <param name="parsefilename">現在parseしているファイル名(ParseEventArgsから)</param>
        void ReadBuffer(string filename, string parsefilename);

        /// <summary>
        /// バッファディレクトリを取得します
        /// </summary>
        /// <param name="parseFileName"></param>
        string GetBufferDirectory(string parseFileName);

        /// <summary>
        /// バッファデータを取得します。
        /// </summary>
        /// <param name="filename">バッファファイル名</param>
        /// <param name="parsefilename">現在parseしているファイル名(ParseEventArgsから)</param>
        /// <returns>バッファファイル名を読み込んでいなければnull</returns>
        NameSpaceInfo GetBuffer(string filename, string parsefilename);

        /// <summary>
        /// 開いているドキュメントで他の情報を取得します。
        /// </summary>
        /// <param name="filename">ドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <returns>なければnull</returns>
        EditorAssistData GetBuffer(string filename);

        /// <summary>
        /// 指定したファイルの指定したインデックスが何行何列かを返します
        /// </summary>
        /// <param name="filename">ドキュメントのファイル名(タブ上の表示あるいは絶対パス)</param>
        /// <param name="index">インデックス</param>
        /// <param name="lineindex">行のインデックス</param>
        /// <param name="columnindex">列のインデックス</param>
        /// <returns>ドキュメントがなければfalse</returns>
        bool GetLineAndColumnIndexFromIndex(string filename, int index, out int lineindex, out int columnindex);

        /// <summary>
        /// 補完表示する単語を決めるときに使用します
        /// </summary>
        /// <param name="e"></param>
        bool CheckWord(CheckWordEventArgs e);

        /// <summary>
        /// 設定を設定します。
        /// </summary>
        /// <param name="extensionName">拡張名。</param>
        /// <param name="key">キー。</param>
        /// <param name="value">値。</param>
        void SetSetting(string extensionName, string key, string value);

        /// <summary>
        /// 設定を取得します。
        /// </summary>
        /// <param name="extensionName">拡張名。</param>
        /// <param name="key">キー。</param>
        /// <returns>設定。存在しない場合は<c>null</c></returns>
        string GetSetting(string extensionName, string key);

        /// <summary>
        /// インクリメンタルサーチを行います。
        /// </summary>
        /// <param name="regex">対象の正規表現。</param>
        void SetIncrementalSearch(Regex regex);

        /// <summary>
        /// インクリメンタルサーチを解除します。
        /// </summary>
        void ClearIncrementalSearch();

        /// <summary>
        /// Invokeします。
        /// </summary>
        /// <param name="action">アクション。</param>
        void InvokeInMainForm(Action action);
    }
    /// <summary>
    /// 文字列の範囲を表す構造体です。
    /// 文字列は[Star,End)で、Endの文字は含まれないことに注意してください。
    /// 無効な範囲の場合はStart = -1,End = -1とします。
    /// </summary>
    public struct Range
    {
        int start;
        int end;
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="start">開始インデックス</param>
        /// <param name="end">終了インデックス</param>
        public Range(int start, int end)
        {
            this.start = start;
            this.end = end;
        }
        /// <summary>
        /// 開始インデックス
        /// </summary>
        public int Start
        {
            get
            {
                return start;
            }
            set
            {
                start = value;
            }
        }
        /// <summary>
        /// 終了インデックス
        /// </summary>
        public int End
        {
            get
            {
                return end;
            }
            set
            {
                end = value;
            }
        }
        /// <summary>
        /// 無効なレンジ
        /// </summary>
        public static Range InvalidRange
        {
            get
            {
                return new Range(-1, -1);
            }
        }
        /// <summary>
        /// 無効なレンジかどうか
        /// </summary>
        public bool IsInvalid
        {
            get
            {
                return (start == -1 && end == -1) ? true : false;
            }
        }
        /// <summary>
        /// 等号チェック
        /// </summary>
        /// <param name="range1">レンジ１</param>
        /// <param name="range2">レンジ２</param>
        /// <returns>等しいならtrue</returns>
        public static bool operator ==(Range range1, Range range2)
        {
            return (range1.start == range2.start && range1.end == range2.end);
        }
        /// <summary>
        /// 等号チェック
        /// </summary>
        /// <param name="range1">レンジ１</param>
        /// <param name="range2">レンジ２</param>
        /// <returns>等しいならtrue</returns>
        public static bool operator !=(Range range1, Range range2)
        {
            return (range1.start != range2.start || range1.end != range2.end);
        }
        /// <summary>
        /// 等価チェック
        /// </summary>
        /// <param name="obj">オブジェクト</param>
        /// <returns>等しいならtrue</returns>
        public override bool Equals(object obj)
        {
            return base.Equals(obj);
        }
        /// <summary>
        /// ハッシュコード
        /// </summary>
        /// <returns></returns>
        public override int GetHashCode()
        {
            return base.GetHashCode();
        }
    }

    /// <summary>
    /// エラータイプ
    /// </summary>
    public enum ErrorType
    {
        /// <summary>
        /// エラー
        /// </summary>
        Error = 0,
        /// <summary>
        /// 警告
        /// </summary>
        Warn = 1
    }

    /// <summary>
    /// パースイベント引数
    /// </summary>
    public class ParseEventArgs
    {
        /// <summary>
        /// コンストラクタ
        /// </summary>
        /// <param name="assistdata"></param>
        /// <param name="parsetext"></param>
        /// <param name="filename"></param>
        /// <param name="extension"></param>
        public ParseEventArgs(EditorAssistData assistdata, string parsetext, string filename, string extension)
        {
            AssistData = assistdata;
            ParseText = parsetext;
            Filename = filename;
            Handled = false;
            Extension = extension;
        }
        /// <summary>
        /// EditorAssistData
        /// </summary>
        public EditorAssistData AssistData
        {
            get;
            private set;
        }
        /// <summary>
        /// パースしているファイル名
        /// </summary>
        public string Filename
        {
            get;
            private set;
        }
        /// <summary>
        /// falseなら正規表現を用いたパースに入ります
        /// </summary>
        public bool Handled
        {
            get;
            set;
        }
        /// <summary>
        /// パースしているファイルの拡張子設定
        /// </summary>
        public string Extension
        {
            get;
            private set;
        }
        /// <summary>
        /// パースする対象のテキスト
        /// </summary>
        public string ParseText
        {
            get;
            private set;
        }
    }

    public class CheckWordEventArgs : EventArgs
    {
        private Document document;
        private string text;
        public CheckWordEventArgs(Document document, string fileName, string extension, int caretIndex, Range range)
        {
            SelectIndex = -1;
            AssistInformations = new List<AssistInformation>();
            CaretIndex = caretIndex;
            Extension = extension;
            FileName = fileName;
            Range = range;
            this.document = document;
        }

        public Document Document
        {
            get
            {
                return document;
            }
        }

        public string Text
        {
            get
            {
                if (text == null)
                {
                    text = document.Text;
                }
                return text;
            }
        }

        public string FileName
        {
            get;
            private set;
        }

        public string Extension
        {
            get;
            private set;
        }

        public Range Range
        {
            get;
            private set;
        }

        public int SelectIndex
        {
            get;
            set;
        }

        public int CaretIndex
        {
            get;
            private set;
        }

        public bool Handled
        {
            get;
            set;
        }

        public List<AssistInformation> AssistInformations
        {
            get;
            private set;
        }
    }
}
