﻿using System;
using System.IO;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Text;
using System.Windows.Forms;
using WeifenLuo.WinFormsUI.Docking;
using System.Threading;
using System.Diagnostics;
using System.Collections;

namespace NotepadNeue
{
    public partial class TextEditWindow : NotepadNeueExtension.ExDockContent
    {
        delegate void Action();
        delegate T Func<T, V1, V2, V3>(V1 arg1, V2 arg2, V3 arg3);

        bool saved = true;
        string filename;
        MainForm fm1;
        ExAzukiControl exac;
        Thread compilingThread;

        public TextEditWindow()
        {
            InitializeComponent();
            //for check compile thread
        }
        public void Initialize(MainForm fm1, MyToolTip mytoolstrip, MyListBox mydatagridview, NotepadUtility nu, ExtensionInfomation exinfo)
        {
            this.fm1 = fm1;
            exac = new ExAzukiControl(fm1, mytoolstrip, mydatagridview, nu);
            this.Controls.Add(exac);
            exac.Dock = DockStyle.Fill;
            exac.ExtensionInfo = exinfo;
            exac.ContentChanged += new EventHandler(exac_ContentChanged);
            exac.DragEnter += new DragEventHandler(exac_DragEnter);
            exac.DragDrop += new DragEventHandler(exac_DragDrop);

            DockHandler.DocumentTabFileDroped += new Action<string[]>(DockHandler_FileDroped);
        }

        void DockHandler_FileDroped(string[] obj)
        {
            OpenFiles(obj);
        }

        void exac_DragEnter(object sender, DragEventArgs e)
        {
            //コントロール内にドラッグされたとき実行される
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                //ドラッグされたデータ形式を調べ、ファイルのときはコピーとする
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                //ファイル以外は受け付けない
                e.Effect = DragDropEffects.None;
            }
        }
        void exac_DragDrop(object sender, DragEventArgs e)
        {
            string[] fileNames = (string[])e.Data.GetData(DataFormats.FileDrop, false);
            OpenFiles(fileNames);
        }

        private void OpenFiles(string[] fileNames)
        {
            if (File.Exists(fileNames[0]))
            {
                fm1.OpenFile(fileNames[0]);
            }
            for (int i = 1; i < fileNames.Length; i++)
            {
                fm1.OpenFileToNewDocument(fileNames[i]);
            }
        }

        void exac_ContentChanged(object sender, EventArgs e)
        {
            if (!TabText.StartsWith("*"))
            {
                TabText = "*" + TabText;
            }
            this.saved = false;
        }
        public ExAzukiControl ExAzukiControl
        {
            get
            {
                return exac;
            }
        }
        public void Save()
        {
            if (TabText.StartsWith("*"))
            {
                TabText = TabText.Remove(0, 1);
            }
            saved = true;
            exac.Document.IsDirty = false;
        }
        public bool Saved
        {
            get
            {
                return saved;
            }
        }
        public string Filename
        {
            get
            {
                return filename;
            }
            set
            {
                filename = value;
                if (TabText.Contains("*"))
                {
                    TabText = "*" + Path.GetFileName(filename);
                }
                else
                {
                    TabText = Path.GetFileName(filename);
                }
                this.ToolTipText = filename;
            }
        }
        public string BatFilename
        {
            get
            {
                return Path.GetFileNameWithoutExtension(filename) + ".bat";
            }
        }
        private void TextEditWindow_FormClosing(object sender, FormClosingEventArgs e)
        {
            if (!CheckStopCompile())
            {
                e.Cancel = true;
                return;
            }

            if (!this.saved)
            {
                DialogResult dr = DialogProvider.AskSave();
                if (dr == DialogResult.Yes)
                {
                    if (fm1.Save(this) == DialogResult.OK)
                    {
                        Save();
                        exac.KillThread();
                        fm1.SafeClose(this);
                    }
                    else
                    {
                        e.Cancel = true;
                    }
                }
                else if (dr == DialogResult.No)
                {
                    Save();
                    exac.KillThread();
                    fm1.SafeClose(this);
                }
                else if (dr == DialogResult.Cancel)
                {
                    e.Cancel = true;
                }
            }
        }
        public void CompileAndExecute(Action<bool> compileEndAction)
        {
            Compile((result) =>
            {
                if (compileEndAction != null)
                {
                    compileEndAction(result);
                }
                if (result)
                {
                    Execute();
                }
            });
        }

        private bool CheckStopCompile()
        {
            if (compilingThread != null && compilingThread.IsAlive)
            {
                DialogResult result = MessageBox.Show("既にコンパイル中です。コンパイルを中止しますか？", "確認", MessageBoxButtons.OKCancel);
                if (result == DialogResult.Cancel)
                {
                    return false;
                }
                compilingThread.Abort();
                compilingThread = null;
            }
            return true;
        }

        public void Compile(Action<bool> compileEndAction)
        {
            ClearErrorMark();
            ClearWarningMark();
            ExAzukiControl.Invalidate();

            // #sysdir# is registered to registry, #usrdir# is userdirectory, #file# is currentfilename without extension
            if (!CheckStopCompile())
            {
                return;
            }
            try
            {
                string batbefore = GetChangedString(exac.ExtensionInfo.BatBeforeCompile);
                if (batbefore != "")
                {
                    batbefore = "call " + '"' + batbefore + '"';
                }
                string cmp = '"' + GetChangedString(exac.ExtensionInfo.Compiler) + '"' + " ";
                string option = GetChangedString(exac.ExtensionInfo.Option);
                option = option + " ";
                string program = GetChangedString(exac.ExtensionInfo.Program);
                if (program != "")
                {
                    program = '"' + program + '"' + " ";
                }
                string programoption = GetChangedString(exac.ExtensionInfo.ProgramOption);
                programoption = programoption + " ";
                string execfilename = GetChangedString(exac.ExtensionInfo.Filename);
                string extra = GetChangedString(exac.ExtensionInfo.Extra);
                try
                {
                    StreamWriter sw = new StreamWriter(fm1.CurrentDirectory + BatFilename, false, Encoding.Default);
                    string temp = "@echo off" + System.Environment.NewLine + batbefore + System.Environment.NewLine + cmp + option +
                        '"' + this.Filename + '"' + System.Environment.NewLine + extra + System.Environment.NewLine + "exit";
                    sw.Write(temp);
                    sw.Close();
                    System.Diagnostics.ProcessStartInfo psi = new System.Diagnostics.ProcessStartInfo();
                    psi.FileName = System.Environment.GetEnvironmentVariable("comspec");
                    psi.CreateNoWindow = false;
                    psi.Arguments = String.Format("/k \"{0}\"", Path.Combine(Path.GetDirectoryName(fm1.CurrentDirectory), BatFilename));
                    psi.WorkingDirectory = Path.GetDirectoryName(this.Filename);
                    //出力を読み取れるようにする
                    psi.RedirectStandardInput = true;
                    psi.RedirectStandardOutput = true;
                    psi.RedirectStandardError = true;
                    psi.UseShellExecute = false;
                    //ウィンドウを表示しないようにする
                    psi.CreateNoWindow = true;
                    compilingThread = new Thread(new ParameterizedThreadStart(ExecuteProcess));
                    compilingThread.Start(new ArrayList() { psi, compileEndAction });
                }
                catch (Exception ex)
                {
                    MessageBox.Show(ex.Message);
                }
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        public void Execute()
        {
            string program = GetChangedString(exac.ExtensionInfo.Program);
            if (program != "")
            {
                program = '"' + program + '"' + " ";
            }
            string programoption = GetChangedString(exac.ExtensionInfo.ProgramOption);
            programoption = programoption + " ";
            string execfilename = GetChangedString(exac.ExtensionInfo.Filename);
            if (String.IsNullOrEmpty(program) && Path.GetExtension(execfilename).ToLower() != ".exe")
            {
                return;
            }
            try
            {
                StreamWriter sw = new StreamWriter(fm1.CurrentDirectory + BatFilename, false, Encoding.Default);
                string temp = "@echo off" + System.Environment.NewLine + program + programoption + '"' + execfilename + '"' + System.Environment.NewLine + "pause" + System.Environment.NewLine + "exit";
                sw.Write(temp);
                sw.Close();
                System.Diagnostics.ProcessStartInfo psi = new System.Diagnostics.ProcessStartInfo();
                psi.FileName = System.Environment.GetEnvironmentVariable("comspec");
                psi.CreateNoWindow = false;
                psi.Arguments = String.Format("/k \"{0}\"", Path.Combine(Path.GetDirectoryName(fm1.CurrentDirectory), BatFilename));
                psi.WorkingDirectory = Path.GetDirectoryName(this.Filename);
                Process.Start(psi);
            }
            catch (Exception ex)
            {
                MessageBox.Show(ex.Message);
            }
        }

        private string GetChangedString(string s)
        {
            s = s.Replace("#sysdir#", fm1.CurrentDirectory);
            s = s.Replace("#filepath#", Path.Combine(Path.GetDirectoryName(Filename), Path.GetFileNameWithoutExtension(Filename)));
            s = s.Replace("#file#", Path.GetFileNameWithoutExtension(Filename));
            s = s.Replace("#dir#", Path.GetDirectoryName(Filename));
            s = s.Replace("#filename#", Path.GetFileName(Filename));
            return s;
        }

        private void ExecuteProcess(Object obj)
        {
            ArrayList list = obj as ArrayList;
            bool ok = false;
            using (System.Diagnostics.Process p = System.Diagnostics.Process.Start(list[0] as ProcessStartInfo))
            {
                string stdOutStr = "", stdErrStr = "";

                System.Threading.Timer timer = new System.Threading.Timer((sync) =>
                {
                    try
                    {
                        if (!p.HasExited)
                        {
                            p.StandardInput.Write(p.StandardInput.NewLine);
                            lock (this)
                            {
                                stdOutStr += p.StandardOutput.ReadToEnd();
                                stdErrStr += p.StandardError.ReadToEnd();
                            }
                        }
                    }
                    catch
                    {
                    }
                }, new Object(), 100, 10);

                if (!p.WaitForExit(this.exac.ExtensionInfo.TimeOutInterval))
                {
                    timer.Dispose();
                    ok = false;
                    ProcessTimeOuted();
                }
                else
                {
                    timer.Dispose();
                    lock (this)
                    {
                        stdOutStr += p.StandardOutput.ReadToEnd();
                        stdErrStr += p.StandardError.ReadToEnd();
                    }
                    ok = ProcessFinished(p.ExitCode, stdOutStr, stdErrStr) == 0;
                }

                if (list.Count >= 2)
                {
                    new Thread(() =>
                    {
                        (list[1] as Action<bool>).Invoke(ok);
                    }).Start();
                }
            }
        }

        private int ProcessFinished(int exitCode, string stdOutStr, string stdErrStr)
        {
            if (InvokeRequired)
            {
                return (int)this.Invoke(new Func<int, int, string, string>(ProcessFinished), exitCode, stdOutStr, stdErrStr);
            }
            string results = String.Format("{0}\r\n{1}", stdOutStr, stdErrStr);
            return fm1.CheckError(this, results, exitCode);
        }

        private void ProcessTimeOuted()
        {
            if (InvokeRequired)
            {
                this.Invoke(new Action(ProcessTimeOuted));
                return;
            }
            MessageBox.Show("コンパイルから" + this.exac.ExtensionInfo.TimeOutInterval + "ミリ秒経過してもプロセスが終了しなかったのでKillしました");
        }

        public void ClearErrorMark()
        {
            ExAzukiControl.Document.Unmark(0, ExAzukiControl.Document.Text.Length, 0);
        }

        public void ClearWarningMark()
        {
            ExAzukiControl.Document.Unmark(0, ExAzukiControl.Document.Text.Length, 1);
        }

        public void AddMark(string fileName, int lineIndex, int columnIndex, int markId)
        {
            if (this.filename.ToLower() != fileName.ToLower())
            {
                return;
            }

            if (lineIndex > 0)
            {
                lineIndex--;
            }
            if (columnIndex > 0)
            {
                columnIndex--;
            }

            try
            {
                int linePosIndex = ExAzukiControl.GetCharIndexFromLineColumnIndex(lineIndex, columnIndex);
                int lineEndPosIndex = ExAzukiControl.Document.GetLineEndIndexFromCharIndex(linePosIndex);
                ExAzukiControl.Document.Mark(linePosIndex, lineEndPosIndex, markId);
            }
            catch
            {

            }
        }

        private void 保存ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            fm1.OverSave();
        }

        private void 閉じるToolStripMenuItem_Click(object sender, EventArgs e)
        {
            this.Close();
        }

        private void このウィンドウ以外すべて閉じるToolStripMenuItem_Click(object sender, EventArgs e)
        {
            fm1.CloseExceptThisWindow(this);
        }
        private void 完全パスのコピーToolStripMenuItem_Click(object sender, EventArgs e)
        {
            Clipboard.SetText(filename);
        }
        private void このファイルを含むフォルダを開くToolStripMenuItem_Click(object sender, EventArgs e)
        {
            if (File.Exists(this.filename))
            {
                System.Diagnostics.Process.Start("EXPLORER.EXE", "/select," + filename);
            }
        }

        private void 水平タブグループの新規作成ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            HorizontalSplit();
        }

        public void HorizontalSplit()
        {
            if (this.Pane.Contents.Count > 1)
            {
                this.Show(this.Pane, DockAlignment.Bottom, 0.5);
            }
        }
        private void 垂直タブグループの新規作成ToolStripMenuItem_Click(object sender, EventArgs e)
        {
            VerticalSplit();
        }
        public void VerticalSplit()
        {
            if (this.Pane.Contents.Count > 1)
            {
                this.Show(this.Pane, DockAlignment.Right, 0.5);
            }
        }
    }
}
