/*
 * IRCNamespaceTest.java
 * IRCCoreTest
 *
 * Created by tarchan on Dec 4, 2006.
 * Copyright (c) 2006 tarchan. All rights reserved.
 */
package com.mac.tarchan.irc;

import java.util.regex.Matcher;
import java.util.regex.Pattern;

import junit.framework.TestCase;

import com.mac.tarchan.irc.IRCChannel;
import com.mac.tarchan.irc.IRCMessage;
import com.mac.tarchan.irc.IRCNamespace;
import com.mac.tarchan.irc.IRCNetwork;
import com.mac.tarchan.irc.IRCUser;


/**
 * IRCNamespaceTest
 *
 * @author tarchan
 */
public class IRCNamespaceTest extends TestCase
{
	public void testIsChannel001()
	{
		assertEquals(true, IRCNamespace.isChannel("&dameTunes"));
	}

	public void testIsChannel002()
	{
		assertEquals(true, IRCNamespace.isChannel("#dameTunes"));
	}

	public void testIsChannel003()
	{
		assertEquals(true, IRCNamespace.isChannel("+dameTunes"));
	}

	public void testIsChannel004()
	{
		assertEquals(true, IRCNamespace.isChannel("!dameTunes"));
	}

	public void testIsChannel005()
	{
		assertEquals(true, IRCNamespace.isChannel("%irodorie"));
	}

	public void testIsChannel006()
	{
		assertEquals(true, IRCNamespace.isChannel("#dame7Tunes"));
	}

	public void testIsNotChannel001()
	{
		assertEquals(false, IRCNamespace.isChannel(" #dameTunes"));
	}

	public void testIsNotChannel002()
	{
		assertEquals(false, IRCNamespace.isChannel("#dame,Tunes"));
	}

	public void testIsNotChannel003()
	{
		assertEquals(false, IRCNamespace.isChannel("#dame Tunes"));
	}

	public void testIsNotChannel004()
	{
		assertEquals(false, IRCNamespace.isChannel("#dameTunes "));
	}

	public void testIsNotChannel005()
	{
		assertEquals(false, IRCNamespace.isChannel("#dame\007Tunes"));
	}

	public void testIsNotChannel006()
	{
		assertEquals(false, IRCNamespace.isChannel(null));
	}

	public void testNormalizeChannel001()
	{
		assertEquals("#irodorie:*.jp", IRCNamespace.normalizeChannel("%irodorie", "jp"));
	}

	public void testNormalizeChannel002()
	{
		assertEquals("#dameTunes", IRCNamespace.normalizeChannel("#dameTunes", "jp"));
	}

	public void testNormalizeNick001()
	{
		assertEquals("tarchan", IRCNamespace.normalizeNick("@tarchan"));
	}

	public void testNormalizeNick002()
	{
		assertEquals("tarchan", IRCNamespace.normalizeNick("tarchan"));
	}

	public void testNick001()
	{
		assertEquals(true, IRCNamespace.isNick("tarchan"));
	}

	public void testNick002()
	{
		assertEquals(true, IRCNamespace.isNick("_846ekaku"));
	}

	public void testNick101()
	{
		assertEquals(false, IRCNamespace.isNick("#dameTunes"));
	}

	public void testURLhttp()
	{
		Matcher m = IRCNamespace.uriMatcher("11:01 yuta_WK> -! http://www.aquaplus.co.jp/th/gallery.html これって京本正樹の黒歴史じゃないのか");
		m.find();
		assertEquals("http://www.aquaplus.co.jp/th/gallery.html", m.group());
	}

	public void testURLttp()
	{
		Matcher m = IRCNamespace.uriMatcher("11:01 yuta_WK> -! ttp://www.aquaplus.co.jp/th/gallery.html これって京本正樹の黒歴史じゃないのか");
		m.find();
		assertEquals("ttp://www.aquaplus.co.jp/th/gallery.html", m.group());
	}

	public void testURLftp()
	{
		Matcher m = IRCNamespace.uriMatcher("11:01 yuta_WK> -! ftp://www.aquaplus.co.jp/th/gallery.html これって京本正樹の黒歴史じゃないのか");
		m.find();
		assertEquals("ftp://www.aquaplus.co.jp/th/gallery.html", m.group());
	}

	public void testURLReplaceCat()
	{
		Pattern p = Pattern.compile("cat");
		Matcher m = p.matcher("one cat two cats in the yard");
		StringBuffer sb = new StringBuffer();
		while (m.find())
		{
			System.out.println(m.group());
			m.appendReplacement(sb, "dog");
		}
		m.appendTail(sb);
		assertEquals("one dog two dogs in the yard", sb.toString());
	}

	public void testURLReplaceHref()
	{
		Matcher m = IRCNamespace.uriMatcher("11:01 yuta_WK> -! http://www.aquaplus.co.jp/th/gallery.html ttp://www.aquaplus.co.jp/th/gallery.html ftp://www.aquaplus.co.jp/th/gallery.htmlこれって京本正樹の黒歴史じゃないのか");
		StringBuffer sb = new StringBuffer();
		while (m.find())
		{
			String src = m.group();
			String url = src;
			if (url.startsWith("ttp")) url = "h" + url;
			String out = "<a href=\"" + url + "\">" + src + "</a>";
			System.out.println(m.group() + " -> " + out);
			m.appendReplacement(sb, out);
		}
		m.appendTail(sb);
		assertEquals("11:01 yuta_WK> -! <a href=\"http://www.aquaplus.co.jp/th/gallery.html\">http://www.aquaplus.co.jp/th/gallery.html</a> <a href=\"http://www.aquaplus.co.jp/th/gallery.html\">ttp://www.aquaplus.co.jp/th/gallery.html</a> <a href=\"ftp://www.aquaplus.co.jp/th/gallery.html\">ftp://www.aquaplus.co.jp/th/gallery.html</a>これって京本正樹の黒歴史じゃないのか", sb.toString());
	}

	public void testURLgif()
	{
		Matcher m = IRCNamespace.imageMatcher("11:01 yuta_WK> -! http://www.aquaplus.co.jp/th/gallery.gif これって京本正樹の黒歴史じゃないのか");
		m.find();
		assertEquals("http://www.aquaplus.co.jp/th/gallery.gif", m.group());
	}

	public void testURLjpeg()
	{
		Matcher m = IRCNamespace.imageMatcher("11:01 yuta_WK> -! http://www.aquaplus.co.jp/th/gallery.jpeg これって京本正樹の黒歴史じゃないのか");
		m.find();
		assertEquals("http://www.aquaplus.co.jp/th/gallery.jpeg", m.group());
	}

	public void testURLjpg()
	{
		Matcher m = IRCNamespace.imageMatcher("11:01 yuta_WK> -! http://www.aquaplus.co.jp/th/gallery.jpg これって京本正樹の黒歴史じゃないのか");
		m.find();
		assertEquals("http://www.aquaplus.co.jp/th/gallery.jpg", m.group());
	}

	public void testURLpng()
	{
		Matcher m = IRCNamespace.imageMatcher("11:01 yuta_WK> -! http://www.aquaplus.co.jp/th/gallery.png これって京本正樹の黒歴史じゃないのか");
		m.find();
		assertEquals("http://www.aquaplus.co.jp/th/gallery.png", m.group());
	}

	public void testURLbmp()
	{
		Matcher m = IRCNamespace.imageMatcher("11:01 yuta_WK> -! http://www.aquaplus.co.jp/th/gallery.bmp これって京本正樹の黒歴史じゃないのか");
		m.find();
		assertEquals("http://www.aquaplus.co.jp/th/gallery.bmp", m.group());
	}

	public void testURLBMP()
	{
		Matcher m = IRCNamespace.imageMatcher("11:01 yuta_WK> -! http://www.aquaplus.co.jp/th/gallery.BMP これって京本正樹の黒歴史じゃないのか");
		if (m.find()) assertEquals("http://www.aquaplus.co.jp/th/gallery.BMP", m.group());
		else fail("No match found");
	}

	public void testCreateUser001()
	{
		IRCNetwork net = IRCNetwork.getNetwork("test");
		IRCMessage msg = new IRCMessage(this, ":irc.huie.hokudai.ac.jp 353 tarc2 @ #dameTunes :@matuken @Kaho @m_mai hideo masan_Zzz ");
		String[] names = msg.getTrailing().split(" ");
		for (String name : names)
		{
			net.getUser(name);
		}
		IRCUser user = net.findUser("matuken");
		assertEquals("matuken", user.getNick());
	}

	public void testCreateChannel001()
	{
		IRCNetwork net = IRCNetwork.getNetwork("test");
		net.getChannel("#javabreak");
		net.getChannel("#dameTunes");
		net.getChannel("%irodorie");
		IRCChannel ch = net.findChannel("#dametunes");
		assertEquals("#dameTunes", ch.getName());
	}
}
