/*
 * IRCConnection.java
 * IRCCore
 *
 * Created by tarchan on Aug 07, 2006.
 * Copyright (c) 2006 tarchan. All rights reserved.
 */
package com.mac.tarchan.irc;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.Reader;
import java.net.InetSocketAddress;
import java.net.URL;
import java.net.URLConnection;
import java.nio.channels.Channels;
import java.nio.channels.SocketChannel;

/**
 * IRCConnection
 * 
 * @author tarchan
 * @author nori090
 */
public class IRCConnection
    extends URLConnection {
    protected SocketChannel socket;

    public IRCConnection( URL url ) {
        super( url );
        setDoInput( true );
        setDoOutput( true );
    }

    public void set( String key, String value ) {
        super.setRequestProperty( key, value );
    }

    public String get( String key ) {
        return super.getRequestProperty( key );
    }

    protected boolean isConnected() {
        return socket != null && socket.isConnected();
    }

    public void connect()
        throws IOException {
        if ( isConnected() ) {
            return;
        }
        InetSocketAddress remote = new InetSocketAddress( url.getHost(), url.getPort() );
        socket = SocketChannel.open( remote );
    }

    public void disconnect() {
        try {
            socket.close();
        }
        catch ( IOException x ) {
            x.printStackTrace();
        }
        finally {
            socket = null;
        }
    }

    /**
     * 入力ストリームを返します。
     */
    public InputStream getInputStream()
        throws IOException {
        return isConnected() ? Channels.newInputStream( socket ) : null;
    }

    /**
     * 出力ストリームを返します。
     */
    public OutputStream getOutputStream()
        throws IOException {
        return isConnected() ? Channels.newOutputStream( socket ) : null;
    }

    /**
     * 入力ストリームを返します。
     * 
     * @return 入力ストリーム。接続していない場合は null
     * @see #connect()
     */
    public Reader reader() {
        return Channels.newReader( socket, "ISO-8859-1" );
    }

    /**
     * デフォルトの文字エンコーディングの出力ストリームを返します。
     * 
     * @return 出力ストリーム。接続していない場合は null
     * @see #connect()
     */
    public IRCWriter writer() {
        return new IRCWriter( Channels.newWriter( socket, "ISO-8859-1" ) );
    }

    /**
     * 指定した文字エンコーディングの出力ストリームを返します。
     * 
     * @param encoding 文字エンコーディング
     * @return 出力ストリーム。接続していない場合は null
     * @see #connect()
     */
    public IRCWriter writer( String encoding ) {
        return new IRCWriter( Channels.newWriter( socket, encoding ) );
    }
}
