/*
 * Handler.java
 * IRCCore
 *
 * Created by tarchan on Aug 07, 2006.
 * Copyright (c) 2006 tarchan. All rights reserved.
 */
package com.mac.tarchan.irc;

import java.io.IOException;
import java.net.URL;
import java.net.URLConnection;
import java.net.URLStreamHandler;

import org.apache.log4j.Logger;

/**
 * Handler {@link IRCConnection}を生成する役割を担います。 {@link URL}から読み込まれるので、パッケージ名、クラス名は固定です。
 * 
 * @author tarchan
 * @author nori090
 */
public class Handler
    extends URLStreamHandler {
    /** debug logger */
    private static Logger log = Logger.getLogger( Handler.class );

    /** irc default connection port */
    public static int DEFAULT_PORT = 6667;

    /**
     * IRC接続を返します。
     * 
     * @return IRC接続
     */
    @Override
    protected URLConnection openConnection( URL url )
        throws IOException {
        return new IRCConnection( url );
    }

    /**
     * デフォルトポートを返します。(6667)
     * 
     * @return デフォルトポート
     */
    @Override
    public int getDefaultPort() {
        return DEFAULT_PORT;
    }

    /**
     * 文字列表現を IRC として構文解析します。
     * 
     * <pre>
     * path = &lt;path&gt;['?'&lt;query&gt;]['#'&lt;ref&gt;]
     * </pre>
     * 
     * @param url 仕様構文解析の結果を受け取る URL
     * @param spec 構文解析する必要のある URL を表す String
     * @param start 構文解析の開始位置を表す文字インデックス。これはプロトコル名の確定を表す「:」(存在する場合) の直後にくる
     * @param limit 構文解析の終了位置を表す文字の位置。これは文字列の終わりか、「#」文字 (存在する場合) の位置である。シャープ記号よりもあとの情報はすべてアンカーを表す
     */
    @Override
    protected void parseURL( URL url, String spec, int start, int limit ) {
        if ( log.isDebugEnabled() ) {
            log.debug( ">>> parse" );
            log.debug( "url=" + url );
            log.debug( "spec=" + spec );
            log.debug( "start=" + start );
            log.debug( "limit=" + limit );
            log.debug( "ch=" + spec.substring( start ) );
            log.debug( "<" );
        }

        // http として構文解析する
        super.parseURL( url, spec, start, limit );

        String protocol = url.getProtocol();
        String host = url.getHost();
        int port = url.getPort();
        String auth = url.getAuthority();
        String userinfo = url.getUserInfo();
        String path = url.getPath();
        String query = url.getQuery();
        String ref = url.getRef();

        if ( log.isDebugEnabled() ) {
            log.debug( ">>> spec=" + spec );
            log.debug( new StringBuffer().append( "protocol=" ).append( protocol ).append( ",host=" ).append( host ).append(
                                                                                                                             ",port=" ).append(
                                                                                                                                                port ).append(
                                                                                                                                                               ",auth=" ).append(
                                                                                                                                                                                  auth ).append(
                                                                                                                                                                                                 ",userinfo=" ).append(
                                                                                                                                                                                                                        userinfo ).append(
                                                                                                                                                                                                                                           ",path=" ).append(
                                                                                                                                                                                                                                                              path ).append(
                                                                                                                                                                                                                                                                             ",query=" ).append(
                                                                                                                                                                                                                                                                                                 query ).append(
                                                                                                                                                                                                                                                                                                                 ",ref=" ).append(
                                                                                                                                                                                                                                                                                                                                   ref ).toString() );
            log.debug( "<" );
        }

        if ( port == -1 )
            port = getDefaultPort();
        if ( userinfo == null )
            userinfo = "";
        if ( ref == null )
            ref = "";

        super.setURL( url, protocol, host, port, auth, userinfo, path, query, ref );
    }
}
