# -*- coding: utf-8 -*-
#
#  install.py - an installer module for ninix
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2012 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import distutils.dir_util
import os
import shutil
import stat
import sys
import logging
import tempfile
import urllib.request
import zipfile

from gi.repository import Gtk
from gi.repository import GObject

import ninix.home
import ninix.version


class URLopener(urllib.request.FancyURLopener):
    version = 'ninix-aya/{0}'.format(ninix.version.VERSION)

urllib.request._urlopener = URLopener()


class InstallError(Exception):
    pass


def fatal(error):
    logging.error(error) # XXX
    raise InstallError(error)


class Installer:

    def __init__(self):
        self.dialog = Gtk.MessageDialog(type=Gtk.MessageType.QUESTION,
                                        buttons=Gtk.ButtonsType.YES_NO)
        self.select_dialog = Gtk.Dialog(
            buttons=(Gtk.STOCK_CANCEL, Gtk.ResponseType.REJECT,
                     Gtk.STOCK_OK, Gtk.ResponseType.ACCEPT))
        ls = Gtk.ListStore(GObject.TYPE_INT, GObject.TYPE_STRING)
        tv = Gtk.TreeView(model=ls)
        tv.set_rules_hint(True)
        renderer = Gtk.CellRendererText()
        col0 = Gtk.TreeViewColumn('No.', renderer, text=0)
        col1 = Gtk.TreeViewColumn('Path', renderer, text=1)
        tv.append_column(col0)
        tv.append_column(col1)
        sw = Gtk.ScrolledWindow()
        sw.set_vexpand(True)
        sw.set_policy(Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        sw.add(tv)
        sw.show_all() # XXX
        self.treeview = tv
        label = Gtk.Label(label=
            'Multiple candidates found.\n'
            'Select the path name of the supplement target.') ## FIXME
        ##label.set_use_markup(True)
        content_area = self.select_dialog.get_content_area()
        content_area.add(label)
        label.show()
        content_area.add(sw)
        self.select_dialog.set_title('Select the target') ## FIXME
        self.select_dialog.set_default_size(-1, 200)

    def check_archive(self, filename):
        # check archive format
        basename, ext = os.path.splitext(filename)
        ext = ext.lower()
        if ext in ['.nar', '.zip']:
            pass
        else:
            fatal('unknown archive format')

    def extract_files(self, filename):
        # extract files from the archive
        tmpdir = tempfile.mkdtemp('ninix-aya')
        shutil.rmtree(tmpdir) # XXX
        try:
            os.makedirs(tmpdir)
        except:
            fatal('cannot make temporary directory')
        url = None
        if filename.startswith('http:') or filename.startswith('ftp:'):
            url = filename
            filename = self.download(filename, tmpdir)
            if filename is None:
                shutil.rmtree(tmpdir)
                fatal('cannot download the archive file')
        try:
            with zipfile.ZipFile(filename) as zf:
                for name in zf.namelist():
                    path = os.path.join(os.fsencode(tmpdir), os.fsencode(name))
                    dname, fname = os.path.split(path)
                    if not os.path.exists(dname):
                        os.makedirs(dname)
                    if not fname: # directory
                        continue
                    buf = zf.read(name)
                    with open(path, 'wb') as of:
                        of.write(buf)
        except:
            shutil.rmtree(tmpdir)
            fatal('cannot extract files from the archive')
        for (dirpath, dirnames, filenames) in os.walk(tmpdir):
            for name in dirnames:
                path = os.path.join(dirpath, name)
                st_mode = os.stat(path).st_mode
                os.chmod(path, st_mode | stat.S_IWUSR | stat.S_IRUSR | stat.S_IXUSR)
            for name in filenames:
                path = os.path.join(dirpath, name)
                st_mode = os.stat(path).st_mode
                os.chmod(path, st_mode | stat.S_IWUSR | stat.S_IRUSR)
        self.rename_files(tmpdir)
        return os.fsencode(tmpdir) # XXX

    def get_file_type(self, tmpdir):
        # check the file type
        inst = ninix.home.read_install_txt(tmpdir)
        if not inst:
            if os.path.exists(os.path.join(tmpdir, b'kinoko.ini')):
                filetype = 'kinoko'
            elif os.path.exists(os.path.join(tmpdir, b'plugin.txt')):
                filetype = 'plugin'
            else:
                fatal('cannot read install.txt from the archive')
        else:
            filetype = inst.get('type')
        if filetype == 'ghost':
            if not os.path.exists(os.path.join(tmpdir, b'ghost', b'master')):
                filetype = 'ghost.inverse'
        elif filetype == 'ghost with balloon':
            filetype = 'ghost.inverse'
        elif filetype in ['shell', 'shell with balloon', 'supplement']:
            if 'accept' in inst:
                filetype = 'supplement'
            else:
                filetype = 'shell.inverse'
        elif filetype == 'balloon':
            pass
        elif filetype == 'plugin':
            pass
        elif filetype == 'skin':
            filetype = 'nekoninni'
        elif filetype == 'katochan':
            pass
        elif filetype == 'kinoko':
            pass
        else:
            fatal('unsupported file type({0})'.format(filetype))
        if filetype in ['shell.inverse', 'ghost.inverse']:
            fatal('unsupported file type({0})'.format(filetype))
        return filetype

    def install(self, filename, homedir):
        self.check_archive(filename)
        homedir = os.fsencode(homedir) # XXX
        tmpdir = self.extract_files(filename)
        filetype = self.get_file_type(tmpdir)
        try:
            func = getattr(self, 'install_{0}'.format(filetype))
            target_dir = func(filename, tmpdir, homedir)
        finally:
            shutil.rmtree(tmpdir)
        return filetype, target_dir


    def download(self, url, basedir):
        #logging.debug('downloading {0}'.format(url))
        try:
            ifile = urllib.request.urlopen(url)
        except IOError:
            return None
        headers = ifile.info()
        #if 'content-length' in headers:
        #    logging.debug(
        #        '(size = {0} bytes)'.format(headers.get('content-length')))
        arcdir = ninix.home.get_archive_dir()
        if not os.path.exists(arcdir):
            os.makedirs(arcdir)
        basedir = arcdir
        filename = os.path.join(basedir, os.path.basename(url))
        try:
            with open(filename, 'wb') as ofile:
                while 1:
                    data = ifile.read(4096)
                    if not data:
                        break
                    ofile.write(data)
        except IOError:
            return None
        ifile.close()
        # check the format of the downloaded file
        self.check_archive(filename) ## FIXME
        try:
            zf = zipfile.ZipFile(filename)
        except:
            return None
        test_zip = zf.testzip()
        zf.close()
        return None if test_zip is not None else filename

    def rename_files(self, basedir):
        if os.name == 'nt': # XXX
            return
        for filename in os.listdir(basedir):
            filename2 = filename.lower()
            path = os.path.join(basedir, filename2)
            if filename != filename2:
                os.rename(os.path.join(basedir, filename), path)
            if os.path.isdir(path):
                self.rename_files(path)

    def list_all_directories(self, top, basedir):
        dirlist = []
        for path in os.listdir(os.path.join(top, basedir)):
            if os.path.isdir(os.path.join(top, basedir, path)):
                dirlist.extend(self.list_all_directories(
                        top, os.path.join(basedir, path)))
                dirlist.append(os.path.join(basedir, path))
        return dirlist

    def remove_files_and_dirs(self, target_dir, mask):
        path = os.path.abspath(target_dir)
        if not os.path.isdir(path):
            return
        os.path.walk(path, self.remove_files, mask)
        dirlist = self.list_all_directories(path, '')
        dirlist.sort()
        dirlist.reverse()
        for name in dirlist:
            current_path = os.path.join(path, name)
            if os.path.isdir(current_path):
                head, tail = os.path.split(current_path)
                if tail not in mask and not os.listdir(current_path):
                    shutil.rmtree(current_path)

    def remove_files(self, mask, top_dir, filelist):
        for name in filelist:
            path = os.path.join(top_dir, name)
            if os.path.isdir(path) or name in mask:
                pass
            else:
                os.remove(path)

    def lower_files(self, top_dir):
        if os.name == 'nt': # XXX
            return    
        n = 0
        for filename in os.listdir(top_dir):
            filename2 = filename.lower()
            path = os.path.join(top_dir, filename2)
            if filename != filename2:
                os.rename(os.path.join(top_dir, filename), path)
                logging.info(
                    'renamed {0}'.format(os.path.join(top_dir, filename)))
                n += 1
            if os.path.isdir(path):
                n += lower_files(path)
        return n

    def confirm(self, message):
        self.dialog.set_markup(message)
        response = self.dialog.run()
        self.dialog.hide()
        return response == Gtk.ResponseType.YES

    def confirm_overwrite(self, path, type_string):
        return self.confirm('Overwrite "{0}"({1})?'.format(os.fsdecode(path), type_string))

    def confirm_removal(self, path, type_string):
        return self.confirm('Remove "{0}"({1})?'.format(os.fsdecode(path), type_string))

    def select(self, candidates):
        assert len(candidates) >= 1
        if len(candidates) == 1:
            return candidates[0]
        ls = self.treeview.get_model()
        ls.clear()
        for i, item in enumerate(candidates):
            ls.append((i, os.fsdecode(item)))
        ts = self.treeview.get_selection()
        ts.select_iter(ls.get_iter_first())
        response = self.select_dialog.run()
        self.select_dialog.hide()
        if response != Gtk.ResponseType.ACCEPT:
            return None
        model, it = ts.get_selected()
        return candidates[model.get_value(it, 0)]

    def install_ghost(self, archive, tmpdir, homedir):
        # find install.txt
        inst = ninix.home.read_install_txt(tmpdir)
        if inst is None:
            fatal('install.txt not found')
        target_dir = inst.get('directory')
        if target_dir is None:
            fatal('"directory" not found in install.txt')
        target_dir = os.fsencode(target_dir)
        prefix = os.path.join(homedir, b'ghost', target_dir)
        tmpdir = os.fsencode(tmpdir)
        ghost_src = os.path.join(tmpdir, b'ghost', b'master')
        shell_src = os.path.join(tmpdir, b'shell')
        ghost_dst = os.path.join(prefix, b'ghost', b'master')
        shell_dst = os.path.join(prefix, b'shell')
        filelist = []
        ##filelist.append((os.path.join(tmpdir, 'install.txt'),
        ##                 os.path.join(prefix, 'install.txt'))) # XXX
        readme_txt = os.path.join(tmpdir, b'readme.txt')
        if os.path.exists(readme_txt):
            filelist.append((readme_txt,
                             os.path.join(prefix, b'readme.txt')))
        thumbnail_png = os.path.join(tmpdir, b'thumbnail.png')
        thumbnail_pnr = os.path.join(tmpdir, b'thumbnail.pnr')
        if os.path.exists(thumbnail_png):
            filelist.append((thumbnail_png,
                             os.path.join(prefix, b'thumbnail.png')))
        elif os.path.exists(thumbnail_pnr):
            filelist.append((thumbnail_pnr,
                             os.path.join(prefix, b'thumbnail.pnr')))
        for path in self.list_all_files(ghost_src, b''):
            filelist.append((os.path.join(ghost_src, path),
                             os.path.join(ghost_dst, path)))
        # find shell
        for path in self.list_all_files(shell_src, b''):
            filelist.append((os.path.join(shell_src, path),
                             os.path.join(shell_dst, path)))
        # find balloon
        balloon_dir = inst and inst.get('balloon.directory')
        if balloon_dir:
            balloon_dir = ninix.home.get_normalized_path(balloon_dir)
            balloon_dst = os.path.join(homedir, b'balloon', balloon_dir)
            balloon_src = inst and inst.get('balloon.source.directory')
            if balloon_src:
                balloon_src = ninix.home.get_normalized_path(balloon_src)
            else:
                balloon_src = balloon_dir
            if os.path.exists(balloon_dst) and \
                    not self. confirm_removal(balloon_dst, 'balloon'):
                pass # don't install balloon
            else:
                if os.path.exists(balloon_dst):
                    # uninstall older versions of the balloon
                    self.remove_files_and_dirs(balloon_dst, [])
                balloon_list = []
                for path in self.list_all_files(os.path.join(tmpdir, balloon_src), ''):
                    balloon_list.append((os.path.join(tmpdir, balloon_src, path),
                                         os.path.join(balloon_dst, path)))
                self.install_files(balloon_list)
        if os.path.exists(prefix):
            inst_dst = ninix.home.read_install_txt(prefix)
            if inst.get_with_type('refresh', int, 0):
                # uninstall older versions of the ghost
                if self.confirm_removal(prefix, 'ghost'):
                    mask = [ninix.home.get_normalized_path(path) for path in \
                                inst.get('refreshundeletemask', '').split(':')]
                    mask.append('HISTORY')
                    self.remove_files_and_dirs(prefix, mask)
                else:
                    return
            else:
                if not self.confirm_overwrite(prefix, 'ghost'):
                    return
        # install files
        logging.info('installing {0} (ghost)'.format(archive))
        self.install_files(filelist)
        # create SETTINGS
        path = os.path.join(prefix, b'SETTINGS')
        if not os.path.exists(path):
            try:
                with open(path, 'w') as f:
                    if balloon_dir:
                        f.write('balloon_directory, {0}\n'.format(balloon_dir))
            except IOError as e:
                code, message = e.args
                logging.error('cannot write {0}'.format(path))
        return os.fsdecode(target_dir)

    def install_supplement(self, archive, tmpdir, homedir):
        inst = ninix.home.read_install_txt(tmpdir)
        if inst and 'accept' in inst:
            logging.info('searching supplement target ...')
            candidates = []
            try:
                dirlist = os.listdir(os.path.join(homedir, b'ghost'))
            except OSError:
                dirlist = []
            for dirname in dirlist:
                path = os.path.join(homedir, b'ghost', dirname)
                if os.path.exists(os.path.join(path, b'shell', b'surface.txt')):
                    continue # ghost.inverse(obsolete)
                desc = ninix.home.read_descript_txt(
                    os.path.join(path, b'ghost', b'master'))
                if desc and desc.get('sakura.name') == inst.get('accept'):
                    candidates.append(dirname)
            if not candidates:
                logging.info('not found')
            else:
                target = self.select(candidates)
                if target is None:
                    return
                path = os.path.join(homedir, b'ghost', target)
                if 'directory' in inst:
                    if inst.get('type') == 'shell':
                        path = os.path.join(path, b'shell', os.fsencode(inst['directory']))
                    else:
                        if 'type' not in inst:
                            logging.error('supplement type not specified')
                        else:
                            logging.error('unsupported supplement type: {0}'.format(inst['type']))
                        return
                logging.info('found')
                if not os.path.exists(path):
                    os.makedirs(path)
                os.remove(os.path.join(tmpdir, b'install.txt'))
                distutils.dir_util.copy_tree(os.fsdecode(tmpdir), os.fsdecode(path))
                return os.fsdecode(target)

    def install_balloon(self, archive, srcdir, homedir):
        # find install.txt
        inst = ninix.home.read_install_txt(srcdir)
        if inst is None:
            fatal('install.txt not found')
        target_dir = inst.get('directory')
        if target_dir is None:
            fatal('"directory" not found in install.txt')
        target_dir = os.fsencode(target_dir)
        dstdir = os.path.join(homedir, b'balloon', target_dir)
        filelist = []
        for path in self.list_all_files(srcdir, b''):
            filelist.append((os.path.join(srcdir, path),
                             os.path.join(dstdir, path)))
        ##filelist.append((os.path.join(srcdir, 'install.txt'),
        ##                 os.path.join(dstdir, 'install.txt')))
        if os.path.exists(dstdir):
            inst_dst = ninix.home.read_install_txt(dstdir)
            if inst.get_with_type('refresh', int, 0):
                # uninstall older versions of the balloon
                if self.confirm_removal(dstdir, 'balloon'):
                    mask = [ninix.home.get_normalized_path(path) for path in \
                                inst.get('refreshundeletemask', '').split(':')]
                    self.remove_files_and_dirs(dstdir, mask)
                else:
                    return
            else:
                if not self.confirm_overwrite(dstdir, 'balloon'):
                    return
        # install files
        logging.info('installing {0} (balloon)'.format(archive))
        self.install_files(filelist)
        return os.fsdecode(target_dir)

    def uninstall_plugin(self, homedir, name):
        try:
            dirlist = os.listdir(os.path.join(homedir, b'plugin'))
        except OSError:
            return
        for subdir in dirlist:
            path = os.path.join(homedir, b'plugin', subdir)
            plugin = ninix.home.read_plugin_txt(path)
            if plugin is None:
                continue
            plugin_name, plugin_dir, startup, menu_items = plugin
            if plugin_name == name:
                plugin_dir = os.path.join(homedir, b'plugin', subdir)
                if self.confirm_removal(plugin_dir, 'plugin'):
                    shutil.rmtree(plugin_dir)

    def install_plugin(self, archive, srcdir, homedir):
        filelist = []
        # find plugin.txt
        plugin = ninix.home.read_plugin_txt(srcdir)
        if plugin is None:
            fatal('failed to read plugin.txt')
        plugin_name, plugin_dir, startup, menu_items = plugin
        # find files
        for filename in os.listdir(srcdir):
            path = os.path.join(srcdir, filename)
            if os.path.isfile(path):
                filelist.append((path, os.path.join(homedir, b'plugin', plugin_dir, filename)))
        # uninstall older versions of the plugin
        self.uninstall_plugin(homedir, plugin_name)
        # install files
        print('installing {0} (plugin)'.format(archive))
        self.install_files(filelist)
        return plugin_dir

    def uninstall_kinoko(self, homedir, name):
        try:
            dirlist = os.listdir(os.path.join(homedir, b'kinoko'))
        except OSError:
            return
        for subdir in dirlist:
            path = os.path.join(homedir, b'kinoko', subdir)
            kinoko = ninix.home.read_kinoko_ini(path)
            if kinoko is None:
                continue
            kinoko_name = kinoko['title']
            if kinoko_name == name:
                kinoko_dir = os.path.join(homedir, b'kinoko', subdir)
                if self.confirm_removal(kinoko_dir, 'kinoko'):
                    shutil.rmtree(kinoko_dir)

    def install_kinoko(self, archive, srcdir, homedir):
        # find kinoko.ini
        kinoko = ninix.home.read_kinoko_ini(srcdir)
        if kinoko is None:
            fatal('failed to read kinoko.ini')
        kinoko_name = kinoko['title']
        if kinoko['extractpath'] is not None:
            dstdir = os.path.join(
                homedir, b'kinoko', os.fsencode(kinoko['extractpath']))
        else:
            dstdir = os.path.join(
                homedir, b'kinoko', os.path.basename(archive)[:-4])
        # find files
        filelist = []
        for filename in os.listdir(srcdir):
            path = os.path.join(srcdir, filename)
            if os.path.isfile(path):
                filelist.append((path, os.path.join(dstdir, filename)))
        # uninstall older versions of the kinoko
        self.uninstall_kinoko(homedir, kinoko_name)
        # install files
        logging.info('installing {0} (kinoko)'.format(archive))
        self.install_files(filelist)
        return dstdir

    def uninstall_nekoninni(self, homedir, dir):
        nekoninni_dir = os.path.join(homedir, b'nekodorif', b'skin', dir)
        if not os.path.exists(nekoninni_dir):
            return
        if self.confirm_removal(nekoninni_dir, 'nekodorif skin'):
            shutil.rmtree(nekoninni_dir)

    def install_nekoninni(self, archive, srcdir, homedir):
        # find install.txt
        inst = ninix.home.read_install_txt(srcdir)
        if inst is None:
            fatal('install.txt not found')
        target_dir = inst.get('directory')
        if target_dir is None:
            fatal('"directory" not found in install.txt')
        target_dir = os.fsencode(target_dir)
        dstdir = os.path.join(homedir, b'nekodorif', b'skin', target_dir)
        # find files
        filelist = []
        for filename in os.listdir(srcdir):
            path = os.path.join(srcdir, filename)
            if os.path.isfile(path):
                filelist.append((path, os.path.join(dstdir, filename)))
        # uninstall older versions of the skin
        self.uninstall_nekoninni(homedir, target_dir)
        # install files
        logging.info('installing {0} (nekodorif skin)'.format(archive))
        self.install_files(filelist)
        return target_dir

    def uninstall_katochan(self, homedir, target_dir):
        katochan_dir = os.path.join(
            homedir, b'nekodorif', b'katochan', target_dir)
        if not os.path.exists(katochan_dir):
            return
        if self.confirm_removal(katochan_dir, 'nekodorif katochan'):
            shutil.rmtree(katochan_dir)

    def install_katochan(self, archive, srcdir, homedir):
        # find install.txt
        inst = ninix.home.read_install_txt(srcdir)
        if inst is None:
            fatal('install.txt not found')
        target_dir = inst.get('directory')
        if target_dir is None:
            fatal('"directory" not found in install.txt')
        target_dir = os.fsencode(target_dir)
        dstdir = os.path.join(homedir, b'nekodorif', b'katochan', target_dir)
        # find files
        filelist = []
        for filename in os.listdir(srcdir):
            path = os.path.join(srcdir, filename)
            if os.path.isfile(path):
                filelist.append((path, os.path.join(dstdir, filename)))
        # uninstall older versions of the skin
        self.uninstall_katochan(homedir, target_dir)
        # install files
        logging.info('installing {0} (nekodorif katochan)'.format(archive))
        self.install_files(filelist)
        return target_dir

    def list_all_files(self, top, target_dir):
        filelist = []
        for path in os.listdir(os.path.join(top, target_dir)):
            if os.path.isdir(os.path.join(top, target_dir, path)):
                filelist.extend(self.list_all_files(
                        top, os.path.join(target_dir, path)))
            else:
                filelist.append(os.path.join(target_dir, path))
        return filelist

    def install_files(self, filelist):
        for from_path, to_path in filelist:
            dirname, filename = os.path.split(to_path)
            if not os.path.exists(dirname):
                os.makedirs(dirname)
            shutil.copy(from_path, to_path)
