/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.util.List;

public class QuadroImpl extends Quadro {

	//
	static final int EQ_TO_LEFT = -1;
	static final int BLANKX = -2;
	static final int BLANKY = -3;
	static final int E2 = -100;
	static final int W2 = -101;
	static final int N2 = -102;
	static final int S2 = -103;
	static final int E3 = -200;
	static final int W3 = -201;
	static final int N3 = -202;
	static final int S3 = -203;
	static final int DONE = -300;
	static final int ENTRY = -301;
	static final int WARP2 = -400;
	static final int WARP3 = -402;

	private int[][] quadro;
	private Object[][] scratch;
	private int curx = 0, cury = 0;
	private Object edge;

	String rootResource;
	String rootPackage;

	//
	QuadroImpl(List<int[]> q) {
		quadro  = new int[q.size()][];
		scratch = new Object[q.size()][];
		for(int i = 0; i < q.size(); i++) {
			quadro[i]  = q.get(i);
			scratch[i] = new Object[q.get(i).length];
		}
	}

	//
	QuadroImpl(char[][] q) {
		quadro  = new int[q.length][];
		scratch = new Object[q.length][];
		for(int i = 0; i < q.length; i++) {
			quadro[i]  = new int[q[i].length];
			scratch[i] = new Object[q[i].length];
			for(int j = 0; j < q[i].length; j++) {
				quadro[i][j] = q[i][j];
			}
		}
	}

	//
	QuadroImpl(String[] q) {
		quadro  = new int[q.length][];
		scratch = new Object[q.length][];
		for(int i = 0; i < q.length; i++) {
			quadro[i]  = new int[q[i].length()];
			scratch[i] = new Object[q[i].length()];
			for(int j = 0; j < q[i].length(); j++) {
				quadro[i][j] = q[i].charAt(j);
			}
		}
	}

	/**
	 * @return the rootResource
	 */
	public String getRootResource() {
		return rootResource;
	}

	/**
	 * @return the rootPackage
	 */
	public String getRootPackage() {
		return rootPackage;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.Quadro#setRootResource(java.lang.String)
	 */
	@Override
	public void setRootResource(String x) {
		rootResource = x;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.Quadro#setRootPackage(java.lang.String)
	 */
	@Override
	public void setRootPackage(String x) {
		rootPackage = x;
	}

	/**
	 * @return the edge
	 */
	public Object getEdge() {
		return edge;
	}

	/**
	 * @param edge the edge to set
	 */
	public void setEdge(Object edge) {
		this.edge = edge;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isInBounds() {
		return (cury >= 0 && cury < quadro.length) &&
				(curx >= 0 && curx < quadro[cury].length);
	}

	/**
	 * 
	 * @return
	 */
	public int get() {
		if(cury < 0 || cury >= quadro.length) {
			return BLANKY;
		} else if(curx < 0 || curx >= quadro[cury].length) {
			return BLANKX;
		} else {
			return quadro[cury][curx];
		}
	}

	/**
	 * 
	 * @return
	 */
	public Object getScratch() {
		if(cury < 0 || cury >= quadro.length) {
			return null;
		} else if(curx < 0 || curx >= quadro[cury].length) {
			return null;
		} else {
			return scratch[cury][curx];
		}
	}

	/**
	 * 
	 * @return
	 */
	public boolean isBlankX() {
		return get() == BLANKX;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isBlankY() {
		return get() == BLANKY;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isLetter() {
		int c = get();

		return c >= 0 && Character.isLetter(c);
	}

	/**
	 * 
	 * @return
	 */
	public boolean isArrow() {
		return "^><v".indexOf(get()) >= 0;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isEntry() {
		return get() == ENTRY;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isDone() {
		return get() == DONE;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isFrame() {
		return "*@=&".indexOf(get()) >= 0;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isArrow2() {
		return get() <= E2 && get() >= S2;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isArrow3() {
		return get() <= E3 && get() >= S3;
	}

	/**
	 * 
	 * @return
	 */
	public boolean isWhitespace() {
		return Character.isWhitespace(get());
	}

	/**
	 * 
	 * @param a
	 */
	public void set(int a) {
		if(isInBounds()) {
			quadro[cury][curx] = a;
		}
	}

	/**
	 * 
	 * @param a
	 */
	public void setScratch(Object a) {
		if(isInBounds()) {
			scratch[cury][curx] = a;
		}
	}

	/**
	 * 
	 * @param d
	 * @return
	 */
	public QuadroImpl move(Quadro.Direction d) {
		switch(d) {
		case WEST:
			curx = curx < 0 ? curx : curx - 1;
			return this;
		case EAST:
			curx = cury < 0 || cury >= quadro.length || curx > quadro[cury].length ?
					curx : curx + 1;
			return this;
		case NORTH:
			cury = cury < 0 ? cury : cury - 1;
			return this;
		case SOUTH:
			cury = cury > quadro.length ? cury : cury + 1;
			return this;
		case CR:
			curx = 0;
			cury = cury > quadro.length ? cury : cury + 1;
			return this;
		default:  throw new RuntimeException();
		}
	}

	/**
	 * 
	 * @param x
	 * @param y
	 * @return
	 */
	public QuadroImpl move(int x, int y) {
		if(cury + y < 0) {
			cury = -1;
		} else if(cury + y > quadro.length) {
			cury = quadro.length;
		}

		if(curx + x < 0) {
			curx = -1;
		} else if(curx + x > quadro[cury].length) {
			curx = quadro[cury].length;
		}
		return this;
	}

	/**
	 * 
	 * @return
	 */
	public QuadroImpl west() {
		return move(Quadro.Direction.WEST);
	}

	/**
	 * 
	 * @return
	 */
	public QuadroImpl east() {
		return move(Quadro.Direction.EAST);
	}

	/**
	 * 
	 * @return
	 */
	public QuadroImpl north() {
		return move(Quadro.Direction.NORTH);
	}

	/**
	 * 
	 * @return
	 */
	public QuadroImpl south() {
		return move(Quadro.Direction.SOUTH);
	}

	/**
	 * 
	 * @return
	 */
	public QuadroImpl cr() {
		return move(Quadro.Direction.CR);
	}

	/* (non-Javadoc)
	 * @see java.lang.Object#toString()
	 */
	public String toString() {
		StringBuffer b = new StringBuffer();

		for(int i = 0; i < quadro.length; i++) {
			for(int j = 0; j < quadro[i].length; j++) {
				if(cury == i && curx == j) {
					b.append('@');
				} else {
					switch(quadro[i][j]) {
					case EQ_TO_LEFT:  break;
					case BLANKX:  b.append('$');  break;
					case BLANKY:  b.append('#');  break;
					case N2:      b.append('^');  break;
					case E2:      b.append('>');  break;
					case S2:      b.append('v');  break;
					case W2:      b.append('<');  break;
					case N3:      b.append('A');  break;
					case E3:      b.append(')');  break;
					case S3:      b.append('w');  break;
					case W3:      b.append('(');  break;
					case DONE:    b.append('0');  break;
					default:
						if(quadro[i][j] >= 0) {
							b.append((char)quadro[i][j]);
						} else {
							b.append('.');
						}
						break;
					}
				}
			}
			b.append('\n');
		}
		return b.toString();
	}

}
