/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.IOException;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAState;
import net.morilib.automata.nfa.NFAAccept;
import net.morilib.automata.nfa.NFAConcatenation;
import net.morilib.automata.nfa.NFAObject;
import net.morilib.automata.nfa.NFAParser;
import net.morilib.automata.nfa.SingleObjectNFA;
import net.morilib.range.CharSets;

public class NinaParser {

	private static class Z {

		Object vertex, edge;

		Z(Object v, Object e) {
			vertex = v;
			edge   = e;
		}

	}

	private static enum S {
		INIT,  PRAG,  PACK,  MACHN, TYPE,  OPTN,  OPTV,
		COMNT,
		LINIT,
		FMNW,  FMNW2, FM_E,  FMNE,  FMNE2, FM_S,
		FMSE,  FMSE2, FM_W,  FMSW,  FMSW2, FM_N,
		FRNW,  FRNW2, FR_E,  FRNE,  FRNE2, FR_S,
		FRSE,  FRSE2, FR_W,  FRSW,  FRSW2, FR_N,
		LBLE,
		AR_N,  AR_E,  AR_EQ, AR_ER, AR_EI, AR_EG, AR_ES, AR_EM, AR_EX,
		AR_S,  AR_W,  AR_WQ, AR_WR, AR_WI, AR_WG, AR_WS, AR_WM, AR_WX,
		AR_ZN, AR_ZS, AR_ZE, AR_ZW,
		F2_W,  F2_N,  F3_W,  F3_N,
		BR_N,  BR_E,  BR_S,  BR_W,
		BR2_N, BR2_E, BR2_S, BR2_W,
		FRR_S, FRR_E, FRR_N, FRR_W,
		ARR_N, ARR_E, ARR_S, ARR_W,
		ARRZN, ARRZS, ARRZE, ARRZW,
		WP_E1, WP_W1,
		TOIX1, TOIY1,
		SER1E, SER1F, SER1G, SER1H,
		SER1W, SER1X, SER1Y,
		WP_E2, WP_W2,
		TOIX2, TOIY2,
		SER2E, SER2F, SER2G, SER2H,
		SER2W, SER2X, SER2Y,
	}

	static final int _PAUSE = 1;
	static final int _PRINT = 2;

	private Quadro q;
	private S etat = S.INIT;
	private StringBuffer buf = new StringBuffer();
	private StringBuffer bf2 = new StringBuffer();
	private NinaAction action;
	private Class<?> type = Character.TYPE;
	private int pause;
	private Map<String, String> pragmas =
			new HashMap<String, String>();
	private int chr;

	//
	NinaParser(Quadro q, NinaAction action) {
		this.q = q;
		this.action = action;
	}

	//
	Object _compileRes(final String n, final NinaAction action) {
		try {
			return compile(Quadro.readResource(n), action, pause);
		} catch(IOException e) {
			throw new NinaParseException(e);
		}
	}

	//
	static Object compile(Quadro q, NinaAction action, int p) {
		NinaParser c = new NinaParser(q, action);
		Object o;

		while((o = c.step()) == null) {
			if((p & _PRINT) != 0) {
				try {
					if((p & _PAUSE) != 0 && System.in.read() == '\r') {
						System.in.read();
					}
					c.printTrace(System.out);
				} catch(IOException e) {
					throw new RuntimeException(e);
				}
			}
		}
		return o;
	}

	/**
	 * 
	 * @param q
	 * @return
	 */
	public static Object complie(Quadro q) {
		return NinaParser.compile(q, null, 0);
	}

	private void metBord(char c) {
		char[] a = new char[1];

		if(type.equals(Character.TYPE)) {
			action.setEdge(q, Integer.valueOf(c));
		} else if(type.equals(String.class)) {
			a[0] = c;
			action.setEdge(q, new String(a));
		} else {
			throw new ClassCastException();
		}
	}

	private void metBord(String s) {
		List<NFAObject<Object, NFAState, Void>> l;
		NFAObject<Object, NFAState, Void> o;

		if(type.equals(Character.TYPE)) {
			if(s.length() == 0) {
				throw new NinaParseException();
			} else if(s.length() == 1) {
				action.setEdge(q, Integer.valueOf(s.charAt(0)));
			} else {
				l = new ArrayList<NFAObject<Object, NFAState, Void>>();
				for(int k = 0; k < s.length(); k++) {
					o = SingleObjectNFA.newInstance(
							(Object)Integer.valueOf(s.charAt(k)));
					l.add(o);
				}
				o = NFAConcatenation.newInstance(l);
				action.setEdge(q, NFAAccept.newInstance(o, null));
			}
		} else if(type.equals(String.class)) {
			action.setEdge(q, s);
		} else if(type.isInstance(Integer.class) ||
				type.equals(Integer.TYPE)) {
			action.setEdge(q, Integer.valueOf(s));
		} else if(type.isInstance(Long.class) ||
				type.equals(Long.TYPE)) {
			action.setEdge(q, Long.valueOf(s));
		} else if(type.isInstance(Float.class) ||
				type.equals(Float.TYPE)) {
			action.setEdge(q, Double.valueOf(s));
		} else if(type.isInstance(Double.class) ||
				type.equals(Double.TYPE)) {
			action.setEdge(q, Double.valueOf(s));
		} else if(type.isInstance(java.math.BigInteger.class)) {
			action.setEdge(q, new java.math.BigInteger(s));
		} else if(type.isInstance(java.math.BigDecimal.class)) {
			action.setEdge(q, new java.math.BigDecimal(s));
		} else {
			throw new ClassCastException();
		}
	}

	private void metBordRes(String s) {
		action.setEdgeResource(q, this, s);
	}

	private void metBordCharset(String s) {
		if(type.equals(Character.TYPE)) {
			action.setEdge(q, CharSets.parse(s));
		} else {
			throw new ClassCastException();
		}
	}

	private void metBordNFA(NFA<Object, NFAState, Void> s) {
		action.setEdge(q, s);
	}

	private void metmealy() {
		if(chr < 0) {
			action.setMealyEdge(buf.reverse().toString());
		} else {
			action.setMealyEdge((char)chr);
		}
	}

	//
	Object step() {
		NinaFrameReader rd;
		NinaAction n;
		String s;
		Object o;
		int c;

		switch(etat) {
		case INIT:
			if(q.get() == '#') {
				buf = new StringBuffer();
				q.east();
				etat = S.PRAG;
			} else if(q.isBlankY()) {
				throw new NinaParseException(this);
			} else if(!q.isBlankX()) {
				etat = S.LINIT;
			}
			break;
		case PRAG:
			if(q.get() == QuadroImpl.EQ_TO_LEFT) {
				q.east();
			} else if(q.isLetter()) {
				buf.appendCodePoint(q.get());
				q.east();
			} else if(!Character.isWhitespace(q.get())) {
				q.east();
				etat = S.COMNT;
			} else if(buf.toString().equals("package")) {
				buf = new StringBuffer();
				q.east();
				etat = S.PACK;
			} else if(buf.toString().equals("machine")) {
				buf = new StringBuffer();
				q.east();
				etat = S.MACHN;
			} else if(buf.toString().equals("type")) {
				buf = new StringBuffer();
				q.east();
				etat = S.TYPE;
			} else if(buf.toString().equals("option")) {
				buf = new StringBuffer();
				q.east();
				etat = S.OPTN;
			}
			break;
		case PACK:
			if(q.get() == QuadroImpl.EQ_TO_LEFT) {
				q.east();
			} else if(q.isLetter() || q.get() == '.') {
				buf.appendCodePoint(q.get());
				q.east();
			} else {
				q.setRootPackage(s = buf.toString().trim());
				q.setRootResource("/" + s.replace('.', '/'));
				pragmas.put("package", s);
				etat = S.COMNT;
			}
			break;
		case MACHN:
			if(q.get() == QuadroImpl.EQ_TO_LEFT) {
				q.east();
			} else if(q.isLetter() || q.get() == '.') {
				buf.appendCodePoint(q.get());
				q.east();
			} else {
				s = buf.toString().trim();
				pragmas.put("machine", s);
				q.options.put("machine", s);
				if(s.indexOf('.') < 0) {
					s = getClass().getPackage().getName() + "." + s;
				}

				n = (NinaAction)Nina.prendClasse(s);
				if(action == null) {
					action = n;
				} else if(!action.getClass().equals(n.getClass())) {
					throw new NinaParseException(this);
				}
				etat = S.COMNT;
			}
			break;
		case TYPE:
			s = null;
			if(q.get() == QuadroImpl.EQ_TO_LEFT) {
				q.east();
			} else if(q.isLetter() || q.get() == '.') {
				buf.appendCodePoint(q.get());
				q.east();
			} else if((s = buf.toString().trim()).equals("char")) {
				type = Character.TYPE;
				etat = S.COMNT;
			} else if((s = buf.toString().trim()).equals("int")) {
				type = Integer.TYPE;
				etat = S.COMNT;
			} else if((s = buf.toString().trim()).equals("long")) {
				type = Long.TYPE;
				etat = S.COMNT;
			} else if((s = buf.toString().trim()).equals("float")) {
				type = Float.TYPE;
				etat = S.COMNT;
			} else if((s = buf.toString().trim()).equals("double")) {
				type = Double.TYPE;
				etat = S.COMNT;
			} else if((s = buf.toString().trim()).equals("string")) {
				type = String.class;
				etat = S.COMNT;
			} else {
				type = (Class<?>)Nina.prendClasse(s);
				etat = S.COMNT;
			}
			if(s != null)  pragmas.put("type", s);
			break;
		case OPTN:
			if(q.get() == QuadroImpl.EQ_TO_LEFT) {
				q.east();
			} else if(q.get() == '=') {
				bf2 = new StringBuffer();
				q.east();
				etat = S.OPTV;
			} else if(q.get() >= 0) {
				buf.appendCodePoint(q.get());
				q.east();
			} else {
				q.options.put(buf.toString(), "");
				etat = S.COMNT;
			}
			break;
		case OPTV:
			if(q.get() == QuadroImpl.EQ_TO_LEFT) {
				q.east();
			} else if(q.get() >= 0) {
				bf2.appendCodePoint(q.get());
				q.east();
			} else {
				q.options.put(buf.toString(), bf2.toString());
				etat = S.COMNT;
			}
			break;
		case COMNT:
			if(q.isBlankX()) {
				q.cr();
				etat = S.INIT;
			} else if(q.isBlankY()) {
				throw new NinaParseException(this);
			} else {
				q.east();
			}
			break;
		case LINIT:
			if(q.get() == '=' || q.get() == '&') {
				etat = S.LBLE;
			} else if(q.isBlankX()) {
				q.cr();
			} else if(q.isBlankY()) {
				throw new NinaParseException(this);
			} else {
				q.east();
			}
			break;
		case LBLE:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				c = q.get();
				q.south().east();
				rd = new NinaFrameReader(q);
				action.labelAdded(q, rd, c == '@' || c == '&');
				rd.close();
				etat = S.FMNW;
			} else {
				q.west();
			}
			break;
		case FMNW:
			if(q.isFrame()) {
				q.east();
				etat = S.FMNW2;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FMNW2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FM_E;
			} else {
				throw new NinaParseException(this);
			}
		case FM_E:
			q.setScratch(action.getVertex());
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.east();
			} else if(q.isDone()) {
				q.west();
				etat = S.FRR_W;
			} else {
				q.west();
				etat = S.FMNE;
			}
			break;
		case FMNE:
			if(q.isFrame()) {
				q.south();
				etat = S.FMNE2;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FMNE2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FM_S;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FM_S:
			q.setScratch(action.getVertex());
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.south();
			} else if(q.isDone()) {
				q.north();
				etat = S.FRR_N;
			} else {
				q.north();
				etat = S.FMSE;
			}
			break;
		case FMSE:
			if(q.isFrame()) {
				q.west();
				etat = S.FMSE2;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FMSE2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FM_W;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FM_W:
			q.setScratch(action.getVertex());
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.west();
			} else if(q.isDone()) {
				q.east();
				etat = S.FRR_E;
			} else {
				q.east();
				etat = S.FMSW;
			}
			break;
		case FMSW:
			if(q.isFrame()) {
				etat = S.FMSW2;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FMSW2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FM_N;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FM_N:
			q.setScratch(action.getVertex());
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.north();
			} else if(q.isDone()) {
				q.south();
				etat = S.FRNW;
			} else {
				q.south();
				etat = S.FRNW;
				action.doneBlockSearching(q);
			}
			break;
		case FRNW:
			if(q.isFrame()) {
				q.east();
				etat = S.FRNW2;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FRNW2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FR_E;
			} else {
				throw new NinaParseException(this);
			}
		case FR_E:
			if(q.get() == '^') {
				action.setVertex(q.getScratch());
				action.setEdge(q, null);
				q.north();
				etat = S.AR_N;
			} else if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.east();
			} else if(q.isDone()) {
				q.west();
				etat = S.FRR_W;
			} else {
				q.west();
				etat = S.FRNE;
			}
			break;
		case FRNE:
			if(q.isFrame()) {
				q.south();
				etat = S.FRNE2;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FRNE2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FR_S;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FR_S:
			if(q.get() == '>') {
				action.setVertex(q.getScratch());
				action.setEdge(q, null);
				q.east();
				etat = S.AR_E;
			} else if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.south();
			} else if(q.isDone()) {
				q.north();
				etat = S.FRR_N;
			} else {
				q.north();
				etat = S.FRSE;
			}
			break;
		case FRSE:
			if(q.isFrame()) {
				q.west();
				etat = S.FRSE2;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FRSE2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FR_W;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FR_W:
			if(q.get() == 'v') {
				action.setVertex(q.getScratch());
				action.setEdge(q, null);
				q.south();
				etat = S.AR_S;
			} else if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.west();
			} else if(q.isDone()) {
				q.east();
				etat = S.FRR_E;
			} else {
				q.east();
				etat = S.FRSW;
			}
			break;
		case FRSW:
			if(q.isFrame()) {
				etat = S.FRSW2;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FRSW2:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				etat = S.FR_N;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case FR_N:
			if(q.get() == '<') {
				action.setVertex(q.getScratch());
				action.setEdge(q, null);
				q.west();
				etat = S.AR_W;
			} else if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.north();
			} else if(q.isDone()) {
				q.south();
				etat = S.FRR_S;
			} else {
				q.south();
				etat = S.FRR_S;
				action.doneBlockSearching(q);
			}
			break;
		case AR_N:
			if(q.get() == '/') {
				q.set(QuadroImpl.S2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() == '\\') {
				q.set(QuadroImpl.S2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() == '|') {
				q.set(QuadroImpl.S2);
				q.north();
			} else if(q.get() == '+') {
				q.setScratch(new Z(action.getVertex(), q.getEdge()));
				q.north();
				etat = S.BR_N;
			} else if(q.get() == '.') {
				q.setEdge(CharSets.ALL_CHAR);
				q.set(QuadroImpl.S2);
				q.north();
			} else if(q.get() == '^') {
				action.link(q, o = q.getScratch());
				if(o != null) {
					q.south();
					etat = S.ARR_S;
				} else {
					q.set(QuadroImpl.ENTRY);
					etat = S.F2_W;
				}
			} else if(q.isLetter()) {
				q.setEdge(Integer.valueOf((char)q.get()));
				q.set(QuadroImpl.S2);
				q.north();
			} else if(q.isArrow2() || q.isArrow3()) {
				q.south();
				etat = S.AR_ZE;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_ZE:
			q.set('^');
			q.east();
			etat = S.FR_E;
			break;
		case AR_E:
			if(q.get() == '/') {
				q.set(QuadroImpl.W2);
				q.north();
				etat = S.AR_N;
			} else if(q.get() == '\\') {
				q.set(QuadroImpl.W2);
				q.south();
				etat = S.AR_S;
			} else if(q.get() == '-') {
				q.set(QuadroImpl.W2);
				q.east();
			} else if(q.get() == '+') {
				q.setScratch(new Z(action.getVertex(), q.getEdge()));
				q.north();
				etat = S.BR_N;
			} else if(q.get() == '.') {
				q.setEdge(CharSets.ALL_CHAR);
				q.set(QuadroImpl.W2);
				q.east();
			} else if(q.get() == '>') {
				action.link(q, o = q.getScratch());
				if(o != null) {
					q.west();
					etat = S.ARR_W;
				} else {
					q.set(QuadroImpl.ENTRY);
					etat = S.F2_N;
				}
			} else if(q.get() == '\'') {
				q.set(QuadroImpl.W2);
				q.east();
				buf = new StringBuffer();
				etat = S.AR_EQ;
			} else if(q.get() == '{') {
				q.set(QuadroImpl.W2);
				q.east();
				buf = new StringBuffer();
				etat = S.AR_ER;
			} else if(q.get() == '[') {
				q.set(QuadroImpl.W2);
				q.east();
				buf = new StringBuffer();
				etat = S.AR_EI;
			} else if(q.get() == '(') {
				q.set(QuadroImpl.WARP2);
				q.east();
				buf = new StringBuffer();
				etat = S.WP_E1;
			} else if(q.isLetter()) {
				metBord((char)q.get());
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_ES;
			} else if(q.isArrow2() || q.isArrow3()) {
				q.west();
				etat = S.AR_ZS;
			} else if(q.get() == Quadro.EQ_TO_LEFT) {
				q.set(QuadroImpl.W2);
				q.east();
			} else if(q.get() == '`') {
				q.set(QuadroImpl.W2);
				q.east();
				buf = new StringBuffer();
				etat = S.AR_EG;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_ZS:
			q.set('>');
			q.south();
			etat = S.FR_S;
			break;
		case AR_EQ:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '\'') {
				metBord(buf.toString());
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_ES;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.W2);
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_ER:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '}') {
				metBordRes(buf.toString());
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.W2);
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_EI:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == ']') {
				metBordCharset(buf.toString());
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.W2);
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_EG:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '`') {
				metBordNFA(NFAParser.parse(buf.toString()));
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.W2);
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_ES:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '/') {
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_EM;
			} else {
				etat = S.AR_E;
			}
			break;
		case AR_EM:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.isWhitespace()) {
				q.set(QuadroImpl.W2);
				q.north().west();
				etat = S.AR_N;
			} else if(q.get() == '\'') {
				q.set(QuadroImpl.W2);
				q.east();
				buf = new StringBuffer();
				etat = S.AR_EX;
			} else if(q.get() >= 0) {
				action.setMealyEdge(q.get());
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_E;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_EX:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '\'') {
				action.setMealyEdge(buf.toString());
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.W2);
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case WP_E1:
			if(q.get() == ')') {
				q.set(QuadroImpl.WARP2);
				etat = S.TOIX1;
			} else if(q.get() >= 0) {
				buf.appendCodePoint(q.get());
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_S:
			if(q.get() == '/') {
				q.set(QuadroImpl.N2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() == '\\') {
				q.set(QuadroImpl.N2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() == '|') {
				q.set(QuadroImpl.N2);
				q.south();
			} else if(q.get() == '+') {
				q.setScratch(new Z(action.getVertex(), q.getEdge()));
				q.north();
				etat = S.BR_N;
			} else if(q.get() == '.') {
				q.setEdge(CharSets.ALL_CHAR);
				q.set(QuadroImpl.N2);
				q.south();
			} else if(q.get() == 'v') {
				action.link(q, o = q.getScratch());
				if(o != null) {
					q.north();
					etat = S.ARR_N;
				} else {
					q.set(QuadroImpl.ENTRY);
					etat = S.F3_W;
				}
			} else if(q.isLetter()) {
				metBord((char)q.get());
				q.set(QuadroImpl.N2);
				q.south();
			} else if(q.isArrow2() || q.isArrow3()) {
				q.north();
				etat = S.AR_ZW;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_ZW:
			q.set('v');
			q.west();
			etat = S.FR_W;
			break;
		case AR_W:
			if(q.get() == '/') {
				q.set(QuadroImpl.E2);
				q.south();
				etat = S.AR_S;
			} else if(q.get() == '\\') {
				q.set(QuadroImpl.E2);
				q.north();
				etat = S.AR_N;
			} else if(q.get() == '-') {
				q.set(QuadroImpl.E2);
				q.west();
			} else if(q.get() == '+') {
				q.setScratch(new Z(action.getVertex(), q.getEdge()));
				q.north();
				etat = S.BR_N;
			} else if(q.get() == '.') {
				q.setEdge(CharSets.ALL_CHAR);
				q.set(QuadroImpl.E2);
				q.west();
			} else if(q.get() == '<') {
				action.link(q, o = q.getScratch());
				if(o != null) {
					q.east();
					etat = S.ARR_E;
				} else {
					q.set(QuadroImpl.ENTRY);
					etat = S.F3_N;
				}
			} else if(q.get() == '\'') {
				buf = new StringBuffer();
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_WQ;
				buf = new StringBuffer();
			} else if(q.get() == '}') {
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_WR;
				buf = new StringBuffer();
			} else if(q.get() == ']') {
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_WI;
				buf = new StringBuffer();
			} else if(q.get() == ')') {
				q.set(QuadroImpl.WARP2);
				q.west();
				buf = new StringBuffer();
				etat = S.WP_W1;
			} else if(q.isLetter()) {
				chr = q.get();
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_WS;
			} else if(q.isArrow2() || q.isArrow3()) {
				q.east();
				etat = S.AR_ZN;
			} else if(q.get() == Quadro.EQ_TO_LEFT) {
				q.set(QuadroImpl.E2);
				q.west();
			} else if(q.get() == '`') {
				q.set(QuadroImpl.E2);
				q.east();
				buf = new StringBuffer();
				etat = S.AR_WG;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_ZN:
			q.set('<');
			q.north();
			etat = S.FR_N;
			break;
		case AR_WQ:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '\'') {
				chr = -1;
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_WS;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.E2);
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_WR:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '{') {
				metBordRes(buf.reverse().toString());
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.E2);
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_WI:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '[') {
				metBordCharset(buf.reverse().toString());
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.E2);
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_WG:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '`') {
				metBordNFA(NFAParser.parse(buf.reverse().toString()));
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.E2);
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_WS:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '/') {
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_WM;
			} else if(chr < 0) {
				metBord(buf.reverse().toString());
				etat = S.AR_W;
			} else {
				metBord((char)chr);
				etat = S.AR_W;
			}
			break;
		case AR_WM:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.isWhitespace()) {
				q.south().east();
				etat = S.AR_S;
			} else if(q.get() == '\'') {
				metmealy();
				q.set(QuadroImpl.E2);
				q.west();
				buf = new StringBuffer();
				etat = S.AR_WX;
			} else if(q.get() == '}') {
				metmealy();
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_WR;
				buf = new StringBuffer();
			} else if(q.get() == ']') {
				metmealy();
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_WI;
				buf = new StringBuffer();
			} else if(q.get() == ')') {
				metmealy();
				q.set(QuadroImpl.WARP2);
				q.west();
				buf = new StringBuffer();
				etat = S.WP_W1;
			} else if(q.get() >= 0) {
				metmealy();
				metBord((char)q.get());
				q.set(QuadroImpl.E2);
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case AR_WX:
			if(q.isBlankX()) {
				throw new NinaParseException(this);
			} else if(q.get() == '\'') {
				metBord(buf.reverse().toString());
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() >= 0) {
				buf.append((char)q.get());
				q.set(QuadroImpl.E2);
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case WP_W1:
			if(q.get() == '(') {
				buf.reverse();
				q.set(QuadroImpl.WARP2);
				etat = S.TOIX1;
			} else if(q.get() >= 0) {
				buf.appendCodePoint(q.get());
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case F2_W:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.west();
			} else if(q.isDone()) {
				q.east();
				etat = S.FRR_E;
			} else {
				q.east();
				etat = S.F2_N;
			}
			break;
		case F2_N:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.north();
			} else if(q.isDone()) {
				q.south();
				etat = S.FRR_S;
			} else {
				q.south();
				etat = S.LBLE;
			}
			break;
		case F3_W:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.west();
			} else {
				q.east();
				etat = S.LBLE;
			}
			break;
		case F3_N:
			if(q.isFrame() || q.isArrow() || q.isEntry()) {
				q.north();
			} else {
				q.south();
				etat = S.F3_W;
			}
			break;
		case FRR_S:
			if(q.isFrame() || q.isArrow() || q.isDone()) {
				q.set(QuadroImpl.DONE);
				q.south();
			} else if(q.isEntry()) {
				q.set(QuadroImpl.DONE);
				q.west();
				etat = S.ARR_W;
			} else {
				q.north().east();
				etat = S.FRR_E;
			}
			break;
		case FRR_E:
			if(q.isFrame() || q.isArrow() || q.isDone()) {
				q.set(QuadroImpl.DONE);
				q.east();
			} else if(q.isEntry()) {
				q.set(QuadroImpl.DONE);
				q.south();
				etat = S.ARR_S;
			} else {
				q.north().west();
				etat = S.FRR_N;
			}
			break;
		case FRR_N:
			if(q.isFrame() || q.isArrow() || q.isDone()) {
				q.set(QuadroImpl.DONE);
				q.north();
			} else if(q.isEntry()) {
				q.set(QuadroImpl.DONE);
				q.east();
				etat = S.ARR_E;
			} else {
				q.south().west();
				etat = S.FRR_W;
			}
			break;
		case FRR_W:
			if(q.isFrame() || q.isArrow() || q.isDone()) {
				q.set(QuadroImpl.DONE);
				q.west();
			} else if(q.isEntry()) {
				q.set(QuadroImpl.DONE);
				q.north();
				etat = S.ARR_N;
			} else {
//				q.east();
//				etat = S.INIT;
				return action.accept();
			}
			break;
		case ARR_N:
			if(q.get() == QuadroImpl.N2) {
				q.set(QuadroImpl.S3);
				q.north();
				etat = S.ARR_N;
			} else if(q.get() == QuadroImpl.E2) {
				q.set(QuadroImpl.W3);
				q.east();
				etat = S.ARR_E;
			} else if(q.get() == QuadroImpl.S2) {
				q.set(QuadroImpl.N3);
				q.south();
				etat = S.ARR_S;
			} else if(q.get() == QuadroImpl.W2) {
				q.set(QuadroImpl.E3);
				q.west();
				etat = S.ARR_W;
			} else if(q.get() == '+') {
				q.setEdge(((Z)q.getScratch()).edge);
				action.setVertex(((Z)q.getScratch()).vertex);
				q.north();
				etat = S.BR_N;
			} else if(q.isArrow()) {
				action.setVertex(q.getScratch());
				q.west();
				etat = S.FR_W;
			} else if(q.isDone()) {
				q.east();
				etat = S.FRR_E;
			} else if(q.isArrow3()) {
				q.south();
				etat = S.ARRZW;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case ARRZW:
			q.set(QuadroImpl.DONE);
			etat = S.FRR_W;
			break;
		case ARR_E:
			if(q.get() == QuadroImpl.N2) {
				q.set(QuadroImpl.S3);
				q.north();
				etat = S.ARR_N;
			} else if(q.get() == QuadroImpl.E2) {
				q.set(QuadroImpl.W3);
				q.east();
				etat = S.ARR_E;
			} else if(q.get() == QuadroImpl.S2) {
				q.set(QuadroImpl.N3);
				q.south();
				etat = S.ARR_S;
			} else if(q.get() == QuadroImpl.W2) {
				q.set(QuadroImpl.E3);
				q.west();
				etat = S.ARR_W;
			} else if(q.get() == QuadroImpl.WARP2) {
				q.set(QuadroImpl.WARP3);
				q.east();
				buf = new StringBuffer();
				etat = S.WP_E2;
			} else if(q.get() == '+') {
				q.setEdge(((Z)q.getScratch()).edge);
				action.setVertex(((Z)q.getScratch()).vertex);
				q.north();
				etat = S.BR_N;
			} else if(q.isArrow()) {
				action.setVertex(q.getScratch());
				q.north();
				etat = S.FR_N;
			} else if(q.isDone()) {
				q.south();
				etat = S.FRR_S;
			} else if(q.isArrow3()) {
				q.west();
				etat = S.ARRZN;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case WP_E2:
			if(q.get() == QuadroImpl.WARP2) {
				q.set(QuadroImpl.WARP3);
				etat = S.TOIX2;
			} else if(q.get() >= 0) {
				buf.appendCodePoint(q.get());
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case ARRZN:
			q.set(QuadroImpl.DONE);
			etat = S.FRR_N;
			break;
		case ARR_S:
			if(q.get() == QuadroImpl.N2) {
				q.set(QuadroImpl.S3);
				q.north();
				etat = S.ARR_N;
			} else if(q.get() == QuadroImpl.E2) {
				q.set(QuadroImpl.W3);
				q.east();
				etat = S.ARR_E;
			} else if(q.get() == QuadroImpl.S2) {
				q.set(QuadroImpl.N3);
				q.south();
				etat = S.ARR_S;
			} else if(q.get() == QuadroImpl.W2) {
				q.set(QuadroImpl.E3);
				q.west();
				etat = S.ARR_W;
			} else if(q.get() == '+') {
				q.setEdge(((Z)q.getScratch()).edge);
				action.setVertex(((Z)q.getScratch()).vertex);
				q.north();
				etat = S.BR_N;
			} else if(q.isArrow()) {
				action.setVertex(q.getScratch());
				q.east();
				etat = S.FR_E;
			} else if(q.isDone()) {
				q.west();
				etat = S.FRR_W;
			} else if(q.isArrow3()) {
				q.north();
				etat = S.ARRZE;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case ARRZE:
			q.set(QuadroImpl.DONE);
			etat = S.FRR_S;
			break;
		case ARR_W:
			if(q.get() == QuadroImpl.N2) {
				q.set(QuadroImpl.S3);
				q.north();
				etat = S.ARR_N;
			} else if(q.get() == QuadroImpl.E2) {
				q.set(QuadroImpl.W3);
				q.east();
				etat = S.ARR_E;
			} else if(q.get() == QuadroImpl.S2) {
				q.set(QuadroImpl.N3);
				q.south();
				etat = S.ARR_S;
			} else if(q.get() == QuadroImpl.W2) {
				q.set(QuadroImpl.E3);
				q.west();
				etat = S.ARR_W;
			} else if(q.get() == QuadroImpl.WARP2) {
				q.set(QuadroImpl.WARP3);
				q.west();
				buf = new StringBuffer();
				etat = S.WP_W2;
			} else if(q.get() == '+') {
				q.setEdge(((Z)q.getScratch()).edge);
				action.setVertex(((Z)q.getScratch()).vertex);
				q.north();
				etat = S.BR_N;
			} else if(q.isArrow()) {
				action.setVertex(q.getScratch());
				q.south();
				etat = S.FR_S;
			} else if(q.isDone()) {
				q.north();
				etat = S.FRR_N;
			} else if(q.isArrow3()) {
				q.east();
				etat = S.ARRZS;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case WP_W2:
			if(q.get() == QuadroImpl.WARP2) {
				buf.reverse();
				q.set(QuadroImpl.WARP3);
				etat = S.TOIX2;
			} else if(q.get() >= 0) {
				buf.appendCodePoint(q.get());
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case ARRZS:
			q.set(QuadroImpl.DONE);
			etat = S.FRR_S;
			break;
		case BR_N:
			if(q.get() == '|') {
				q.set(QuadroImpl.S2);
				q.north();
				etat = S.AR_N;
			} else {
				q.south().east();
				etat = S.BR_E;
			}
			break;
		case BR_E:
			if(q.get() == '-') {
				q.set(QuadroImpl.W2);
				q.east();
				etat = S.AR_E;
			} else {
				q.west().south();
				etat = S.BR_S;
			}
			break;
		case BR_S:
			if(q.get() == '|') {
				q.set(QuadroImpl.N2);
				q.south();
				etat = S.AR_S;
			} else {
				q.north().west();
				etat = S.BR_W;
			}
			break;
		case BR_W:
			if(q.get() == '-') {
				q.set(QuadroImpl.E2);
				q.west();
				etat = S.AR_W;
			} else {
				q.east().north();
				etat = S.BR2_N;
			}
			break;
		case BR2_N:
			if(q.get() == QuadroImpl.N2) {
				q.set(QuadroImpl.S3);
				q.north();
				etat = S.ARR_N;
			} else {
				q.south().east();
				etat = S.BR2_E;
			}
			break;
		case BR2_E:
			if(q.get() == QuadroImpl.E2) {
				q.set(QuadroImpl.W3);
				q.east();
				etat = S.ARR_E;
			} else {
				q.west().south();
				etat = S.BR2_S;
			}
			break;
		case BR2_S:
			if(q.get() == QuadroImpl.S2) {
				q.set(QuadroImpl.N3);
				q.south();
				etat = S.ARR_S;
			} else {
				q.north().west();
				etat = S.BR2_W;
			}
			break;
		case BR2_W:
			if(q.get() == QuadroImpl.W2) {
				q.set(QuadroImpl.E3);
				q.west();
				etat = S.ARR_W;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case TOIX1:
			if(q.isBlankX()) {
				q.east();
				etat = S.TOIY1;
			} else {
				q.west();
			}
			break;
		case TOIY1:
			if(q.isBlankY() || q.get() == '#') {
				q.south();
				etat = S.SER1E;
			} else {
				q.north();
			}
			break;
		case SER1E:
			if(q.isBlankY()) {
				throw new NinaParseException(this);
			} else if(q.isBlankX()) {
				q.cr();
			} else if(q.get() == '(') {
				q.east();
				etat = S.SER1F;
				bf2 = new StringBuffer();
			} else if(q.get() == '-') {
				q.east();
				etat = S.SER1W;
			} else {
				q.east();
			}
			break;
		case SER1F:
			if(q.get() == ')') {
				if(buf.toString().equals(bf2.toString())) {
					q.west();
					etat = S.SER1G;
				} else {
					etat = S.SER1E;
				}
			} else if(q.get() >= 0) {
				bf2.appendCodePoint(q.get());
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case SER1G:
			if(q.get() == '(') {
				q.set(QuadroImpl.WARP2);
				q.east();
				etat = S.SER1H;
			} else if(q.get() >= 0) {
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case SER1H:
			if(q.get() == ')') {
				q.set(QuadroImpl.WARP2);
				q.east();
				etat = S.AR_E;
			} else if(q.get() >= 0) {
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case SER1W:
			if(q.isBlankY()) {
				throw new NinaParseException(this);
			} else if(q.isBlankX()) {
				q.cr();
			} else if(q.get() == '(') {
				q.east();
				etat = S.SER1X;
				bf2 = new StringBuffer();
			} else if(q.get() != '-') {
				q.east();
				etat = S.SER1E;
			} else {
				q.east();
			}
			break;
		case SER1X:
			if(q.get() == ')') {
				if(buf.toString().equals(bf2.toString())) {
					q.set(QuadroImpl.WARP2);
					q.west();
					etat = S.SER1Y;
				} else {
					etat = S.SER1W;
				}
			} else if(q.get() >= 0) {
				bf2.appendCodePoint(q.get());
				q.east();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case SER1Y:
			if(q.get() == '(') {
				q.set(QuadroImpl.WARP2);
				q.west();
				etat = S.AR_W;
			} else if(q.get() >= 0) {
				q.west();
			} else {
				throw new NinaParseException(this);
			}
			break;
		case TOIX2:
			if(q.isBlankX()) {
				q.east();
				etat = S.TOIY2;
			} else {
				q.west();
			}
			break;
		case TOIY2:
			if(q.isBlankY() || q.get() == '#') {
				q.south();
				etat = S.SER2E;
			} else {
				q.north();
			}
			break;
		case SER2E:
			if(q.isBlankY()) {
				throw new NinaParseException(this);
			} else if(q.isBlankX()) {
				q.cr();
			} else if(q.get() == QuadroImpl.WARP2) {
				q.east();
				etat = S.SER2F;
				bf2 = new StringBuffer();
			} else if(q.get() == QuadroImpl.W2) {
				q.east();
				etat = S.SER2W;
			} else {
				q.east();
			}
			break;
		case SER2F:
			if(q.get() >= 0) {
				bf2.appendCodePoint(q.get());
				q.east();
			} else if(q.get() != QuadroImpl.WARP2) {
				q.east();
				etat = S.SER2E;
			} else if(buf.toString().equals(bf2.toString())) {
				q.west();
				etat = S.SER2G;
			} else {
				etat = S.SER2E;
			}
			break;
		case SER2G:
			if(q.get() >= 0) {
				q.west();
			} else if(q.get() == QuadroImpl.WARP2) {
				q.set(QuadroImpl.WARP3);
				q.east();
				etat = S.SER2H;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case SER2H:
			if(q.get() >= 0) {
				q.east();
			} else if(q.get() == QuadroImpl.WARP2) {
				q.set(QuadroImpl.WARP3);
				q.east();
				etat = S.ARR_E;
			} else {
				throw new NinaParseException(this);
			}
			break;
		case SER2W:
			if(q.isBlankY()) {
				throw new NinaParseException(this);
			} else if(q.isBlankX()) {
				q.cr();
			} else if(q.get() == QuadroImpl.WARP2) {
				q.east();
				etat = S.SER2X;
				bf2 = new StringBuffer();
			} else if(q.get() != QuadroImpl.W2) {
				q.east();
				etat = S.SER2E;
			} else {
				q.east();
			}
			break;
		case SER2X:
			if(q.get() >= 0) {
				bf2.appendCodePoint(q.get());
				q.east();
			} else if(q.get() != QuadroImpl.WARP2) {
				q.east();
				etat = S.SER2E;
			} else if(buf.toString().equals(bf2.toString())) {
				q.set(QuadroImpl.WARP3);
				q.west();
				etat = S.SER2Y;
			} else {
				etat = S.SER2E;
			}
			break;
		case SER2Y:
			if(q.get() >= 0) {
				q.west();
			} else if(q.get() == QuadroImpl.WARP2) {
				q.set(QuadroImpl.WARP3);
				q.west();
				etat = S.ARR_W;
			} else {
				throw new NinaParseException(this);
			}
			break;
		}
		return null;
	}

	/**
	 * 
	 * @param s
	 * @return
	 */
	public String getPragma(String s) {
		return pragmas.get(s);
	}

	/**
	 * 
	 * @param pr
	 */
	public void printTrace(PrintStream pr) {
		pr.println(etat);
		pr.println(q.toString());
		pr.println(q.get() < 0 ? q.get() : (char)q.get());
	}

}
