/*
 * Copyright 2009-2010 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina.translate;

import java.io.IOException;
import java.io.InputStream;
import java.io.PrintStream;
import java.io.Reader;

import net.morilib.automata.DFA;
import net.morilib.automata.NFAState;
import net.morilib.automata.dfa.ConvertedRangeDFA;
import net.morilib.nina.Nina;
import net.morilib.nina.NinaNFA;
import net.morilib.nina.cmd.NinatOptions;
import net.morilib.nina.translate.sh.ShNinatBuiltInCommands;
import net.morilib.nina.translate.sh.ShNinatFileSystem;
import net.morilib.sh.DefaultShRuntime;
import net.morilib.sh.ShEnvironment;
import net.morilib.sh.ShFacade;
import net.morilib.sh.ShFileSystem;
import net.morilib.sh.ShProcess;
import net.morilib.sh.ShRootEnvironment;
import net.morilib.sh.ShRuntime;
import net.morilib.sh.ShSyntaxException;
import net.morilib.sh.misc.XtraceStream;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/23
 */
public abstract class AbstractNinaTranslator
implements NinaTranslator {

	/**
	 * 
	 */
	protected NinatOptions options;

	/**
	 * 
	 */
	protected DFA<Object, NFAState, Void> dfa;

	/**
	 * 
	 * @param lang
	 * @return
	 */
	public static NinaTranslator getTranslator(String lang) {
		Class<?> c;
		String p;

		try {
			p = AbstractNinaTranslator.class.getPackage().getName();
			c = Class.forName(p + ".NinaTranslator" + lang);
			return (NinaTranslator)c.newInstance();
		} catch(ClassNotFoundException e) {
			return null;
		} catch(InstantiationException e) {
			throw new RuntimeException(e);
		} catch(IllegalAccessException e) {
			throw new RuntimeException(e);
		}
	}

	/**
	 * 
	 * @param dfa
	 */
	protected abstract void printStates(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	protected abstract void printAcceptStates(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	protected abstract void printAcceptToken(PrintStream out);

	/**
	 * 
	 * @param dfa
	 */
	protected abstract InputStream openScript() throws IOException;

	/**
	 * 
	 * @param dfa
	 */
	protected abstract PrintStream openOutput() throws IOException;

	/* (non-Javadoc)
	 * @see net.morilib.nina.compiler.NinaTranslater#translate(java.io.Reader, java.io.PrintWriter)
	 */
	@Override
	public void translate(Reader rd,
			NinatOptions opts) throws IOException {
		PrintStream ous = null;
		InputStream ins = null;
		NinaNFA n;
		char[] a;
		ShNinatBuiltInCommands cmd;
		ShEnvironment env;
		ShFileSystem fs;
		XtraceStream qs;
		ShRuntime run;

		n   = Nina.compile(rd);
		dfa = ConvertedRangeDFA.convertDFA(n);
		options = opts;

		try {
			ins  = openScript();
			ous  = openOutput();
			env  = new ShRootEnvironment();
			a    = opts.getFilename().toCharArray();
			a[0] = Character.toUpperCase(a[0]);
			env.bind("CLASSNAME", new String(a));
			fs   = new ShNinatFileSystem();
			cmd  = new ShNinatBuiltInCommands();
			cmd.putCommand("print_states", new ShProcess() {

				@Override
				public int main(ShEnvironment env, ShFileSystem fs,
						InputStream stdin, PrintStream stdout,
						PrintStream stderr,
						String... args) throws IOException {
					printStates(stdout);
					return 0;
				}

			});
			cmd.putCommand("print_accepts", new ShProcess() {

				@Override
				public int main(ShEnvironment env, ShFileSystem fs,
						InputStream stdin, PrintStream stdout,
						PrintStream stderr,
						String... args) throws IOException {
					printAcceptStates(stdout);
					return 0;
				}

			});
			cmd.putCommand("print_token", new ShProcess() {

				@Override
				public int main(ShEnvironment env, ShFileSystem fs,
						InputStream stdin, PrintStream stdout,
						PrintStream stderr,
						String... args) throws IOException {
					printAcceptToken(stdout);
					return 0;
				}

			});
			run  = new DefaultShRuntime(cmd);
			qs   = new XtraceStream(System.out);
			ShFacade.execute(env, fs, cmd, run, ins, System.in, ous,
					System.err, qs);
		} catch(ShSyntaxException e) {
			throw new RuntimeException(e);
		} finally {
			if(ins != null)  ins.close();
			if(ous != null)  ous.close();
		}
	}

}
