/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.util.Collections;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;

import net.morilib.automata.NFA;
import net.morilib.automata.NFAEdges;
import net.morilib.automata.NFAState;
import net.morilib.automata.TextBound;
import net.morilib.range.Interval;
import net.morilib.range.Range;

public class NinaSingleNFA
implements NFA<Object, NFAState, Void>, NFAEdges<Object> {

	NFAState initial;
	NinaMap<Object, Set<NFAState>> map;

	NinaSingleNFA(NFAState init) {
		initial = init;
		map = new NinaMap<Object, Set<NFAState>>();
	}

	@Override
	public boolean isState(NFAState o) {
		return initial.equals(o);
	}

	@Override
	public Set<NFAState> getStates(NFAState state, Object alphabet) {
		Set<NFAState> s;

		if(!initial.equals(state) || (s = map.get(alphabet)) == null) {
			return Collections.emptySet();
		} else {
			return new HashSet<NFAState>(s);
		}
	}

	@Override
	public Set<NFAState> getStates(NFAState state, Range rng) {
		Set<NFAState> s;

		if(!initial.equals(state) || (s = map.get(rng)) == null) {
			return Collections.emptySet();
		} else {
			return new HashSet<NFAState>(s);
		}
	}

	@Override
	public Set<NFAState> getStates(NFAState state, EnumSet<TextBound> bound) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getStatesEpsilon(NFAState state) {
		Set<NFAState> s;

		if(!initial.equals(state)) {
			return Collections.emptySet();
		} else if((s = map.get(null)) == null) {
			return Collections.emptySet();
		} else {
			return new HashSet<NFAState>(s);
		}
	}

	@Override
	public Set<NFAState> getStatesBound(NFAState state,
			EnumSet<TextBound> bound) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getInitialStates() {
		return Collections.singleton(initial);
	}

	@Override
	public boolean isInitialState(NFAState o) {
		return initial.equals(o);
	}

	@Override
	public boolean isFinal(NFAState state) {
		return getAcceptedStates().contains(state);
	}

	@Override
	public boolean isFinalAny(Set<NFAState> states) {
		Set<NFAState> t = new HashSet<NFAState>(states);

		t.retainAll(getAcceptedStates());
		return !t.isEmpty();
	}

	@Override
	public NFAEdges<Object> getEdges(NFAState state) {
		return this;
	}

	@Override
	public Set<Interval> nextAlphabets(NFAState state) {
		if(initial.equals(state)) {
			return map.getKeyRange().intervals();
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Iterable<Interval> nextAlphabets(Set<NFAState> states) {
		if(states.contains(initial)) {
			return map.getKeyRange().intervals();
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Set<Object> nextDiscreteAlphabets(NFAState state) {
		if(initial.equals(state)) {
			return map.getDiscreteKeys();
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Iterable<Object> nextDiscreteAlphabets(
			Set<NFAState> states) {
		if(states.contains(initial)) {
			return map.getDiscreteKeys();
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public Set<NFAState> getAcceptedStates() {
		Set<NFAState> r = new HashSet<NFAState>();

		for(Set<NFAState> s : map.valueSet())  r.addAll(s);
		return r;
	}

	@Override
	public Set<Void> getMatchTag(NFAState state) {
		return Collections.emptySet();
	}

	@Override
	public Set<Void> getMatchTagEnd(NFAState state) {
		return Collections.emptySet();
	}

	@Override
	public Set<NFAState> getAccept(NFAState state) {
		if(isAccepted(state)) {
			return Collections.singleton(state);
		} else {
			return Collections.emptySet();
		}
	}

	@Override
	public boolean isAccepted(NFAState state) {
		return getAcceptedStates().contains(state);
	}

	@Override
	public Set<NFAState> goNext(Object alphabet) {
		return map.get(alphabet);
	}

	@Override
	public Set<NFAState> goNext(int alphabet) {
		return map.get(Integer.valueOf(alphabet));
	}

	@Override
	public Set<NFAState> goNext(char alphabet) {
		return map.get(Character.valueOf(alphabet));
	}

	@Override
	public Set<NFAState> goNextEpsilon() {
		return map.get(Range.O);
	}

	@Override
	public Set<? extends Range> nextAlphabets() {
		return map.getKeyRange().intervals();
	}

	@Override
	public boolean isNextEpsilon() {
		return map.get(Range.O) != null;
	}

	public String toString() {
		return map.toString();
	}

}
