/*
 * Copyright 2013 Yuichiro Moriguchi
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package net.morilib.nina;

import java.io.BufferedReader;
import java.io.IOException;
import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import net.morilib.automata.DFA;
import net.morilib.automata.DFAState;
import net.morilib.automata.TextBound;
import net.morilib.automata.dfa.DFAs;
import net.morilib.range.Interval;
import net.morilib.range.Range;
import net.morilib.util.TreeSectionMap;

/**
 *
 *
 * @author MORIGUCHI, Yuichiro 2013/10/14
 */
public class DFABuilder implements NinaAction {

	static class DBS implements DFAState<Object, Object, Void> {

		TreeSectionMap<Interval, Object, DBS> edges;
		boolean initial, accepted;

		void linkAlphabet(DBS d, Object o) {
			if(edges == null) {
				edges = new TreeSectionMap<Interval, Object, DBS>(
						Interval.newPoint(o), d);
			} else if(edges.map(o) == null) {
				edges.insert(Interval.newPoint(o), d);
			} else {
				throw new NinaNondeterministicException();
			}
		}

		void linkAlphabet(DBS d, Range o) {
			for(Interval v : o.intervals()) {
				if(edges == null) {
					edges = new TreeSectionMap<Interval, Object, DBS>(
							v, d);
				} else {
					for(Interval w : edges.keys()) {
						if(!o.independentOf(w)) {
							throw new NinaNondeterministicException();
						}
					}
					edges.insert(v, d);
				}
			}
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#go(java.lang.Object)
		 */
		@Override
		public DFAState<Object, Object, Void> go(Object a) {
			DFAState<Object, Object, Void> d;

			d = edges != null ? edges.map(a) : null;
			return d != null ?
					d : DFAs.<Object, Object, Void>deadState();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#goInt(int)
		 */
		@Override
		public DFAState<Object, Object, Void> goInt(int x) {
			return go(Integer.valueOf(x));
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#goChar(char)
		 */
		@Override
		public DFAState<Object, Object, Void> goChar(char x) {
			return go(Integer.valueOf(x));
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#goBound(net.morilib.automata.TextBound)
		 */
		@Override
		public DFAState<Object, Object, Void> goBound(TextBound b) {
			return DFAs.deadState();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#isInitialState()
		 */
		@Override
		public boolean isInitialState() {
			return initial;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#getAccepted()
		 */
		@Override
		public Set<Object> getAccepted() {
			return accepted ?
					Collections.<Object>singleton(this) : Collections.emptySet();
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#isDead()
		 */
		@Override
		public boolean isDead() {
			return edges == null;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#isAccepted()
		 */
		@Override
		public boolean isAccepted() {
			return accepted;
		}

		/* (non-Javadoc)
		 * @see net.morilib.automata.DFAState#getAlphabets()
		 */
		@Override
		public Set<Object> getAlphabets() {
			Set<Object> r = new HashSet<Object>();
			Object a, b;
			int x;

			for(Interval v : edges.keys()) {
				a = v.getInfimumBound();
				b = v.getSupremumBound();
				if(a.equals(b)) {
					r.add(a);
				} else if(a instanceof Integer) {
					x = ((Integer)a).intValue();
					for(int i = x; v.contains(i); i++) {
						r.add(Integer.valueOf(i));
					}
				} else {
					throw new NinaParseException();
				}
			}
			return r;
		}

	};

	private DBS initial, vertex;
	private String label;

	private DBS prendEtat() {
		DBS s;

		s = new DBS();
		return s;
	}
	
	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#labelAdded(net.morilib.nina.NinaEvent, java.lang.String, boolean)
	 */
	@Override
	public void labelAdded(NinaEvent q, NinaFrameReader rd,
			boolean accept) {
		BufferedReader br = new BufferedReader(rd);

		try {
			label = br.readLine();
		} catch (IOException e) {
			throw new RuntimeException(e);
		}

		if(vertex == null) {
			// set initial state if it is not set
			vertex = prendEtat();
			vertex.initial = true;
			initial = vertex;
		}
		vertex.accepted = accept;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#link(net.morilib.nina.NinaEvent, java.lang.Object)
	 */
	@Override
	public void link(NinaEvent q, Object ver) {
		DBS v = vertex;

		vertex = (ver != null) ? (DBS)ver : prendEtat();
		if(q.getEdge() instanceof Range) {
			v.linkAlphabet(vertex, (Range)q.getEdge());
		} else {
			v.linkAlphabet(vertex, q.getEdge());
		}
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getLabel()
	 */
	@Override
	public String getLabel() {
		return label;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#getVertex()
	 */
	@Override
	public Object getVertex() {
		return vertex;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setVertex(java.lang.Object)
	 */
	@Override
	public void setVertex(Object o) {
		vertex = (DBS)o;
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#doneBlockSearching(net.morilib.nina.NinaEvent)
	 */
	@Override
	public void doneBlockSearching(NinaEvent q) {
		vertex = (DBS)q.getScratch();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdge(net.morilib.nina.NinaEvent, java.lang.Object)
	 */
	@Override
	public void setEdge(NinaEvent q, Object o) {
		q.setEdge(o);
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#setEdgeResource(net.morilib.nina.NinaEvent, net.morilib.nina.NinaParser, java.lang.String)
	 */
	@Override
	public void setEdgeResource(NinaEvent q, NinaParser p, String s) {
		throw new NinaNondeterministicException();
	}

	/* (non-Javadoc)
	 * @see net.morilib.nina.NinaAction#accept()
	 */
	@Override
	public Object accept() {
		return new DFA<Object, Object, Void>() {

			@Override
			public DFAState<Object, Object, Void> getInitialState() {
				return initial;
			}

		};
	}

}
