/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import java.util.List;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link StaticInvokeMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class StaticInvokeMetaDataTest{

    @DataProvider(name = "StaticInvokeMetaData")
    public Object[][] dataStaticInvokeMetaData(){
        return new Object[][]{
            {null},
            {new AttributeMetaData(null)},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.StaticInvokeMetaData#StaticInvokeMetaData(jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#.*"},
          dataProvider = "StaticInvokeMetaData")
    public void testStaticInvokeMetaData(MetaData parent){
        StaticInvokeMetaData metaData = new StaticInvokeMetaData(parent);
        assertEquals(metaData.getParent(), parent);
        assertNull(metaData.getCode());
        assertNull(metaData.getName());
        assertEquals(metaData.getCallState(), Service.State.CREATED.toString());
        assertEquals(metaData.getCallStateValue(), Service.State.CREATED);
        assertEquals(metaData.getArguments().size(), 0);
        assertNull(metaData.getTarget());
    }

    @DataProvider(name = "SetCode")
    public Object[][] dataSetCode(){
        return new Object[][]{
            {null},
            {"test.Hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.StaticInvokeMetaData#setCode(java.lang.String)",
                    "jp.ossc.nimbus.core.StaticInvokeMetaData#getCode()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.StaticInvokeMetaData#StaticInvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetCode")
    public void testSetCode(String code){
        StaticInvokeMetaData metaData = new StaticInvokeMetaData(null);
        metaData.setCode(code);
        assertEquals(metaData.getCode(), code);
    }
    
    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\"/>"), new Object[]{"test.Hoge", "fuga", new MetaData[0]}},
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\"></static-invoke>"), new Object[]{"test.Hoge", "fuga", new MetaData[0]}},
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\"><argument>hoge</argument></static-invoke>"), new Object[]{"test.Hoge", "fuga", new MetaData[]{importXML(new ArgumentMetaData(null, null), "<argument>hoge</argument>")}}},
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\"><argument type=\"int\">0</argument><argument type=\"int\">10</argument></static-invoke>"), new Object[]{"test.Hoge", "fuga", new MetaData[]{importXML(new ArgumentMetaData(null, null), "<argument type=\"int\">0</argument>"), importXML(new ArgumentMetaData(null, null), "<argument type=\"int\">10</argument>")}}},
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\" unknown=\"hoge\"/>"), new Object[]{"test.Hoge", "fuga", new MetaData[0]}},
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\"><unknown/></static-invoke>"), new Object[]{"test.Hoge", "fuga", new MetaData[0]}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.StaticInvokeMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#ArgumentMetaData\\(.*\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.ArgumentMetaData#toXML\\(java\\.lang\\.StringBuilder\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.StaticInvokeMetaData#StaticInvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.StaticInvokeMetaData#getCode\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        StaticInvokeMetaData metaData = new StaticInvokeMetaData(null);
        metaData.importXML(element);
        assertEquals(metaData.getCode(), assertValues[0]);
        assertEquals(metaData.getName(), assertValues[1]);
        assertEquals(metaData.getCallState(), Service.State.CREATED.toString());
        assertEquals(metaData.getCallStateValue(), Service.State.CREATED);
        assertNull(metaData.getTarget());
        List<ArgumentMetaData> arguments = metaData.getArguments();
        MetaData[] args = (MetaData[])assertValues[2];
        if(args.length == 0){
            assertTrue(arguments.size() == 0);
        }else{
            assertEquals(arguments.size(), args.length);
            for(int i = 0; i < arguments.size(); i++){
                assertMetaDataEquals(arguments.get(i), args[i]);
            }
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<static-invoke/>")},
            {createElement("<static-invoke code=\"test.Hoge\"/>")},
            {createElement("<static-invoke name=\"fuga\"/>")},
            {createElement("<static-invoke code=\"\" name=\"fuga\"/>")},
            {createElement("<static-invoke code=\"test.Hoge\" name=\"\"/>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.StaticInvokeMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.StaticInvokeMetaData#StaticInvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        StaticInvokeMetaData metaData = new StaticInvokeMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\"/>"), "<static-invoke code=\"test.Hoge\" name=\"fuga\"/>"},
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\"><argument>hoge</argument></static-invoke>"), "<static-invoke code=\"test.Hoge\" name=\"fuga\">" + System.getProperty("line.separator") + "    <argument>hoge</argument>" + System.getProperty("line.separator") + "</static-invoke>"},
            {createElement("<static-invoke code=\"test.Hoge\" name=\"fuga\"><argument type=\"int\">0</argument><argument type=\"int\">10</argument></static-invoke>"), "<static-invoke code=\"test.Hoge\" name=\"fuga\">" + System.getProperty("line.separator") + "    <argument type=\"int\">0</argument>" + System.getProperty("line.separator") + "    <argument type=\"int\">10</argument>" + System.getProperty("line.separator") + "</static-invoke>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.StaticInvokeMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.InvokeMetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.StaticInvokeMetaData#StaticInvokeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.StaticInvokeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        StaticInvokeMetaData metaData = new StaticInvokeMetaData(null);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}