/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import static org.testng.Assert.*;

import org.testng.annotations.Test;
import org.testng.annotations.DataProvider;
import org.w3c.dom.Element;

import static jp.ossc.nimbus.TestUtility.*;

/**
 * {@link AttributeMetaData}eXgP[XB<p>
 * 
 * @author M.Takata
 */
public class AttributeMetaDataTest{

    @DataProvider(name = "AttributeMetaData")
    public Object[][] dataAttributeMetaData(){
        return new Object[][]{
            {null},
            {new ObjectMetaData()},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.AttributeMetaData#AttributeMetaData(jp.ossc.nimbus.core.MetaData)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*"},
          dataProvider = "AttributeMetaData")
    public void testAttributeMetaData(MetaData parent){
        AttributeMetaData metaData = new AttributeMetaData(parent);
        assertEquals(metaData.getParent(), parent);
    }

    @DataProvider(name = "SetName")
    public Object[][] dataSetName(){
        return new Object[][]{
            {null},
            {"Hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.AttributeMetaData#setName(java.lang.String)",
                    "jp.ossc.nimbus.core.AttributeMetaData#getName()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetName")
    public void testSetName(String name){
        AttributeMetaData metaData = new AttributeMetaData(null);
        metaData.setName(name);
        assertEquals(metaData.getName(), name);
    }

    @DataProvider(name = "SetType")
    public Object[][] dataSetType(){
        return new Object[][]{
            {null},
            {"java.lang.String"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.AttributeMetaData#setType(java.lang.String)",
                    "jp.ossc.nimbus.core.AttributeMetaData#getType()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetType")
    public void testSetType(String type){
        AttributeMetaData metaData = new AttributeMetaData(null);
        metaData.setType(type);
        assertEquals(metaData.getType(), type);
    }

    @DataProvider(name = "SetValue")
    public Object[][] dataSetValue(){
        return new Object[][]{
            {null},
            {"hoge"}
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.AttributeMetaData#setValue(java.lang.Object)",
                    "jp.ossc.nimbus.core.AttributeMetaData#getValue()",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          dataProvider = "SetValue")
    public void testSetValue(Object value){
        AttributeMetaData metaData = new AttributeMetaData(null);
        metaData.setValue(value);
        assertEquals(metaData.getValue(), value);
    }

    @DataProvider(name = "ImportXML")
    public Object[][] dataImportXML() throws Exception{
        return new Object[][]{
            {createElement("<attribute name=\"Hoge\"/>"), new Object[]{"Hoge", null, ""}},
            {createElement("<attribute name=\"Hoge\"></attribute>"), new Object[]{"Hoge", null, ""}},
            {createElement("<attribute name=\"Hoge\" type=\"java.lang.String\"></attribute>"), new Object[]{"Hoge", "java.lang.String", ""}},
            {createElement("<attribute name=\"Hoge\">hoge</attribute>"), new Object[]{"Hoge", null, "hoge"}},
            {createElement("<attribute name=\"Hoge\" type=\"org.w3c.dom.Element\"><hoge/></attribute>"), new Object[]{"Hoge", "org.w3c.dom.Element", createElement("<hoge/>")}},
            {createElement("<attribute name=\"Hoge\"><service-ref>hoge</service-ref></attribute>"), new Object[]{"Hoge", null, importXML(new ServiceRefMetaData(null), "<service-ref>hoge</service-ref>")}},
            {createElement("<attribute name=\"Hoge\"><object code=\"java.lang.String\"/></attribute>"), new Object[]{"Hoge", null, importXML(new ObjectMetaData(), "<object code=\"java.lang.String\"/>")}},
            {createElement("<attribute name=\"Hoge\"><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></attribute>"), new Object[]{"Hoge", null, importXML(new StaticInvokeMetaData(null), "<static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/>")}},
            {createElement("<attribute name=\"Hoge\"><static-field-ref code=\"java.lang.Integer\" name=\"MAX_VALUE\"/></attribute>"), new Object[]{"Hoge", null, importXML(new StaticFieldRefMetaData(null), "<static-field-ref code=\"java.lang.Integer\" name=\"MAX_VALUE\"/>")}},
            {createElement("<attribute name=\"Hoge\"><unknown/></attribute>"), new Object[]{"Hoge", null, ""}},
            {createElement("<attribute name=\"Hoge\" unknown=\"hoge\"/>"), new Object[]{"Hoge", null, ""}},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.AttributeMetaData#importXML(org.w3c.dom.Element)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#getName\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#getType\\(\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#getValue\\(\\)"},
          dataProvider = "ImportXML")
    public void testImportXML(
        Element element,
        Object[] assertValues
    ) throws DeploymentException{
        AttributeMetaData metaData = new AttributeMetaData(null);
        metaData.importXML(element);
        assertEquals(metaData.getName(), assertValues[0]);
        assertEquals(metaData.getType(), assertValues[1]);
        if(metaData.getValue() instanceof Element
                && assertValues[2] instanceof Element){
            assertEquals(((Element)metaData.getValue()).getTagName(), ((Element)assertValues[2]).getTagName());
        }else if(metaData.getValue() instanceof MetaData
                && assertValues[2] instanceof MetaData){
            assertMetaDataEquals((MetaData)metaData.getValue(), (MetaData)assertValues[2]);
        }else{
            assertEquals(metaData.getValue(), assertValues[2]);
        }
    }

    @DataProvider(name = "ImportXMLError")
    public Object[][] dataImportXMLError() throws Exception{
        return new Object[][]{
            {null},
            {createElement("<hoge/>")},
            {createElement("<attribute/>")},
            {createElement("<attribute name=\"\"/>")},
            {createElement("<attribute name=\"Hoge\" type=\"\"/>")},
            {createElement("<attribute name=\"Hoge\" type=\"org.w3c.dom.Element\"><hoge/><fuga/></attribute>")},
            {createElement("<attribute name=\"Hoge\"><service-ref>hoge</service-ref><service-ref>hoge</service-ref></attribute>")},
            {createElement("<attribute name=\"Hoge\"><object code=\"java.lang.String\"/><object code=\"java.lang.String\"/></attribute>")},
            {createElement("<attribute name=\"Hoge\"><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/><static-invoke code=\"java.util.Calendar\" name=\"getInstance\"/></attribute>")},
            {createElement("<attribute name=\"Hoge\"><static-field-ref code=\"java.lang.Integer\" name=\"MAX_VALUE\"/><static-field-ref code=\"java.lang.Integer\" name=\"MAX_VALUE\"/></attribute>")},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.AttributeMetaData#importXML(org.w3c.dom.Element)",
                    "Error"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)"},
          expectedExceptions = {NullPointerException.class, DeploymentException.class},
          dataProvider = "ImportXMLError")
    public void testImportXMLError(Element element) throws DeploymentException{
        AttributeMetaData metaData = new AttributeMetaData(null);
        metaData.importXML(element);
    }

    @DataProvider(name = "ToXML")
    public Object[][] dataToXML() throws Exception{
        return new Object[][]{
            {createElement("<attribute name=\"Hoge\"/>"), "<attribute name=\"Hoge\"></attribute>"},
            {createElement("<attribute name=\"Hoge\"></attribute>"), "<attribute name=\"Hoge\"></attribute>"},
            {createElement("<attribute name=\"Hoge\">fuga</attribute>"), "<attribute name=\"Hoge\">fuga</attribute>"},
            {createElement("<attribute name=\"Hoge\" type=\"java.lang.String\">hoge</attribute>"), "<attribute name=\"Hoge\" type=\"java.lang.String\">hoge</attribute>"},
            {createElement("<attribute name=\"Hoge\">hoge\nfuga</attribute>"), "<attribute name=\"Hoge\">" + System.getProperty("line.separator") + "    hoge" + System.getProperty("line.separator") + "    fuga" + System.getProperty("line.separator") + "</attribute>"},
            {createElement("<attribute name=\"Hoge\"><service-ref>hoge</service-ref></attribute>"), "<attribute name=\"Hoge\">" + System.getProperty("line.separator") + "    <service-ref>Nimbus#hoge</service-ref>" + System.getProperty("line.separator") + "</attribute>"},
            {createElement("<attribute name=\"Hoge\"><unknown/></attribute>"), "<attribute name=\"Hoge\"></attribute>"},
        };
    }
    @Test(groups = {"jp.ossc.nimbus.core.AttributeMetaData#toXML(java.lang.StringBuilder)",
                    "Normal"},
          dependsOnGroups = {"jp\\.ossc\\.nimbus\\.core\\.MetaData#.*",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#AttributeMetaData\\(jp\\.ossc\\.nimbus\\.core\\.MetaData\\)",
                             "jp\\.ossc\\.nimbus\\.core\\.AttributeMetaData#importXML\\(org\\.w3c\\.dom\\.Element\\)"},
          dataProvider = "ToXML")
    public void testToXML(Element element, String expected) throws DeploymentException{
        AttributeMetaData metaData = new AttributeMetaData(null);
        metaData.importXML(element);
        StringBuilder buf = new StringBuilder();
        metaData.toXML(buf);
        assertEquals(buf.toString(), expected);
    }
}
