/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.util.converter;

import java.io.*;
import java.util.*;

import jp.ossc.nimbus.beans.dataset.*;
import jp.ossc.nimbus.io.*;

/**
 * R[hXgCSVRo[^B<p>
 * AAlXgR[hXǵAT|[gȂB<br>
 * 
 * @author M.Takata
 */
public class RecordListCSVConverter implements BindingStreamConverter, StreamStringConverter{
    
    /**
     * R[hXgCSV\ϊʒ萔B<p>
     */
    public static final int RECORDLIST_TO_CSV = OBJECT_TO_STREAM;
    
    /**
     * CSVR[hXg\ϊʒ萔B<p>
     */
    public static final int CSV_TO_RECORDLIST = STREAM_TO_OBJECT;
    
    /**
     * ϊʁB<p>
     */
    protected int convertType;
    
    /**
     * R[hXgCSVϊɎgp镶GR[fBOB<p>
     */
    protected String characterEncodingToStream;
    
    /**
     * CSVR[hXgϊɎgp镶GR[fBOB<p>
     */
    protected String characterEncodingToObject;
    
    /**
     * XL[}`肩ǂ̃tOB<p>
     * R[hXgCSVϊsۂɁACSVɃXL[}`邩ǂ킷BtruȅꍇAXL[}`BftHǵAfalseB<br>
     */
    protected boolean isExistsSchema;
    
    /**
     * CSVwb_肩ǂ̃tOB<p>
     * R[hXgCSVϊsۂɁACSVɃwb_邩ǂ킷BtruȅꍇAwb_BftHǵAfalseB<br>
     */
    protected boolean isExistsHeader;
    
    /**
     * XL[}ɑ݂ȂvpeB𖳎邩ǂ̃tOB<p>
     * ftHǵAfalseŁAϊG[ƂB<br>
     */
    protected boolean isIgnoreUnknownProperty;
    
    protected CSVWriter writer;
    private CSVReader reader;
    
    /**
     * R[hXgCSVϊsRo[^𐶐B<p>
     */
    public RecordListCSVConverter(){
        this(RECORDLIST_TO_CSV);
    }
    
    /**
     * w肳ꂽϊʂ̃Ro[^𐶐B<p>
     *
     * @param type ϊ
     * @see #RECORDLIST_TO_CSV
     * @see #CSV_TO_RECORDLIST
     */
    public RecordListCSVConverter(int type){
        convertType = type;
    }
    
    /**
     * ϊʂݒ肷B<p>
     *
     * @param type ϊ
     * @see #RECORDLIST_TO_CSV
     * @see #CSV_TO_RECORDLIST
     */
    @Override
    public void setConvertType(int type){
        convertType = type;
    }
    
    /**
     * ϊʂ擾B<p>
     *
     * @return ϊ
     * @see #setConvertType(int)
     */
    public int getConvertType(){
        return convertType;
    }
    
    /**
     * R[hXgCSVϊɎgp镶GR[fBOݒ肷B<p>
     * 
     * @param encoding GR[fBO
     */
    public void setCharacterEncodingToStream(String encoding){
        characterEncodingToStream = encoding;
    }
    
    /**
     * R[hXgCSVϊɎgp镶GR[fBO擾B<p>
     * 
     * @return GR[fBO
     */
    public String getCharacterEncodingToStream(){
        return characterEncodingToStream;
    }
    
    /**
     * CSVR[hXgϊɎgp镶GR[fBOݒ肷B<p>
     * 
     * @param encoding GR[fBO
     */
    public void setCharacterEncodingToObject(String encoding){
        characterEncodingToObject = encoding;
    }
    
    /**
     * CSVR[hXgϊɎgp镶GR[fBO擾B<p>
     * 
     * @return GR[fBO
     */
    public String getCharacterEncodingToObject(){
        return characterEncodingToObject;
    }
    
    /**
     * R[hXgCSVϊsۂɁACSVɃXL[}`邩ǂݒ肷B<p>
     * ftHǵAfalseB<br>
     *
     * @param isExists XL[}`ꍇtrue
     */
    public void setExistsSchema(boolean isExists){
        isExistsSchema = isExists;
    }
    
    /**
     * R[hXgCSVϊsۂɁACSVɃXL[}`邩ǂ𔻒肷B<p>
     *
     * @return truȅꍇXL[}`
     */
    public boolean isExistsSchema(){
        return isExistsSchema;
    }
    
    /**
     * R[hXgCSVϊsۂɁACSVɃwb_邩ǂݒ肷B<p>
     * ftHǵAfalseB<br>
     *
     * @param isExists wb_ꍇtrue
     */
    public void setExistsHeader(boolean isExists){
        isExistsHeader = isExists;
    }
    
    /**
     * R[hXgCSVϊsۂɁACSVɃwb_邩ǂ𔻒肷B<p>
     *
     * @return truȅꍇwb_
     */
    public boolean isExistsHeader(){
        return isExistsHeader;
    }
    
    /**
     * XL[}ɑ݂ȂvpeB𖳎邩ǂݒ肷B<p>
     * ftHǵAfalseŁAϊG[ƂȂB<br>
     * 
     * @param isIgnore truȅꍇA
     */
    public void setIgnoreUnknownProperty(boolean isIgnore){
        isIgnoreUnknownProperty = isIgnore;
    }
    
    /**
     * XL[}ɑ݂ȂvpeB𖳎邩ǂ𔻒肷B<p>
     * 
     * @return truȅꍇA
     */
    public boolean isIgnoreUnknownProperty(){
        return isIgnoreUnknownProperty;
    }
    
    /**
     * CSV`̕oCSVWriter̃ev[gݒ肷B<p>
     * 
     * @param writer CSVWriter
     */
    public void setCSVWriter(CSVWriter writer){
        this.writer = writer;
    }

    /**
     * CSV`̕oCSVWriter̃ev[g擾B<p>
     * 
     * @return CSVWriter
     */
    public CSVWriter getCSVWriter(){
        return writer;
    }

    /**
     * CSV`̕ǂݍCSVReader̃ev[gݒ肷B<p>
     * 
     * @param reader CSVReader
     */
    public void setReader(CSVReader reader){
        this.reader = reader;
    }

    /**
     * CSV`̕ǂݍCSVReader̃ev[g擾B<p>
     * 
     * @return CSVReader
     */
    public CSVReader getReader(){
        return reader;
    }

    /**
     * w肳ꂽIuWFNgϊB<p>
     *
     * @param obj ϊΏۂ̃IuWFNg
     * @return ϊ̃IuWFNg
     * @exception ConvertException ϊɎsꍇ
     */
    @Override
    public Object convert(Object obj) throws ConvertException{
        if(obj == null){
            return null;
        }
        switch(convertType){
        case RECORDLIST_TO_CSV:
            return convertToStream(obj);
        case CSV_TO_RECORDLIST:
            if(obj instanceof File){
                return toRecordList((File)obj);
            }else if(obj instanceof InputStream){
                return toRecordList((InputStream)obj);
            }else{
                throw new ConvertException(
                    "Invalid input type : " + obj.getClass()
                );
            }
        default:
            throw new ConvertException(
                "Invalid convert type : " + convertType
            );
        }
    }
    
    /**
     * R[hXgCSVXg[֕ϊB<p>
     *
     * @param obj R[hXg
     * @return ϊʂǂݎ̓Xg[
     * @exception ConvertException ϊɎsꍇ
     */
    @Override
    public InputStream convertToStream(Object obj) throws ConvertException{
        if(!(obj instanceof RecordList)){
            throw new ConvertException("Input is not RecordList." + obj);
        }
        RecordList recList = (RecordList)obj;
        RecordSchema schema = recList.getRecordSchema();
        if(schema == null){
            throw new ConvertException("Schema is null." + recList);
        }
        ByteArrayOutputStream baos = new ByteArrayOutputStream();
        try{
            OutputStreamWriter osw = null;
            if(characterEncodingToStream == null){
                osw = new OutputStreamWriter(baos);
            }else{
                osw = new OutputStreamWriter(baos, characterEncodingToStream);
            }
            CSVWriter writer = this.writer == null ? new CSVWriter() : this.writer.cloneWriter();
            writer.setWriter(osw);
            if(isExistsSchema){
                writer.writeElement(schema.getSchema());
                writer.newLine();
            }else if(isExistsHeader){
                for(int i = 0, imax = schema.getPropertySize(); i < imax; i++){
                    writer.writeElement(schema.getPropertyName(i));
                }
                writer.newLine();
            }
            for(int i = 0, imax = recList.size(); i < imax; i++){
                Record record = recList.getRecord(i);
                if(record != null){
                    for(int j = 0, jmax = schema.getPropertySize(); j < jmax; j++){
                        writer.writeElement(record.getFormatProperty(j));
                    }
                    if(i != imax - 1){
                        writer.newLine();
                    }
                }
            }
            writer.close();
        }catch(IOException e){
            throw new ConvertException(e);
        }catch(DataSetException e){
            throw new ConvertException(e);
        }
        return new ByteArrayInputStream(baos.toByteArray());
    }
    
    /**
     * CSVXg[烌R[hXg֕ϊB<p>
     *
     * @param is ̓Xg[
     * @return R[hXg
     * @exception ConvertException ϊɎsꍇ
     */
    @Override
    public Object convertToObject(InputStream is) throws ConvertException{
        return toRecordList(is);
    }
    
    protected RecordList toRecordList(File file) throws ConvertException{
        try{
            return toRecordList(new FileInputStream(file));
        }catch(IOException e){
            throw new ConvertException(e);
        }
    }
    
    protected RecordList toRecordList(InputStream is) throws ConvertException{
        return toRecordList(is, null);
    }
    
    protected RecordList toRecordList(InputStream is, RecordList recList)
     throws ConvertException{
        CSVReader reader = null;
        try{
            InputStreamReader isr = null;
            if(characterEncodingToObject == null){
                isr = new InputStreamReader(is);
            }else{
                isr = new InputStreamReader(is, characterEncodingToObject);
            }
            reader = this.reader == null ? new CSVReader() : this.reader.cloneReader();
            reader.setReader(isr);
            if(recList == null){
                recList = new RecordList();
            }
            RecordSchema schema = recList.getRecordSchema();
            List<String> csv = new ArrayList<String>();
            List<String> propertyNames = new ArrayList<String>();
            if(isExistsSchema){
                csv = reader.readCSVLineList(csv);
                if(csv == null){
                    return recList;
                }
                if(csv.size() != 0){
                    if(schema == null){
                        recList.setSchema((String)csv.get(0));
                        schema = recList.getRecordSchema();
                    }else{
                        schema = RecordSchema.getInstance((String)csv.get(0));
                    }
                }
            }else{
                if(schema == null || isExistsHeader){
                    csv = reader.readCSVLineList(csv);
                    if(csv == null){
                        return recList;
                    }
                    final StringBuilder schemaBuf = new StringBuilder();
                    for(int i = 0, imax = csv.size(); i < imax; i++){
                        schemaBuf.append(':');
                        if(isExistsHeader){
                            schemaBuf.append(csv.get(i));
                        }else{
                            schemaBuf.append(i);
                        }
                        schemaBuf.append(',');
                        schemaBuf.append(String.class.getName());
                        if(i != imax - 1){
                            schemaBuf.append('\n');
                        }
                    }
                    if(schema == null){
                        recList.setSchema(schemaBuf.toString());
                        schema = recList.getRecordSchema();
                    }else if(isExistsHeader){
                        schema = RecordSchema.getInstance(schemaBuf.toString());
                    }
                    if(!isExistsHeader){
                        Record record = recList.createRecord();
                        for(int i = 0, imax = csv.size(); i < imax; i++){
                            record.setProperty(i, csv.get(i));
                        }
                        recList.addRecord(record);
                    }
                }
            }
            for(int i = 0, imax = schema.getPropertySize(); i < imax; i++){
                propertyNames.add(schema.getPropertyName(i));
            }
            RecordSchema targetSchema = recList.getRecordSchema();
            while((csv = reader.readCSVLineList(csv)) != null){
                Record record = recList.createRecord();
                int size = csv.size();
                for(int i = 0, imax = propertyNames.size(); i < imax; i++){
                    if(i >= size){
                        continue;
                    }
                    String name = (String)propertyNames.get(i);
                    if(targetSchema.getPropertyIndex(name) == -1){
                        if(isIgnoreUnknownProperty){
                            continue;
                        }
                    }
                    record.setParseProperty(
                        name,
                        csv.get(i)
                    );
                }
                recList.addRecord(record);
            }
            reader.close();
        }catch(IOException e){
            throw new ConvertException(e);
        }catch(DataSetException e){
            throw new ConvertException(e);
        }finally{
            if(reader != null){
                try{
                    reader.close();
                }catch(IOException e){}
            }
        }
        return recList;
    }
    
    /**
     * w肳ꂽR[hXg֕ϊB<p>
     * 
     * @param is ̓Xg[
     * @param returnType ϊΏۂ̃R[hXg
     * @return ϊꂽR[hXg
     * @throws ConvertException ϊɎsꍇ
     */
    @Override
    public Object convertToObject(InputStream is, Object returnType)
     throws ConvertException{
        if(returnType != null && !(returnType instanceof RecordList)){
            throw new ConvertException("ReturnType is not RecordList." + returnType);
        }
        return toRecordList(is, (RecordList)returnType);
    }
}
