/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.context;

import java.util.Map;
import java.util.HashMap;
import java.util.Collection;
import java.util.List;
import java.util.ArrayList;
import java.io.Externalizable;
import java.io.IOException;
import java.io.ObjectOutput;
import java.io.ObjectInput;

import jp.ossc.nimbus.beans.dataset.Record;
import jp.ossc.nimbus.beans.dataset.RecordList;
import jp.ossc.nimbus.beans.dataset.RecordSchema;
import jp.ossc.nimbus.beans.dataset.PropertySchemaDefineException;

/**
 * LReLXgp̃R[hXgB<p>
 * XVT|[gB<br>
 *
 * @author M.Takata
 */
public class SharedContextRecordList extends RecordList implements SharedContextValueDifferenceSupport{
    
    protected int updateVersion;
    
    /**
     * `̃R[hXg𐶐B<p>
     */
    public SharedContextRecordList(){
    }
    
    /**
     * `̃R[hXg𐶐B<p>
     *
     * @param name R[h
     */
    public SharedContextRecordList(String name){
        super(name);
    }
    
    /**
     * ̃R[hXg𐶐B<p>
     *
     * @param name R[h
     * @param schema XL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public SharedContextRecordList(String name, String schema)
     throws PropertySchemaDefineException{
        super(name, schema);
    }
    
    /**
     * ̃R[hXg𐶐B<p>
     *
     * @param name R[h
     * @param schema XL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public SharedContextRecordList(String name, RecordSchema schema)
     throws PropertySchemaDefineException{
        super(name, schema);
    }
    
    /**
     * ̃R[hXg𐶐B<p>
     *
     * @param name R[h
     * @param clazz R[hNX
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public SharedContextRecordList(String name, Class<?> clazz)
     throws PropertySchemaDefineException{
        super(name, clazz);
    }
    
    @Override
    public void setUpdateVersion(int version){
        updateVersion = version;
    }
    
    @Override
    public int getUpdateVersion(){
        return updateVersion;
    }
    
    @Override
    public Record createRecord(){
        if(recordClass == null){
            return new SharedContextRecord(recordSchema);
        }else{
            return super.createRecord();
        }
    }
    
    @Override
    public boolean add(Record o){
        if(o != null){
            ((SharedContextRecord)o).setRecordList(this);
            ((SharedContextRecord)o).setIndex(size());
        }
        return super.add(o);
    }
    
    @Override
    public void add(int index, Record element){
        super.add(index, element);
        if(element != null){
            ((SharedContextRecord)element).setRecordList(this);
            ((SharedContextRecord)element).setIndex(index);
        }
        for(int i = index + 1, imax = size(); i < imax; i++){
            SharedContextRecord record = (SharedContextRecord)get(i);
            if(record != null){
                record.setIndex(record.getIndex() + 1);
            }
        }
    }
    
    @Override
    public Record set(int index, Record element){
        Record result = super.set(index, element);
        if(result != null){
            ((SharedContextRecord)result).setRecordList(null);
            ((SharedContextRecord)result).setIndex(-1);
        }
        return result;
    }
    
    @Override
    public boolean remove(Object o){
        boolean result = super.remove(o);
        if(result && o != null){
            ((SharedContextRecord)o).setRecordList(null);
            ((SharedContextRecord)o).setIndex(-1);
        }
        for(int i = 0, imax = size(); i < imax; i++){
            SharedContextRecord record = (SharedContextRecord)get(i);
            if(record != null){
                record.setIndex(i);
            }
        }
        return result;
    }
    
    @Override
    public Record remove(int index){
        Record result = super.remove(index);
        if(result != null){
            ((SharedContextRecord)result).setRecordList(null);
            ((SharedContextRecord)result).setIndex(-1);
        }
        for(int i = index, imax = size(); i < imax; i++){
            SharedContextRecord record = (SharedContextRecord)get(i);
            if(record != null){
                record.setIndex(record.getIndex() - 1);
            }
        }
        return result;
    }
    
    @Override
    public boolean addAll(Collection<? extends Record> c){
        boolean result = super.addAll(c);
        if(result){
            for(int i = 0, imax = size(); i < imax; i++){
                SharedContextRecord record = (SharedContextRecord)get(i);
                if(record != null){
                    record.setRecordList(this);
                    record.setIndex(i);
                }
            }
        }
        return result;
    }
    
    @Override
    public boolean addAll(int index, Collection<? extends Record> c){
        boolean result = super.addAll(index, c);
        if(result){
            for(int i = index, imax = size(); i < imax; i++){
                SharedContextRecord record = (SharedContextRecord)get(i);
                if(record != null){
                    record.setRecordList(this);
                    record.setIndex(i);
                }
            }
        }
        return result;
    }
    
    @Override
    public boolean removeAll(Collection<?> c){
        boolean result = super.removeAll(c);
        if(result){
            for(int i = 0, imax = size(); i < imax; i++){
                SharedContextRecord record = (SharedContextRecord)get(i);
                if(record != null){
                    record.setIndex(i);
                }
            }
        }
        return result;
    }
    
    @Override
    public boolean retainAll(Collection<?> c){
        boolean result = super.retainAll(c);
        if(result){
            for(int i = 0, imax = size(); i < imax; i++){
                SharedContextRecord record = (SharedContextRecord)get(i);
                if(record != null){
                    record.setIndex(i);
                }
            }
        }
        return result;
    }
    
    /**
     * w肳ꂽR[hǉꍇ̍擾B<p>
     *
     * @param record R[h
     * @param diff 
     * @return 
     * @exception SharedContextUpdateException ̎擾Ɏsꍇ
     */
    public SharedContextValueDifference updateAdd(Record record, SharedContextValueDifference diff) throws SharedContextUpdateException{
        if(diff == null){
            diff = new Difference();
        }else if(!(diff instanceof Difference)){
            throw new SharedContextUpdateException("Unsupported type. class=" + diff.getClass().getName());
        }
        ((Difference)diff).add(this, record);
        return diff;
    }
    
    /**
     * w肳ꂽR[hAw肵CfbNXɑ}ꍇ̍擾B<p>
     *
     * @param index CfbNX
     * @param record R[h
     * @param diff 
     * @return 
     * @exception SharedContextUpdateException ̎擾Ɏsꍇ
     */
    public SharedContextValueDifference updateAdd(int index, Record record, SharedContextValueDifference diff) throws SharedContextUpdateException{
        if(diff == null){
            diff = new Difference();
        }else if(!(diff instanceof Difference)){
            throw new SharedContextUpdateException("Unsupported type. class=" + diff.getClass().getName());
        }
        ((Difference)diff).add(this, index, record);
        return diff;
    }
    
    /**
     * w肳ꂽR[hAw肵CfbNX̃R[hƍւꍇ̍擾B<p>
     *
     * @param index CfbNX
     * @param record R[h
     * @param diff 
     * @return 
     * @exception SharedContextUpdateException ̎擾Ɏsꍇ
     */
    public SharedContextValueDifference updateSet(int index, Record record, SharedContextValueDifference diff) throws SharedContextUpdateException{
        if(diff == null){
            diff = new Difference();
        }else if(!(diff instanceof Difference)){
            throw new SharedContextUpdateException("Unsupported type. class=" + diff.getClass().getName());
        }
        ((Difference)diff).set(this, index, record);
        return diff;
    }
    
    /**
     * w肳ꂽCfbNX̃R[h폜ꍇ̍擾B<p>
     *
     * @param index CfbNX
     * @param record R[h
     * @param diff 
     * @return 
     * @exception SharedContextUpdateException ̎擾Ɏsꍇ
     */
    public SharedContextValueDifference updateRemove(int index, SharedContextValueDifference diff) throws SharedContextUpdateException{
        if(diff == null){
            diff = new Difference();
        }else if(!(diff instanceof Difference)){
            throw new SharedContextUpdateException("Unsupported type. class=" + diff.getClass().getName());
        }
        ((Difference)diff).remove(this, index);
        return diff;
    }
    
    public SharedContextValueDifference updateRemove(Record record, SharedContextValueDifference diff) throws SharedContextUpdateException{
        if(diff == null){
            diff = new Difference();
        }else if(!(diff instanceof Difference)){
            throw new SharedContextUpdateException("Unsupported type. class=" + diff.getClass().getName());
        }
        ((Difference)diff).remove(this, record);
        return diff;
    }
    
    public SharedContextValueDifference updateClear(SharedContextValueDifference diff) throws SharedContextUpdateException{
        if(diff == null){
            diff = new Difference();
        }else if(!(diff instanceof Difference)){
            throw new SharedContextUpdateException("Unsupported type. class=" + diff.getClass().getName());
        }
        ((Difference)diff).clear(this);
        return diff;
    }
    
    @Override
    public boolean update(SharedContextValueDifference diff) throws SharedContextUpdateException{
        if(!(diff instanceof Difference)){
            throw new SharedContextUpdateException("Unsupported type. class=" + diff.getClass().getName());
        }
        return ((Difference)diff).updateRecordList(this);
    }
    
    public void writeExternal(ObjectOutput out) throws IOException{
        super.writeExternal(out);
        SharedContextRecord.writeInt(out, updateVersion);
    }
    
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
        super.readExternal(in);
        updateVersion = SharedContextRecord.readInt(in);
    }
    
    /**
     * R[hXgB<p>
     *
     * @author M.Takata
     */
    public static class Difference implements SharedContextValueDifference, Externalizable{
        private int updateVersion;
        private Map<Integer,SharedContextRecord.Difference> recordDiffMap;
        private List<Transaction> transactionList;
        
        @Override
        public int getUpdateVersion(){
            return updateVersion;
        }
        
        /**
         * R[hǉXVi[B<p>
         *
         * @param list R[hXg
         * @param record ǉ郌R[h
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void add(SharedContextRecordList list, Record record) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            transactionList.add(new AddTransaction(record));
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * R[hǉXVi[B<p>
         *
         * @param list R[hXg
         * @param index ǉ郌R[h̃CfbNX
         * @param record ǉ郌R[h
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void add(SharedContextRecordList list, int index, Record record) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            if(index < 0 || index > list.size() - 1){
                throw new SharedContextUpdateException("Illegal index. index=" + index + ", size=" + list.size());
            }
            transactionList.add(new AddTransaction(index, record));
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * R[hւXVi[B<p>
         *
         * @param list R[hXg
         * @param index ւ郌R[h̃CfbNX
         * @param record ւ郌R[h
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void set(SharedContextRecordList list, int index, Record record) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            if(index < 0 || index > list.size() - 1){
                throw new SharedContextUpdateException("Illegal index. index=" + index + ", size=" + list.size());
            }
            transactionList.add(new SetTransaction(index, record));
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * R[h폜XVi[B<p>
         * w肳ꂽR[hR[hXgɊ܂܂ĂȂꍇ́AB<br>
         *
         * @param list R[hXg
         * @param val 폜郌R[h
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void remove(SharedContextRecordList list, Object val) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            int index = -1;
            if(val != null){
                SharedContextRecord record = (SharedContextRecord)val;
                index = record.getIndex();
            }
            if(index == -1){
                index = list.indexOf(val);
            }
            if(index != -1){
                transactionList.add(new RemoveTransaction(index));
            }
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * R[h폜XVi[B<p>
         *
         * @param list R[hXg
         * @param index 폜郌R[h̃CfbNX
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void remove(SharedContextRecordList list, int index) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            if(index < 0 || index > list.size() - 1){
                throw new SharedContextUpdateException("Illegal index. index=" + index + ", size=" + list.size());
            }
            transactionList.add(new RemoveTransaction(index));
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * R[hXgSč폜XVi[B<p>
         *
         * @param list R[hXg
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void clear(SharedContextRecordList list) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            transactionList.add(new ClearTransaction());
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * R[h̏WǉXVi[B<p>
         *
         * @param list R[hXg
         * @param c ǉ郌R[h̏W
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void addAll(SharedContextRecordList list, Collection<? extends Record> c) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            transactionList.add(new AddAllTransaction(c));
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * R[h̏WǉXVi[B<p>
         *
         * @param list R[hXg
         * @param index ǉCfbNX
         * @param c ǉ郌R[h̏W
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void addAll(SharedContextRecordList list, int index, Collection<? extends Record> c) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            if(index < 0 || index > list.size() - 1){
                throw new SharedContextUpdateException("Illegal index. index=" + index + ", size=" + list.size());
            }
            transactionList.add(new AddAllTransaction(index, c));
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * R[h̏W폜XVi[B<p>
         *
         * @param list R[hXg
         * @param c 폜郌R[h̏W
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void removeAll(SharedContextRecordList list, Collection<?> c) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            transactionList.add(new RemoveAllTransaction(c));
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * w肳ꂽR[h̏Ŵ݂cXVi[B<p>
         *
         * @param list R[hXg
         * @param c cR[h̏W
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void retainAll(SharedContextRecordList list, Collection<?> c) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            transactionList.add(new RetainAllTransaction(c));
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * w肳ꂽR[h̍XVi[B<p>
         *
         * @param list R[hXg
         * @param index R[h̃CfbNX
         * @param diff R[h̍
         * @exception SharedContextUpdateException XV̊i[Ɏsꍇ
         */
        public void updateRecord(SharedContextRecordList list, int index, SharedContextRecord.Difference diff) throws SharedContextUpdateException{
            if(transactionList == null){
                transactionList = new ArrayList<Transaction>();
            }
            if(index < 0 || index > list.size() - 1){
                throw new SharedContextUpdateException("Illegal index. index=" + index + ", size=" + list.size());
            }
            if(recordDiffMap == null){
                recordDiffMap = new HashMap<Integer,SharedContextRecord.Difference>();
            }
            Integer key = new Integer(index);
            if(!recordDiffMap.containsKey(key)){
                recordDiffMap.put(key, diff);
                transactionList.add(new UpdateTransaction(index, diff));
            }
            updateVersion = list.getUpdateVersion() + 1;
        }
        
        /**
         * w肳ꂽR[h̍擾B<p>
         *
         * @param index R[h̃CfbNX
         * @return BȂꍇ́Anull
         */
        protected SharedContextRecord.Difference getRecordDifference(int index){
            return recordDiffMap == null ? null : (SharedContextRecord.Difference)recordDiffMap.get(new Integer(index));
        }
        
        /**
         * w肳ꂽR[hXgɍXV𔽉fB<p>
         *
         * @param list XVΏۂ̃R[hXg
         * @exception SharedContextUpdateException XV̔fɎsꍇ
         */
        public boolean updateRecordList(SharedContextRecordList list) throws SharedContextUpdateException{
            if(transactionList != null && transactionList.size() != 0){
                if(SharedContextRecord.compareToUpdateVersion(list.getUpdateVersion(), updateVersion) >= 0){
                    return true;
                }else if(list.getUpdateVersion() + 1 != updateVersion){
                    return false;
                }
                for(int i = 0, imax = transactionList.size(); i < imax; i++){
                    transactionList.get(i).execute(list);
                }
            }
            list.setUpdateVersion(updateVersion);
            return true;
        }
        
        /**
         * XVꂽ𔻒肷B<p>
         *
         * @return XVꂽꍇ́Atrue
         */
        @Override
        public boolean isUpdate(){
            return transactionList != null && transactionList.size() != 0;
        }
        
        /**
         * XṼgUNVXg擾B<p>
         *
         * @return XVgUNṼXg
         */
        public List<Transaction> getTransactionList(){
            return transactionList;
        }
        
        /**
         * w肳ꂽʂ̃gUNVXg擾B<p>
         * 
         * @param type gUNV
         * @return XVgUNṼXgBw肳ꂽʂ̃gUNV݂Ȃꍇ́Anull
         */
        @SuppressWarnings("unchecked")
        protected <T> List<T> getTransactionList(int type){
            if(transactionList == null || transactionList.size() == 0){
                return null;
            }
            List<T> result = null;
            for(int i = 0, imax = transactionList.size(); i < imax; i++){
                Transaction transaction = transactionList.get(i);
                if(transaction.getType() == type){
                    if(result == null){
                        result = new ArrayList<T>();
                    }
                    result.add((T)transaction);
                }
            }
            return result;
        }
        
        /**
         * ǉgUNṼXg擾B<p>
         *
         * @return ǉgUNṼXgB݂Ȃꍇ́Anull
         */
        public List<AddTransaction> getAddTransactionList(){
            return this.<AddTransaction>getTransactionList(Transaction.ADD);
        }
        
        /**
         * ւgUNṼXg擾B<p>
         *
         * @return ւgUNṼXgB݂Ȃꍇ́Anull
         */
        public List<SetTransaction> getSetTransactionList(){
            return this.<SetTransaction>getTransactionList(Transaction.SET);
        }
        
        /**
         * 폜gUNṼXg擾B<p>
         *
         * @return 폜gUNṼXgB݂Ȃꍇ́Anull
         */
        public List<RemoveTransaction> getRemoveTransactionList(){
            return this.<RemoveTransaction>getTransactionList(Transaction.REMOVE);
        }
        
        /**
         * SǉgUNṼXg擾B<p>
         *
         * @return SǉgUNṼXgB݂Ȃꍇ́Anull
         */
        public List<AddAllTransaction> getAddAllTransactionList(){
            return this.<AddAllTransaction>getTransactionList(Transaction.ADDALL);
        }
        
        /**
         * S폜gUNṼXg擾B<p>
         *
         * @return S폜gUNṼXgB݂Ȃꍇ́Anull
         */
        public List<RemoveAllTransaction> getRemoveAllTransactionList(){
            return this.<RemoveAllTransaction>getTransactionList(Transaction.REMOVEALL);
        }
        
        /**
         * wvfŴ݂cgUNṼXg擾B<p>
         *
         * @return wvfŴ݂cgUNṼXgB݂Ȃꍇ́Anull
         */
        public List<RetainAllTransaction> getRetainAllTransactionList(){
            return this.<RetainAllTransaction>getTransactionList(Transaction.RETAINALL);
        }
        
        /**
         * R[hXVgUNṼXg擾B<p>
         *
         * @return R[hXVgUNṼXgB݂Ȃꍇ́Anull
         */
        public List<UpdateTransaction> getUpdateTransactionList(){
            return this.<UpdateTransaction>getTransactionList(Transaction.UPDATE);
        }
        
        /**
         * S폜gUNV݂邩肷B<p>
         *
         * @return S폜gUNV݂ꍇ́Atrue
         */
        public boolean isClear(){
            List<ClearTransaction> trans = this.<ClearTransaction>getTransactionList(Transaction.CLEAR);
            return trans == null || trans.size() == 0 ? false : true;
        }
        
        @Override
        public void writeExternal(ObjectOutput out) throws IOException{
            if(transactionList == null || transactionList.size() == 0){
                SharedContextRecord.writeInt(out, 0);
            }else{
                SharedContextRecord.writeInt(out, transactionList.size());
                for(int i = 0, imax = transactionList.size(); i < imax; i++){
                    Transaction tran = (Transaction)transactionList.get(i);
                    out.write(tran.getType());
                    tran.writeExternal(out);
                }
            }
            SharedContextRecord.writeInt(out, updateVersion);
        }
        
        @Override
        public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
            int size = SharedContextRecord.readInt(in);
            if(size > 0){
                transactionList = new ArrayList<Transaction>(size);
                for(int i = 0; i < size; i++){
                    final int type = in.read();
                    Transaction tran = null;
                    switch(type){
                    case Transaction.ADD:
                        tran = new AddTransaction();
                        break;
                    case Transaction.SET:
                        tran = new SetTransaction();
                        break;
                    case Transaction.REMOVE:
                        tran = new RemoveTransaction();
                        break;
                    case Transaction.CLEAR:
                        tran = new ClearTransaction();
                        break;
                    case Transaction.ADDALL:
                        tran = new AddAllTransaction();
                        break;
                    case Transaction.REMOVEALL:
                        tran = new RemoveAllTransaction();
                        break;
                    case Transaction.RETAINALL:
                        tran = new RetainAllTransaction();
                        break;
                    case Transaction.UPDATE:
                        tran = new UpdateTransaction();
                        break;
                    default:
                        throw new IOException("Unknown transaction. type=" + type);
                    }
                    tran.readExternal(in);
                    transactionList.add(tran);
                }
            }
            updateVersion = SharedContextRecord.readInt(in);
        }
        
        /**
         * gUNVB<p>
         *
         * @author M.Takata
         */
        public static interface Transaction extends Externalizable{
            
            /**
             * XVʁFǉB<p>
             */
            public static final byte ADD       = 1;
            
            /**
             * XVʁFւB<p>
             */
            public static final byte SET       = 2;
            
            /**
             * XVʁF폜B<p>
             */
            public static final byte REMOVE    = 3;
            
            /**
             * XVʁFS폜B<p>
             */
            public static final byte CLEAR     = 4;
            
            /**
             * XVʁFWǉB<p>
             */
            public static final byte ADDALL    = 5;
            
            /**
             * XVʁFW폜B<p>
             */
            public static final byte REMOVEALL = 6;
            
            /**
             * XVʁFWcB<p>
             */
            public static final byte RETAINALL = 7;
            
            /**
             * XVʁFR[hXVB<p>
             */
            public static final byte UPDATE    = 8;
            
            /**
             * gUNVʂ擾B<p>
             *
             * @return gUNV
             * @see #ADD
             * @see #SET
             * @see #REMOVE
             * @see #CLEAR
             * @see #ADDALL
             * @see #REMOVEALL
             * @see #RETAINALL
             * @see #UPDATE
             */
            public byte getType();
            
            /**
             * R[hXgɃgUNV𔽉fB<p>
             *
             * @param list R[hXg
             * @exception SharedContextUpdateException XV̔fɎsꍇ
             */
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException;
        }
        
        /**
         * ǉgUNVB<p>
         *
         * @author M.Takata
         */
        public static class AddTransaction implements Transaction{
            private int index = -1;
            private Record record;
            public AddTransaction(){
            }
            public AddTransaction(Record record){
                this.record = record;
            }
            public AddTransaction(int index, Record record){
                this.index = index;
                this.record = record;
            }
            @Override
            public byte getType(){
                return ADD;
            }
            @Override
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException{
                if(index == -1){
                    list.add(record);
                }else{
                    list.add(index, record);
                }
            }
            
            /**
             * ǉ郌R[h̃CfbNX擾B<p>
             *
             * @return R[h̃CfbNXBɒǉꍇ́A-1
             */
            public int getIndex(){
                return index;
            }
            
            /**
             * ǉ郌R[h擾B<p>
             *
             * @return R[h
             */
            public Record getRecord(){
                return record;
            }
            
            @Override
            public void writeExternal(ObjectOutput out) throws IOException{
                SharedContextRecord.writeInt(out, index);
                out.writeObject(record);
            }
            @Override
            public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
                index = SharedContextRecord.readInt(in);
                record = (Record)in.readObject();
            }
        }
        
        /**
         * ւgUNVB<p>
         *
         * @author M.Takata
         */
        public static class SetTransaction implements Transaction{
            private int index;
            private Record record;
            public SetTransaction(){
            }
            public SetTransaction(int index, Record record){
                this.index = index;
                this.record = record;
            }
            @Override
            public byte getType(){
                return SET;
            }
            @Override
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException{
                list.set(index, record);
            }
            
            /**
             * ւ郌R[h̃CfbNX擾B<p>
             *
             * @return R[h̃CfbNX
             */
            public int getIndex(){
                return index;
            }
            
            /**
             * ւ郌R[h擾B<p>
             *
             * @return R[h
             */
            public Record getRecord(){
                return record;
            }
            
            @Override
            public void writeExternal(ObjectOutput out) throws IOException{
                SharedContextRecord.writeInt(out, index);
                out.writeObject(record);
            }
            @Override
            public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
                index = SharedContextRecord.readInt(in);
                record = (Record)in.readObject();
            }
        }
        
        /**
         * 폜gUNVB<p>
         *
         * @author M.Takata
         */
        public static class RemoveTransaction implements Transaction{
            private int index;
            public RemoveTransaction(){
            }
            public RemoveTransaction(int index){
                this.index = index;
            }
            @Override
            public byte getType(){
                return REMOVE;
            }
            @Override
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException{
                list.remove(index);
            }
            
            /**
             * 폜R[h̃CfbNX擾B<p>
             *
             * @return R[h̃CfbNX
             */
            public int getIndex(){
                return index;
            }
            
            @Override
            public void writeExternal(ObjectOutput out) throws IOException{
                SharedContextRecord.writeInt(out, index);
            }
            @Override
            public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
                index = SharedContextRecord.readInt(in);
            }
        }
        
        /**
         * S폜gUNVB<p>
         *
         * @author M.Takata
         */
        public static class ClearTransaction implements Transaction{
            public ClearTransaction(){
            }
            @Override
            public byte getType(){
                return CLEAR;
            }
            @Override
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException{
                list.clear();
            }
            @Override
            public void writeExternal(ObjectOutput out) throws IOException{
            }
            @Override
            public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
            }
        }
        
        /**
         * WǉgUNVB<p>
         *
         * @author M.Takata
         */
        public static class AddAllTransaction implements Transaction{
            private int index = -1;
            private Collection<? extends Record> c;
            public AddAllTransaction(){
            }
            public AddAllTransaction(Collection<? extends Record> c){
                this.c = new ArrayList<Record>(c);
            }
            public AddAllTransaction(int index, Collection<? extends Record> c){
                this.index = index;
                this.c = new ArrayList<Record>(c);
            }
            @Override
            public byte getType(){
                return ADDALL;
            }
            @Override
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException{
                if(index == -1){
                    list.addAll(c);
                }else{
                    list.addAll(index, c);
                }
            }
            
            /**
             * ǉ郌R[hW̃CfbNX擾B<p>
             *
             * @return R[hW̃CfbNXBɒǉꍇ́A-1
             */
            public int getIndex(){
                return index;
            }
            
            /**
             * ǉ郌R[hW擾B<p>
             *
             * @return R[hW
             */
            public Collection<? extends Record> getRecords(){
                return c;
            }
            
            @Override
            public void writeExternal(ObjectOutput out) throws IOException{
                SharedContextRecord.writeInt(out, index);
                out.writeObject(c);
            }
            @SuppressWarnings("unchecked")
            @Override
            public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
                index = SharedContextRecord.readInt(in);
                c = (Collection<? extends Record>)in.readObject();
            }
        }
        
        /**
         * W폜gUNVB<p>
         *
         * @author M.Takata
         */
        public static class RemoveAllTransaction implements Transaction{
            private Collection<?> c;
            public RemoveAllTransaction(){
            }
            public RemoveAllTransaction(Collection<?> c){
                this.c = new ArrayList<Object>(c);
            }
            @Override
            public byte getType(){
                return REMOVEALL;
            }
            @Override
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException{
                list.removeAll(c);
            }
            
            /**
             * 폜郌R[hW擾B<p>
             *
             * @return R[hW
             */
            public Collection<?> getRecords(){
                return c;
            }
            
            @Override
            public void writeExternal(ObjectOutput out) throws IOException{
                out.writeObject(c);
            }
            @Override
            public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
                c = (Collection<?>)in.readObject();
            }
        }
        
        /**
         * WcgUNVB<p>
         *
         * @author M.Takata
         */
        public static class RetainAllTransaction implements Transaction{
            private Collection<?> c;
            public RetainAllTransaction(){
            }
            public RetainAllTransaction(Collection<?> c){
                this.c = new ArrayList<Object>(c);
            }
            @Override
            public byte getType(){
                return RETAINALL;
            }
            @Override
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException{
                list.retainAll(c);
            }
            
            /**
             * cR[hW擾B<p>
             *
             * @return R[hW
             */
            public Collection<?> getRecords(){
                return c;
            }
            
            @Override
            public void writeExternal(ObjectOutput out) throws IOException{
                out.writeObject(c);
            }
            @Override
            public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
                c = (Collection<?>)in.readObject();
            }
        }
        
        /**
         * R[hXVgUNVB<p>
         *
         * @author M.Takata
         */
        public static class UpdateTransaction implements Transaction{
            private int index;
            private SharedContextRecord.Difference diff;
            public UpdateTransaction(){
            }
            public UpdateTransaction(int index, SharedContextRecord.Difference diff){
                this.index = index;
                this.diff = diff;
            }
            @Override
            public byte getType(){
                return UPDATE;
            }
            
            /**
             * XVꂽ𔻒肷B<p>
             *
             * @return XVꂽꍇ́Atrue
             */
            public boolean isUpdate(){
                return diff.isUpdate();
            }
            
            @Override
            public void execute(SharedContextRecordList list) throws SharedContextUpdateException{
                SharedContextRecord record = null;
                try{
                    record = (SharedContextRecord)list.getRecord(index);
                }catch(IndexOutOfBoundsException e){
                    throw new SharedContextUpdateException(e);
                }
                record.update(diff);
            }
            
            /**
             * XV郌R[h̃CfbNX擾B<p>
             *
             * @return R[h̃CfbNX
             */
            public int getIndex(){
                return index;
            }
            
            /**
             * XV郌R[h̍擾B<p>
             *
             * @return R[h̍
             */
            public SharedContextRecord.Difference getRecordDifference(){
                return diff;
            }
            
            @Override
            public void writeExternal(ObjectOutput out) throws IOException{
                SharedContextRecord.writeInt(out, index);
                out.writeObject(diff);
            }
            @Override
            public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
                index = SharedContextRecord.readInt(in);
                diff = (SharedContextRecord.Difference)in.readObject();
            }
        }
    }
}