/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.io.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.service.io.Externalizer;

/**
 * t@CLbVQƁB<p>
 * LbVIuWFNg𒼗񉻂ăt@CɕێB<br>
 *
 * @author M.Takata
 */
public class FileCachedReference<E> extends DefaultCachedReference<E>
 implements Serializable{
    
    private static final long serialVersionUID = -1164595696487398731L;
    
    private transient Externalizer<E> externalizer;
    
    /**
     * w肳ꂽt@CLbVIuWFNg𒼗񉻂t@CƂĕێLbVQƂ𐶐B<p>
     *
     * @param file LbVIuWFNg𒼗񉻂t@C
     */
    public FileCachedReference(File file){
        super(file);
    }
    
    /**
     * w肳ꂽt@CLbVIuWFNg𒼗񉻂t@CƂĕێLbVQƂ𐶐B<p>
     *
     * @param file LbVIuWFNg𒼗񉻂t@C
     * @param ext 񉻂sExternalizer
     */
    public FileCachedReference(File file, Externalizer<E> ext){
        super(file);
        externalizer = ext;
    }
    
    /**
     * w肳ꂽt@CɎw肳ꂽLbVIuWFNg𒼗񉻂ĕێLbVQƂ𐶐B<p>
     *
     * @param file LbVIuWFNg𒼗񉻂t@C
     * @param obj LbVIuWFNg
     * @exception IOException 񉻂Ɏsꍇ
     */
    public FileCachedReference(File file, E obj) throws IOException{
        super(file);
        serializeObject(file, obj);
    }
    
    /**
     * w肳ꂽt@CɎw肳ꂽLbVIuWFNg𒼗񉻂ĕێLbVQƂ𐶐B<p>
     *
     * @param file LbVIuWFNg𒼗񉻂t@C
     * @param obj LbVIuWFNg
     * @param ext 񉻂sExternalizer
     * @exception IOException 񉻂Ɏsꍇ
     */
    public FileCachedReference(File file, E obj, Externalizer<E> ext) throws IOException{
        super(file);
        externalizer = ext;
        serializeObject(file, obj);
    }
    
    /**
     * IuWFNgwt@CɃVACYĊi[B<p>
     * 
     * @param file LbVIuWFNg𒼗񉻂t@C
     * @param obj LbVIuWFNg
     * @exception IOException 񉻂Ɏsꍇ
     */
    protected void serializeObject(File file, E obj)
     throws IOException{
        OutputStream os = new FileOutputStream(file);
        try{
            if(externalizer == null){
                ObjectOutputStream oos = new ObjectOutputStream(os);
                try{
                    if(obj != null){
                        synchronized(obj){
                            oos.writeObject(obj);
                        }
                    }else{
                        oos.writeObject(obj);
                    }
                    oos.flush();
                }finally{
                    try{
                        oos.close();
                    }catch(IOException e){}
                }
            }else{
                if(obj != null){
                    synchronized(obj){
                        externalizer.writeExternal(obj, os);
                    }
                }else{
                    externalizer.writeExternal(obj, os);
                }
            }
        }finally{
            if(os!=null){
                try{
                    os.close();
                }catch(IOException e){}
            }
        }
    }
    
    /**
     * IuWFNg𕜌B<p>
     * 
     * @return IuWFNg
     * @exception IOException LbVt@C̕Ɏsꍇ
     * @exception ClassNotFoundException LbVt@C̕ʂ̃NX ݂Ȃꍇ
     */
    public E deserializeObject() throws IOException, ClassNotFoundException{
        return deserializeObject(getFile());
    }
    
    /**
     * w肳ꂽt@CIuWFNg𕜌B<p>
     * 
     * @param file LbVIuWFNg񉻂Ăt@C
     * @return IuWFNg
     * @exception IOException LbVt@C̕Ɏsꍇ
     * @exception ClassNotFoundException LbVt@C̕ʂ̃NX ݂Ȃꍇ
     */
    @SuppressWarnings("unchecked")
    protected E deserializeObject(File file)
     throws IOException, ClassNotFoundException{
        E entry = null;
        InputStream is = new FileInputStream(file);
        try{
            if(externalizer == null){
                ObjectInputStream ois = new ObjectInputStream(is);
                try{
                    entry = (E)ois.readObject();
                }finally{
                    try{
                        ois.close();
                    }catch(IOException e){}
                }
            }else{
                entry = externalizer.readExternal(is);
            }
        }finally{
            if(is!=null){
                try{
                    is.close() ;
                }catch(IOException ex){
                }
            }
        }
        return entry;
    }
    
    /**
     * 񉻂LbVIuWFNg̃t@C擾B<p>
     *
     * @return 񉻂LbVIuWFNg̃t@C
     */
    public File getFile(){
        return (File)super.get(null, false);
    }
    
    /**
     * LbVꂽIuWFNg擾B<p>
     * truȅꍇ́A{@link #addCacheAccessListener(CacheAccessListener)}œo^ꂽ{@link CacheAccessListener}ɒʒmBAAœnꂽĂяoIuWFNgʒmCacheAccessListener̃CX^XƓꍇ́AʒmȂB<br>
     * gێLbVIuWFNgnullłȂꍇ́AFileIuWFNgɃLXg{@link #deserializeObject(File)}ĂяoĖ{̃LbVIuWFNg𕜌ĕԂBAAɎsꍇ́AnullԂBgێLbVIuWFNgnull̏ꍇ́A{@link #addLinkedReference(LinkedReference)}œo^ꂽ{@link LinkedReference}擾݂B擾łꍇ́AlɕĕԂB擾łȂꍇ́AnullԂB<br>
     *
     * @param source LbV擾邱̃\bȟĂяoIuWFNg
     * @param notify LbVANZXXiɒʒmꍇtrue
     * @return LbVIuWFNg
     */
    @Override
    public E get(Object source, boolean notify){
        final E obj = super.get(source, notify);
        if(obj instanceof File){
            File file = (File)obj;
            if(file.exists() && file.canRead()){
                try{
                    return deserializeObject(file);
                }catch(IOException e){
                    return null;
                }catch(ClassNotFoundException e){
                    return null;
                }
            }else{
                return null;
            }
        }else{
            return obj;
        }
    }
    
    @Override
    public void set(Object source, E obj)
     throws IllegalCachedReferenceException{
        notifyChange(source, obj);
        if(obj != null){
            try{
                serializeObject((File)cacheObj, obj);
            }catch(IOException e){
                throw new IllegalCachedReferenceException(e);
            }
        }else{
            ((File)cacheObj).delete();
        }
    }
    
    @Override
    public void remove(Object source){
        if(cacheObj != null){
            notifyRemoved(source);
            File file = (File)cacheObj;
            if(file.exists() && file.canRead()){
                ((File)cacheObj).delete();
            }
            cacheObj = null;
            if(linkedReferences != null){
                linkedReferences.clear();
            }
        }
    }
    
    private void writeObject(ObjectOutputStream out) throws IOException{
        out.defaultWriteObject();
        writeSet(out, linkedReferences);
        writeSet(out, removeListeners);
        writeSet(out, accessListeners);
        writeSet(out, changeListeners);
        ServiceName name = null;
        if(externalizer != null){
            name = getServiceName(externalizer);
        }
        if(name != null){
            out.writeObject(name);
        }else{
            out.writeObject(externalizer);
        }
    }
    
    @SuppressWarnings("unchecked")
    private void readObject(ObjectInputStream in)
     throws IOException, ClassNotFoundException{
        in.defaultReadObject();
        linkedReferences = readSet(in);
        removeListeners = readSet(in);
        accessListeners = readSet(in);
        changeListeners = readSet(in);
        Object obj = in.readObject();
        if(obj != null){
            if(obj instanceof ServiceName){
                externalizer = ServiceManagerFactory.getServiceObject(
                    (ServiceName)obj
                );
            }else{
                externalizer = (Externalizer<E>)obj;
            }
        }
    }
}