/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.aop.interceptor.servlet;

import java.util.*;
import java.text.SimpleDateFormat;
import javax.servlet.ServletRequest;
import javax.servlet.http.HttpServletRequest;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.beans.*;
import jp.ossc.nimbus.daemon.*;
import jp.ossc.nimbus.service.aop.*;
import jp.ossc.nimbus.service.aop.interceptor.*;
import jp.ossc.nimbus.service.writer.*;

/**
 * HTTPNGXggNXC^[Zv^B<p>
 * HTTPNGXgɑ΂āAgNX擾C^[Zv^łB<br>
 * ̃C^[Zv^Ŏ擾ł郁gNX́AȉłB<br>
 * <ul>
 *     <li>NGXgURI</li>
 *     <li>NGXg񐔁i퉞j</li>
 *     <li>NGXg񐔁iExceptionj</li>
 *     <li>NGXg񐔁iErrorj</li>
 *     <li>ŏINGXg</li>
 *     <li>ŏIException</li>
 *     <li>ŏIError</li>
 *     <li>ō</li>
 *     <li>ōԎ</li>
 *     <li>ŒᏈ</li>
 *     <li>ŒᏈԎ</li>
 *     <li>Ϗ</li>
 *     <li>iϏԁ~NGXg񐔁jŕ]ꂽʁi~j</li>
 * </ul>
 * ȉɁAgNX擾C^[Zv^̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;server&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="HttpServletRequestMetricsInterceptor"
 *                  code="jp.ossc.nimbus.service.aop.interceptor.servlet.HttpServletRequestMetricsInterceptorService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/server&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class HttpServletRequestMetricsInterceptorService extends ServletFilterInterceptorService
 implements DaemonRunnable<Object>, HttpServletRequestMetricsInterceptorServiceMBean{
    
    private static final long serialVersionUID = -8746150658855433357L;
    
    private static final Comparator<MetricsInfo> COMP = new MetricsInfoComparator();
    private static final String LINE_SEP = System.getProperty("line.separator");
    
    private Map<String,MetricsInfo> metricsInfos;
    private boolean isEnabled = true;
    private boolean isCalculateOnlyNormal;
    private String dateFormat = DEFAULT_DATE_FORMAT;
    private long outputInterval = 60000;
    private boolean isResetByOutput;
    private Properties pathAndCategoryServiceNameMapping;
    private Map<String,Category> pathAndCategoryMap;
    private Daemon writerDaemon;
    private ServiceName categoryServiceName;
    private Category metricsCategory;
    private boolean isOutputCount = true;
    private boolean isOutputExceptionCount = false;
    private boolean isOutputErrorCount = false;
    private boolean isOutputLastTime = false;
    private boolean isOutputLastExceptionTime = false;
    private boolean isOutputLastErrorTime = false;
    private boolean isOutputBestPerformance = true;
    private boolean isOutputBestPerformanceTime = false;
    private boolean isOutputWorstPerformance = true;
    private boolean isOutputWorstPerformanceTime = false;
    private boolean isOutputAveragePerformance = true;
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setEnabled(boolean enable){
        isEnabled = enable;
    }
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isEnabled(){
        return isEnabled;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setCalculateOnlyNormal(boolean isCalc){
        isCalculateOnlyNormal = isCalc;
    }
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isCalculateOnlyNormal(){
        return isCalculateOnlyNormal;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setDateFormat(String format){
        dateFormat = format;
    }
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public String getDateFormat(){
        return dateFormat;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public String displayMetricsInfo(){
        final MetricsInfo[] infos = metricsInfos.values()
            .toArray(new MetricsInfo[metricsInfos.size()]);
        Arrays.sort(infos, COMP);
        final SimpleDateFormat format
             = new SimpleDateFormat(dateFormat);
        final StringBuffer buf = new StringBuffer();
        buf.append("\"No.\"");
        if(isOutputCount){
            buf.append(",\"Count\"");
        }
        if(isOutputExceptionCount){
            buf.append(",\"ExceptionCount\"");
        }
        if(isOutputErrorCount){
            buf.append(",\"ErrorCount\"");
        }
        if(isOutputLastTime){
            buf.append(",\"LastTime\"");
        }
        if(isOutputLastExceptionTime){
            buf.append(",\"LastExceptionTime\"");
        }
        if(isOutputLastErrorTime){
            buf.append(",\"LastErrorTime\"");
        }
        if(isOutputBestPerformance){
            buf.append(",\"Best performance[ms]\"");
        }
        if(isOutputBestPerformanceTime){
            buf.append(",\"Best performance time\"");
        }
        if(isOutputWorstPerformance){
            buf.append(",\"Worst performance[ms]\"");
        }
        if(isOutputWorstPerformanceTime){
            buf.append(",\"Worst performance time\"");
        }
        if(isOutputAveragePerformance){
            buf.append(",\"Average performance[ms]\"");
        }
        buf.append(",\"Path\"");
        buf.append(LINE_SEP);
        for(int i = 0; i < infos.length; i++){
            buf.append('"').append(i + 1).append('"');
            if(isOutputCount){
                buf.append(',').append('"').append(infos[i].getCount()).append('"');
            }
            if(isOutputExceptionCount){
                buf.append(',').append('"').append(infos[i].getExceptionCount())
                    .append('"');
            }
            if(isOutputErrorCount){
                buf.append(',').append('"').append(infos[i].getErrorCount())
                    .append('"');
            }
            if(isOutputLastTime){
                if(infos[i].getLastTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"')
                        .append(format.format(new Date(infos[i].getLastTime())))
                        .append('"');
                }
            }
            if(isOutputLastExceptionTime){
                if(infos[i].getLastExceptionTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"')
                        .append(format.format(
                            new Date(infos[i].getLastExceptionTime()))
                        ).append('"');
                }
            }
            if(isOutputLastErrorTime){
                if(infos[i].getLastErrorTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append('"').append(',')
                        .append(format.format(new Date(infos[i].getLastErrorTime())))
                        .append('"');
                }
            }
            if(isOutputBestPerformance){
                buf.append(',').append('"').append(infos[i].getBestPerformance())
                    .append('"');
            }
            if(isOutputBestPerformanceTime){
                if(infos[i].getBestPerformanceTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"').append(format.format(
                        new Date(infos[i].getBestPerformanceTime())
                        )).append('"');
                }
            }
            if(isOutputWorstPerformance){
                buf.append(',').append('"').append(infos[i].getWorstPerformance())
                    .append('"');
            }
            if(isOutputWorstPerformanceTime){
                if(infos[i].getWorstPerformanceTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"').append(format.format(
                        new Date(infos[i].getWorstPerformanceTime())
                        )).append('"');
                }
            }
            if(isOutputAveragePerformance){
                buf.append(',').append('"').append(infos[i].getAveragePerformance())
                    .append('"');
            }
            buf.append(',').append('"').append(infos[i].getKey()).append('"');
            buf.append(LINE_SEP);
        }
        return buf.toString();
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void reset(){
        metricsInfos.clear();
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public MetricsInfo getMetricsInfo(String path){
        return metricsInfos.get(path);
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public Map<String,MetricsInfo> getMetricsInfos(){
        if(metricsInfos == null){
            return new HashMap<String,MetricsInfo>();
        }
        synchronized(metricsInfos){
            return new HashMap<String,MetricsInfo>(metricsInfos);
        }
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputInterval(long interval){
        outputInterval = interval;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public long getOutputInterval(){
        return outputInterval;
    }
    
    public void setResetByOutput(boolean isReset){
        isResetByOutput = isReset;
    }
    
    public boolean isResetByOutput(){
        return isResetByOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setPathAndCategoryServiceNameMapping(Properties mapping){
        pathAndCategoryServiceNameMapping = mapping;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public Properties getPathAndCategoryServiceNameMapping(){
        return pathAndCategoryServiceNameMapping;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setCategoryServiceName(ServiceName name){
        categoryServiceName = name;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public ServiceName getCategoryServiceName(){
        return categoryServiceName;
    }
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputCount(boolean isOutput){
        isOutputCount = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputCount(){
        return isOutputCount;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputExceptionCount(boolean isOutput){
        isOutputExceptionCount = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputExceptionCount(){
        return isOutputExceptionCount;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputErrorCount(boolean isOutput){
        isOutputErrorCount = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputErrorCount(){
        return isOutputErrorCount;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputLastTime(boolean isOutput){
        isOutputLastTime = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputLastTime(){
        return isOutputLastTime;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputLastExceptionTime(boolean isOutput){
        isOutputLastExceptionTime = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputLastExceptionTime(){
        return isOutputLastExceptionTime;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputLastErrorTime(boolean isOutput){
        isOutputLastErrorTime = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputLastErrorTime(){
        return isOutputLastErrorTime;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputBestPerformance(boolean isOutput){
        isOutputBestPerformance = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputBestPerformance(){
        return isOutputBestPerformance;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputBestPerformanceTime(boolean isOutput){
        isOutputBestPerformanceTime = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputBestPerformanceTime(){
        return isOutputBestPerformanceTime;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputWorstPerformance(boolean isOutput){
        isOutputWorstPerformance = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputWorstPerformance(){
        return isOutputWorstPerformance;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputWorstPerformanceTime(boolean isOutput){
        isOutputWorstPerformanceTime = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputWorstPerformanceTime(){
        return isOutputWorstPerformanceTime;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public void setOutputAveragePerformance(boolean isOutput){
        isOutputAveragePerformance = isOutput;
    }
    
    // HttpServletRequestMetricsInterceptorServiceMBeanJavaDoc
    public boolean isOutputAveragePerformance(){
        return isOutputAveragePerformance;
    }
    
    /**
     * T[rX̐sB<p>
     *
     * @exception Exception Ɏsꍇ
     */
    public void createService() throws Exception{
        metricsInfos = Collections.synchronizedMap(new HashMap<String,MetricsInfo>());
        pathAndCategoryMap = new HashMap<String,Category>();
    }
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception JnɎsꍇ
     */
    public void startService() throws Exception{
        metricsInfos.clear();
        if(pathAndCategoryServiceNameMapping != null
            && pathAndCategoryServiceNameMapping.size() != 0){
            final ServiceNameEditor nameEditor = new ServiceNameEditor();
            nameEditor.setServiceManagerName(getServiceManagerName());
            final Iterator<Object> paths
                 = pathAndCategoryServiceNameMapping.keySet().iterator();
            while(paths.hasNext()){
                final String path = (String)paths.next();
                final String nameStr = pathAndCategoryServiceNameMapping
                    .getProperty(path);
                nameEditor.setAsText(nameStr);
                final ServiceName name = (ServiceName)nameEditor.getValue();
                final Category category = ServiceManagerFactory
                    .getServiceObject(name);
                pathAndCategoryMap.put(path, category);
            }
        }
        
        if(categoryServiceName != null){
            metricsCategory = ServiceManagerFactory
                .getServiceObject(categoryServiceName);
        }
        
        if((pathAndCategoryMap != null && pathAndCategoryMap.size() != 0)
             || metricsCategory != null){
            writerDaemon = new Daemon(this);
            writerDaemon.setName("Nimbus MetricsWriteDaemon " + getServiceNameObject());
            writerDaemon.start();
        }
    }
    /**
     * T[rX̒~sB<p>
     * 擾gNXAWo͂ɏo͂B
     *
     * @exception Exception ~Ɏsꍇ
     */
    public void stopService() throws Exception{
        System.out.println(displayMetricsInfo());
        
        if(writerDaemon != null){
            writerDaemon.stop();
            writerDaemon = null;
        }
        
        pathAndCategoryMap.clear();
    }
    /**
     * T[rX̔jsB<p>
     *
     * @exception Exception jɎsꍇ
     */
    public void destroyService() throws Exception{
        metricsInfos = null;
        pathAndCategoryMap = null;
    }
    
    /**
     * gNX擾āÃC^[Zv^ĂяoB<p>
     * T[rXJnĂȂꍇ{@link #setEnabled(boolean) setEnabled(false)}ɐݒ肳Ăꍇ́AgNX擾s킸Ɏ̃C^[Zv^ĂяoB<br>
     *
     * @param context ĂяõReLXg
     * @param chain ̃C^[Zv^Ăяo߂̃`F[
     * @return Ăяoʂ̖߂l
     * @exception Throwable ĂяoŗOꍇA܂͂̃C^[Zv^ŔCӂ̗OꍇBAA{Ăяo鏈throwȂRuntimeExceptionȊO̗OthrowĂAĂяoɂ͓`dȂB
     */
    public Object invokeFilter(
        ServletFilterInvocationContext context,
        InterceptorChain chain
    ) throws Throwable{
        long start = 0;
        boolean isError = false;
        boolean isException = false;
        if(getState() == State.STARTED && isEnabled()){
            start = System.currentTimeMillis();
        }
        try{
            return chain.invokeNext(context);
        }catch(Exception e){
            isException = true;
            throw e;
        }catch(Error err){
            isError = true;
            throw err;
        }finally{
            if(getState() == State.STARTED && isEnabled()){
                long end = System.currentTimeMillis();
                ServletRequest sr = context.getServletRequest();
                if(sr instanceof HttpServletRequest){
                    HttpServletRequest hsr = (HttpServletRequest)sr;
                    String path = hsr.getPathInfo();
                    if(path == null){
                        path = hsr.getServletPath();
                    }
                    MetricsInfo metricsInfo = null;
                    synchronized(metricsInfos){
                        metricsInfo = metricsInfos.get(path);
                        if(metricsInfo == null){
                            metricsInfo = new MetricsInfo(
                                path,
                                isCalculateOnlyNormal
                            );
                            metricsInfos.put(path, metricsInfo);
                        }
                        metricsInfo.calculate(end - start, isException, isError);
                    }
                }
            }
        }
    }
    
    /**
     * f[JnɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onStart() {
        return true;
    }
    
    /**
     * f[~ɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onStop() {
        return true;
    }
    
    /**
     * f[fɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onSuspend() {
        return true;
    }
    
    /**
     * f[ĊJɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onResume() {
        return true;
    }
    
    /**
     * o͊ԊuX[vB<p>
     * 
     * @param ctrl DaemonControlIuWFNg
     * @return null
     */
    public Object provide(DaemonControl ctrl){
        try{
            Thread.sleep(outputInterval);
        }catch(InterruptedException e){
        }
        return null;
    }
    
    /**
     * o͐悪ݒ肳Ă΁AB<p>
     *
     * @param dequeued null
     * @param ctrl DaemonControlIuWFNg
     */
    public void consume(Object dequeued, DaemonControl ctrl){
        if(pathAndCategoryMap != null && pathAndCategoryMap.size() != 0){
            final Iterator<String> paths = pathAndCategoryMap.keySet().iterator();
            while(paths.hasNext()){
                final String path = paths.next();
                final Category category
                     = pathAndCategoryMap.get(path);
                final MetricsInfo info = metricsInfos.get(path);
                if(info != null && category != null){
                    try{
                        category.write(createRecord(info));
                    }catch(MessageWriteException e){
                        // TODO Oo
                    }
                }
            }
        }
        if(metricsCategory != null){
            final MetricsInfo[] infos = metricsInfos.values()
                .toArray(new MetricsInfo[metricsInfos.size()]);
            Arrays.sort(infos, COMP);
            for(int i = 0; i < infos.length; i++){
                try{
                    metricsCategory.write(createRecord(i + 1, infos[i]));
                }catch(MessageWriteException e){
                    // TODO Oo
                }
            }
        }
        if(isResetByOutput){
            final MetricsInfo[] infos = metricsInfos.values()
                .toArray(new MetricsInfo[metricsInfos.size()]);
            for(int i = 0; i < infos.length; i++){
                infos[i].reset();
            }
        }
    }
    
    private Map<String,Object> createRecord(MetricsInfo info){
        return createRecord(-1, info);
    }
    private Map<String,Object> createRecord(int order, MetricsInfo info){
        final Map<String,Object> record = new HashMap<String,Object>();
        if(order > 0){
            record.put(RECORD_KEY_ORDER, new Integer(order));
        }
        record.put(RECORD_KEY_PATH, info.getKey());
        if(isOutputCount){
            record.put(RECORD_KEY_COUNT, new Long(info.getCount()));
        }
        if(isOutputExceptionCount){
            record.put(
                RECORD_KEY_EXCEPTION_COUNT,
                new Long(info.getExceptionCount())
            );
        }
        if(isOutputErrorCount){
            record.put(
                RECORD_KEY_ERROR_COUNT,
                new Long(info.getErrorCount())
            );
        }
        if(isOutputLastTime){
            record.put(
                RECORD_KEY_LAST_TIME,
                info.getLastTime() == 0 ? null : new Date(info.getLastTime())
            );
        }
        if(isOutputLastExceptionTime){
            record.put(
                RECORD_KEY_LAST_EXCEPTION_TIME,
                info.getLastExceptionTime() == 0
                     ? null : new Date(info.getLastExceptionTime())
            );
        }
        if(isOutputLastErrorTime){
            record.put(
                RECORD_KEY_LAST_ERROR_TIME,
                info.getLastErrorTime() == 0
                     ? null : new Date(info.getLastErrorTime())
            );
        }
        if(isOutputBestPerformance){
            record.put(
                RECORD_KEY_BEST_PERFORMANCE,
                new Long(info.getBestPerformance())
            );
        }
        if(isOutputBestPerformanceTime){
            record.put(
                RECORD_KEY_BEST_PERFORMANCE_TIME,
                info.getBestPerformanceTime() == 0
                     ? null : new Date(info.getBestPerformanceTime())
            );
        }
        if(isOutputWorstPerformance){
            record.put(
                RECORD_KEY_WORST_PERFORMANCE,
                new Long(info.getWorstPerformance())
            );
        }
        if(isOutputWorstPerformanceTime){
            record.put(
                RECORD_KEY_WORST_PERFORMANCE_TIME,
                info.getWorstPerformanceTime() == 0
                     ? null : new Date(info.getWorstPerformanceTime())
            );
        }
        if(isOutputAveragePerformance){
            record.put(
                RECORD_KEY_AVERAGE_PERFORMANCE,
                new Long(info.getAveragePerformance())
            );
        }
        return record;
    }
    
    /**
     * ȂB<p>
     */
    public void garbage(){
    }
    
    private static class MetricsInfoComparator implements Comparator<MetricsInfo>{
        public int compare(MetricsInfo info1, MetricsInfo info2){
            final long sortKey1 = info1.getAveragePerformance() * info1.getCount();
            final long sortKey2 = info2.getAveragePerformance() * info2.getCount();
            if(sortKey1 > sortKey2){
                return -1;
            }else if(sortKey1 < sortKey2){
                return 1;
            }else{
                return 0;
            }
        }
    }
}
