/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.sql;

import java.util.*;
import java.text.SimpleDateFormat;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.daemon.*;
import jp.ossc.nimbus.service.writer.*;
import jp.ossc.nimbus.service.aop.interceptor.MetricsInfo;

/**
 * SQLgNXWvT[rXB<p>
 * SQĽĂяoɑ΂āAgNX擾T[rXłB<br>
 * ̃T[rXŎ擾ł郁gNX́AȉłB<br>
 * <ul>
 *     <li>SQL</li>
 *     <li>Ăяo񐔁i퉞j</li>
 *     <li>Ăяo񐔁iExceptionj</li>
 *     <li>Ăяo񐔁iErrorj</li>
 *     <li>ŏIĂяo</li>
 *     <li>ŏIException</li>
 *     <li>ŏIError</li>
 *     <li>ō</li>
 *     <li>ōԎ</li>
 *     <li>ŒᏈ</li>
 *     <li>ŒᏈԎ</li>
 *     <li>Ϗ</li>
 *     <li>iϏԁ~Ăяo񐔁jŕ]ꂽʁi~j</li>
 * </ul>
 * ȉɁAT[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;server&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="SQLMetricsCollector"
 *                  code="jp.ossc.nimbus.service.connection.SQLMetricsCollectorService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/server&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class SQLMetricsCollectorService extends ServiceBase
 implements SQLMetricsCollector, DaemonRunnable<Object>, SQLMetricsCollectorServiceMBean{
    
    private static final long serialVersionUID = -114715010029870256L;
    
    private static final Comparator<MetricsInfo> COMP = new MetricsInfoComparator();
    private static final String LINE_SEP = System.getProperty("line.separator");
    private static final String LINE_FEED_STRING = "\\n";
    private static final String CARRIAGE_RETURN_STRING = "\\r";
    private static final String CSV_ENCLOSE_STRING = "\"\"";
    private static final char CSV_ENCLOSE_CHAR = '"';
    private static final char CARRIAGE_RETURN  = '\r';
    private static final char LINE_FEED  = '\n';
    
    private Map<String, MetricsInfo> metricsInfos;
    private boolean isEnabled = true;
    private boolean isCalculateOnlyNormal;
    private String dateFormat = DEFAULT_DATE_FORMAT;
    private long outputInterval = 60000;
    private boolean isResetByOutput;
    private Daemon writerDaemon;
    private ServiceName categoryServiceName;
    private Category metricsCategory;
    private boolean isOutputCount = true;
    private boolean isOutputExceptionCount = false;
    private boolean isOutputErrorCount = false;
    private boolean isOutputLastTime = false;
    private boolean isOutputLastExceptionTime = false;
    private boolean isOutputLastErrorTime = false;
    private boolean isOutputBestPerformance = true;
    private boolean isOutputBestPerformanceTime = false;
    private boolean isOutputWorstPerformance = true;
    private boolean isOutputWorstPerformanceTime = false;
    private boolean isOutputAveragePerformance = true;
    private int maxMetricsSize = -1;
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setEnabled(boolean enable){
        isEnabled = enable;
    }
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isEnabled(){
        return isEnabled;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setCalculateOnlyNormal(boolean isCalc){
        isCalculateOnlyNormal = isCalc;
    }
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isCalculateOnlyNormal(){
        return isCalculateOnlyNormal;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setDateFormat(String format){
        dateFormat = format;
    }
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public String getDateFormat(){
        return dateFormat;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public String displayMetricsInfo(){
        final MetricsInfo[] infos = (MetricsInfo[])metricsInfos.values()
            .toArray(new MetricsInfo[metricsInfos.size()]);
        Arrays.sort(infos, COMP);
        final SimpleDateFormat format
             = new SimpleDateFormat(dateFormat);
        final StringBuilder buf = new StringBuilder();
        buf.append("\"No.\"");
        if(isOutputCount){
            buf.append(",\"Count\"");
        }
        if(isOutputExceptionCount){
            buf.append(",\"ExceptionCount\"");
        }
        if(isOutputErrorCount){
            buf.append(",\"ErrorCount\"");
        }
        if(isOutputLastTime){
            buf.append(",\"LastTime\"");
        }
        if(isOutputLastExceptionTime){
            buf.append(",\"LastExceptionTime\"");
        }
        if(isOutputLastErrorTime){
            buf.append(",\"LastErrorTime\"");
        }
        if(isOutputBestPerformance){
            buf.append(",\"Best performance[ms]\"");
        }
        if(isOutputBestPerformanceTime){
            buf.append(",\"Best performance time\"");
        }
        if(isOutputWorstPerformance){
            buf.append(",\"Worst performance[ms]\"");
        }
        if(isOutputWorstPerformanceTime){
            buf.append(",\"Worst performance time\"");
        }
        if(isOutputAveragePerformance){
            buf.append(",\"Average performance[ms]\"");
        }
        buf.append(",\"SQL\"");
        buf.append(LINE_SEP);
        for(int i = 0; i < infos.length; i++){
            buf.append('"').append(i + 1).append('"');
            if(isOutputCount){
                buf.append(',').append('"').append(infos[i].getCount()).append('"');
            }
            if(isOutputExceptionCount){
                buf.append(',').append('"').append(infos[i].getExceptionCount())
                    .append('"');
            }
            if(isOutputErrorCount){
                buf.append(',').append('"').append(infos[i].getErrorCount())
                    .append('"');
            }
            if(isOutputLastTime){
                if(infos[i].getLastTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"')
                        .append(format.format(new Date(infos[i].getLastTime())))
                        .append('"');
                }
            }
            if(isOutputLastExceptionTime){
                if(infos[i].getLastExceptionTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"')
                        .append(format.format(
                            new Date(infos[i].getLastExceptionTime()))
                        ).append('"');
                }
            }
            if(isOutputLastErrorTime){
                if(infos[i].getLastErrorTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append('"').append(',')
                        .append(format.format(new Date(infos[i].getLastErrorTime())))
                        .append('"');
                }
            }
            if(isOutputBestPerformance){
                buf.append(',').append('"').append(infos[i].getBestPerformance())
                    .append('"');
            }
            if(isOutputBestPerformanceTime){
                if(infos[i].getBestPerformanceTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"').append(format.format(
                        new Date(infos[i].getBestPerformanceTime())
                        )).append('"');
                }
            }
            if(isOutputWorstPerformance){
                buf.append(',').append('"').append(infos[i].getWorstPerformance())
                    .append('"');
            }
            if(isOutputWorstPerformanceTime){
                if(infos[i].getWorstPerformanceTime() == 0){
                    buf.append(",\"\"");
                }else{
                    buf.append(',').append('"').append(format.format(
                        new Date(infos[i].getWorstPerformanceTime())
                        )).append('"');
                }
            }
            if(isOutputAveragePerformance){
                buf.append(',').append('"').append(infos[i].getAveragePerformance())
                    .append('"');
            }
            buf.append(',').append('"').append(escape(infos[i].getKey())).append('"');
            buf.append(LINE_SEP);
        }
        return buf.toString();
    }
    
    protected String escape(String str){
        if(str == null){
            return null;
        }
        final int length = str.length();
        if(length == 0){
            return str;
        }
        final StringBuilder buf = new StringBuilder();
        for(int i = 0; i < length; i++){
            final char c = str.charAt(i);
            switch(c){
            case CARRIAGE_RETURN:
                buf.append(CARRIAGE_RETURN_STRING);
                break;
            case LINE_FEED:
                buf.append(LINE_FEED_STRING);
                break;
            case CSV_ENCLOSE_CHAR:
                buf.append(CSV_ENCLOSE_STRING);
                break;
            default:
                buf.append(c);
            }
        }
        return buf.toString();
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void reset(){
        metricsInfos.clear();
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public MetricsInfo getMetricsInfo(String sql){
        return (MetricsInfo)metricsInfos.get(sql);
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputInterval(long interval){
        outputInterval = interval;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public long getOutputInterval(){
        return outputInterval;
    }
    
    public void setResetByOutput(boolean isReset){
        isResetByOutput = isReset;
    }
    
    public boolean isResetByOutput(){
        return isResetByOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setCategoryServiceName(ServiceName name){
        categoryServiceName = name;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public ServiceName getCategoryServiceName(){
        return categoryServiceName;
    }
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputCount(boolean isOutput){
        isOutputCount = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputCount(){
        return isOutputCount;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputExceptionCount(boolean isOutput){
        isOutputExceptionCount = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputExceptionCount(){
        return isOutputExceptionCount;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputErrorCount(boolean isOutput){
        isOutputErrorCount = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputErrorCount(){
        return isOutputErrorCount;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputLastTime(boolean isOutput){
        isOutputLastTime = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputLastTime(){
        return isOutputLastTime;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputLastExceptionTime(boolean isOutput){
        isOutputLastExceptionTime = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputLastExceptionTime(){
        return isOutputLastExceptionTime;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputLastErrorTime(boolean isOutput){
        isOutputLastErrorTime = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputLastErrorTime(){
        return isOutputLastErrorTime;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputBestPerformance(boolean isOutput){
        isOutputBestPerformance = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputBestPerformance(){
        return isOutputBestPerformance;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputBestPerformanceTime(boolean isOutput){
        isOutputBestPerformanceTime = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputBestPerformanceTime(){
        return isOutputBestPerformanceTime;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputWorstPerformance(boolean isOutput){
        isOutputWorstPerformance = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputWorstPerformance(){
        return isOutputWorstPerformance;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputWorstPerformanceTime(boolean isOutput){
        isOutputWorstPerformanceTime = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputWorstPerformanceTime(){
        return isOutputWorstPerformanceTime;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setOutputAveragePerformance(boolean isOutput){
        isOutputAveragePerformance = isOutput;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public boolean isOutputAveragePerformance(){
        return isOutputAveragePerformance;
    }
    
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public void setMaxMetricsSize(int max){
        maxMetricsSize = max;
    }
    // SQLMetricsCollectorServiceMBeanJavaDoc
    public int getMaxMetricsSize(){
        return maxMetricsSize;
    }
    
    /**
     * T[rX̐sB<p>
     *
     * @exception Exception Ɏsꍇ
     */
    public void createService() throws Exception{
        metricsInfos = Collections.synchronizedMap(new HashMap<String, MetricsInfo>());
    }
    /**
     * T[rX̊JnsB<p>
     *
     * @exception Exception JnɎsꍇ
     */
    public void startService() throws Exception{
        metricsInfos.clear();
        
        if(categoryServiceName != null){
            metricsCategory = (Category)ServiceManagerFactory
                .getServiceObject(categoryServiceName);
        }
        
        if(metricsCategory != null){
            writerDaemon = new Daemon(this);
            writerDaemon.setName("Nimbus MetricsWriteDaemon " + getServiceNameObject());
            writerDaemon.start();
        }
    }
    /**
     * T[rX̒~sB<p>
     * 擾gNXAWo͂ɏo͂B
     *
     * @exception Exception ~Ɏsꍇ
     */
    public void stopService() throws Exception{
        System.out.println(displayMetricsInfo());
        
        if(writerDaemon != null){
            writerDaemon.stop();
            writerDaemon = null;
        }
    }
    /**
     * T[rX̔jsB<p>
     *
     * @exception Exception jɎsꍇ
     */
    public void destroyService() throws Exception{
        metricsInfos = null;
    }
    
    public void register(String sql, long performance){
        register(sql, performance, false, false);
    }
    
    public void registerException(String sql, long performance){
        register(sql, performance, true, false);
    }
    
    public void registerError(String sql, long performance){
        register(sql, performance, false, true);
    }
    
    private void register(String sql, long performance, boolean isException, boolean isError){
        synchronized(metricsInfos){
            MetricsInfo metricsInfo = (MetricsInfo)metricsInfos.get(sql);
            if(metricsInfo == null){
                if(maxMetricsSize > 0 && metricsInfos.size() > maxMetricsSize){
                    return;
                }
                metricsInfo = new MetricsInfo(
                    sql,
                    isCalculateOnlyNormal
                );
                metricsInfos.put(sql, metricsInfo);
            }
            metricsInfo.calculate(performance, isException, isError);
        }
    }
    
    /**
     * f[JnɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onStart() {
        return true;
    }
    
    /**
     * f[~ɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onStop() {
        return true;
    }
    
    /**
     * f[fɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onSuspend() {
        return true;
    }
    
    /**
     * f[ĊJɌĂяoB<p>
     * 
     * @return trueԂ
     */
    public boolean onResume() {
        return true;
    }
    
    /**
     * o͊ԊuX[vB<p>
     * 
     * @param ctrl DaemonControlIuWFNg
     * @return null
     */
    public Object provide(DaemonControl ctrl){
        try{
            Thread.sleep(outputInterval);
        }catch(InterruptedException e){
        }
        return null;
    }
    
    /**
     * o͐悪ݒ肳Ă΁AB<p>
     *
     * @param dequeued null
     * @param ctrl DaemonControlIuWFNg
     */
    public void consume(Object dequeued, DaemonControl ctrl){
        if(metricsCategory != null){
            final MetricsInfo[] infos = (MetricsInfo[])metricsInfos.values()
                .toArray(new MetricsInfo[metricsInfos.size()]);
            Arrays.sort(infos, COMP);
            for(int i = 0; i < infos.length; i++){
                try{
                    metricsCategory.write(createRecord(i + 1, infos[i]));
                }catch(MessageWriteException e){
                    // TODO Oo
                }
            }
        }
        if(isResetByOutput){
            final MetricsInfo[] infos = (MetricsInfo[])metricsInfos.values()
                .toArray(new MetricsInfo[metricsInfos.size()]);
            for(int i = 0; i < infos.length; i++){
                infos[i].reset();
            }
        }
    }
    
    private Map<String, Object> createRecord(int order, MetricsInfo info){
        final Map<String, Object> record = new HashMap<String, Object>();
        if(order > 0){
            record.put(RECORD_KEY_ORDER, order);
        }
        record.put(RECORD_KEY_SQL, info.getKey());
        if(isOutputCount){
            record.put(RECORD_KEY_COUNT, new Long(info.getCount()));
        }
        if(isOutputExceptionCount){
            record.put(
                RECORD_KEY_EXCEPTION_COUNT,
                new Long(info.getExceptionCount())
            );
        }
        if(isOutputErrorCount){
            record.put(
                RECORD_KEY_ERROR_COUNT,
                new Long(info.getErrorCount())
            );
        }
        if(isOutputLastTime){
            record.put(
                RECORD_KEY_LAST_TIME,
                info.getLastTime() == 0 ? null : new Date(info.getLastTime())
            );
        }
        if(isOutputLastExceptionTime){
            record.put(
                RECORD_KEY_LAST_EXCEPTION_TIME,
                info.getLastExceptionTime() == 0
                     ? null : new Date(info.getLastExceptionTime())
            );
        }
        if(isOutputLastErrorTime){
            record.put(
                RECORD_KEY_LAST_ERROR_TIME,
                info.getLastErrorTime() == 0
                     ? null : new Date(info.getLastErrorTime())
            );
        }
        if(isOutputBestPerformance){
            record.put(
                RECORD_KEY_BEST_PERFORMANCE,
                new Long(info.getBestPerformance())
            );
        }
        if(isOutputBestPerformanceTime){
            record.put(
                RECORD_KEY_BEST_PERFORMANCE_TIME,
                info.getBestPerformanceTime() == 0
                     ? null : new Date(info.getBestPerformanceTime())
            );
        }
        if(isOutputWorstPerformance){
            record.put(
                RECORD_KEY_WORST_PERFORMANCE,
                new Long(info.getWorstPerformance())
            );
        }
        if(isOutputWorstPerformanceTime){
            record.put(
                RECORD_KEY_WORST_PERFORMANCE_TIME,
                info.getWorstPerformanceTime() == 0
                     ? null : new Date(info.getWorstPerformanceTime())
            );
        }
        if(isOutputAveragePerformance){
            record.put(
                RECORD_KEY_AVERAGE_PERFORMANCE,
                new Long(info.getAveragePerformance())
            );
        }
        return record;
    }
    
    /**
     * ȂB<p>
     */
    public void garbage(){
    }
    
    private static class MetricsInfoComparator implements Comparator<MetricsInfo>{
        public int compare(MetricsInfo info1, MetricsInfo info2){
            final long sortKey1 = info1.getAveragePerformance() * info1.getCount();
            final long sortKey2 = info2.getAveragePerformance() * info2.getCount();
            if(sortKey1 > sortKey2){
                return -1;
            }else if(sortKey1 < sortKey2){
                return 1;
            }else{
                return 0;
            }
        }
    }
}
