/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.log;

import java.util.*;

import jp.ossc.nimbus.core.*;
import jp.ossc.nimbus.daemon.*;
import jp.ossc.nimbus.service.context.*;
import jp.ossc.nimbus.service.message.*;
import jp.ossc.nimbus.service.queue.Queue;
import jp.ossc.nimbus.service.queue.DefaultQueueService;
import jp.ossc.nimbus.service.writer.*;

/**
 * OT[rXB<p>
 * 
 * @author H.Nakano
 */
public class DefaultLoggerService extends ServiceBase
 implements Logger, DefaultLoggerServiceMBean{
    
    private static final long serialVersionUID = 2114672628671738940L;
    /** 󕶎萔 */
    protected static final String EMPTY_STRING = "";
    /** ʏContextL[w肳ĂȂꍇ̎ʏ񕶎 */
    protected static final String NONE_ID = "NONE";
    
    private static final String LINE_SEP = System.getProperty("line.separator");
    private static final String MSG_CAUSE = "Caused by: ";
    private static final String SERVLET_EXCEPTION_NAME = "javax.servlet.ServletException";
    private static final String GET_ROOT_CAUSE_METHOD = "getRootCause";
    private static final String JMS_EXCEPTION_NAME = "javax.jms.JMSException";
    private static final String GET_LINKED_EXCEPTION_METHOD = "getLinkedException";
    
    /**
     * JeS{@link LogCategory}T[rX̃}bsOB<p>
     * <table border="1">
     *   <tr bgcolor="#CCCCFF"><th colspan="2">L[</th><th colspan="2">l</th></tr>
     *   <tr bgcolor="#CCCCFF"><th>^</th><th>e</th><th>^</th><th>e</th></tr>
     *   <tr><td>String</td><td>JeS</td><td>LogCategory</td><td>JeST[rX</td></tr>
     * </table>
     */
    private Map<String, LogCategory> categoryMap;
    
    /**
     * ftHg̃JeS{@link LogCategory}T[rX̃}bsOB<p>
     * <table border="1">
     *   <tr bgcolor="#CCCCFF"><th colspan="2">L[</th><th colspan="2">l</th></tr>
     *   <tr bgcolor="#CCCCFF"><th>^</th><th>e</th><th>^</th><th>e</th></tr>
     *   <tr><td>String</td><td>JeS</td><td>LogCategory</td><td>JeST[rX</td></tr>
     * </table>
     */
    private Map<String, LogCategory> defaultCategoryMap;
    
    /**
     * JeST[rXzB<p>
     */
    private ServiceName[] categoryNames;
    
    /**
     * {@link Queue}T[rXB<p>
     */
    private ServiceName queueServiceName;
    
    /**
     * {@link #getQueueServiceName()}null̏ꍇAftHg{@link Queue}T[rXƂĐ{@link DefaultQueueService}T[rXB<p>
     */
    private DefaultQueueService<LogEnqueuedRecord> defaultQueue;
    
    /**
     * {@link Queue}IuWFNgB<p>
     */
    private Queue<LogEnqueuedRecord> queue;
    
    /**
     * {@link MessageRecordFactory}T[rXB<p>
     */
    private ServiceName messageFactoryServiceName;
    
    /**
     * {@link MessageRecordFactory}T[rXB<p>
     */
    private MessageRecordFactory messageFactory;
    
    /**
     * {@link #getMessageRecordFactoryServiceName()}null̏ꍇAftHg{@link MessageRecordFactory}T[rXƂĐ{@link LogMessageRecordFactoryService}T[rXB<p>
     */
    private LogMessageRecordFactoryService defaultMessageFactory;
    
    /**
     * {@link Context}T[rXB<p>
     */
    private ServiceName contextServiceName;
    
    /**
     * {@link Context}T[rXB<p>
     */
    private Context<Object, Object> context;
    
    /**
     * ftHg{@link MessageWriter}T[rXB<p>
     */
    private ServiceName defaultMessageWriterServiceName;
    
    /**
     * ftHg{@link MessageWriter}T[rXB<p>
     */
    private MessageWriter defaultMessageWriter;
    
    /**
     * {@link #getDefaultMessageWriterServiceName()}null̏ꍇAftHg{@link MessageWriter}T[rXƂĐ{@link ConsoleMessageWriterService}T[rXB<p>
     */
    private ConsoleMessageWriterService consoleWriter;
    
    /**
     * ftHg{@link WritableRecordFactory}T[rXB<p>
     */
    private ServiceName defaultWritableRecordFactoryServiceName;
    
    /**
     * ftHg{@link WritableRecordFactory}T[rXB<p>
     */
    private WritableRecordFactory defaultWritableRecordFactory;
    
    /**
     * {@link #getDefaultWritableRecordFactoryServiceName()}null̏ꍇAftHg{@link WritableRecordFactory}T[rXƂĐ{@link DefaultWritableRecordFactoryService}T[rXB<p>
     */
    private DefaultWritableRecordFactoryService defaultWritableRecordFactoryService;
    
    /**
     * {@link #DEBUG_METHOD_CATEGORY}JeS̃Oo͂s{@link MessageWriter}T[rXB<p>
     */
    private ServiceName debugMessageWriterServiceName;
    
    /**
     * {@link #SYSTEM_DEBUG_CATEGORY}JeS̃Oo͂s{@link MessageWriter}T[rXB<p>
     */
    private ServiceName systemDebugMessageWriterServiceName;
    
    /**
     * {@link #SYSTEM_INFO_CATEGORY}JeS̃Oo͂s{@link MessageWriter}T[rXB<p>
     */
    private ServiceName systemInfoMessageWriterServiceName;
    
    /**
     * {@link #SYSTEM_WARN_CATEGORY}JeS̃Oo͂s{@link MessageWriter}T[rXB<p>
     */
    private ServiceName systemWarnMessageWriterServiceName;
    
    /**
     * {@link #SYSTEM_ERROR_CATEGORY}JeS̃Oo͂s{@link MessageWriter}T[rXB<p>
     */
    private ServiceName systemErrorMessageWriterServiceName;
    
    /**
     * {@link #SYSTEM_FATAL_CATEGORY}JeS̃Oo͂s{@link MessageWriter}T[rXB<p>
     */
    private ServiceName systemFatalMessageWriterServiceName;
    
    /**
     * {@link #DEBUG_METHOD_CATEGORY}JeS̃Oo̓tH[}bgs{@link WritableRecordFactory}T[rXB<p>
     */
    private ServiceName debugRecordFactoryServiceName;
    
    /**
     * {@link #SYSTEM_DEBUG_CATEGORY}JeS̃Oo̓tH[}bgs{@link WritableRecordFactory}T[rXB<p>
     */
    private ServiceName systemDebugRecordFactoryServiceName;
    
    /**
     * {@link #SYSTEM_INFO_CATEGORY}JeS̃Oo̓tH[}bgs{@link WritableRecordFactory}T[rXB<p>
     */
    private ServiceName systemInfoRecordFactoryServiceName;
    
    /**
     * {@link #SYSTEM_WARN_CATEGORY}JeS̃Oo̓tH[}bgs{@link WritableRecordFactory}T[rXB<p>
     */
    private ServiceName systemWarnRecordFactoryServiceName;
    
    /**
     * {@link #SYSTEM_ERROR_CATEGORY}JeS̃Oo̓tH[}bgs{@link WritableRecordFactory}T[rXB<p>
     */
    private ServiceName systemErrorRecordFactoryServiceName;
    
    /**
     * {@link #SYSTEM_FATAL_CATEGORY}JeS̃Oo̓tH[}bgs{@link WritableRecordFactory}T[rXB<p>
     */
    private ServiceName systemFatalRecordFactoryServiceName;
    
    /**
     * {@link Daemon}IuWFNgB<p>
     */
    private Daemon daemon;
    
    /**
     * tH[}bgContextL[i[WB<p>
     */
    private Set<String> contextKeys = new HashSet<String>();
    
    /** VXeODEBUGOo̓tO */
    private boolean isSystemDebugEnabled = false;
    
    /** VXeOINFOOo̓tO */
    private boolean isSystemInfoEnabled = true;
    
    /** VXeOWARNOo̓tO */
    private boolean isSystemWarnEnabled = true;
    
    /** VXeOERROROo̓tO */
    private boolean isSystemErrorEnabled = true;
    
    /** VXeOFATALOo̓tO */
    private boolean isSystemFatalEnabled = true;
    
    protected String defaultFormat = DEFAULT_FORMAT;
    
    private boolean isDaemon = true;
    
    // LogServiceMBeanJavaDoc
    public void setDefaultMessageWriterServiceName(ServiceName name){
        defaultMessageWriterServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getDefaultMessageWriterServiceName(){
        return defaultMessageWriterServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setDefaultWritableRecordFactoryServiceName(ServiceName name){
        defaultWritableRecordFactoryServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getDefaultWritableRecordFactoryServiceName(){
        return defaultWritableRecordFactoryServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setCategoryServiceNames(ServiceName[] names){
        categoryNames = names;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName[] getCategoryServiceNames(){
        return categoryNames;
    }
    
    // LogServiceMBeanJavaDoc
    public void setQueueServiceName(ServiceName name){
        queueServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getQueueServiceName(){
        return queueServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setMessageRecordFactoryServiceName(ServiceName name){
        messageFactoryServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getMessageRecordFactoryServiceName(){
        return messageFactoryServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setContextServiceName(ServiceName name){
        contextServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getContextServiceName(){
        return contextServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setContextFormatKeys(String[] keys){
        if(keys != null){
            for(String key : keys){
                if(key != null){
                    contextKeys.add(key);
                }
            }
        }
    }
    
    // LogServiceMBeanJavaDoc
    public String[] getContextFormatKeys(){
        return contextKeys.toArray(new String[contextKeys.size()]);
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemDebugEnabled(boolean isEnabled){
        isSystemDebugEnabled = isEnabled;
        setEnabled(
            SYSTEM_DEBUG_CATEGORY,
            isEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public boolean isSystemDebugEnabled(){
        return isEnabled(
            SYSTEM_DEBUG_CATEGORY,
            isSystemDebugEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemInfoEnabled(boolean isEnabled){
        isSystemInfoEnabled = isEnabled;
        setEnabled(
            SYSTEM_INFO_CATEGORY,
            isEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public boolean isSystemInfoEnabled(){
        return isEnabled(
            SYSTEM_INFO_CATEGORY,
            isSystemInfoEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemWarnEnabled(boolean isEnabled){
        isSystemWarnEnabled = isEnabled;
        setEnabled(
            SYSTEM_WARN_CATEGORY,
            isEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public boolean isSystemWarnEnabled(){
        return isEnabled(
            SYSTEM_WARN_CATEGORY,
            isSystemWarnEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemErrorEnabled(boolean isEnabled){
        isSystemErrorEnabled = isEnabled;
        setEnabled(
            SYSTEM_ERROR_CATEGORY,
            isEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public boolean isSystemErrorEnabled(){
        return isEnabled(
            SYSTEM_ERROR_CATEGORY,
            isSystemErrorEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemFatalEnabled(boolean isEnabled){
        isSystemFatalEnabled = isEnabled;
        setEnabled(
            SYSTEM_FATAL_CATEGORY,
            isEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public boolean isSystemFatalEnabled(){
        return isEnabled(
            SYSTEM_FATAL_CATEGORY,
            isSystemFatalEnabled
        );
    }
    
    // LogServiceMBeanJavaDoc
    public void setDebugMessageWriterServiceName(ServiceName name){
        debugMessageWriterServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getDebugMessageWriterServiceName(){
        return debugMessageWriterServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemDebugMessageWriterServiceName(ServiceName name){
        systemDebugMessageWriterServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemDebugMessageWriterServiceName(){
        return systemDebugMessageWriterServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemInfoMessageWriterServiceName(ServiceName name){
        systemInfoMessageWriterServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemInfoMessageWriterServiceName(){
        return systemInfoMessageWriterServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemWarnMessageWriterServiceName(ServiceName name){
        systemWarnMessageWriterServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemWarnMessageWriterServiceName(){
        return systemWarnMessageWriterServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemErrorMessageWriterServiceName(ServiceName name){
        systemErrorMessageWriterServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemErrorMessageWriterServiceName(){
        return systemErrorMessageWriterServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemFatalMessageWriterServiceName(ServiceName name){
        systemFatalMessageWriterServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemFatalMessageWriterServiceName(){
        return systemFatalMessageWriterServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setDebugWritableRecordFactoryServiceName(
        ServiceName name
    ){
        debugRecordFactoryServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getDebugWritableRecordFactoryServiceName(){
        return debugRecordFactoryServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemDebugWritableRecordFactoryServiceName(
        ServiceName name
    ){
        systemDebugRecordFactoryServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemDebugWritableRecordFactoryServiceName(){
        return systemDebugRecordFactoryServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemInfoWritableRecordFactoryServiceName(ServiceName name){
        systemInfoRecordFactoryServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemInfoWritableRecordFactoryServiceName(){
        return systemInfoRecordFactoryServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemWarnWritableRecordFactoryServiceName(ServiceName name){
        systemWarnRecordFactoryServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemWarnWritableRecordFactoryServiceName(){
        return systemWarnRecordFactoryServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemErrorWritableRecordFactoryServiceName(
        ServiceName name
    ){
        systemErrorRecordFactoryServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemErrorWritableRecordFactoryServiceName(){
        return systemErrorRecordFactoryServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setSystemFatalWritableRecordFactoryServiceName(
        ServiceName name
    ){
        systemFatalRecordFactoryServiceName = name;
    }
    
    // LogServiceMBeanJavaDoc
    public ServiceName getSystemFatalWritableRecordFactoryServiceName(){
        return systemFatalRecordFactoryServiceName;
    }
    
    // LogServiceMBeanJavaDoc
    public void setDaemon(boolean isDaemon){
        this.isDaemon = isDaemon;
    }
    // LogServiceMBeanJavaDoc
    public boolean isDaemon(){
        return isDaemon;
    }
    
    /**
     * sB<p>
     * ̃\bhɂ́Aȉ̎sĂB<br>
     * <ol>
     *   <li>{@link Daemon}CX^X𐶐B</li>
     *   <li>JeSǗpMapCX^X𐶐B</li>
     * </ol>
     * 
     * @exception Exception ɎsꍇB
     */
    public void createService() throws Exception{
        daemon = new Daemon(new LogWriteDaemon());
        daemon.setName("Nimbus LogWriteDaemon " + getServiceNameObject());
        categoryMap = new HashMap<String, LogCategory>();
        defaultCategoryMap = new HashMap<String, LogCategory>();
    }
    
    /**
     * JnsB<p>
     * ̃\bhɂ́Aȉ̎sĂB<br>
     * <ol>
     *   <li>{@link #setDefaultMessageWriterServiceName(ServiceName)}ŃftHg{@link MessageWriter}T[rX̖Oݒ肳Ăꍇ́A{@link ServiceManager}MessageWriter擾{@link #setDefaultMessageWriterService(MessageWriter)}Őݒ肷B܂AftHgMessageWriterT[rX̖Oݒ肳ĂȂꍇ́A{@link ConsoleMessageWriterService}𐶐{@link #setDefaultMessageWriterService(MessageWriter)}Őݒ肷B</li>
     *   <li>{@link #setDefaultWritableRecordFactoryServiceName(ServiceName)}ŃftHg{@link WritableRecordFactory}T[rX̖Oݒ肳Ăꍇ́A{@link ServiceManager}WritableRecordFactory擾{@link #setDefaultWritableRecordFactoryService(WritableRecordFactory)}Őݒ肷B܂AftHgWritableRecordFactoryT[rX̖Oݒ肳ĂȂꍇ́A{@link WritableRecordFactoryService}𐶐{@link #setDefaultWritableRecordFactoryService(WritableRecordFactory)}Őݒ肷B</li>
     *   <li>VXeJeS𐶐ēo^B</li>
     *   <li>{@link #setCategoryServiceNames(ServiceName[])}Őݒ肳ꂽJeSo^B</li>
     *   <li>{@link #setQueueServiceName(ServiceName)}{@link Queue}T[rX̖Oݒ肳Ăꍇ́A{@link ServiceManager}Queue擾{@link #setQueueService(Queue)}Őݒ肷B܂AQueueT[rX̖Oݒ肳ĂȂꍇ́A{@link DefaultQueueService}𐶐{@link #setQueueService(Queue)}Őݒ肷B</li>
     *   <li>{@link #setMessageRecordFactoryServiceName(ServiceName)}{@link MessageRecordFactory}T[rX̖Oݒ肳Ăꍇ́A{@link ServiceManager}MessageRecordFactory擾{@link #setMessageRecordFactoryService(MessageRecordFactory)}Őݒ肷B܂AMessageRecordFactoryT[rX̖Oݒ肳ĂȂꍇ́A{@link DefaultMessageRecordFactoryService}𐶐{@link #setMessageRecordFactoryService(MessageRecordFactory)}Őݒ肷B</li>
     *   <li>{@link #setContextServiceName(ServiceName)}{@link Context}T[rX̖Oݒ肳Ăꍇ́A{@link ServiceManager}Context擾{@link #setContextService(Context)}Őݒ肷B</li>
     *   <li>{@link Daemon}NB</li>
     * </ol>
     * 
     * @exception Exception JnɎsꍇB
     */
    public void startService() throws Exception{
        
        // ftHgMessageWriterT[rX̐܂͎擾
        if(getDefaultMessageWriterServiceName() == null){
            if(getConsoleWriterService() == null){
                final ConsoleMessageWriterService consoleWriter
                     = new ConsoleMessageWriterService();
                consoleWriter.setOutput(ConsoleMessageWriterService.OUTPUT_STDOUT);
                consoleWriter.create();
                consoleWriter.start();
                setConsoleWriterService(consoleWriter);
            }else{
                getConsoleWriterService().start();
            }
            setDefaultMessageWriterService(getConsoleWriterService());
        }else{
            setDefaultMessageWriterService(
                ServiceManagerFactory
                    .<MessageWriter>getServiceObject(getDefaultMessageWriterServiceName())
            );
        }
        
        // ftHgWritableRecordFactoryT[rX̐܂͎擾
        if(getDefaultWritableRecordFactoryServiceName() == null){
            if(getWritableRecordFactoryService() == null){
                final DefaultWritableRecordFactoryService recordFactory
                     = new DefaultWritableRecordFactoryService();
                recordFactory.setFormat(getDefaultFormat());
                recordFactory.create();
                recordFactory.start();
                setWritableRecordFactoryService(recordFactory);
            }else{
                getWritableRecordFactoryService().start();
            }
            setDefaultWritableRecordFactoryService(
                getWritableRecordFactoryService()
            );
        }else{
            setDefaultWritableRecordFactoryService(
                ServiceManagerFactory.<WritableRecordFactory>getServiceObject(
                    getDefaultWritableRecordFactoryServiceName()
                )
            );
        }
        
        // VXeJeS̓o^
        initDefaultCategory();
        
        // [U`JeS̓o^
        final ServiceName[] categoryNames = getCategoryServiceNames();
        if(categoryNames != null){
            for(ServiceName categoryName : categoryNames){
                final LogCategory category = ServiceManagerFactory
                    .<LogCategory>getServiceObject(categoryName);
                addCategoryService(category);
            }
        }
        
        // QueueT[rX̐܂͎擾
        if(getQueueServiceName() == null){
            if(getDefaultQueueService() == null){
                final DefaultQueueService<LogEnqueuedRecord> defaultQueue
                     = new DefaultQueueService<LogEnqueuedRecord>();
                defaultQueue.create();
                defaultQueue.start();
                setDefaultQueueService(defaultQueue);
            }else{
                getDefaultQueueService().start();
            }
            setQueueService(getDefaultQueueService());
        }else{
            setQueueService(ServiceManagerFactory
                    .<Queue<LogEnqueuedRecord>>getServiceObject(getQueueServiceName())
            );
        }
        
        // MessageRecordFactoryT[rX̐܂͎擾
        if(getMessageRecordFactoryServiceName() == null){
            if(getDefaultMessageRecordFactoryService() == null){
                final LogMessageRecordFactoryService logMessageFactory
                     = new LogMessageRecordFactoryService();
                logMessageFactory.create();
                logMessageFactory.start();
                setDefaultMessageRecordFactoryService(logMessageFactory);
            }else{
                getDefaultMessageRecordFactoryService().start();
            }
            setMessageRecordFactoryService(defaultMessageFactory);
        }else{
            setMessageRecordFactoryService(
                ServiceManagerFactory
                    .<MessageRecordFactory>getServiceObject(getMessageRecordFactoryServiceName())
            );
        }
        
        // ContextT[rX̎擾
        if(getContextServiceName() != null){
            setContextService(ServiceManagerFactory
                    .<Context<Object, Object>>getServiceObject(getContextServiceName())
            );
        }
        
        // L[擾҂Jn
        queue.accept();
        
        daemon.setDaemon(isDaemon);
        
        // f[N
        daemon.start();
    }
    
    /**
     * ̃OT[rXŃftHgŎ{@link LogCategory}𐶐ēo^B<p>
     *
     * @param defaultMessageWriter ftHgMessageWriterBmessageWriterNameɗLȃT[rXw肳ĂȂꍇLogCategoryŎgpB
     * @param defaultRecordFactory ftHgWritableRecordFactoryBrecordFactoryNameɗLȃT[rXw肳ĂȂꍇLogCategoryŎgpB
     * @param messageWriterName LogCategoryŎgpMessageWriterT[rX
     * @param recordFactoryName LogCategoryŎgpWritableRecordFactoryT[rX
     * @param categoryName JeS
     * @param priorityMin D揇ʔ͈͂̍ŏl
     * @param priorityMax D揇ʔ͈͂̍ől
     * @param label JeS̗D揇ʔ͈͂̃x
     * @param isEnabled o͂Lɂ邩ǂ̃tOBo͂Ԃɂꍇ true
     * @exception Exception JeST[rX̐EJnɎsꍇ
     */
    protected void addDefaultCategory(
        MessageWriter defaultMessageWriter,
        WritableRecordFactory defaultRecordFactory,
        ServiceName messageWriterName,
        ServiceName recordFactoryName,
        String categoryName,
        int priorityMin,
        int priorityMax,
        String label,
        boolean isEnabled
    ) throws Exception{
        MessageWriter messageWriter = defaultMessageWriter;
        WritableRecordFactory recordFactory = defaultRecordFactory;
        if(messageWriterName != null){
            messageWriter = ServiceManagerFactory
                .<MessageWriter>getServiceObject(messageWriterName);
        }
        if(recordFactoryName != null){
            recordFactory = ServiceManagerFactory
                .<WritableRecordFactory>getServiceObject(recordFactoryName);
        }
        final SimpleLogCategoryService category = new SimpleLogCategoryService();
        category.setCategoryName(categoryName);
        category.setPriorityRangeValue(priorityMin, priorityMax);
        category.setLabel(priorityMin, priorityMax, label);
        category.create();
        category.start();
        category.setMessageWriterService(messageWriter);
        category.setWritableRecordFactoryService(recordFactory);
        
        addDefaultCategoryService(category);
        
        setEnabled(categoryName, isEnabled);
    }
    
    protected void initDefaultCategory() throws Exception{
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getSystemDebugMessageWriterServiceName(),
            getSystemDebugWritableRecordFactoryServiceName(),
            SYSTEM_DEBUG_CATEGORY,
            SYSTEM_DEBUG_CATEGORY_PRIORITY_MIN,
            SYSTEM_DEBUG_CATEGORY_PRIORITY_MAX,
            SYSTEM_DEBUG_CATEGORY_LABEL,
            isSystemDebugEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getSystemInfoMessageWriterServiceName(),
            getSystemInfoWritableRecordFactoryServiceName(),
            SYSTEM_INFO_CATEGORY,
            SYSTEM_INFO_CATEGORY_PRIORITY_MIN,
            SYSTEM_INFO_CATEGORY_PRIORITY_MAX,
            SYSTEM_INFO_CATEGORY_LABEL,
            isSystemInfoEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getSystemWarnMessageWriterServiceName(),
            getSystemWarnWritableRecordFactoryServiceName(),
            SYSTEM_WARN_CATEGORY,
            SYSTEM_WARN_CATEGORY_PRIORITY_MIN,
            SYSTEM_WARN_CATEGORY_PRIORITY_MAX,
            SYSTEM_WARN_CATEGORY_LABEL,
            isSystemWarnEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getSystemErrorMessageWriterServiceName(),
            getSystemErrorWritableRecordFactoryServiceName(),
            SYSTEM_ERROR_CATEGORY,
            SYSTEM_ERROR_CATEGORY_PRIORITY_MIN,
            SYSTEM_ERROR_CATEGORY_PRIORITY_MAX,
            SYSTEM_ERROR_CATEGORY_LABEL,
            isSystemErrorEnabled()
        );
        addDefaultCategory(
            getDefaultMessageWriterService(),
            getDefaultWritableRecordFactoryService(),
            getSystemFatalMessageWriterServiceName(),
            getSystemFatalWritableRecordFactoryServiceName(),
            SYSTEM_FATAL_CATEGORY,
            SYSTEM_FATAL_CATEGORY_PRIORITY_MIN,
            SYSTEM_FATAL_CATEGORY_PRIORITY_MAX,
            SYSTEM_FATAL_CATEGORY_LABEL,
            isSystemFatalEnabled()
        );
    }
    
    /**
     * ~sB<p>
     * ̃\bhɂ́Aȉ̎sĂB<br>
     * <ol>
     *   <li>ftHgMessageWriterT[rX𖳖T[rXƂĐĂꍇÃT[rX~B</li>
     *   <li>ftHgWritableRecordFactoryT[rX𖳖T[rXƂĐĂꍇÃT[rX~B</li>
     *   <li>QueueT[rX𖳖T[rXƂĐĂꍇA
        ̃T[rX~B</li>
     *   <li>MessageRecordFactoryT[rX𖳖T[rXƂĐĂꍇA
        ̃T[rX~B</li>
     *   <li>JeS폜B</li>
     *   <li>{@link Daemon}~B</li>
     * </ol>
     * 
     * @exception Exception ~ɎsꍇB
     */
    public void stopService(){
        
        // f[~
        daemon.stop();
        
        // L[擾҂J
        queue.release();
        
        // ftHgMessageWriterT[rX𖳖T[rXƂĐ
        // ꍇÃT[rX~
        if(getDefaultMessageWriterService() == getConsoleWriterService()){
            getConsoleWriterService().stop();
        }
        
        // ftHgWritableRecordFactoryT[rX𖳖T[rXƂ
        // ĂꍇÃT[rX~
        if(getDefaultWritableRecordFactoryService()
             == getWritableRecordFactoryService()){
            getWritableRecordFactoryService().stop();
        }
        
        // QueueT[rX𖳖T[rXƂĐĂꍇA
        // ̃T[rX~
        if(getQueueService() == getDefaultQueueService()){
            getDefaultQueueService().stop();
        }
        
        // MessageRecordFactoryT[rX𖳖T[rXƂĐĂꍇA
        // ̃T[rX~
        if(getMessageRecordFactoryService()
             == getDefaultMessageRecordFactoryService()){
            getDefaultMessageRecordFactoryService().stop();
        }
        
        // JeS폜
        categoryMap.clear();
        defaultCategoryMap.clear();
    }
    
    /**
     * jsB<p>
     * ̃\bhɂ́Aȉ̎sĂB<br>
     * <ol>
     *   <li>ftHgMessageWriterT[rX𖳖T[rXƂĐĂꍇÃT[rXjB</li>
     *   <li>ftHgWritableRecordFactoryT[rX𖳖T[rXƂĐĂꍇÃT[rXjB</li>
     *   <li>QueueT[rX𖳖T[rXƂĐĂꍇA
        ̃T[rXjB</li>
     *   <li>MessageRecordFactoryT[rX𖳖T[rXƂĐĂꍇA
        ̃T[rXjB</li>
     *   <li>JeSjB</li>
     *   <li>{@link Daemon}jB</li>
     * </ol>
     * 
     * @exception Exception jɎsꍇB
     */
    public void destroyService(){
        
        // ftHgMessageWriterT[rX𖳖T[rXƂĐ
        // ꍇÃT[rXj
        if(getDefaultMessageWriterService() == getConsoleWriterService()
            && getConsoleWriterService() != null){
            getConsoleWriterService().destroy();
            setConsoleWriterService(null);
        }
        
        // ftHgWritableRecordFactoryT[rX𖳖T[rXƂ
        // ĂꍇÃT[rXj
        if(getDefaultWritableRecordFactoryService()
             == getWritableRecordFactoryService()
             && getWritableRecordFactoryService() != null){
            getWritableRecordFactoryService().destroy();
            setWritableRecordFactoryService(null);
        }
        
        // QueueT[rX𖳖T[rXƂĐĂꍇA
        // ̃T[rXj
        if(getQueueService() == getDefaultQueueService()
            && getDefaultQueueService() != null){
            getDefaultQueueService().destroy();
            setDefaultQueueService(null);
        }
        
        // MessageRecordFactoryT[rX𖳖T[rXƂĐĂꍇA
        // ̃T[rXj
        if(getMessageRecordFactoryService()
             == getDefaultMessageRecordFactoryService()
            && getDefaultMessageRecordFactoryService() != null){
            getDefaultMessageRecordFactoryService().destroy();
            setDefaultMessageRecordFactoryService(null);
        }
        
        // JeSǗMapj
        categoryMap = null;
        defaultCategoryMap = null;
        
        // f[j
        daemon = null;
    }
    
    /**
     * JeS{@link LogCategory}̃}bsO擾B<p>
     *
     * @return JeS{@link LogCategory}̃}bsO
     */
    protected Map<String, LogCategory> getCategoryMap(){
        return categoryMap;
    }
    
    /**
     * ftHg̃JeS{@link LogCategory}̃}bsO擾B<p>
     *
     * @return JeS{@link LogCategory}̃}bsO
     */
    protected Map<String, LogCategory> getDefaultCategoryMap(){
        return defaultCategoryMap;
    }
    
    public void setDefaultMessageWriterService(MessageWriter writer){
        defaultMessageWriter = writer;
    }
    
    public MessageWriter getDefaultMessageWriterService(){
        return defaultMessageWriter;
    }
    
    /**
     * ftHgMessageWriterw肳ĂȂꍇɎgp{@link ConsoleMessageWriterService}擾B<p>
     * ConsoleWriterServicéAT[rXƂĐB܂A{@link #setDefaultMessageWriterServiceName(ServiceName)}ŃftHgMessageWriterw肳Ăꍇ́AnullԂꍇB<br>
     *
     * @return ConsoleWriterServiceIuWFNgBĂȂꍇ́AnullԂB
     */
    protected ConsoleMessageWriterService getConsoleWriterService(){
        return consoleWriter;
    }
    
    /**
     * ftHgMessageWriterw肳ĂȂꍇɎgp{@link ConsoleMessageWriterService}ݒ肷B<p>
     *
     * @param consoleWriter ConsoleWriterServiceIuWFNg
     */
    protected void setConsoleWriterService(ConsoleMessageWriterService consoleWriter){
        this.consoleWriter = consoleWriter;
    }
    
    public void setDefaultWritableRecordFactoryService(
        WritableRecordFactory recordFactory
    ){
        defaultWritableRecordFactory = recordFactory;
    }
    
    public WritableRecordFactory getDefaultWritableRecordFactoryService(){
        return defaultWritableRecordFactory;
    }
    
    /**
     * ftHgWritableRecordFactoryw肳ĂȂꍇɎgp{@link DefaultWritableRecordFactoryService}擾B<p>
     * DefaultWritableRecordFactoryServicéAT[rXƂĐB܂A{@link #setDefaultWritableRecordFactoryServiceName(ServiceName)}ŃftHgWritableRecordFactoryw肳Ăꍇ́AnullԂꍇB<br>
     *
     * @return DefaultWritableRecordFactoryServiceIuWFNgBĂȂꍇ́AnullԂB
     */
    protected DefaultWritableRecordFactoryService getWritableRecordFactoryService(){
        return defaultWritableRecordFactoryService;
    }
    
    /**
     * ftHgWritableRecordFactoryw肳ĂȂꍇɎgp{@link DefaultWritableRecordFactoryService}ݒ肷B<p>
     *
     * @param logRecordFactory DefaultWritableRecordFactoryServiceIuWFNg
     */
    protected void setWritableRecordFactoryService(
        DefaultWritableRecordFactoryService logRecordFactory
    ){
        this.defaultWritableRecordFactoryService = logRecordFactory;
    }
    
    public void setCategoryServices(LogCategory[] categories){
        if(categoryMap != null){
            categoryMap.clear();
            // w肳ꂽJeSo^
            if(categories != null){
                for(LogCategory category : categories){
                    if(category != null){
                        addCategoryService(category);
                    }
                }
            }
        }
    }
    
    public LogCategory[] getCategoryServices(){
        final Map<String, LogCategory> result = new HashMap<String, LogCategory>();
        if(defaultCategoryMap != null){
            result.putAll(defaultCategoryMap);
        }
        if(categoryMap != null){
            result.putAll(categoryMap);
        }
        return result.values().toArray(
            new LogCategory[categoryMap.size()]
        );
    }
    
    // LogServiceMBeanJavaDoc
    public void addCategoryService(LogCategory category){
        if(categoryMap != null && category != null){
            categoryMap.put(category.getCategoryName(), category);
        }
    }
    
    /**
     * ftHg{@link LogCategory}T[rXǉB<p>
     *
     * @parma category LogCategoryT[rX
     */
    public void addDefaultCategoryService(LogCategory category){
        if(defaultCategoryMap != null && category != null){
            defaultCategoryMap.put(category.getCategoryName(), category);
        }
    }
    
    public LogCategory getCategoryService(String name){
        LogCategory category = null;
        if(categoryMap != null){
            category = categoryMap.get(name);
        }
        if(category == null && defaultCategoryMap != null){
            category = defaultCategoryMap.get(name);
        }
        return category;
    }
    
    public void setQueueService(Queue<LogEnqueuedRecord> queue){
        this.queue = queue;
    }
    
    public Queue<LogEnqueuedRecord> getQueueService(){
        return queue;
    }
    
    /**
     * Queuew肳ĂȂꍇɎgp{@link DefaultQueueService}擾B<p>
     * DefaultQueueServicéAT[rXƂĐB܂A{@link #setQueueServiceName(ServiceName)}Queuew肳Ăꍇ́AnullԂꍇB<br>
     *
     * @return DefaultQueueServiceIuWFNgBĂȂꍇnullԂB
     */
    protected DefaultQueueService<LogEnqueuedRecord> getDefaultQueueService(){
        return defaultQueue;
    }
    
    /**
     * Queuew肳ĂȂꍇɎgp{@link DefaultQueueService}ݒ肷B<p>
     *
     * @param queue DefaultQueueServiceIuWFNg
     */
    protected void setDefaultQueueService(DefaultQueueService<LogEnqueuedRecord> queue){
        defaultQueue = queue;
    }
    
    /**
     * MessageRecordFactoryw肳ĂȂꍇɎgp{@link LogMessageRecordFactoryService}擾B<p>
     * LogMessageRecordFactoryServicéAT[rXƂĐB܂A{@link #setMessageRecordFactoryServiceName(ServiceName)}MessageRecordFactoryw肳Ăꍇ́AnullԂꍇB<br>
     *
     * @return LogMessageRecordFactoryServiceIuWFNgBĂȂꍇnullԂB
     */
    protected LogMessageRecordFactoryService getDefaultMessageRecordFactoryService(){
        return defaultMessageFactory;
    }
    
    /**
     * MessageRecordFactoryw肳ĂȂꍇɎgp{@link jp.ossc.nimbus.service.message.LogMessageRecordFactoryService LogMessageRecordFactoryService}ݒ肷B<p>
     *
     * @param message LogMessageRecordFactoryServiceIuWFNg
     */
    protected void setDefaultMessageRecordFactoryService(
        LogMessageRecordFactoryService message
    ){
        defaultMessageFactory = message;
    }
    
    protected void setMessageRecordFactoryService(MessageRecordFactory factory){
        messageFactory = factory;
    }
    protected MessageRecordFactory getMessageRecordFactoryService(){
        return messageFactory;
    }
    
    // LogServiceMBeanJavaDoc
    public void setContextService(Context<Object, Object> context){
        this.context = context;
    }
    
    // LogServiceMBeanJavaDoc
    public Context<Object, Object> getContextService(){
        return context;
    }
    
    // LogServiceMBeanJavaDoc
    public void addContextFormatKey(String key){
        if(key != null){
            contextKeys.add(key);
        }
    }
    
    // LogServiceMBeanJavaDoc
    public void removeContextFormatKey(String key){
        if(key != null){
            contextKeys.remove(key);
        }
    }
    
    // LogServiceMBeanJavaDoc
    public void clearContextFormatKeys(){
        contextKeys.clear();
    }
    
    protected String getDefaultFormat(){
        return defaultFormat;
    }
    
    /**
     * w肳ꂽL[{@link Context}T[rXl擾B<p>
     *
     * @param key L[
     * @return {@link Context}T[rX擾l
     */
    protected Object getContextFormatValue(String key){
        final Context<Object, Object> context = getContextService();
        if(context != null){
            return context.get(key);
        }
        return null;
    }
    
    /**
     * w肳ꂽ{@link LogMessageRecord}o͂邩肷B<p>
     *
     * @param messageRecord LogMessageRecord
     * @return o͂ꍇ true
     */
    protected boolean isWrite(LogMessageRecord messageRecord){
        final int priority = messageRecord.getPriority();
        for(String categoryName : messageRecord.getCategories()){
            final LogCategory category = getCategoryService(categoryName);
            if(category == null){
                continue;
            }
            if(category.isEnabled() && category.isValidPriorityRange(priority)){
                return true;
            }
        }
        return false;
    }
    
    /**
     * ÕL[ɑ}{@link LogEnqueuedRecord}𐶐B<p>
     * {@link #write(LogMessageRecord, Locale, String[], Throwable)}ĂяoB<br>
     *
     * @param messageRecord Oo͗v̂LogMessageRecordIuWFNg
     * @param locale Oo͂Ɏgp郁bZ[W̃P[
     * @param throwable Oo͂ɎgpOBObZ[WłȂꍇ́AnullB
     * @param embeds Oo͂̃bZ[WɎgp閄ߍݕBߍ݂̂ȂbZ[W̏ꍇ́AnullB
     * @return ÕL[ɑ}LogEnqueuedRecord
     */
    protected LogEnqueuedRecord createLogEnqueuedRecord(
        LogMessageRecord messageRecord,
        Locale locale,
        Throwable throwable,
        String... embeds
    ){
        return new LogEnqueuedRecord(messageRecord, locale, throwable, embeds);
    }
    
    /**
     * ÕL[ɑ}{@link LogEnqueuedRecord}𐶐ăL[ɑ}B<p>
     * LogEnqueuedRecord̐́A{@link #createLogEnqueuedRecord(LogMessageRecord, Locale, String, Throwable)}ĂяočsB<br>
     * L[ւ̑}́A{@link #enqueue(LogEnqueuedRecord)}ĂяočsB<br>
     *
     * @param messageRecord o͂LogMessageRecord
     * @param locale Oo͂Ɏgp郁bZ[W̃P[
     * @param embeds Oo͂̃bZ[WɎgp閄ߍݕBߍ݂̂ȂbZ[W̏ꍇ́AnullB
     * @param throwable Oo͂ɎgpOBObZ[WłȂꍇ́AnullB
     */
    protected void write(
        LogMessageRecord messageRecord,
        Locale locale,
        Throwable throwable,
        String... embeds
    ){
        final LogEnqueuedRecord enqueuedRecord = createLogEnqueuedRecord(
            messageRecord,
            locale,
            throwable,
            embeds
        );
        enqueue(enqueuedRecord);
    }
    
    /**
     * ÕL[ɑ}OsB<p>
     * {@link #FORMAT_DATE_KEY}ɑΉ{@link Date}IuWFNg𐶐āA{@link LogEnqueuedRecord#addWritableElement(Object, Object)}{@link jp.ossc.nimbus.service.writer.WritableElement WritableElement}ƂĒǉB<br>
     * ܂A{@link #setContextFormatKeys(String[])}Őݒ肳ꂽL[gāA{@link #setContextServiceName(ServiceName)}Ŏw肳ꂽ{@link Context}T[rXIuWFNg擾B̃IuWFNgAReLXgtH[}bgƂ{@link LogEnqueuedRecord#addWritableElement(Object, Object)}{@link jp.ossc.nimbus.service.writer.WritableElement WritableElement}ƂĒǉB<br>
     *
     * @param enqueuedRecord L[ɑ}LogEnqueuedRecord
     */
    protected void preEnqueue(LogEnqueuedRecord enqueuedRecord){
        enqueuedRecord.addWritableElement(
            FORMAT_DATE_KEY,
            new Date()
        );
        final String[] keys = getContextFormatKeys();
        if(keys != null){
            for(String key : keys){
                if(key != null){
                    final Object val = getContextFormatValue(key);
                    if(val != null){
                        enqueuedRecord.addWritableElement(
                            key,
                            val
                        );
                    }
                }
            }
        }
    }
    
    /**
     * ÕL[ɑ}B<p>
     * L[}OɁA{@link #preEnqueue(LogEnqueuedRecord)}ĂяoB<br>
     * 
     * @param enqueuedRecord LogEnqueuedRecordIuWFNg
     */
    protected void enqueue(LogEnqueuedRecord enqueuedRecord){
        preEnqueue(enqueuedRecord);
        queue.push(enqueuedRecord);
    }
    
    /**
     * ÕL[ȍsB<p>
     * {@link LogMessageRecord#makeMessage(Locale, Object[])}ŏo̓bZ[W𐶐BbZ[W{@link #FORMAT_MESSAGE_KEY}ɑΉ郁bZ[WƂāA{@link LogEnqueuedRecord#addWritableElement(Object, Object)}{@link jp.ossc.nimbus.service.writer.WritableElement WritableElement}ƂĒǉB<br>
     * ܂A{@link #FORMAT_CODE_KEY}ɑΉ郁bZ[WR[h擾āA{@link LogEnqueuedRecord#addWritableElement(Object, Object)}{@link jp.ossc.nimbus.service.writer.WritableElement WritableElement}ƂĒǉB<br>
     *
     * @param dequeuedRecord LogEnqueuedRecordIuWFNg
     */
    protected void postDequeue(LogEnqueuedRecord dequeuedRecord){
        final LogMessageRecord messageRecord
             = dequeuedRecord.getLogMessageRecord();
        final Locale locale = dequeuedRecord.getLocale();
        final String[] embeds = dequeuedRecord.getEmbedStringArray();
        final Throwable throwable = dequeuedRecord.getThrowable();
        String message = messageRecord.makeMessage(locale, (Object[])embeds);
        if(throwable != null && messageRecord.isPrintStackTrace()){
            final StringBuilder buf = new StringBuilder(message);
            buf.append(LINE_SEP);
            buf.append(getStackTraceString(throwable));
            message = buf.toString();
        }
        dequeuedRecord.addWritableElement(
            FORMAT_CODE_KEY,
            messageRecord.getMessageCode()
        );
        dequeuedRecord.addWritableElement(
            FORMAT_MESSAGE_KEY,
            message
        );
    }
    
    /**
     * w肳ꂽ{@link LogMessageRecord}o͂{@link LogCategory}擾B<p>
     *
     * @param messageRecord LogMessageRecord
     * @return o͂LogCategory̔z
     */
    protected LogCategory[] getWriteCategories(LogMessageRecord messageRecord){
        final List<LogCategory> result = new ArrayList<LogCategory>();
        final int priority = messageRecord.getPriority();
        for(String categoryName : messageRecord.getCategories()){
            final LogCategory category = getCategoryService(categoryName);
            if(category != null && category.isEnabled()
                && category.isValidPriorityRange(priority)){
                result.add(category);
            }
        }
        return result.toArray(new LogCategory[result.size()]);
    }
    
    /**
     * L[oɁAJeS{@link WritableRecord}𐶐B<p>
     * {@link #dequeue(LogEnqueuedRecord)}ĂяoB<br>
     *
     * @param dequeuedRecord L[oLogEnqueuedRecordIuWFNg
     * @param category LogCategoryIuWFNg
     */
    protected Map<Object, Object> createWritableElementMap(
        LogEnqueuedRecord dequeuedRecord,
        LogCategory category
    ){
        final LogMessageRecord messageRecord
             = dequeuedRecord.getLogMessageRecord();
        dequeuedRecord.addWritableElement(
            FORMAT_PRIORITY_KEY,
            category.getLabel(messageRecord.getPriority())
        );
        dequeuedRecord.addWritableElement(
            FORMAT_CATEGORY_KEY,
            category.getCategoryName()
        );
        return dequeuedRecord.getWritableElements();
    }
    
    /**
     * L[oꂽ{@link LogEnqueuedRecord}JeS{@link WritableRecord}𐶐{@link MessageWriter}ɏo͂˗B<p>
     * {@link #postDequeue(LogEnqueuedRecord)}ĂяoāAL[ȍsB<br>
     * ܂A{@link #getWriteCategories(LogMessageRecord)}ŏoׂ͂{@link LogCategory}擾āAJeS{@link #createWritableElementMap(LogEnqueuedRecord, LogCategory)}Map𐶐BMapJeS{@link LogCategory#write(int, Map)}gāAo͂˗B<br>
     *
     * @param dequeuedRecord L[oLogEnqueuedRecordIuWFNg
     */
    protected void dequeue(LogEnqueuedRecord dequeuedRecord){
        postDequeue(dequeuedRecord);
        final LogMessageRecord messageRecord
             = dequeuedRecord.getLogMessageRecord();
        final LogCategory[] categories = getWriteCategories(messageRecord);
        for(LogCategory category : categories){
            try{
                category.write(
                    messageRecord.getPriority(),
                    createWritableElementMap(dequeuedRecord, category)
                );
            }catch(MessageWriteException e){
                // 
            }
        }
    }
    
    // LoggerJavaDoc
    public void write(String logCode, Object... embeds) {
        write(logCode, Locale.getDefault(), null, embeds);
    }
    
    /**
     * ObjectzStringzɕϊB<p>
     *
     * @param vals Objectz
     * @return Stringz
     */
    protected static String[] convertStringArray(Object[] vals){
        String[] strings = null;
        if(vals != null){
            strings = new String[vals.length];
            for(int i = 0; i < vals.length; i++){
                if(vals[i] != null){
                    strings[i] = vals[i].toString();
                }
            }
        }
        return strings;
    }
    
    // LoggerJavaDoc
    public void write(String logCode, Locale lo, Object... embeds){
        final LogMessageRecord  messageRecord
             = (LogMessageRecord)messageFactory.findMessageRecord(logCode);
        if(messageRecord == null || !isWrite(messageRecord)){
            return;
        }
        write(messageRecord, lo, null, convertStringArray(embeds));
    }
    
    // LoggerJavaDoc
    public void write(String logCode, Throwable oException, Object... embeds) {
        write(logCode, Locale.getDefault(), oException, embeds);
    }
    
    // LoggerJavaDoc
    public void write(
        String logCode,
        Locale lo,
        Throwable oException,
        Object... embeds
    ){
        final LogMessageRecord  messageRecord
             = (LogMessageRecord)messageFactory.findMessageRecord(logCode);
        if(messageRecord == null || !isWrite(messageRecord)){
            return;
        }
        write(messageRecord, lo, oException, convertStringArray(embeds));
    }
    
    // LoggerJavaDoc
    public boolean isWrite(String logCode){
        final LogMessageRecord  messageRecord
             = (LogMessageRecord)messageFactory.findMessageRecord(logCode);
        if(messageRecord == null || !isWrite(messageRecord)){
            return false;
        }
        return true;
    }
    
    /**
     * ÕX^bNg[X擾B<p>
     *
     * @param e O
     * @return X^bNg[X
     */
    protected static String getStackTraceString(Throwable e){
        final StringBuilder buf = new StringBuilder();
        buf.append(e).append(LINE_SEP);
        final StackTraceElement[] elemss = e.getStackTrace();
        if(elemss != null){
            for(int i = 0, max = elemss.length; i < max; i++){
                buf.append('\t');
                buf.append(elemss[i]);
                if(i != max - 1){
                    buf.append(LINE_SEP);
                }
            }
        }
        for(Throwable ee = getCause(e); ee != null; ee = getCause(ee)){
            buf.append(LINE_SEP).append(MSG_CAUSE)
                .append(ee).append(LINE_SEP);
            final StackTraceElement[] elems = ee.getStackTrace();
            if(elems != null){
                for(int i = 0, max = elems.length; i < max; i++){
                    buf.append('\t');
                    buf.append(elems[i]);
                    if(i != max - 1){
                        buf.append(LINE_SEP);
                    }
                }
            }
        }
        return buf.toString();
    }
    
    private static Throwable getCause(Throwable th){
        Throwable cause = null;
        String thClassName = th.getClass().getName();
        if(thClassName.equals(SERVLET_EXCEPTION_NAME)){
            // OServletException̏ꍇ́A[ǧ擾
            try{
                cause = (Throwable)th.getClass()
                    .getMethod(GET_ROOT_CAUSE_METHOD, (Class[])null).invoke(th, (Object[])null);
            }catch(NoSuchMethodException e){
            }catch(IllegalAccessException e){
            }catch(java.lang.reflect.InvocationTargetException e){
            }
        }else if(thClassName.equals(JMS_EXCEPTION_NAME)){
            // OJMSException̏ꍇ́ANO擾
            try{
                cause = (Exception)th.getClass()
                    .getMethod(GET_LINKED_EXCEPTION_METHOD, (Class[])null).invoke(th, (Object[])null);
            }catch(NoSuchMethodException e){
            }catch(IllegalAccessException e){
            }catch(java.lang.reflect.InvocationTargetException e){
            }
        }else{
            cause = th.getCause();
        }
        return cause == th ? null : cause;
    }
    
    /**
     * wJeS̗D揇ʔ͈̗͂L/ݒ肷B<p>
     *
     * @param categoryName JeS
     * @param isEnabled Lɂꍇ true
     */
    protected void setEnabled(
        String categoryName,
        boolean isEnabled
    ){
        final LogCategory category = getCategoryService(categoryName);
        if(category == null){
            return;
        }
        category.setEnabled(isEnabled);
    }
    
    /**
     * w肳ꂽJeS̗D揇ʔ͈͂L𒲂ׂB<p>
     *
     * @param categoryName JeS
     * @param defaultEnabled JeS݂Ȃꍇ̖߂l
     */
    protected boolean isEnabled(
        String categoryName,
        boolean defaultEnabled
    ){
        final LogCategory category = getCategoryService(categoryName);
        if(category == null){
            return defaultEnabled;
        }
        return category.isEnabled();
    }
    
    protected class LogWriteDaemon implements DaemonRunnable<LogEnqueuedRecord>{
        
        /**
         * f[JnɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStart() {
            return true;
        }
        
        /**
         * f[~ɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onStop() {
            return true;
        }
        
        /**
         * f[fɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onSuspend() {
            return true;
        }
        
        /**
         * f[ĊJɌĂяoB<p>
         * 
         * @return trueԂ
         */
        public boolean onResume() {
            return true;
        }
        
        /**
         * L[PoĕԂB<p>
         * 
         * @param ctrl DaemonControlIuWFNg
         * @return {@link LogEnqueuedRecord}IuWFNg
         */
        public LogEnqueuedRecord provide(DaemonControl ctrl){
            return queue.get(10000);
        }
        
        /**
         * dequeuedœnꂽIuWFNgB<p>
         * dequeuedœnꂽIuWFNg{@link LogEnqueuedRecord}ɃLXg{@link #dequeue(LogEnqueuedRecord)}ĂяoB<br>
         *
         * @param dequeued L[oꂽIuWFNg
         * @param ctrl DaemonControlIuWFNg
         */
        public void consume(LogEnqueuedRecord dequeued, DaemonControl ctrl){
            if(dequeued == null){
                return;
            }
            try{
                dequeue(dequeued);
            }catch(Throwable e){
                e.printStackTrace();
            }
        }
        
        /**
         * L[̒gfoB<p>
         */
        public void garbage(){
            if(queue != null){
                while(queue.size() > 0){
                    consume(queue.get(0), daemon);
                }
            }
        }
    }
    
    protected static class LogEnqueuedRecord implements java.io.Serializable{
        
        private static final long serialVersionUID = 6657674897069127746L;
        
        private LogMessageRecord messageRecord;
        private Map<Object, Object> elements;
        private List<String> embeds;
        private Locale locale;
        private Throwable throwable;
        
        public LogEnqueuedRecord(){
            this(null, null, null);
        }
        
        public LogEnqueuedRecord(
            LogMessageRecord messageRecord,
            Locale locale,
            Throwable throwable,
            String... embeds
        ){
            this.messageRecord = messageRecord;
            this.elements = new HashMap<Object, Object>();
            this.embeds = new ArrayList<String>();
            setEmbedStringArray(embeds);
            this.locale = locale;
            if(locale == null){
                this.locale = Locale.getDefault();
            }
            this.throwable = throwable;
        }
        
        public LogMessageRecord getLogMessageRecord(){
            return messageRecord;
        }
        public void setLogMessageRecord(LogMessageRecord record){
            messageRecord = record;
        }
        public void setEmbedStringArray(String[] embeds){
            if(embeds == null){
                return;
            }
            this.embeds.clear();
            for(String embed : embeds){
                this.embeds.add(embed);
            }
        }
        public String[] getEmbedStringArray(){
            if(embeds.size() == 0){
                return null;
            }
            return embeds.toArray(new String[embeds.size()]);
        }
        public void addEmbedString(String embed){
            embeds.add(embed);
        }
        public void addWritableElement(Object key, Object element){
            elements.put(key, element);
        }
        public Map<Object, Object> getWritableElements(){
            return elements;
        }
        public void setLocale(Locale locale){
            this.locale = locale;
        }
        public Locale getLocale(){
            return locale;
        }
        public void setThrowable(Throwable throwable){
            this.throwable = throwable;
        }
        public Throwable getThrowable(){
            return throwable;
        }
        public String toString(){
            StringBuilder buf = new StringBuilder(super.toString());
            buf.append('{');
            buf.append("messageRecord=").append(messageRecord);
            buf.append(",elements=").append(elements);
            if(embeds == null){
                buf.append(",embeds=").append(embeds);
            }else{
                buf.append(",embeds=[");
                for(String embed : embeds){
                    buf.append(embed).append(',');
                }
                buf.deleteCharAt(buf.length() - 1);
                buf.append(']');
            }
            buf.append(",locale=").append(locale);
            buf.append(",throwable=").append(throwable);
            return buf.toString();
        }
    }
}
