/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.service.cache;

import java.util.*;

import jp.ossc.nimbus.core.ServiceBase;

/**
 * Least Recently Used ӂASYT[rXB<p>
 * ȉɁALRUłӂΏۂƂȂLbVIuWFNg肷邠ӂASYT[rX̃T[rX`B<br>
 * <pre>
 * &lt;?xml version="1.0" encoding="Shift_JIS"?&gt;
 * 
 * &lt;nimbus&gt;
 *     
 *     &lt;manager name="Sample"&gt;
 *         
 *         &lt;service name="LRUOverflowAlgorithm"
 *                  code="jp.ossc.nimbus.service.cache.LRUOverflowAlgorithmService"/&gt;
 *         
 *     &lt;/manager&gt;
 *     
 * &lt;/nimbus&gt;
 * </pre>
 *
 * @author M.Takata
 */
public class LRUOverflowAlgorithmService<E> extends ServiceBase
 implements OverflowAlgorithm<E>, CacheRemoveListener<E>, CacheAccessListener<E>,
            java.io.Serializable, LRUOverflowAlgorithmServiceMBean{
    
    private static final long serialVersionUID = 2274633140923055371L;
    
    private Map<CachedReference<E>, TimeCachedReference<E>> referenceMap;
    private List<TimeCachedReference<E>> referenceList;
    
    // LRUOverflowAlgorithmServiceMBeanJavaDoc
    @Override
    public int size(){
        return referenceMap == null ? 0 : referenceMap.size();
    }
    
    /**
     * T[rX̐sB<p>
     * CX^Xϐ̏sB
     *
     * @exception Exception T[rX̐Ɏsꍇ
     */
    @Override
    public void createService() throws Exception{
        referenceMap = Collections.synchronizedMap(new HashMap<CachedReference<E>, TimeCachedReference<E>>());
        referenceList = Collections.synchronizedList(new ArrayList<TimeCachedReference<E>>());
    }
    
    /**
     * T[rX̔jsB<p>
     * CX^Xϐ̊JsB
     *
     * @exception Exception T[rX̔jɎsꍇ
     */
    @Override
    public void destroyService() throws Exception{
        reset();
        referenceMap = null;
        referenceList = null;
    }
    
    /**
     * LbVQƂǉB<p>
     * œnꂽLbVQƂێBɁA{@link CachedReference#addCacheAccessListener(CacheAccessListener)}ŁA{@link CacheAccessListener}ƂĎgo^B܂A{@link CachedReference#addCacheRemoveListener(CacheRemoveListener)}ŁA{@link CacheRemoveListener}ƂĎgo^B<br>
     *
     * @param ref LbVQ
     */
    @Override
    public void add(CachedReference<E> ref){
        if(referenceMap == null || ref == null){
            return;
        }
        synchronized(referenceMap){
            if(!referenceMap.containsKey(ref)){
                TimeCachedReference<E> tmc = new TimeCachedReference<E>(ref);
                referenceMap.put(ref, tmc);
                referenceList.add(tmc);
                ref.addCacheAccessListener(this);
                ref.addCacheRemoveListener(this);
            }
        }
    }
    
    /**
     * LbVQƂ폜B<p>
     * œnꂽLbVQƂŕێĂꍇ́AjBɁA{@link CachedReference#removeCacheAccessListener(CacheAccessListener)}ŁA{@link CacheAccessListener}ƂĎgo^B܂A{@link CachedReference#removeCacheRemoveListener(CacheRemoveListener)}ŁA{@link CacheRemoveListener}ƂĎgo^B<br>
     *
     * @param ref LbVQ
     */
    @Override
    public void remove(CachedReference<E> ref){
        if(referenceMap == null || ref == null){
            return;
        }
        synchronized(referenceMap){
            if(referenceMap.containsKey(ref)){
                TimeCachedReference<E> tmc = referenceMap.remove(ref);
                referenceList.remove(tmc);
                ref.removeCacheAccessListener(this);
                ref.removeCacheRemoveListener(this);
            }
        }
    }
    
    /**
     * QƂĂȂԂłLbVQƂӂꂳB<p>
     * {@link #add(CachedReference)}œnꂽLbVQƂ̒AQƂĂȂԂłLbVQƂAӂLbVQƂƂĕԂB<br>
     *
     * @return QƂĂȂԂłLbVQ
     */
    @Override
    public CachedReference<E> overflow(){
        if(referenceMap == null){
            return null;
        }
        synchronized(referenceMap){
            if(referenceMap.size() != 0){
                Collections.sort(referenceList);
                TimeCachedReference<E> tmc = referenceList.remove(0);
                referenceMap.remove(tmc.getCachedReference());
                return tmc.getCachedReference();
            }
            return null;
        }
    }
    
    /**
     * QƂĂȂԂłLbVQƂӂꂳB<p>
     * {@link #add(CachedReference)}œnꂽLbVQƂ̒AQƂĂȂԂłLbVQƂAӂLbVQƂƂĕԂB<br>
     *
     * @param size ӂꐔ
     * @return QƂĂȂԂłLbVQ
     */
    @SuppressWarnings("unchecked")
    public CachedReference<E>[] overflow(int size){
        if(referenceMap == null || referenceMap.size() == 0){
            return null;
        }
        synchronized(referenceMap){
            if(referenceMap.size() != 0){
                final CachedReference<E>[] result = new CachedReference[Math.min(referenceMap.size(), size)];
                Collections.sort(referenceList);
                for(int i = 0; i < result.length; i++){
                    TimeCachedReference<E> tmc = referenceList.remove(0);
                    referenceMap.remove(tmc.getCachedReference());
                    result[i] = tmc.getCachedReference();
                }
                return result;
            }
            return null;
        }
    }
    
    /**
     * ӂASYs邽߂ɕێĂB<p>
     * {@link #add(CachedReference)}œnꂽLbVQƂSĔjB<br>
     */
    @Override
    public void reset(){
        synchronized(referenceMap){
            if(referenceMap != null){
                referenceMap.clear();
            }
            if(referenceList != null){
                referenceList.clear();
            }
        }
    }
    
    /**
     * LbV폜ꂽLbVQƂ̒ʒm󂯂B<p>
     * {@link #remove(CachedReference)}ĂяoB<br>
     *
     * @param ref LbV폜ꂽLbVQ
     */
    @Override
    public void removed(CachedReference<E> ref){
        remove(ref);
    }
    
    /**
     * QƂꂽLbVQƂ̒ʒm󂯂B<p>
     * ŕێĂLbVQƂ̃XgɊ܂܂ꍇ́AŌɈړB<br>
     *
     * @param ref QƂꂽLbVQ
     */
    @Override
    public void accessed(CachedReference<E> ref){
        if(referenceMap == null){
            return;
        }
        synchronized(referenceMap){
            if(referenceMap != null && referenceMap.containsKey(ref)){
                TimeCachedReference<E> tmcRef = referenceMap.get(ref);
                if(tmcRef != null){
                    tmcRef.access();
                }
            }
        }
    }
    
    private static class TimeCachedReference<E>
     implements java.io.Serializable, Comparable<TimeCachedReference<E>>{
        
        private static final long serialVersionUID = 1731034355455490827L;
        private CachedReference<E> reference;
        private long lastAccessTime;
        
        public TimeCachedReference(CachedReference<E> ref){
            reference = ref;
            lastAccessTime = System.currentTimeMillis();
        }
        public CachedReference<E> getCachedReference(){
            return reference;
        }
        
        public void access(){
            lastAccessTime = System.currentTimeMillis();
        }
        
        public long getLastAccessTime(){
            return lastAccessTime;
        }
        
        public int compareTo(TimeCachedReference<E> comp) {
            if(comp == null){
                return 1;
            }
            if(comp == this){
                return 0;
            }
            final long diff = lastAccessTime - comp.getLastAccessTime();
            if(diff == 0){
                return 0;
            }else if(diff > 0){
                return 1;
            }else{
                return -1;
            }
        }
     }
}
