/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.core;

import java.util.*;
import java.net.*;
import java.io.*;

import jp.ossc.nimbus.service.log.*;
import jp.ossc.nimbus.service.message.*;
import jp.ossc.nimbus.service.repository.*;
import jp.ossc.nimbus.util.WaitSynchronizeMonitor;

/**
 * T[rXǗt@NgB<p>
 * {@link ServiceLoader}ɃT[rX`̃[hvāAT[rXǗ
 * {@link ServiceManager}𐶐t@NgłB<br>
 * ServiceManageŕA{@link Repository}ŊǗAOŃANZXłB<br>
 * <p>
 * ܂AServiceManagerɑ΂āA̖OServiceManager̊estaticɁA
 * ANZX郉bp[Iȋ@\B<br>
 *
 * @author M.Takata
 */
public class ServiceManagerFactory implements Serializable{
    
    private static final long serialVersionUID = -1120514470640321429L;
    
    private static final String USAGE_RESOURCE
         = "jp/ossc/nimbus/core/ServiceManagerFactoryUsage.txt";
    
    /**
     * ftHg̃Oo͂s{@link DefaultLoggerService}IuWFNgB<p>
     */
    static final DefaultLoggerService DEFAULT_LOGGER;
    
    /**
     * ftHg̃bZ[Ws{@link DefaultMessageRecordFactoryService}IuWFNgB<p>
     */
    static final DefaultMessageRecordFactoryService DEFAULT_MESSAGE;
    
    // bZ[WID`
    private static final String SVCMF = "SVCMF";
    private static final String SVCMF0 = SVCMF + 0;
    private static final String SVCMF00 = SVCMF0 + 0;
    private static final String SVCMF000 = SVCMF00 + 0;
    private static final String SVCMF0000 = SVCMF000 + 0;
    private static final String SVCMF00001 = SVCMF0000 + 1;
    private static final String SVCMF00002 = SVCMF0000 + 2;
    private static final String SVCMF00003 = SVCMF0000 + 3;
    private static final String SVCMF00004 = SVCMF0000 + 4;
    private static final String SVCMF00005 = SVCMF0000 + 5;
    private static final String SVCMF00006 = SVCMF0000 + 6;
    private static final String SVCMF00007 = SVCMF0000 + 7;
    private static final String SVCMF00008 = SVCMF0000 + 8;
    private static final String SVCMF00009 = SVCMF0000 + 9;
    private static final String SVCMF00010 = SVCMF000 + 10;
    private static final String SVCMF00011 = SVCMF000 + 11;
    private static final String SVCMF00012 = SVCMF000 + 12;
    private static final String SVCMF00013 = SVCMF000 + 13;
    private static final String SVCMF00014 = SVCMF000 + 14;
    private static final String SVCMF00015 = SVCMF000 + 15;
    private static final String SVCMF00016 = SVCMF000 + 16;
    private static final String SVCMF00017 = SVCMF000 + 17;
    private static final String SVCMF00018 = SVCMF000 + 18;
    private static final String SVCMF00019 = SVCMF000 + 19;
    private static final String SVCMF00020 = SVCMF000 + 20;
    private static final String SVCMF00021 = SVCMF000 + 21;
    private static final String SVCMF00022 = SVCMF000 + 22;
    private static final String SVCMF00023 = SVCMF000 + 23;
    private static final String SVCMF00024 = SVCMF000 + 24;
    private static final String SVCMF00025 = SVCMF000 + 25;
    private static final String SVCMF00026 = SVCMF000 + 26;
    private static final String SVCMF00027 = SVCMF000 + 27;
    private static final String SVCMF00028 = SVCMF000 + 28;
    private static final String SVCMF00029 = SVCMF000 + 29;
    private static final String SVCMF00030 = SVCMF000 + 30;
    private static final String SVCMF00031 = SVCMF000 + 31;
    
    /**
     * {@link ServiceLoader}̎NXw肷VXevpeB̃L[B<p>
     */
    private static final String LOADER_IMPL_CLASS_KEY
         = "jp.ossc.nimbus.core.loader";
    
    /**
     * {@link ServiceManager}̎NXw肷VXevpeB̃L[B<p>
     */
    private static final String MANAGER_IMPL_CLASS_KEY
         = "jp.ossc.nimbus.core.manager";
    
    /**
     * Oo͂s{@link Logger}̃bp[IuWFNgB<p>
     */
    private static LoggerWrapper logger;
    
    /**
     * bZ[Ws{@link MessageRecordFactory}̃bp[IuWFNgB<p>
     */
    private static MessageRecordFactoryWrapper message;
    
    /**
     * sB<p>
     */
    private static final String LINE_SEPARAOTR
         = System.getProperty("line.separator");
    
    /**
     * ҋ@̃T[rX̌ƂȂT[rX\ۂ̐ړB<p>
     */
    private static final String CAUSE_SERVICES = " causes ";
    
    /**
     * NɎsT[rX̌ƂȂO\ۂ̐ړB<p>
     */
    private static final String CAUSE_THROWABLE = " cause ";
    
    /**
     * ServiceLoader̃ftHgNXB<p>
     */
    private static final Class<DefaultServiceLoaderService> DEFAULT_SERVICE_LOADER_CLASS
         = DefaultServiceLoaderService.class;
    
    /**
     * T[rX`[h{@link ServiceLoader}Ǘ}bvB<p>
     * <table border="1">
     *   <tr bgcolor="#CCCCFF"><th colspan="2">L[</th><th colspan="2">l</th></tr>
     *   <tr bgcolor="#CCCCFF"><th>^</th><th>e</th><th>^</th><th>e</th></tr>
     *   <tr><td>java.net.URL</td><td>T[rX`URL</td><td>ServiceLoader</td><td>L[URL̃T[rX`[h{@link ServiceLoader}</td></tr>
     * </table>
     */
    private static final Map<URL, ServiceLoader> loaders = Collections.synchronizedMap(new HashMap<URL, ServiceLoader>());
    
    /**
     * {@link ServiceManager}Ǘ{@link Repository}B<p>
     * ftHgł́AMapRepositotyB<br>
     */
    private static Repository repository = new DefaultRepository();
    
    private static class DefaultRepository implements Repository{
        private final Map<String, Object> managerMap = Collections.synchronizedMap(new HashMap<String, Object>());
        
        @SuppressWarnings("unchecked")
        public <T> T get(String name){
            return (T)managerMap.get(name);
        }
        
        public boolean register(String name, Object manager){
            if(managerMap.containsKey(name)){
                return false;
            }
            managerMap.put(name, manager);
            return true;
        }
        
        public boolean unregister(String name){
            managerMap.remove(name);
            return true;
        }
        
        public boolean isRegistered(String name){
            return managerMap.containsKey(name);
        }
        
        public Set<String> nameSet(){
            return new HashSet<String>(managerMap.keySet());
        }
        
        @SuppressWarnings("unchecked")
        public <T> Set<T> registeredSet(){
            return new HashSet<T>((Collection<T>)managerMap.values());
        }
    };
    
    /**
     * ServiceManagerFactoryɓo^ꂽo^ԃXĩXgB<p>
     */
    private static List<RegistrationListener> registrationListeners = new ArrayList<RegistrationListener>();
    
    /**
     * {@link ServiceLoader}NXB<p>
     */
    private static Class<? extends ServiceLoader> loaderClass = DEFAULT_SERVICE_LOADER_CLASS;
    
    /**
     * {@link ServiceManager}NXB<p>
     */
    private static Class<? extends ServiceManager> managerClass;
    
    static{
        DEFAULT_LOGGER = new DefaultLoggerService();
        try{
            DEFAULT_LOGGER.create();
            DEFAULT_LOGGER.start();
            DEFAULT_LOGGER.setSystemDebugEnabled(false);
            DEFAULT_LOGGER.setSystemInfoEnabled(true);
            DEFAULT_LOGGER.setSystemWarnEnabled(true);
            DEFAULT_LOGGER.setSystemErrorEnabled(true);
            DEFAULT_LOGGER.setSystemFatalEnabled(true);
        }catch(Exception e){
            e.printStackTrace();
        }
        logger = new LoggerWrapper(DEFAULT_LOGGER);
        
        DEFAULT_MESSAGE = new DefaultMessageRecordFactoryService();
        try{
            DEFAULT_MESSAGE.create();
            DEFAULT_MESSAGE.start();
        }catch(Exception e){
            e.printStackTrace();
        }
        message = new MessageRecordFactoryWrapper(DEFAULT_MESSAGE);
    }
    
    private static Properties properties = new Properties();
    
    /**
     * RXgN^B<p>
     */
    private ServiceManagerFactory(){
        super();
    }
    
    /**
     * ftHg̃T[rX`[hB<p>
     * {@link #loadManager(URL)}nullŌĂяoB<br>
     * ̃\bhɂă[hT[rX`t@ĆA{@link Utility#getDefaultServiceURL()}Ŏ擾URL̃T[rX`t@CłB
     * 
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     */
    public static synchronized boolean loadManager(){
        return loadManager((URL)null);
    }
    
    /**
     * w肵pX̃T[rX`[hB<p>
     * w肵pX́A{@link Utility#convertServicePathToURL(String)}URLɕϊA{@link #loadManager(URL)}ĂяoB<br>
     *
     * @param path T[rX`t@C̃pX
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     * @exception IllegalArgumentException w肵pXsȏꍇA܂݂͑Ȃꍇ
     */
    public static synchronized boolean loadManager(String path){
        return loadManager(path, false, false);
    }
    
    /**
     * w肵URL̃T[rX`[hB<p>
     * {@link #loadManager(URL, boolean)}AloadManager(url, false)ŌĂяoB<br>
     *
     * @param url T[rX`t@CURL
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     */
    public static synchronized boolean loadManager(URL url){
        return loadManager(url, false);
    }
    
    /**
     * w肵pX̃T[rX`[hB<p>
     * w肵pX́A{@link Utility#convertServicePathToURL(String)}URLɕϊA{@link #loadManager(URL, boolean)}ĂяoB<br>
     *
     * @param path T[rX`t@C̃pX
     * @param isReload Ƀ[hT[rX`ă[hꍇɂ́Atrue
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     */
    public static synchronized boolean loadManager(
        String path,
        boolean isReload
    ){
        return loadManager(
            path,
            isReload,
            false
        );
    }
    
    /**
     * w肵URL̃T[rX`[hB<p>
     * {@link #loadManager(URL, boolean, boolean)}AloadManager(url, isReload, false)ŌĂяoB<br>
     *
     * @param url T[rX`t@CURL
     * @param isReload Ƀ[hT[rX`ă[hꍇɂ́Atrue
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     */
    public static synchronized boolean loadManager(URL url, boolean isReload){
        return loadManager(url, isReload, false);
    }
    
    /**
     * w肵pX̃T[rX`[hB<p>
     * w肵pX́A{@link Utility#convertServicePathToURL(String)}URLɕϊA{@link #loadManager(URL, boolean, boolean)}ĂяoB<br>
     *
     * @param path T[rX`t@C̃pX
     * @param isReload Ƀ[hT[rX`ă[hꍇɂ́Atrue
     * @param isValidate T[rX`t@C]邩ǂB]ꍇtrue
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     */
    public static synchronized boolean loadManager(
        String path,
        boolean isReload,
        boolean isValidate
    ){
        URL url = null;
        try{
            url = Utility.convertServicePathToURL(path);
        }catch(IllegalArgumentException e){
            logger.write(SVCMF00030, e, path);
            return false;
        }
        if(url == null){
            logger.write(SVCMF00030, path);
            return false;
        }
        return loadManager(url, isReload, isValidate);
    }
    
    /**
     * w肵pX̃T[rX`[hB<p>
     * w肵pX́A{@link Utility#convertServicePathToURL(String)}URLɕϊA{@link #loadManager(URL, boolean, boolean)}ĂяoB<br>
     *
     * @param path T[rX`t@C̃pX
     * @param config T[rX[_\
     * @param isReload Ƀ[hT[rX`ă[hꍇɂ́Atrue
     * @param isValidate T[rX`t@C]邩ǂB]ꍇtrue
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     */
    public static synchronized boolean loadManager(
        String path,
        ServiceLoaderConfig config,
        boolean isReload,
        boolean isValidate
    ){
        URL url = null;
        try{
            url = Utility.convertServicePathToURL(path);
        }catch(IllegalArgumentException e){
            logger.write(SVCMF00030, e, path);
            return false;
        }
        if(url == null){
            logger.write(SVCMF00030, path);
            return false;
        }
        return loadManager(url, config, isReload, isValidate);
    }
    
    /**
     * w肵URL̃T[rX`[hB<p>
     *
     * @param url T[rX`t@CURL
     * @param isReload Ƀ[hT[rX`ă[hꍇɂ́Atrue
     * @param isValidate T[rX`t@C]邩ǂB]ꍇtrue
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     */
    public static synchronized boolean loadManager(
        URL url,
        boolean isReload,
        boolean isValidate
    ){
        return loadManager(url, null, isReload, isValidate);
    }
    
    /**
     * w肵URL̃T[rX`[hB<p>
     *
     * @param url T[rX`t@CURL
     * @param config T[rX[_\
     * @param isReload Ƀ[hT[rX`ă[hꍇɂ́Atrue
     * @param isValidate T[rX`t@C]邩ǂB]ꍇtrue
     * @return [hɐꍇtrueBAAṒAKT[rX`ɒ`ꂽT[rXSĐɋNۏ؂̂ł́A܂BT[rX`̃[hɎgpServiceLoaderɋNꂽ܂B
     */
    @SuppressWarnings("unchecked")
    public static synchronized boolean loadManager(
        URL url,
        ServiceLoaderConfig config,
        boolean isReload,
        boolean isValidate
    ){
        logger.write(
            SVCMF00001,
            url, isReload ? Boolean.TRUE : Boolean.FALSE
        );
        
        if(url == null){
            url = Utility.getDefaultServiceURL();
        }
        
        if(url == null){
            return false;
        }
        
        ServiceLoader loader = null;
        if(!loaders.containsKey(url)){
            Class<? extends ServiceLoader> loaderClass = null;
            String loaderClassName
                 = System.getProperty(LOADER_IMPL_CLASS_KEY);
            if(loaderClassName != null && loaderClassName.length() != 0){
                try{
                    loaderClass = (Class<? extends ServiceLoader>)Class.forName(
                        loaderClassName,
                        true,
                        NimbusClassLoader.getInstance()
                    );
                }catch(ClassNotFoundException e){
                    logger.write(
                        SVCMF00002,
                        e,
                        ServiceLoader.class, loaderClassName
                    );
                }catch(IllegalArgumentException e){
                    logger.write(SVCMF00004, e, loaderClassName);
                }
            }
            if(loaderClass == null
                && config != null
                && config.getServiceLoaderClassName() != null){
                loaderClassName = config.getServiceLoaderClassName();
                try{
                    loaderClass = (Class<? extends ServiceLoader>)Class.forName(
                        loaderClassName,
                        true,
                        NimbusClassLoader.getInstance()
                    );
                }catch(ClassNotFoundException e){
                    logger.write(
                        SVCMF00002,
                        e,
                        ServiceLoader.class, loaderClassName
                    );
                }catch(IllegalArgumentException e){
                    logger.write(SVCMF00004, e, loaderClassName);
                }
            }
            if(loaderClass == null){
                loaderClass = getServiceLoaderClass();
            }
            try{
                loader = loaderClass.newInstance();
            }catch(InstantiationException e){
                logger.write(SVCMF00005, e, getServiceLoaderClass());
                return false;
            }catch(IllegalAccessException e){
                logger.write(SVCMF00006, e, getServiceLoaderClass());
                return false;
            }
            String managerClassName
                 = System.getProperty(MANAGER_IMPL_CLASS_KEY);
            if((managerClassName == null || managerClassName.length() == 0)
                && getServiceManagerClass() != null){
                managerClassName = getServiceManagerClass().getName();
            }
            if(managerClassName != null && managerClassName.length() != 0){
                try{
                    loader.setServiceManagerClassName(managerClassName);
                }catch(ClassNotFoundException e){
                    logger.write(
                        SVCMF00002,
                        e,
                        ServiceManager.class, managerClassName
                    );
                }catch(IllegalArgumentException e){
                    logger.write(SVCMF00031, e, managerClassName);
                }
            }
            try{
                loader.setServiceURL(url);
            }catch(IllegalArgumentException e){
                logger.write(SVCMF00007, e, url);
                return false;
            }
        }else if(isReload){
            loader = (ServiceLoader)loaders.get(url);
            loader.stop();
            loader.destroy();
            unregisterLoader(loader);
        }else{
            return true;
        }
        
        loader.setValidate(isValidate);
        loader.setConfig(config);
        
        try{
            loader.create();
            loader.start();
        }catch(Exception e){
            logger.write(SVCMF00008, e, url);
            loader.destroy();
            return false;
        }
        
        registerLoader(loader);
        logger.write(SVCMF00009, url);
        return true;
    }
    
    /**
     * ftHg̃T[rX`A[hB<p>
     * {@link #unloadManager(URL)}nullŌĂяoB<br>
     *
     * @return T[rX`̃A[hsꍇtrue
     */
    public static synchronized boolean unloadManager(){
        return unloadManager((URL)null);
    }
    
    /**
     * w肳ꂽpX̃T[rX`A[hB<p>
     * w肵pX́A{@link Utility#convertServicePathToURL(String)}URLɕϊA{@link #unloadManager(URL)}ĂяoB<br>
     *
     * @param path T[rX`t@C̃pX
     * @return T[rX`̃A[hsꍇtrue
     */
    public static synchronized boolean unloadManager(String path){
        URL url = null;
        try{
            url = Utility.convertServicePathToURL(path);
        }catch(IllegalArgumentException e){
            try{
                url = new File(path).toURI().toURL();
            }catch(MalformedURLException ee){
                // ̗O͔Ȃ͂
                return false;
            }
        }
        return unloadManager(url);
    }
    
    /**
     * w肳ꂽpX̃T[rX`A[hB<p>
     *
     * @param url T[rX`t@CURL
     * @return T[rX`̃A[hsꍇtrue
     */
    public static synchronized boolean unloadManager(URL url){
        
        logger.write(SVCMF00010, url);
        
        if(url == null){
            url = Utility.getDefaultServiceURL();
        }
        
        if(url == null){
            return false;
        }
        
        if(!loaders.containsKey(url)){
            
            logger.write(SVCMF00011, url);
            return false;
        }else{
            Service service = (Service)loaders.get(url);
            service.stop();
            service.destroy();
            
            logger.write(SVCMF00012, url);
        }
        return true;
    }
    
    /**
     * [hT[rX`ɒ`ꂽT[rXSċNĂ邩ׂB<p>
     * @return SċNĂꍇtrue
     */
    public static boolean checkLoadManagerCompleted(){
        return checkLoadManagerCompleted(null);
    }
    
    /**
     * [hT[rX`ɒ`ꂽT[rXSċNĂ邩ׂB<p>
     * NĂȂT[rX{@link ServiceName}ƂāAnotStartedɊi[ĕԂB<br>
     * 
     * @param notStarted NłȂT[rX̏Wi[Zbg
     * @return SċNĂꍇtrue
     */
    public static boolean checkLoadManagerCompleted(Set<ServiceName> notStarted){
        
        logger.write(SVCMF00013);
        final Set<ServiceName> tmpNotStarted = new HashSet<ServiceName>();
        final ServiceManager[] managers = findManagers();
        final StringBuilder message = new StringBuilder();
        for(int i = 0, max = managers.length; i < max; i++){
            final ServiceManager manager = managers[i];
            final String managerName = manager.getServiceName();
            if(manager.existFailedService()){
                final Iterator<String> failedServices
                     = manager.getFailedServices().iterator();
                while(failedServices.hasNext()){
                    final String failedService
                         = (String)failedServices.next();
                    final ServiceName name
                         = new ServiceName(managerName, failedService);
                    tmpNotStarted.add(name);
                    message.append(name);
                    final Throwable cause
                         = manager.getFailedCause(failedService);
                    if(cause != null){
                        message.append(CAUSE_THROWABLE);
                        message.append(cause);
                    }
                    if(failedServices.hasNext()){
                        message.append(LINE_SEPARAOTR);
                    }
                }
                if(i != max - 1){
                    message.append(LINE_SEPARAOTR);
                }
            }
        }
        boolean mustInsertLine = message.length() != 0;
        for(int i = 0, max = managers.length; i < max; i++){
            final ServiceManager manager = managers[i];
            final String managerName = manager.getServiceName();
            if(manager.existWaitingService()){
                final Iterator<String> waitingServices
                     = manager.getWaitingServices().iterator();
                while(waitingServices.hasNext()){
                    final String waitingService
                         = (String)waitingServices.next();
                    final ServiceName name
                         = new ServiceName(managerName, waitingService);
                    if(!tmpNotStarted.contains(name)
                        && !waitingService.equals(managerName)){
                        tmpNotStarted.add(name);
                    }else{
                        continue;
                    }
                    if(mustInsertLine){
                        message.append(LINE_SEPARAOTR);
                        mustInsertLine = false;
                    }
                    message.append(name);
                    final Set<ServiceName> causes = manager.getWaitingCauses(waitingService);
                    if(causes != null && causes.size() != 0){
                        message.append(CAUSE_SERVICES);
                        message.append(causes);
                    }
                    if(waitingServices.hasNext()){
                        message.append(LINE_SEPARAOTR);
                    }
                }
                if(i != max - 1){
                    message.append(LINE_SEPARAOTR);
                }
            }
        }
        final boolean isSuccess = tmpNotStarted.size() == 0;
        if(!isSuccess){
            if(notStarted != null){
                notStarted.addAll(tmpNotStarted);
            }
            logger.write(SVCMF00014, message.toString());
        }
        return isSuccess;
    }
    
    /**
     * [hT[rX`ɒ`ꂽT[rXSċNĂ邩ׂB<p>
     * @param managerNames `FbN}l[W̏W
     * @return SċNĂꍇtrue
     */
    public static boolean checkLoadManagerCompletedBy(Set<String> managerNames){
        return checkLoadManagerCompletedBy(managerNames, null);
    }
    
    /**
     * [hT[rX`ɒ`ꂽT[rXSċNĂ邩ׂB<p>
     * NĂȂT[rX{@link ServiceName}ƂāAnotStartedɊi[ĕԂB<br>
     * 
     * @param managerNames `FbN}l[W̏W
     * @param notStarted NłȂT[rX̏Wi[Zbg
     * @return SċNĂꍇtrue
     */
    public static boolean checkLoadManagerCompletedBy(
        Set<String> managerNames,
        Set<ServiceName> notStarted
    ){
        logger.write(SVCMF00013);
        final Set<ServiceName> tmpNotStarted = new HashSet<ServiceName>();
        final ServiceManager[] managers = findManagers();
        final StringBuilder message = new StringBuilder();
        for(int i = 0, max = managers.length; i < max; i++){
            final ServiceManager manager = managers[i];
            final String managerName = manager.getServiceName();
            if(!managerNames.contains(managerName)){
                continue;
            }
            if(manager.existFailedService()){
                final Iterator<String> failedServices
                     = manager.getFailedServices().iterator();
                while(failedServices.hasNext()){
                    final String failedService
                         = (String)failedServices.next();
                    final ServiceName name
                         = new ServiceName(managerName, failedService);
                    tmpNotStarted.add(name);
                    message.append(name);
                    final Throwable cause
                         = manager.getFailedCause(failedService);
                    if(cause != null){
                        message.append(CAUSE_THROWABLE);
                        message.append(cause);
                    }
                    if(failedServices.hasNext()){
                        message.append(LINE_SEPARAOTR);
                    }
                }
                if(i != max - 1){
                    message.append(LINE_SEPARAOTR);
                }
            }
        }
        boolean mustInsertLine = message.length() != 0;
        for(int i = 0, max = managers.length; i < max; i++){
            final ServiceManager manager = managers[i];
            final String managerName = manager.getServiceName();
            if(!managerNames.contains(managerName)){
                continue;
            }
            if(manager.existWaitingService()){
                final Iterator<String> waitingServices
                     = manager.getWaitingServices().iterator();
                while(waitingServices.hasNext()){
                    final String waitingService
                         = (String)waitingServices.next();
                    final ServiceName name
                         = new ServiceName(managerName, waitingService);
                    if(!tmpNotStarted.contains(name)
                        && !waitingService.equals(managerName)){
                        tmpNotStarted.add(name);
                    }else{
                        continue;
                    }
                    if(mustInsertLine){
                        message.append(LINE_SEPARAOTR);
                        mustInsertLine = false;
                    }
                    message.append(name);
                    final Set<ServiceName> causes = manager.getWaitingCauses(waitingService);
                    if(causes != null && causes.size() != 0){
                        message.append(CAUSE_SERVICES);
                        message.append(causes);
                    }
                    if(waitingServices.hasNext()){
                        message.append(LINE_SEPARAOTR);
                    }
                }
                if(i != max - 1){
                    message.append(LINE_SEPARAOTR);
                }
            }
        }
        final boolean isSuccess = tmpNotStarted.size() == 0;
        if(!isSuccess){
            if(notStarted != null){
                notStarted.addAll(tmpNotStarted);
            }
            logger.write(SVCMF00014, message.toString());
        }
        return isSuccess;
    }
    
    /**
     * [hT[rX`ɒ`ꂽT[rXSċNĂ邩ׂB<p>
     * @param managerName `FbN}l[W
     * @return SċNĂꍇtrue
     */
    public static boolean checkLoadManagerCompletedBy(String managerName){
        return checkLoadManagerCompletedBy(managerName, null);
    }
    
    /**
     * [hT[rX`ɒ`ꂽT[rXSċNĂ邩ׂB<p>
     * NĂȂT[rX{@link ServiceName}ƂāAnotStartedɊi[ĕԂB<br>
     * 
     * @param managerName `FbN}l[W
     * @param notStarted NłȂT[rX̏Wi[Zbg
     * @return SċNĂꍇtrue
     */
    public static boolean checkLoadManagerCompletedBy(
        String managerName,
        Set<ServiceName> notStarted
    ){
        logger.write(SVCMF00013);
        final Set<ServiceName> tmpNotStarted = new HashSet<ServiceName>();
        final ServiceManager manager = findManager(managerName);
        final StringBuilder message = new StringBuilder();
        if(manager.existFailedService()){
            final Iterator<String> failedServices
                 = manager.getFailedServices().iterator();
            while(failedServices.hasNext()){
                final String failedService
                     = (String)failedServices.next();
                final ServiceName name
                     = new ServiceName(managerName, failedService);
                tmpNotStarted.add(name);
                message.append(name);
                final Throwable cause
                     = manager.getFailedCause(failedService);
                if(cause != null){
                    message.append(CAUSE_THROWABLE);
                    message.append(cause);
                }
                if(failedServices.hasNext()){
                    message.append(LINE_SEPARAOTR);
                }
            }
        }
        boolean mustInsertLine = message.length() != 0;
        if(manager.existWaitingService()){
            final Iterator<String> waitingServices
                 = manager.getWaitingServices().iterator();
            while(waitingServices.hasNext()){
                final String waitingService
                     = (String)waitingServices.next();
                final ServiceName name
                     = new ServiceName(managerName, waitingService);
                if(!tmpNotStarted.contains(name)
                    && !waitingService.equals(managerName)){
                    tmpNotStarted.add(name);
                }else{
                    continue;
                }
                if(mustInsertLine){
                    message.append(LINE_SEPARAOTR);
                    mustInsertLine = false;
                }
                message.append(name);
                final Set<ServiceName> causes = manager.getWaitingCauses(waitingService);
                if(causes != null && causes.size() != 0){
                    message.append(CAUSE_SERVICES);
                    message.append(causes);
                }
                if(waitingServices.hasNext()){
                    message.append(LINE_SEPARAOTR);
                }
            }
        }
        final boolean isSuccess = tmpNotStarted.size() == 0;
        if(!isSuccess){
            if(notStarted != null){
                notStarted.addAll(tmpNotStarted);
            }
            logger.write(SVCMF00014, message.toString());
        }
        return isSuccess;
    }
    
    /**
     * [hꂽSĂServiceManagerTB<p>
     *
     * @return [hꂽSĂServiceManager̔zBP[hĂȂꍇ́AO̔zԂB
     */
    public static ServiceManager[] findManagers(){
        final Set<ServiceManager> managerSet = repository.registeredSet();
        return managerSet.toArray(new ServiceManager[managerSet.size()]);
    }
    
    /**
     * ftHg̖OServiceManagerTB<p>
     * ŌAftHg̖ÓA{@link ServiceManager#DEFAULT_NAME}łB<br>
     *
     * @return ftHg̖OServiceManagerBȂꍇ́AnullԂB
     */
    public static ServiceManager findManager(){
        return findManager(ServiceManager.DEFAULT_NAME);
    }
    
    /**
     * w肳ꂽOServiceManagerTB<p>
     *
     * @param name ServiceManager̖O
     * @return w肳ꂽOServiceManagerBȂꍇ́AnullԂB
     */
    public static ServiceManager findManager(String name){
        if(name == null){
            return null;
        }
        return (ServiceManager)repository.get(name);
    }
    
    /**
     * ServiceManagero^B<p>
     * ServiceManager̃ftHgNXł{@link DefaultServiceManagerService}gpB<br>
     *
     * @param name ServiceManager̓o^
     * @return o^łꍇtrue
     * @see #registerManager(String, ServiceManager)
     */
    public static boolean registerManager(String name){
        final DefaultServiceLoaderService loader
             = new DefaultServiceLoaderService();
        final NimbusMetaData serverData = new NimbusMetaData(loader, null);
        final ManagerMetaData managerData = new ManagerMetaData(loader, serverData);
        managerData.setName(name);
        serverData.addManager(managerData);
        loader.setServerMetaData(serverData);
        try{
            loader.create();
            loader.start();
        }catch(Exception e){
            // NȂ͂
            loader.destroy();
            return false;
        }
        return true;
    }
    
    /**
     * ServiceManagero^B<p>
     * ㏑o^ł邩ǂ́AServiceManager̊Ǘɗp{@link Repository}̎ɈˑBftHgRepositorýA㏑o^͋ȂB<br>
     * <p>
     * o^łꍇ́A{@link #processRegisterd(ServiceManager)}ĂяoāAo^Ă{@link RegistrationListener}ɓo^ʒmB<br>
     *
     * @param name ServiceManager̓o^
     * @param manager o^ServiceManagerIuWFNg
     * @return o^łꍇtrue
     */
    public static boolean registerManager(String name, ServiceManager manager){
        logger.write(SVCMF00015, new Object[]{name, manager});
        final boolean result = repository.register(name, manager);
        if(result){
            logger.write(SVCMF00016, name);
            if(manager != null){
                processRegisterd(manager);
            }
        }else{
            logger.write(SVCMF00017, name);
        }
        return result;
    }
    
    /**
     * w肵OServiceManager̓o^B<p>
     * o^łꍇ́A{@link #processUnregisterd(ServiceManager)}ĂяoāAo^Ă{@link RegistrationListener}ɓo^ʒmB<br>
     *
     * @param name ServiceManager̓o^
     * @return o^łꍇtrue
     */
    public static boolean unregisterManager(String name){
        logger.write(SVCMF00018, name);
        final ServiceManager manager = findManager(name);
        final boolean result = repository.unregister(name);
        if(result){
            logger.write(SVCMF00019, name);
            if(manager != null){
                processUnregisterd(manager);
            }
        }else{
            logger.write(SVCMF00020, name);
        }
        return result;
    }
    
    /**
     * w肳ꂽÕ}l[Wo^Ă邩ׂB<p>
     *
     * @param name }l[W
     * @return o^Ăꍇtrue
     */
    public static boolean isRegisteredManager(String name){
        return repository.isRegistered(name); 
    }
    
    /**
     * w肳ꂽOServiceManagerAw肳ꂽT[rX̃T[rX擾B<p>
     * ServiceLoaderŃ[hꂽT[rX́A{@link Service}C^tF[XĂȂĂAServiceC^tF[Xbp[ł܂ēo^B<br>
     * ̃\bh́A̓𐶂AServiceLoaderŃ[hꂽT[rXServiceIuWFNgƂĎ擾郁\bhłB<br>
     * ʏAT[rX̋NA~Ȃǂ̏sꍇɁÃ\bhŃT[rX擾B<br>
     *
     * @param managerName ServiceManager̖O
     * @param serviceName T[rX
     * @return T[rX
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     */
    public static Service getService(String managerName, String serviceName)
     throws ServiceNotFoundException{
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.getService(serviceName);
        }
        throw new ServiceNotFoundException(managerName, serviceName);
    }
    
    /**
     * w肳ꂽT[rX̃T[rX擾B<p>
     *
     * @param serviceName T[rX
     * @return T[rX
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     * @see #getService(String, String)
     */
    public static Service getService(ServiceName serviceName)
     throws ServiceNotFoundException{
        if(serviceName == null){
            throw new ServiceNotFoundException(null, null);
        }
        return getService(
            serviceName.getServiceManagerName(),
            serviceName.getServiceName()
        );
    }
    
    /**
     * ftHg̖OServiceManagerAw肳ꂽT[rX̃T[rX擾B<p>
     *
     * @param serviceName T[rX
     * @return T[rX
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     * @see #getService(String, String)
     */
    public static Service getService(String serviceName)
     throws ServiceNotFoundException{
        final ServiceManager manager = findManager();
        if(manager != null){
            return manager.getService(serviceName);
        }
        throw new ServiceNotFoundException(
            ServiceManager.DEFAULT_NAME,
            serviceName
        );
    }
    
    /**
     * w肳ꂽOServiceManagerAw肳ꂽT[rX̃T[rX̒`擾B<p>
     *
     * @param managerName ServiceManager̖O
     * @param serviceName T[rX
     * @return T[rX`
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     */
    public static ServiceMetaData getServiceMetaData(
        String managerName,
        String serviceName
    ) throws ServiceNotFoundException{
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.getServiceMetaData(serviceName);
        }
        throw new ServiceNotFoundException(managerName, serviceName);
    }
    
    /**
     * w肳ꂽT[rX̃T[rX̒`擾B<p>
     *
     * @param serviceName T[rX
     * @return T[rX`
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     */
    public static ServiceMetaData getServiceMetaData(ServiceName serviceName)
     throws ServiceNotFoundException{
        if(serviceName == null){
            throw new ServiceNotFoundException(null);
        }
        return getServiceMetaData(
            serviceName.getServiceManagerName(),
            serviceName.getServiceName()
        );
    }
    
    /**
     * ftHg̖OServiceManagerAw肳ꂽT[rX̃T[rX`擾B<p>
     *
     * @param serviceName T[rX
     * @return T[rX`
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     */
    public static ServiceMetaData getServiceMetaData(
        String serviceName
    ) throws ServiceNotFoundException{
        final ServiceManager manager = findManager();
        if(manager != null){
            return manager.getServiceMetaData(serviceName);
        }
        throw new ServiceNotFoundException(
            ServiceManager.DEFAULT_NAME,
            serviceName
        );
    }
    
    /**
     * w肳ꂽOServiceManagerAw肳ꂽT[rX̃T[rXIuWFNg擾B<p>
     * ServiceLoaderŃ[hꂽT[rX́A{@link Service}C^tF[XĂȂĂAServiceC^tF[Xbp[ł܂ēo^B<br>
     * {@link #getService(String, String)}\bhł́AServiceLoaderŃ[hꂽT[rXServiceIuWFNgƂĎ擾邪Ã\bh́AServiceLoaderŃ[hꂽT[rXIuWFNĝ̂擾B<br>
     * ʏAT[rX̃AvP[V̋@\gpꍇɁÃ\bhŃT[rX擾āAKvȃC^tF[XɃLXgĎgpB<br>
     *
     * @param managerName ServiceManager̖O
     * @param serviceName T[rX
     * @return T[rX
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     */
    public static <T> T getServiceObject(
        String managerName,
        String serviceName
    ) throws ServiceNotFoundException{
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.<T>getServiceObject(serviceName);
        }
        throw new ServiceNotFoundException(managerName, serviceName);
    }
    
    /**
     * w肳ꂽT[rX̃T[rXIuWFNg擾B<p>
     *
     * @param serviceName T[rX
     * @return T[rXIuWFNg
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     * @see #getServiceObject(String, String)
     */
    public static <T> T getServiceObject(ServiceName serviceName)
     throws ServiceNotFoundException{
        if(serviceName == null){
            throw new ServiceNotFoundException(serviceName);
        }
        return ServiceManagerFactory.<T>getServiceObject(
            serviceName.getServiceManagerName(),
            serviceName.getServiceName()
        );
    }
    
    /**
     * ftHg̖OServiceManagerAw肳ꂽT[rX̃T[rXIuWFNg擾B<p>
     *
     * @param serviceName T[rX
     * @return T[rXIuWFNg
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     * @see #getServiceObject(String, String)
     */
    public static <T> T getServiceObject(
        String serviceName
    ) throws ServiceNotFoundException{
        final ServiceManager manager = findManager();
        if(manager != null){
            return manager.<T>getServiceObject(serviceName);
        }
        throw new ServiceNotFoundException(
            ServiceManager.DEFAULT_NAME,
            serviceName
        );
    }
    
    /**
     * w肳ꂽOServiceManagerAw肳ꂽT[rX̃T[rX̏ԕύXʒmServiceStateListenable擾B<p>
     * ServiceLoaderŃ[hꂽT[rX́AServiceStateListenableĂĂǂBw肳ꂽT[rXServiceStateListenableĂꍇ́AԂBĂȂꍇ́AnullԂB<br>
     *
     * @param managerName ServiceManager̖O
     * @param serviceName T[rX
     * @return ServiceStateListenableIuWFNg
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     */
    public static ServiceStateListenable getServiceStateListenable(
        String managerName,
        String serviceName
    ) throws ServiceNotFoundException{
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.getServiceStateListenable(serviceName);
        }
        throw new ServiceNotFoundException(managerName, serviceName);
    }
    
    /**
     * w肳ꂽT[rX̃T[rX̏ԕύXʒmServiceStateListenable擾B<p>
     *
     * @param serviceName T[rX
     * @return ServiceStateListenableIuWFNg
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     * @see #getServiceStateListenable(String, String)
     */
    public static ServiceStateListenable getServiceStateListenable(
        ServiceName serviceName
    ) throws ServiceNotFoundException{
        if(serviceName == null){
            throw new ServiceNotFoundException(null, null);
        }
        return getServiceStateListenable(
            serviceName.getServiceManagerName(),
            serviceName.getServiceName()
        );
    }
    
    /**
     * ftHg̖OServiceManagerAw肳ꂽT[rX̃T[rX̏ԕύXʒmServiceStateListenable擾B<p>
     *
     * @param serviceName T[rX
     * @return ServiceStateListenableIuWFNg
     * @exception ServiceNotFoundException w肳ꂽT[rXȂꍇ
     * @see #getServiceStateListenable(String, String)
     */
    public static ServiceStateListenable getServiceStateListenable(
        String serviceName
    ) throws ServiceNotFoundException{
        final ServiceManager manager = findManager();
        if(manager != null){
            return manager.getServiceStateListenable(serviceName);
        }
        throw new ServiceNotFoundException(
            ServiceManager.DEFAULT_NAME,
            serviceName
        );
    }
    
    /**
     * w肳ꂽT[rX`^f[^ɏ]T[rXAw肳ꂽOServiceManagerɁAw肳ꂽT[rXŃT[rXƂēo^B<p>
     *
     * @param managerName ServiceManager̖O
     * @param serviceData T[rX`^f[^
     * @return o^łꍇ́AtrueBw肳ꂽOServiceManager݂ȂꍇAServiceManagero^Ɏsꍇ́AfalseԂB
     * @exception Exception T[rX̃CX^XɎsꍇ
     */
    public static boolean registerService(
        String managerName,
        ServiceMetaData serviceData
    ) throws Exception{
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.registerService(serviceData);
        }
        return false;
    }
    
    /**
     * w肳ꂽIuWFNgAw肳ꂽOServiceManagerɁAw肳ꂽT[rXŃT[rXƂēo^B<p>
     * w肳ꂽIuWFNgServiceC^tF[XĂȂꍇ́AServiceC^tF[Xbp[ł܂ēo^B<br>
     *
     * @param managerName ServiceManager̖O
     * @param serviceName T[rX
     * @param obj T[rXIuWFNg
     * @return o^łꍇ́AtrueBw肳ꂽOServiceManager݂ȂꍇAServiceManagero^Ɏsꍇ́AfalseԂB
     * @exception Exception T[rX̓o^Ɏsꍇ
     */
    public static boolean registerService(
        String managerName,
        String serviceName,
        Object obj
    ) throws Exception{
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.registerService(serviceName, obj);
        }
        return false;
    }
    
    /**
     * w肳ꂽT[rXAw肳ꂽOServiceManagerɁAw肳ꂽT[rXœo^B<p>
     *
     * @param managerName ServiceManager̖O
     * @param serviceName T[rX
     * @param service T[rX
     * @return o^łꍇ́AtrueBw肳ꂽOServiceManager݂ȂꍇAServiceManagero^Ɏsꍇ́AfalseԂB
     * @exception Exception T[rX̓o^Ɏsꍇ
     */
    public static boolean registerService(
        String managerName,
        String serviceName,
        Service service
    ) throws Exception{
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.registerService(serviceName, service);
        }
        return false;
    }
    
    
    /**
     * w肳ꂽOServiceManagerAw肳ꂽT[rX̃T[rX̓o^B<p>
     *
     * @param managerName ServiceManager̖O
     * @param serviceName T[rX
     * @return o^łꍇ́AtrueB܂Aw肳ꂽOServiceManager݂ȂꍇtrueBServiceManagero^̉Ɏsꍇ́AfalseԂB
     */
    public static boolean unregisterService(
        String managerName,
        String serviceName
    ){
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.unregisterService(serviceName);
        }
        return true;
    }
    
    /**
     * w肳ꂽOServiceManagerɁAw肳ꂽT[rX̃T[rXo^Ă邩ׂB<p>
     *
     * @param managerName }l[W
     * @param serviceName T[rX
     * @return o^Ăꍇtrue
     */
    public static boolean isRegisteredService(
        String managerName,
        String serviceName
    ){
        final ServiceManager manager = findManager(managerName);
        if(manager != null){
            return manager.isRegisteredService(serviceName);
        }
        return false;
    }
    
    /**
     * w肳ꂽT[rX̃T[rXo^Ă邩ׂB<p>
     *
     * @param serviceName T[rX
     * @return o^Ăꍇtrue
     */
    public static boolean isRegisteredService(ServiceName serviceName){
        if(serviceName == null){
            return false;
        }
        final ServiceManager manager = findManager(
            serviceName.getServiceManagerName()
        );
        if(manager != null){
            return manager.isRegisteredService(serviceName.getServiceName());
        }
        return false;
    }
    
    /**
     * ServiceManager̊ǗɎgpRepositoryT[rXݒ肷B<p>
     * ݂Repositoryɓo^ĂServiceManagerASĐVRepositoryɓo^łꍇ̂݁ARepositoryύXBłAo^ɎsServiceManager݂ꍇɂ́ȀԂɖ߂B<br>
     *
     * @param name ServiceManager̊ǗɎgpRepositoryT[rX̃T[rX
     * @return Repository̓ւɐꍇtrue
     */
    public static boolean setManagerRepository(ServiceName name){
        return setManagerRepository(
            name.getServiceManagerName(),
            name.getServiceName()
        );
    }
    
    /**
     * ServiceManager̊ǗɎgpRepositoryT[rXݒ肷B<p>
     * ݂Repositoryɓo^ĂServiceManagerASĐVRepositoryɓo^łꍇ̂݁ARepositoryύXBłAo^ɎsServiceManager݂ꍇɂ́ȀԂɖ߂B<br>
     *
     * @param manager ServiceManager̊ǗɎgpRepositoryT[rXo^Ă}l[W
     * @param service ServiceManager̊ǗɎgpRepositoryT[rX̃T[rX
     * @return Repository̓ւɐꍇtrue
     */
    public static boolean setManagerRepository(
        final String manager,
        final String service
    ){
        if(isRegisteredService(manager, service)
             && getService(manager, service).getState() == Service.State.STARTED
        ){
            return setManagerRepository(
                ServiceManagerFactory.<Repository>getServiceObject(manager, service)
            );
        }else{
            addServiceStateListener(
                manager,
                service,
                new ServiceStateListener(){
                    public void stateChanged(ServiceStateChangeEvent e)
                     throws Exception{
                        ServiceManagerFactory.removeServiceStateListener(
                            manager,
                            service,
                            this
                        );
                        ServiceManagerFactory
                            .setManagerRepository(manager, service);
                    }
                    public boolean isEnabledState(Service.State st){
                        return st == Service.State.STARTED;
                    }
                }
            );
            return false;
        }
    }
    
    /**
     * ServiceManager̊ǗɎgpRepositoryݒ肷B<p>
     * ݂Repositoryɓo^ĂServiceManagerASĐVRepositoryɓo^łꍇ̂݁ARepositoryύXBłAo^ɎsServiceManager݂ꍇɂ́ȀԂɖ߂B<br>
     *
     * @param newRep ServiceManager̊ǗɎgpRepositoryIuWFNg
     * @return Repository̓ւɐꍇtrue
     */
    public static boolean setManagerRepository(Repository newRep){
        logger.write(SVCMF00021, newRep);
        synchronized(repository){
            if(newRep == null){
                newRep = new DefaultRepository();
            }
            if(repository.equals(newRep)){
                return true;
            }
            boolean success = true;
            final Set<String> registered = new HashSet<String>();
            Iterator<String> names = repository.nameSet().iterator();
            while(names.hasNext()){
                final String name = (String)names.next();
                final ServiceManager manager = repository.get(name);
                if(manager != null){
                    if(!newRep.register(name, manager)){
                        logger.write(SVCMF00022, name);
                        success = false;
                    }else{
                        registered.add(name);
                    }
                }
            }
            if(!success){
                logger.write(SVCMF00023, newRep);
                names = registered.iterator();
                while(names.hasNext()){
                    final String name = (String)names.next();
                    newRep.unregister(name);
                }
                return false;
            }
            logger.write(SVCMF00024, newRep);
            names = newRep.nameSet().iterator();
            while(names.hasNext()){
                final String name = (String)names.next();
                repository.unregister(name);
            }
            repository = newRep;
        }
        return true;
    }
    
    /**
     * T[rX`[hServiceLoadero^B<p>
     * 
     * @param loader o^ServiceLoaderIuWFNg
     */
    public static void registerLoader(ServiceLoader loader){
        final URL url = loader.getServiceURL();
        if(!loaders.containsKey(url)){
            loaders.put(url, loader);
        }
    }
    
    /**
     * T[rX`[hServiceLoader̓o^B<p>
     * 
     * @param loader o^ServiceLoaderIuWFNg
     */
    public static void unregisterLoader(ServiceLoader loader){
        loaders.remove(loader.getServiceURL());
    }
    
    /**
     * w肳ꂽURL̃T[rX`[hServiceLaoder擾B<p>
     *
     * @param url T[rX`URL
     * @return w肳ꂽURL̃T[rX`[hServiceLaoderB݂Ȃꍇ́AnullԂB
     */
    public static ServiceLoader getLoader(URL url){
        return loaders.get(url);
    }
    
    /**
     * T[rX`[hSĂServiceLaoder擾B<p>
     *
     * @return T[rX`[hServiceLaodeȑWB݂Ȃꍇ́ȀWԂB
     */
    public static Collection<ServiceLoader> getLoaders(){
        return new HashSet<ServiceLoader>(loaders.values());
    }
    
    /**
     * T[rX`t@C[hServiceLoaderNXݒ肷B<p>
     * AAVXevpeB"jp.ossc.nimbus.core.loader"Ŏw肵ServiceLoaderNX̕D悳B<br>
     *
     * @param loader T[rX`t@C[hServiceLoaderNX
     */
    public static void setServiceLoaderClass(Class<? extends ServiceLoader> loader)
     throws IllegalArgumentException{
        if(loader == null){
            loaderClass = DEFAULT_SERVICE_LOADER_CLASS;
        }else if(ServiceLoader.class.isAssignableFrom(loader)){
            loaderClass = loader;
        }else{
            throw new IllegalArgumentException(
                message.findMessage(SVCMF00003, loader)
            );
        }
        logger.write(SVCMF00025, loaderClass);
    }
    
    /**
     * T[rX`t@C[hServiceLoaderNX擾B<p>
     *
     * @return T[rX`t@C[hServiceLoaderNX
     */
    public static Class<? extends ServiceLoader> getServiceLoaderClass(){
        return loaderClass;
    }
    
    /**
     * ̃t@NgŐServiceManagerNXݒ肷B<p>
     * AAVXevpeB"jp.ossc.nimbus.core.manager"Ŏw肵ServiceManagerNX̕D悳B<br>
     *
     * @param manager ̃t@NgŐServiceManagerNX
     */
    public static void setServiceManagerClass(Class<? extends ServiceManager> manager)
     throws IllegalArgumentException{
        if(manager == null){
            managerClass = null;
        }else if(ServiceManager.class.isAssignableFrom(manager)){
            managerClass = manager;
        }else{
            throw new IllegalArgumentException(
                message.findMessage(SVCMF00027, manager)
            );
        }
        logger.write(SVCMF00026, managerClass);
    }
    
    /**
     * T[rX`t@C[hServiceManagerNX擾B<p>
     *
     * @return ̃t@NgŐServiceManagerNX
     */
    public static Class<? extends ServiceManager> getServiceManagerClass(){
        return managerClass;
    }
    
    /**
     * Oo͂sLogger擾B<p>
     *
     * @return LoggerIuWFNg
     */
    public static Logger getLogger(){
        return logger;
    }
    
    /**
     * Oo͂sLoggerT[rXݒ肷B<p>
     *
     * @param name T[rX̃T[rX
     */
    public static void setLogger(ServiceName name){
        setLogger(name.getServiceManagerName(), name.getServiceName());
    }
    
    /**
     * Oo͂sLoggerT[rXݒ肷B<p>
     *
     * @param manager LoggerT[rXo^ĂServiceManager̖O
     * @param service LoggerT[rX̃T[rX
     */
    public static void setLogger(final String manager, final String service){
        if(isRegisteredService(manager, service)
             && getService(manager, service).getState() == Service.State.STARTED
        ){
            final Logger newLogger = (Logger)getServiceObject(manager, service);
            final Service newLoggerService = getService(manager, service);
            logger.write(SVCMF00028, manager, service);
            logger.setLogger(newLogger, newLoggerService);
        }else{
            addServiceStateListener(
                manager,
                service,
                new ServiceStateListener(){
                    public void stateChanged(ServiceStateChangeEvent e)
                     throws Exception{
                        ServiceManagerFactory.removeServiceStateListener(
                            manager,
                            service,
                            this
                        );
                        ServiceManagerFactory.setLogger(manager, service);
                    }
                    public boolean isEnabledState(Service.State st){
                        return st == Service.State.STARTED;
                    }
                }
            );
        }
    }
    
    /**
     * bZ[WsMessageRecordFactory擾B<p>
     *
     * @return MessageRecordFactoryIuWFNg
     */
    public static MessageRecordFactory getMessageRecordFactory(){
        return message;
    }
    
    /**
     * bZ[WsMessageRecordFactoryT[rXݒ肷B<p>
     *
     * @param name MessageRecordFactoryT[rX̃T[rX
     */
    public static void setMessageRecordFactory(ServiceName name){
        setMessageRecordFactory(
            name.getServiceManagerName(),
            name.getServiceName()
        );
    }
    
    /**
     * bZ[WsMessageRecordFactoryT[rXݒ肷B<p>
     *
     * @param manager MessageRecordFactoryT[rXo^ĂServiceManager̖O
     * @param service MessageRecordFactoryT[rX̃T[rX
     */
    public static void setMessageRecordFactory(
        final String manager,
        final String service
    ){
        if(isRegisteredService(manager, service)
             && getService(manager, service).getState() == Service.State.STARTED
        ){
            final MessageRecordFactory newMessage
                 = (MessageRecordFactory)getServiceObject(manager, service);
            final Service newMessageService = getService(manager, service);
            logger.write(SVCMF00029, manager, service);
            message.setMessageRecordFactory(newMessage, newMessageService);
        }else{
            addServiceStateListener(
                manager,
                service,
                new ServiceStateListener(){
                    public void stateChanged(ServiceStateChangeEvent e)
                     throws Exception{
                        ServiceManagerFactory.removeServiceStateListener(
                            manager,
                            service,
                            this
                        );
                        ServiceManagerFactory.setMessageRecordFactory(
                            manager,
                            service
                        );
                    }
                    public boolean isEnabledState(Service.State st){
                        return st == Service.State.STARTED;
                    }
                }
            );
        }
    }
    
    /**
     * o^ԂύXꂽĎRegistrationListenerǉB<p>
     *
     * @param listener RegistrationListenerIuWFNg
     */
    public static void addRegistrationListener(RegistrationListener listener){
        if(!registrationListeners.contains(listener)){
            registrationListeners.add(listener);
        }
    }
    
    /**
     * o^ԂύXꂽĎRegistrationListener폜B<p>
     *
     * @param listener RegistrationListenerIuWFNg
     */
    public static void removeRegistrationListener(
        RegistrationListener listener
    ){
        registrationListeners.remove(listener);
    }
    
    /**
     * ServiceManagero^ꂽRegistrationListenerɒʒmB<p>
     * 
     * @param manager o^ꂽServiceManager
     * @see RegistrationListener
     * @see RegistrationEvent
     */
    protected static void processRegisterd(ServiceManager manager){
        final Iterator<RegistrationListener> listeners
             = new ArrayList<RegistrationListener>(registrationListeners).iterator();
        while(listeners.hasNext()){
            final RegistrationListener listener
                 = (RegistrationListener)listeners.next();
            listener.registered(new RegistrationEvent(manager));
        }
    }
    
    /**
     * ServiceManager폜ꂽRegistrationListenerɒʒmB<p>
     * 
     * @param manager 폜ꂽServiceManager
     * @see RegistrationListener
     * @see RegistrationEvent
     */
    protected static void processUnregisterd(ServiceManager manager){
        final Iterator<RegistrationListener> listeners
             = new ArrayList<RegistrationListener>(registrationListeners).iterator();
        while(listeners.hasNext()){
            final RegistrationListener listener
                 = (RegistrationListener)listeners.next();
            listener.unregistered(new RegistrationEvent(manager));
        }
    }
    
    /**
     * T[rX̏ԂύXꂽĎServiceStateListenerǉB<p>
     * w肵{@link ServiceManager}o^ĂȂꍇARegistrationListenero^BServiceManagero^ƁA{@link ServiceManager#addServiceStateListener(String, ServiceStateListener)}ŁAServiceStateListenero^B<br>
     *
     * @param managerName }l[W
     * @param serviceName T[rX
     * @param listener ServiceStateListenerIuWFNg
     */
    public static void addServiceStateListener(
        final String managerName,
        final String serviceName,
        final ServiceStateListener listener
    ){
        if(isRegisteredManager(managerName)){
            final ServiceManager manager = findManager(managerName);
            manager.addServiceStateListener(serviceName, listener);
            return;
        }
        addRegistrationListener(
            new RegistrationListener(){
                public void registered(RegistrationEvent e){
                    final ServiceManager manager
                         = (ServiceManager)e.getRegistration();
                    if(!manager.getServiceName().equals(managerName)){
                        return;
                    }
                    removeRegistrationListener(this);
                    manager.addServiceStateListener(serviceName, listener);
                }
                public void unregistered(RegistrationEvent e){
                }
            }
        );
    }
    
    /**
     * T[rX̏ԂύXꂽĎServiceStateListenerǉB<p>
     * w肵{@link ServiceManager}o^ĂȂꍇARegistrationListenero^BServiceManagero^ƁA{@link ServiceManager#addServiceStateListener(String, ServiceStateListener)}ŁAServiceStateListenero^B<br>
     *
     * @param serviceName T[rX
     * @param listener ServiceStateListenerIuWFNg
     */
    public static void addServiceStateListener(
        ServiceName serviceName,
        ServiceStateListener listener
    ){
        addServiceStateListener(
            serviceName.getServiceManagerName(),
            serviceName.getServiceName(),
            listener
        );
    }
    
    /**
     * T[rX̏ԂύXꂽĎServiceStateListenerǉB<p>
     * ftHg{@link ServiceManager}o^ĂȂꍇARegistrationListenero^BftHgServiceManagero^ƁA{@link ServiceManager#addServiceStateListener(String, ServiceStateListener)}ŁAServiceStateListenero^B<br>
     *
     * @param serviceName T[rX
     * @param listener ServiceStateListenerIuWFNg
     */
    public static void addServiceStateListener(
        final String serviceName,
        final ServiceStateListener listener
    ){
        final ServiceManager manager = findManager(serviceName);
        if(manager != null){
            manager.addServiceStateListener(serviceName, listener);
            return;
        }
        addRegistrationListener(
            new RegistrationListener(){
                public void registered(RegistrationEvent e){
                    final ServiceManager manager
                         = (ServiceManager)e.getRegistration();
                    if(!manager.getServiceName()
                        .equals(ServiceManager.DEFAULT_NAME)){
                        return;
                    }
                    removeRegistrationListener(this);
                    manager.addServiceStateListener(serviceName, listener);
                }
                public void unregistered(RegistrationEvent e){
                }
            }
        );
    }
    
    /**
     * T[rX̏ԂύXꂽĎServiceStateListener폜B<p>
     *
     * @param managerName }l[W
     * @param serviceName T[rX
     * @param listener ServiceStateListenerIuWFNg
     */
    public static void removeServiceStateListener(
        String managerName,
        String serviceName,
        ServiceStateListener listener
    ){
        if(isRegisteredManager(managerName)){
            final ServiceManager manager = findManager(managerName);
            manager.removeServiceStateListener(serviceName, listener);
        }
    }
    
    /**
     * T[rX̏ԂύXꂽĎServiceStateListener폜B<p>
     *
     * @param serviceName T[rX
     * @param listener ServiceStateListenerIuWFNg
     */
    public static void removeServiceStateListener(
        ServiceName serviceName,
        ServiceStateListener listener
    ){
        removeServiceStateListener(
            serviceName.getServiceManagerName(),
            serviceName.getServiceName(),
            listener
        );
    }
    
    /**
     * T[rX̏ԂύXꂽĎServiceStateListener폜B<p>
     *
     * @param serviceName T[rX
     * @param listener ServiceStateListenerIuWFNg
     */
    public static void removeServiceStateListener(
        String serviceName,
        ServiceStateListener listener
    ){
        final ServiceManager manager = findManager();
        if(manager != null){
            manager.removeServiceStateListener(serviceName, listener);
        }
    }
    
    /**
     * T[ovpeB擾B<p>
     * 
     * @param name vpeB
     * @return T[ovpeB
     */
    public static String getProperty(String name){
        return properties.getProperty(name);
    }
    
    /**
     * T[ovpeBݒ肷B<p>
     * 
     * @param name vpeB
     * @param value T[ovpeB
     */
    public static void setProperty(String name, String value){
        properties.setProperty(name, value);
    }
    
    /**
     * T[ovpeB폜B<p>
     * 
     * @param name vpeB
     */
    public static void clearProperty(String name){
        properties.remove(name);
    }
    
    /**
     * gp@Wo͂ɕ\B<p>
     */
    private static void usage(){
        try{
            System.out.println(
                getResourceString(USAGE_RESOURCE)
            );
        }catch(IOException e){
            e.printStackTrace();
        }
    }
    /**
     * \[X𕶎ƂēǂݍށB<p>
     *
     * @param name \[X
     * @exception IOException \[X݂Ȃꍇ
     */
    private static String getResourceString(String name) throws IOException{
        
        // \[X̓̓Xg[擾
        InputStream is = ServiceManagerFactory.class.getClassLoader()
            .getResourceAsStream(name);
        
        // bZ[W̓ǂݍ
        StringBuilder buf = new StringBuilder();
        BufferedReader reader = null;
        final String separator = System.getProperty("line.separator");
        try{
            reader = new BufferedReader(new InputStreamReader(is));
            String line = null;
            while((line = reader.readLine()) != null){
                buf.append(line).append(separator);
            }
        }finally{
            if(reader != null){
                try{
                    reader.close();
                }catch(IOException e){
                }
            }
        }
        return unicodeConvert(buf.toString());
    }
    
    /**
     * jR[hGXP[v܂ł\̂镶ftHgGR[fBO̕ɕϊB<p>
     *
     * @param str jR[hGXP[v܂ł\̂镶
     * @return ftHgGR[fBO̕
     */
    private static String unicodeConvert(String str){
        char c;
        int len = str.length();
        StringBuilder buf = new StringBuilder(len);
        
        for(int i = 0; i < len; ){
            c = str.charAt(i++);
            if(c == '\\'){
                c = str.charAt(i++);
                if(c == 'u'){
                    int value = 0;
                    for(int j = 0; j < 4; j++){
                        c = str.charAt(i++);
                        switch(c){
                        case '0':
                        case '1':
                        case '2':
                        case '3':
                        case '4':
                        case '5':
                        case '6':
                        case '7':
                        case '8':
                        case '9':
                            value = (value << 4) + (c - '0');
                            break;
                        case 'a':
                        case 'b':
                        case 'c':
                        case 'd':
                        case 'e':
                        case 'f':
                            value = (value << 4) + 10 + (c - 'a');
                            break;
                        case 'A':
                        case 'B':
                        case 'C':
                        case 'D':
                        case 'E':
                        case 'F':
                            value = (value << 4) + 10 + (c - 'A');
                            break;
                        default:
                            throw new IllegalArgumentException(
                                "Failed to convert unicode : " + c
                            );
                        }
                    }
                    buf.append((char)value);
                }else{
                    switch(c){
                    case 't':
                        c = '\t';
                        break;
                    case 'r':
                        c = '\r';
                        break;
                    case 'n':
                        c = '\n';
                        break;
                    case 'f':
                        c = '\f';
                        break;
                    default:
                    }
                    buf.append(c);
                }
            }else{
                buf.append(c);
            }
        }
        return buf.toString();
    }
    
    /**
     * RpCR}hsB<p>
     * <pre>
     * R}hgp@F
     *  java jp.ossc.nimbus.core.ServiceManagerFactory [options] [paths]
     * 
     * [options]
     * 
     *  [-validate]
     *   T[rX`DTDŌ؂B
     * 
     *  [-server]
     *   CXbhҋ@āAT[oƂēB
     * 
     *  [-help]
     *   wv\܂B
     * 
     * [paths]
     *  [hT[rX`t@C̃pX
     * 
     * gp : 
     *    java -classpath classes;lib/nimbus.jar jp.ossc.nimbus.core.ServiceManagerFactory service-definition.xml
     * </pre>
     *
     * @param args R}h
     * @exception Exception RpCɖ肪ꍇ
     */
    public static void main(String[] args) throws Exception{
        
        if(args.length != 0 && args[0].equals("-help")){
            usage();
            return;
        }
        
        final List<String> servicePaths = new ArrayList<String>();
        boolean validate = false;
        boolean server = false;
        for(int i = 0; i < args.length; i++){
            if(args[i].equals("-server")){
                server = true;
            }else if(args[i].equals("-validate")){
                validate = true;
            }else{
                servicePaths.add(args[i]);
            }
        }
        
        if(servicePaths.size() == 0){
            usage();
            return;
        }
        
        Runtime.getRuntime().addShutdownHook(
            new Thread(
                new Runnable(){
                    public void run(){
                        for(int i = servicePaths.size(); --i >= 0;){
                            ServiceManagerFactory.unloadManager((String)servicePaths.get(i));
                        }
                    }
                }
            )
        );
        
        for(int i = 0, max = servicePaths.size(); i < max; i++){
            if(!ServiceManagerFactory.loadManager((String)servicePaths.get(i), false, validate)){
                Thread.sleep(1000);
                System.exit(-1);
            }
        }
        if(!ServiceManagerFactory.checkLoadManagerCompleted()){
            Thread.sleep(1000);
            System.exit(-1);
        }
        if(server){
            WaitSynchronizeMonitor lock = new WaitSynchronizeMonitor();
            synchronized(lock){
               lock.initMonitor();
                try{
                   lock.waitMonitor();
                }catch(InterruptedException ignore){}
            }
        }
    }
}