/*
 * This software is distributed under following license based on modified BSD
 * style license.
 * ----------------------------------------------------------------------
 * 
 * Copyright 2009 The Nimbus2 Project. All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer. 
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE NIMBUS PROJECT ``AS IS'' AND ANY EXPRESS
 * OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN
 * NO EVENT SHALL THE NIMBUS PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 * THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 * 
 * The views and conclusions contained in the software and documentation are
 * those of the authors and should not be interpreted as representing official
 * policies, either expressed or implied, of the Nimbus2 Project.
 */
package jp.ossc.nimbus.beans.dataset;

import java.util.*;
import java.io.*;
import java.lang.reflect.Field;

import org.apache.commons.jexl2.*;

import jp.ossc.nimbus.beans.PropertyAccess;
import jp.ossc.nimbus.service.codemaster.PartUpdate;
import jp.ossc.nimbus.service.codemaster.PartUpdateRecords;
import jp.ossc.nimbus.service.codemaster.CodeMasterUpdateKey;


/**
 * R[hXgB<p>
 * f[^ZbǧJԂ\f[^\BeanŁA{@link Record R[h}̃XgłB<br>
 * JԂ\̗vfł郌R[h́AXL[}`ɂāAǂ̂悤ȃR[hivpeBA^ȂǁjJԂ̂𓮓IɌłB<br>
 * ȉɃTvR[hB<br>
 * <pre>
 *     import jp.ossc.nimbus.beans.dataset.*;
 *     
 *     // R[hXg𐶐
 *     RecordList recordList = new RecordList();
 *     
 *     // R[hXg̃XL[}ȉ̂悤ɒ`
 *     //   vpeB  ^
 *     //        A        int
 *     //        B        java.lang.String
 *     //        C        java.lang.String
 *     recordList.setSchema(
 *         ":A,int\n"
 *             + ":B,java.lang.String\n"
 *             + ":C,java.lang.String"
 *     );
 *     
 *     // R[h1𐶐āAlݒ肷
 *     Record record1 = recordList.createRecord();
 *     record1.setProperty("A", 1);
 *     record1.setProperty("B", "hoge1");
 *     record1.setProperty("C", "fuga1");
 *     recordList.addRecord(record1);
 *     // R[h2𐶐āAlݒ肷
 *     Record record2 = recordList.createRecord();
 *     record2.setProperty("A", 2);
 *     record2.setProperty("B", "hoge2");
 *     record2.setProperty("C", "fuga2");
 *     recordList.addRecord(record2);
 * </pre>
 * 
 * @author M.Takata
 */
public class RecordList implements Externalizable, List<Record>, RandomAccess, Cloneable, PartUpdate{
    
    private static final long serialVersionUID = 6399184480196775369L;
    
    /**
     * L[ɂCfbNX\\񖼁B<p>
     */
    public static final String PRIMARY_KEY_INDEX_NAME = "$PRIMARY_KEY";
    
    /**
     * ̃vpeB֐\B<p>
     */
    protected static final String PROP_FUNCTION_NAME = "prop";
    
    /**
     * R[hB<p>
     */
    protected String name;
    
    /**
     * XL[}B<p>
     */
    protected String schema;
    
    /**
     * R[hXL[}B<p>
     */
    protected RecordSchema recordSchema;
    
    /**
     * R[h̃XgB<p>
     */
    protected List<Record> records = Collections.synchronizedList(new ArrayList<Record>());
    
    /**
     * p̃CfbNX}bvB<p>
     */
    protected Map<String, Expression> conditionIndexMap;
    
    /**
     * p̌ʃ}bvB<p>
     */
    protected Map<String, List<Record>> conditionIndexSearchMap;
    
    /**
     * L[p̃CfbNX}bvB<p>
     */
    protected Map<String, KeyIndex> keyIndexMap;
    
    /**
     * L[p̌ʃ}bvB<p>
     */
    protected Map<String, Map<Object, List<Record>>> keyIndexSearchMap;
    
    /**
     * XVJEgB<p>
     */
    protected int modCount = 0;
    
    protected int[] partUpdateOrderBy;
    
    protected boolean[] partUpdateIsAsc;
    
    /**
     * `̃R[hXg𐶐B<p>
     */
    public RecordList(){
    }
    
    /**
     * `̃R[hXg𐶐B<p>
     *
     * @param name R[h
     */
    public RecordList(String name){
        this.name = name;
    }
    
    /**
     * ̃R[hXg𐶐B<p>
     *
     * @param name R[h
     * @param schema XL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public RecordList(String name, String schema)
     throws PropertySchemaDefineException{
        this.name = name;
        setSchema(schema);
    }
    
    /**
     * ̃R[hXg𐶐B<p>
     *
     * @param name R[h
     * @param schema XL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public RecordList(String name, RecordSchema schema)
     throws PropertySchemaDefineException{
        this.name = name;
        setRecordSchema(schema);
    }
    
    /**
     * R[h擾B<p>
     *
     * @return R[h
     */
    public String getName(){
        return name;
    }
    
    /**
     * R[hݒ肷B<p>
     *
     * @param name R[h
     */
    public void setName(String name){
        this.name = name;
    }
    
    /**
     * R[h̃XL[}ݒ肷B<p>
     *
     * @param schema R[h̃XL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public void setSchema(String schema) throws PropertySchemaDefineException{
        setRecordSchema(RecordSchema.getInstance(schema));
    }
    
    /**
     * R[h̃XL[}擾B<p>
     *
     * @return R[h̃XL[}
     */
    public String getSchema(){
        return schema;
    }
    
    /**
     * R[hXL[}ݒ肷B<p>
     *
     * @param schema R[hXL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public void setRecordSchema(RecordSchema schema) throws PropertySchemaDefineException{
        if(size() != 0){
            throw new PropertySchemaDefineException("Record already exists.");
        }
        recordSchema = schema;
        this.schema = schema == null ? null : schema.getSchema();
        List<String> primaryKeyNames = null;
        final PropertySchema[] primaryKeys
            = recordSchema.getPrimaryKeyPropertySchemata();
        if(primaryKeys != null){
            for(PropertySchema prop : primaryKeys){
                if(primaryKeyNames == null){
                    primaryKeyNames = new ArrayList<String>();
                }
                primaryKeyNames.add(prop.getName());
            }
        }
        if(primaryKeyNames == null){
            removeKeyIndex(PRIMARY_KEY_INDEX_NAME);
        }else{
            setKeyIndex(
                PRIMARY_KEY_INDEX_NAME,
                primaryKeyNames.toArray(new String[primaryKeyNames.size()])
            );
        }
    }
    
    /**
     * R[hXL[}uB<p>
     *
     * @param schema R[h̃XL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public void replaceSchema(String schema) throws PropertySchemaDefineException{
        replaceRecordSchema(RecordSchema.getInstance(schema));
    }
    
    /**
     * R[hXL[}uB<p>
     *
     * @param schema R[hXL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public void replaceRecordSchema(RecordSchema schema) throws PropertySchemaDefineException{
        
        if(recordSchema != null && schema != null && size() != 0){
            for(Record record : records){
                record.replaceRecordSchema(schema);
            }
        }
        setRecordSchema(schema);
    }
    
    /**
     * R[hXL[}ǉB<p>
     *
     * @param schema R[h̃XL[}
     * @exception PropertySchemaDefineException vpeB̃XL[}`Ɏsꍇ
     */
    public void appendSchema(
        String schema
    ) throws PropertySchemaDefineException{
        if(recordSchema == null){
            setSchema(schema);
        }else{
            replaceRecordSchema(
                recordSchema.appendSchema(schema)
            );
        }
    }
    
    /**
     * R[hXL[}擾B<p>
     *
     * @return R[hXL[}
     */
    public RecordSchema getRecordSchema(){
        return recordSchema;
    }
    
    /**
     * VR[h𐶐B<p>
     *
     * @return VR[h
     */
    public Record createRecord(){
        return new Record(recordSchema);
    }
    
    /**
     * w肳ꂽCfbNX̃R[h擾B<p>
     *
     * @param index CfbNX
     * @return R[h
     */
    public Record getRecord(int index){
        return (Record)get(index);
    }
    
    /**
     * R[hǉB<p>
     *
     * @param r R[h
     */
    public void addRecord(Record r){
        add(r);
    }
    
    /**
     * w肳ꂽCfbNXɃR[h}B<p>
     *
     * @param index CfbNX
     * @param r R[h
     */
    public void addRecord(int index, Record r){
        add(index, r);
    }
    
    /**
     * w肳ꂽCfbNX̃R[huB<p>
     * AÃ\bhŒuꂽR[h́ACfbNX^̑ΏۂɂȂȂB<br>
     *
     * @param index CfbNX
     * @param r R[h
     * @return uꂽÂR[h
     */
    public Record setRecord(int index, Record r){
        return (Record)set(index, r);
    }
    
    /**
     * w肳ꂽR[h폜B<p>
     *
     * @param r R[h
     */
    public void removeRecord(Record r){
        remove(r);
    }
    
    /**
     * w肳ꂽCfbNX̃R[h폜B<p>
     *
     * @param index CfbNX
     * @return 폜ꂽR[h
     */
    public Record removeRecord(int index){
        return (Record)remove(index);
    }
    
    /**
     * CfbNXw肵āAL[ɂCfbNXݒ肷B<p>
     * ɂ́AR[h~ςۂɌCfbNX^ƁA~ςꂽR[hɃtANZXČ郊A^B<br>
     * ̃Zb^[́ACfbNX^̂߂̏ݒŝŁAR[h~ςOɐݒ肵ĂȂ΂ȂȂB<br>
     * CfbNX^̗_́A~ώɓɌs邽߁Aۂ̌ɂ́A炩ߌꂽʂoł邽߁AȌ\ɂȂ鎖łB<br>
     * tɌ_́Aݒ肳OɒǉꂽR[hAvpeB̒lύXR[h́ACfbNXݒ肳ꂽʂɂ́AfȂB܂A炩ߐݒ肷Kv邽߁AIɕςꍇ́AΉłȂB̂悤ȏꍇ́AA^({@link #realSearch(String, Map)})gpB<br>
     * ̃Zb^[ɑΉCfbNX^́A{@link #searchByKeyIndex(String, Record)}ōsB<br>
     *
     * @param name CfbNX
     * @param propertyNames vpeBz
     * @exception DataSetException CfbNX̐ݒɎsꍇ
     * @see #searchByKeyIndex(String, Record)
     */
    public void setKeyIndex(String name, String... propertyNames) throws DataSetException{
        
        if(recordSchema == null){
            throw new DataSetException("Schema not initalize.");
        }
        final int[] propertyIndexes = new int[propertyNames.length];
        for(int i = 0; i < propertyNames.length; i++){
            propertyIndexes[i] = recordSchema.getPropertyIndex(propertyNames[i]);
            if(propertyIndexes[i] == -1){
                throw new DataSetException("No such property " + propertyNames[i]);
            }
        }
        setKeyIndex(name, propertyIndexes);
    }
    
    /**
     * CfbNXw肵āAL[ɂCfbNXݒ肷B<p>
     * ɂ́AR[h~ςۂɌCfbNX^ƁA~ςꂽR[hɃtANZXČ郊A^B<br>
     * ̃Zb^[́ACfbNX^̂߂̏ݒŝŁAR[h~ςOɐݒ肵ĂȂ΂ȂȂB<br>
     * CfbNX^̗_́A~ώɓɌs邽߁Aۂ̌ɂ́A炩ߌꂽʂoł邽߁AȌ\ɂȂ鎖łB<br>
     * tɌ_́Aݒ肳OɒǉꂽR[hAvpeB̒lύXR[h́A~ςꂽʂɂ́AfȂB܂A炩ߐݒ肷Kv邽߁AIɕςꍇ́AΉłȂB̂悤ȏꍇ́AA^({@link #realSearch(String, Map)})gpB<br>
     * ̃Zb^[ɑΉCfbNX^́A{@link #searchByKeyIndex(String, Record)}ōsB<br>
     *
     * @param name CfbNX
     * @param propertyIndexes vpeBCfbNXz
     * @exception DataSetException CfbNX̐ݒɎsꍇ
     * @see #searchByKeyIndex(String, Record)
     */
    public void setKeyIndex(String name, int... propertyIndexes) throws DataSetException{
        
        if(keyIndexMap == null){
            keyIndexMap = Collections.synchronizedMap(new HashMap<String, KeyIndex>());
        }
        
        if(recordSchema == null){
            throw new DataSetException("Schema not initalize.");
        }
        
        final int fieldSize = recordSchema.getPropertySize();
        for(int index : propertyIndexes){
            if(index >= fieldSize){
                throw new DataSetException("No such property " + index);
            }
        }
        
        final KeyIndex condition = new KeyIndex(propertyIndexes);
        keyIndexMap.put(name, condition);
        
        if(keyIndexSearchMap == null){
            keyIndexSearchMap = Collections.synchronizedMap(new HashMap<String, Map<Object, List<Record>>>());
        }else{
            keyIndexSearchMap.remove(name);
        }
        if(size() != 0){
            analyzeSearchKeyIndex(name);
        }
    }
    
    /**
     * L[ɂCfbNX̃CfbNXz擾B<p>
     *
     * @return CfbNXz
     */
    public String[] getKeyIndexNames(){
        return keyIndexMap == null || keyIndexMap.size() == 0 ? new String[0]
            : keyIndexMap.keySet().toArray(new String[keyIndexMap.size()]);
    }
    
    /**
     * CfbNXw肵āACfbNX̃L[ƂȂvpeBz擾B<p>
     *
     * @param name CfbNX
     * @return vpeBz
     */
    public String[] getKeyIndexPropertyNames(String name){
        final int[] indexes = getKeyIndexPropertyIndexes(name);
        if(indexes == null || recordSchema == null){
            return null;
        }
        
        final String[] propNames = new String[indexes.length];
        for(int i = 0; i < indexes.length; i++){
            propNames[i] = recordSchema.getPropertyName(indexes[i]);
        }
        return propNames;
    }
    
    /**
     * CfbNXw肵āACfbNX̃L[ƂȂvpeBCfbNXz擾B<p>
     *
     * @param name CfbNX
     * @return vpeBCfbNXz
     */
    public int[] getKeyIndexPropertyIndexes(String name){
        if(keyIndexMap == null || !keyIndexMap.containsKey(name)){
            return null;
        }
        final KeyIndex condition = keyIndexMap.get(name);
        return condition.propertyIndexes;
    }
    
    /**
     * CfbNXw肵āAL[ɂCfbNX폜B<p>
     *
     * @param name CfbNX
     */
    public void removeKeyIndex(String name){
        if(keyIndexMap != null){
            keyIndexMap.remove(name);
        }
        if(keyIndexSearchMap != null){
            keyIndexSearchMap.remove(name);
        }
    }
    
    /**
     * SẴL[ɂCfbNX폜B<p>
     */
    public void clearKeyIndex(){
        if(keyIndexMap != null){
            keyIndexMap.clear();
        }
        if(keyIndexSearchMap != null){
            keyIndexSearchMap.clear();
        }
    }
    
    /**
     * CfbNXw肵āAɂCfbNXݒ肷B<p>
     * ɂ́AR[h~ςۂɌCfbNX^ƁA~ςꂽR[hɃtANZXČ郊A^B<br>
     * ̃Zb^[́ACfbNX^̂߂̏ݒŝŁAR[h~ςOɐݒ肵ĂȂ΂ȂȂB<br>
     * CfbNX^̗_́A~ώɓɌs邽߁Aۂ̌ɂ́A炩ߌꂽʂoł邽߁AȌ\ɂȂ鎖łB<br>
     * tɌ_́Aݒ肳OɒǉꂽR[hAvpeB̒lύXR[h́A~ςꂽʂɂ́AfȂB܂A炩ߐݒ肷Kv邽߁AIɕςꍇ́AΉłȂB̂悤ȏꍇ́AA^({@link #realSearch(String, Map)})gpB<br>
     * ̃Zb^[ɑΉCfbNX^́A{@link #searchByConditionIndex(String)}ōsB<br>
     * <p>
     * ́A<a href="http://jakarta.apache.org/commons/jexl/">Jakarta Commons Jexl</a>̎gpB<br>
     * CfbNX^ł́AR[h̗̒lA񖼂w肷鎖ŁAŎQƂ鎖łB<br>
     * <pre>
     *  FA == '1' and B &gt;= 3
     * </pre>
     *
     * @param name CfbNX
     * @param condition 
     * @exception DataSetException sȏꍇ
     * @see #searchByConditionIndex(String)
     */
    public void setConditionIndex(String name, String condition)
     throws DataSetException{
        
        if(conditionIndexMap == null){
            conditionIndexMap = Collections.synchronizedMap(new HashMap<String, Expression>());
        }
        
        if(recordSchema == null){
            throw new DataSetException("Schema not initalize.");
        }
        
        try{
            JexlEngine jexl = new JexlEngine();
            jexl.setSilent(true);
            Map<String, Object> funcs = new HashMap<String, Object>();
            PropertyAccess propAccess = new PropertyAccess();
            funcs.put(PROP_FUNCTION_NAME, propAccess);
            jexl.setFunctions(funcs);
            conditionIndexMap.put(name, jexl.createExpression(condition));
        }catch(Exception e){
            throw new DataSetException(e);
        }
        
        if(conditionIndexSearchMap == null){
            conditionIndexSearchMap = Collections.synchronizedMap(new HashMap<String, List<Record>>());
        }else{
            conditionIndexSearchMap.remove(name);
        }
        if(size() != 0){
            analyzeSearchConditionIndex(name);
        }
    }
    
    /**
     * ɂCfbNX̃CfbNXz擾B<p>
     *
     * @return CfbNXz
     */
    public String[] getConditionIndexNames(){
        return conditionIndexMap == null || conditionIndexMap.size() == 0 ? new String[0]
            : conditionIndexMap.keySet().toArray(new String[conditionIndexMap.size()]);
    }
    
    /**
     * CfbNXw肵āACfbNX̏擾B<p>
     *
     * @param name CfbNX
     * @return 
     */
    public String getConditionIndex(String name){
        if(conditionIndexMap == null){
            return null;
        }
        Expression exp = conditionIndexMap.get(name);
        return exp == null ? null : exp.getExpression();
    }
    
    /**
     * CfbNXw肵āAɂCfbNX폜B<p>
     *
     * @param name CfbNX
     */
    public void removeConditionIndex(String name){
        if(conditionIndexMap != null){
            conditionIndexMap.remove(name);
        }
        if(conditionIndexSearchMap != null){
            conditionIndexSearchMap.remove(name);
        }
    }
    
    /**
     * SĂ̏ɂCfbNX폜B<p>
     */
    public void clearConditionIndex(){
        if(conditionIndexMap != null){
            conditionIndexMap.clear();
        }
        if(conditionIndexSearchMap != null){
            conditionIndexSearchMap.clear();
        }
    }
    
    /**
     * vC}L[ŌB<p>
     * vC}L[L[s߂ɂ́AXL[}`ɂāAj[NL[tOݒ肷KvB<br>
     *
     * @param key L[R[h
     * @return ʁBɍvR[h
     */
    public Record searchByPrimaryKey(Record key){
        final List<Record> list = searchByKeyIndex(PRIMARY_KEY_INDEX_NAME, key);
        return list == null || list.size() == 0 ? null : list.get(0);
    }
    
    /**
     * w肳ꂽCfbNX̃L[CfbNX^̌ʂ擾B<p>
     * ~ό^L[s߂ɂ́A{@link #setKeyIndex(String, String[])}A{@link #setKeyIndex(String, int[])}Ōݒ肵ĂKvB<br>
     *
     * @param name CfbNX
     * @param key L[R[h
     * @return ʁBɍvR[h̔z
     * @see #setKeyIndex(String, String[])
     * @see #setKeyIndex(String, int[])
     */
    public List<Record> searchByKeyIndex(String name, Record key){
        final List<Record> result = new ArrayList<Record>();
        if(keyIndexSearchMap == null
             || keyIndexSearchMap.size() == 0){
            return result;
        }
        final Map<Object, List<Record>> map = keyIndexSearchMap.get(name);
        if(map == null){
            return result;
        }
        final KeyIndex condition = keyIndexMap.get(name);
        if(condition == null){
            return result;
        }
        Object recKey = condition.createKey(key);
        if(recKey == null){
            return result;
        }
        List<Record> values = map.get(recKey);
        if(values == null){
            return result;
        }
        result.addAll(values);
        return result;
    }
    
    /**
     * w肳ꂽCfbNX̏CfbNX^̌ʂ擾B<p>
     * CfbNX^s߂ɂ́A{@link #setConditionIndex(String, String)}Ōݒ肵ĂKvB<br>
     *
     * @param name CfbNX
     * @return ʁBɍvR[h̃Xg
     * @see #setConditionIndex(String, String)
     */
    public List<Record> searchByConditionIndex(String name){
        final List<Record> result = new ArrayList<Record>();
        if(conditionIndexSearchMap == null
             || conditionIndexSearchMap.size() == 0){
            return result;
        }
        final List<Record> values = conditionIndexSearchMap.get(name);
        if(values == null){
            return result;
        }
        result.addAll(values);
        return result;
    }
    
    /**
     * A^L[sB<p>
     * R[hXgAw肵vpeB̒lAL[R[h̒lƈvR[hB<br>
     *
     * @param key L[R[h
     * @param propertyNames L[ƂȂvpeB
     * @return ʁBɍvR[h̔z
     */
    public List<Record> searchByKey(Record key, String... propertyNames){
        final List<Record> result = new ArrayList<Record>();
        if(size() == 0){
            return result;
        }
        if(recordSchema == null){
            throw new DataSetException("Schema not initalize.");
        }
        final int[] propertyIndexes = new int[propertyNames.length];
        for(int i = 0; i < propertyNames.length; i++){
            propertyIndexes[i] = recordSchema.getPropertyIndex(propertyNames[i]);
            if(propertyIndexes[i] == -1){
                throw new DataSetException("No such property " + propertyNames[i]);
            }
        }
        return searchByKey(key, propertyIndexes);
    }
    
    /**
     * A^L[sB<p>
     * R[hXgAw肵vpeB̒lAL[R[h̒lƈvR[hB<br>
     *
     * @param key L[R[h
     * @param propertyNames L[ƂȂvpeB
     * @return ʁBɍvR[h̔z
     */
    public List<Record> searchByKey(Record key, int... propertyIndexes){
        final List<Record> result = new ArrayList<Record>();
        if(size() == 0){
            return result;
        }
        if(recordSchema == null){
            throw new DataSetException("Schema not initalize.");
        }
        for(int i = 0, imax = size(); i < imax; i++){
            Record rd = getRecord(i);
            boolean isMatch = true;
            for(int j = 0; j < propertyIndexes.length; j++){
                Object val1 = key.getProperty(propertyIndexes[j]);
                Object val2 = rd.getProperty(propertyIndexes[j]);
                if(val1 == null && val2 == null){
                    continue;
                }else if(val1 == null && val2 != null
                    || val1 != null && val2 == null
                    || !val1.equals(val2)
                ){
                    isMatch = false;
                    break;
                }
            }
            if(isMatch){
                result.add(rd);
            }
        }
        return result;
    }
    
    /**
     * A^sB<p>
     * R[hXgAɍv郌R[hB<br>
     * ܂Aɂ́AR[h~ςۂɌCfbNX^ƁA~ςꂽR[h烊AɌ郊A^B<br>
     * A^̗_́AɁAIɕςϐw肵A̕ϐlvalueMapŗ^鎖ł鎖łB<br>
     * <p>
     * ́A<a href="http://jakarta.apache.org/commons/jexl/">Jakarta Commons Jexl</a>̎gpB<br>
     * A^ł́AR[h̗̒lA񖼂w肷鎖ŁAŎQƂ鎖ł̂ɉāACӂ̕ϐɒ`A̒lvalueMapŗ^鎖łB<br>
     * <pre>
     *  FA == '1' and B &gt;= 3
     * </pre>
     *
     * @param condition 
     * @param valueMap ̕ϐ}bv
     * @return ʁBɍvR[h̃Xg
     * @exception DataSetException sȏꍇ
     */
    public List<Record> searchByCondition(String condition, Map<String, Object> valueMap) throws DataSetException{
        final List<Record> result = new ArrayList<Record>();
        if(size() == 0){
            return result;
        }
        if(recordSchema == null){
            throw new DataSetException("Schema not initalize.");
        }
        try{
            JexlEngine jexl = new JexlEngine();
            jexl.setSilent(true);
            Map<String, Object> funcs = new HashMap<String, Object>();
            PropertyAccess propAccess = new PropertyAccess();
            funcs.put(PROP_FUNCTION_NAME, propAccess);
            jexl.setFunctions(funcs);
            final Expression exp
                 = jexl.createExpression(condition);
            final JexlContext context = valueMap == null ? new MapContext()
                : new MapContext(valueMap);
            for(int i = 0, imax = size(); i < imax; i++){
                Record rd = getRecord(i);
                for(int j = 0, jmax = recordSchema.getPropertySize();
                        j < jmax; j++){
                    final PropertySchema prop = recordSchema.getPropertySchema(j);
                    final String propName = prop.getName();
                    context.set(propName, rd.getProperty(propName));
                }
                final Boolean ret = (Boolean)exp.evaluate(context);
                if(ret != null && ret.booleanValue()){
                    result.add(rd);
                }
            }
        }catch(Exception e){
            throw new DataSetException(e);
        }
        return result;
    }
    
    /**
     * CfbNX^̍ĉ͂s܂B<p>
     *
     * @param name CfbNX
     * @exception DataSetException sȏꍇ
     */
    public void analyzeSearchConditionIndex(String name)
     throws DataSetException{
        if(size() == 0
             || conditionIndexMap == null
             || conditionIndexMap.size() == 0
             || !conditionIndexMap.containsKey(name)
        ){
            return;
        }
        final Expression exp = conditionIndexMap.get(name);
        List<Record> values = conditionIndexSearchMap.get(name);
        if(values == null){
            values = new ArrayList<Record>();
            conditionIndexSearchMap.put(name, values);
        }
        values.clear();
        
        try{
            final JexlContext context = new MapContext();
            for(Record record : records){
                for(int i = 0, imax = recordSchema.getPropertySize();
                        i < imax; i++){
                    final PropertySchema prop
                         = recordSchema.getPropertySchema(i);
                    final String propName = prop.getName();
                    context.set(
                        propName,
                        record.getProperty(propName)
                    );
                }
                Object ret = exp.evaluate(context);
                if(!(ret instanceof Boolean)){
                    throw new DataSetException(
                        "Illegal condition : " + exp.getExpression()
                    );
                }
                boolean isMatch = ((Boolean)ret).booleanValue();
                if(isMatch){
                    values.add(record);
                }
            }
        }catch(DataSetException e){
            throw e;
        }catch(Exception e){
            throw new DataSetException(e);
        }
    }
    
    /**
     * L[CfbNX^̍ĉ͂s܂B<p>
     *
     * @param name CfbNX
     * @exception DataSetException ĉ͂Ɏsꍇ
     */
    public void analyzeSearchKeyIndex(String name)
     throws DataSetException{
        if(size() == 0
             || keyIndexMap == null
             || keyIndexMap.size() == 0
             || !keyIndexMap.containsKey(name)
        ){
            return;
        }
        final KeyIndex condition = keyIndexMap.get(name);
        Map<Object, List<Record>> map = keyIndexSearchMap.get(name);
        if(map == null){
            map = Collections.synchronizedMap(new HashMap<Object, List<Record>>());
            keyIndexSearchMap.put(name, map);
        }else{
            map.clear();
        }
        boolean isPrimaryKeyIndex = PRIMARY_KEY_INDEX_NAME.equals(name);
        for(Record record : records){
            final Object key = condition.createKey(record);
            if(key == null){
                continue;
            }
            List<Record> values = map.get(key);
            if(values == null){
                values = new ArrayList<Record>();
                map.put(key, values);
            }
            if(isPrimaryKeyIndex && values.size() != 0){
                throw new DataSetException("Dupilicate primary key. " + key);
            }
            values.add(record);
        }
    }
    
    /**
     * CfbNX^̒~ςs܂B<p>
     *
     * @param record ~ς郌R[h
     * @exception DataSetException sȏꍇ
     */
    protected void addSearchConditionIndex(Record record)
     throws DataSetException{
        if(conditionIndexMap == null
             || conditionIndexMap.size() == 0){
            return;
        }
        
        if(recordSchema == null){
            throw new DataSetException("Schema not initalize.");
        }
        try{
            for(Map.Entry<String, Expression> entry : conditionIndexMap.entrySet()){
                final String name = entry.getKey();
                final Expression exp = entry.getValue();
                final JexlContext context = new MapContext();
                for(int i = 0, imax = recordSchema.getPropertySize();
                        i < imax; i++){
                    final PropertySchema prop
                         = recordSchema.getPropertySchema(i);
                    final String propName = prop.getName();
                    context.set(
                        propName,
                        record.getProperty(propName)
                    );
                }
                Object ret = exp.evaluate(context);
                if(!(ret instanceof Boolean)){
                    throw new DataSetException(
                        "Illegal condition : " + exp.getExpression()
                    );
                }
                boolean isMatch = ((Boolean)ret).booleanValue();
                if(isMatch){
                    List<Record> values = conditionIndexSearchMap.get(name);
                    if(values == null){
                        values = new ArrayList<Record>();
                        conditionIndexSearchMap.put(name, values);
                    }
                    values.add(record);
                }
            }
        }catch(DataSetException e){
            throw e;
        }catch(Exception e){
            throw new DataSetException(e);
        }
    }
    
    /**
     * CfbNX^̒~ς폜܂B<p>
     *
     * @param record ~ς폜郌R[h
     * @exception DataSetException sȏꍇ
     */
    protected void removeSearchConditionIndex(Record record)
     throws DataSetException{
        if(conditionIndexSearchMap == null
             || conditionIndexSearchMap.size() == 0){
            return;
        }
        
        if(recordSchema == null){
            throw new DataSetException("Schema not initalize.");
        }
        for(List<Record> list : conditionIndexSearchMap.values()){
            list.remove(record);
        }
    }
    
    /**
     * L[CfbNX^̒~ςs܂B<p>
     *
     * @param record ~ς郌R[h
     * @exception DataSetException sȏꍇ
     */
    protected void addSearchKeyIndex(Record record)
     throws DataSetException{
        if(keyIndexMap == null
             || keyIndexMap.size() == 0){
            return;
        }
        for(Map.Entry<String, KeyIndex> entry : keyIndexMap.entrySet()){
            final String name = entry.getKey();
            final boolean isPrimaryKeyIndex = PRIMARY_KEY_INDEX_NAME.equals(name);
            final KeyIndex condition = entry.getValue();
            final Object key = condition.createKey(record);
            if(key == null){
                continue;
            }
            Map<Object, List<Record>> map = keyIndexSearchMap.get(name);
            if(map == null){
                map = Collections.synchronizedMap(new HashMap<Object, List<Record>>());
                keyIndexSearchMap.put(name, map);
            }
            List<Record> values = map.get(key);
            if(values == null){
                values = new ArrayList<Record>();
                map.put(key, values);
            }
            if(isPrimaryKeyIndex && values.size() != 0){
                throw new DataSetException("Dupilicate primary key. " + key);
            }
            values.add(record);
        }
    }
    
    /**
     * L[CfbNX^̒~ς폜܂B<p>
     *
     * @param record ~ς폜郌R[h
     * @exception DataSetException sȏꍇ
     */
    protected void removeSearchKeyIndex(Record record)
     throws DataSetException{
        if(keyIndexMap == null
             || keyIndexMap.size() == 0){
            return;
        }
        for(Map.Entry<String, KeyIndex> entry : keyIndexMap.entrySet()){
            final String name = entry.getKey();
            final KeyIndex condition = entry.getValue();
            final Object key = condition.createKey(record);
            if(key == null){
                continue;
            }
            Map<Object, List<Record>> map = keyIndexSearchMap.get(name);
            if(map == null){
                continue;
            }
            List<Record> values = map.get(key);
            if(values == null){
                continue;
            }
            values.remove(record);
        }
    }
    
    /**
     * w肳ꂽvpeB̃vpeB\[gL[ɂāAŃ\[gB<p>
     *
     * @param orderBy \[gL[ƂȂvpeBz
     */
    public void sort(String... orderBy){
        sort(orderBy, null);
    }
    
    /**
     * w肳ꂽvpeBCfbNX̃vpeB\[gL[ɂāAŃ\[gB<p>
     *
     * @param orderBy \[gL[ƂȂvpeBCfbNXz
     */
    public void sort(int... orderBy){
        sort(orderBy, null);
    }
    
    /**
     * w肳ꂽvpeBCfbNX̃vpeB\[gL[ɂă\[gB<p>
     *
     * @param orderBy \[gL[ƂȂvpeBCfbNXz
     * @param isAsc \[gꍇtrueB~\[gꍇ́Afalse
     */
    public void sort(int[] orderBy, boolean[] isAsc){
        if(records.size() < 2){
            return;
        }
        Collections.sort(records, new RecordComparator(recordSchema, orderBy, isAsc));
    }
    
    /**
     * w肳ꂽvpeB̃vpeB\[gL[ɂă\[gB<p>
     *
     * @param orderBy \[gL[ƂȂvpeBz
     * @param isAsc \[gꍇtrueB~\[gꍇ́Afalse
     */
    public void sort(String[] orderBy, boolean[] isAsc){
        if(records.size() < 2){
            return;
        }
        Collections.sort(records, new RecordComparator(orderBy, isAsc));
    }
    
    /**
     * w肳ꂽvpeB̃vpeB\[gL[ɂāAŃ\[gB<p>
     *
     * @param records \[gΏۂ̃R[h̃Xg
     * @param orderBy \[gL[ƂȂvpeBz
     */
    public static void sort(List<Record> records, String... orderBy){
        sort(records, orderBy, null);
    }
    
    /**
     * w肳ꂽvpeBCfbNX̃vpeB\[gL[ɂāAŃ\[gB<p>
     *
     * @param records \[gΏۂ̃R[h̃Xg
     * @param orderBy \[gL[ƂȂvpeBCfbNXz
     */
    public static void sort(List<Record> records, int... orderBy){
        sort(records, orderBy, null);
    }
    
    /**
     * w肳ꂽvpeBCfbNX̃vpeB\[gL[ɂă\[gB<p>
     *
     * @param records \[gΏۂ̃R[h̃Xg
     * @param orderBy \[gL[ƂȂvpeBCfbNXz
     * @param isAsc \[gꍇtrueB~\[gꍇ́Afalse
     */
    public static void sort(List<Record> records, int[] orderBy, boolean[] isAsc){
        if(records.size() < 2){
            return;
        }
        Collections.sort(
            records,
            new RecordComparator(
                records.get(0).getRecordSchema(),
                orderBy,
                isAsc
            )
        );
    }
    
    /**
     * w肳ꂽvpeB̃vpeB\[gL[ɂă\[gB<p>
     *
     * @param records \[gΏۂ̃R[h̃Xg
     * @param orderBy \[gL[ƂȂvpeBz
     * @param isAsc \[gꍇtrueB~\[gꍇ́Afalse
     */
    public static void sort(List<Record> records, String[] orderBy, boolean[] isAsc){
        if(records.size() < 2){
            return;
        }
        Collections.sort(
            records,
            new RecordComparator(
                orderBy,
                isAsc
            )
        );
    }
    
    // java.util.ListJavaDoc
    @Override
    public int size(){
        return records.size();
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean isEmpty(){
        return records.isEmpty();
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean contains(Object o){
        return records.contains(o);
    }
    
    // java.util.ListJavaDoc
    @Override
    public Iterator<Record> iterator(){
        return new RecordIterator();
    }
    
    // java.util.ListJavaDoc
    @Override
    public ListIterator<Record> listIterator(){
        return listIterator(0);
    }
    
    // java.util.ListJavaDoc
    @Override
    public ListIterator<Record> listIterator(int index){
        return new RecordListIterator(index);
    }
    
    // java.util.ListJavaDoc
    @Override
    public List<Record> subList(int fromIndex, int toIndex){
        return records.subList(fromIndex, toIndex);
    }
    
    // java.util.ListJavaDoc
    @Override
    public Object[] toArray(){
        return records.toArray();
    }
    
    // java.util.ListJavaDoc
    @Override
    public <T> T[] toArray(T[] a){
        return records.<T>toArray(a);
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean add(Record rec){
        if(rec == null){
            return false;
        }
        boolean isAdd = records.add(rec);
        addSearchConditionIndex(rec);
        addSearchKeyIndex(rec);
        if(isAdd){
            modCount++;
        }
        return isAdd;
    }
    
    // java.util.ListJavaDoc
    @Override
    public void add(int index, Record rec){
        if(rec == null){
            return;
        }
        records.add(index, rec);
        addSearchConditionIndex(rec);
        addSearchKeyIndex(rec);
        modCount++;
    }
    
    // java.util.ListJavaDoc
    @Override
    public Record set(int index, Record rec){
        Record old = null;
        if(index < size()){
            old = records.get(index);
            if(old != null){
                removeSearchConditionIndex(old);
                removeSearchKeyIndex(old);
            }
        }
        old = records.set(index, rec);
        addSearchConditionIndex(rec);
        addSearchKeyIndex(rec);
        return old;
    }
    
    // java.util.ListJavaDoc
    @Override
    public Record get(int index){
        return records.get(index);
    }
    
    // java.util.ListJavaDoc
    @Override
    public int indexOf(Object o){
        return records.indexOf(o);
    }
    
    // java.util.ListJavaDoc
    @Override
    public int lastIndexOf(Object o){
        return records.lastIndexOf(o);
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean remove(Object o){
        boolean isRemoved = records.remove(o);
        if(isRemoved){
            removeSearchConditionIndex((Record)o);
            removeSearchKeyIndex((Record)o);
            modCount++;
        }
        return isRemoved;
    }
    
    // java.util.ListJavaDoc
    @Override
    public Record remove(int index){
        Record old = records.remove(index);
        if(old != null){
            removeSearchConditionIndex(old);
            removeSearchKeyIndex(old);
            modCount++;
        }
        return old;
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean containsAll(Collection<?> c){
        return records.containsAll(c);
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean addAll(Collection<? extends Record> c){
        if(c == null || c.size() == 0){
            return false;
        }
        Record[] vals = c.toArray(new Record[c.size()]);
        boolean result = false;
        for(Record val : vals){
            result |= add(val);
        }
        if(result){
            modCount++;
        }
        return result;
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean addAll(int index, Collection<? extends Record> c){
        if(c == null || c.size() == 0){
            return false;
        }
        Record[] vals = c.toArray(new Record[c.size()]);
        for(int i = vals.length; --i >= 0;){
            add(index, vals[i]);
        }
        modCount++;
        return true;
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean removeAll(Collection<?> c){
        boolean isRemoved = false;
        for(Object o : c){
            isRemoved |= remove(o);
        }
        if(isRemoved){
            modCount++;
        }
        return isRemoved;
    }
    
    // java.util.ListJavaDoc
    @Override
    public boolean retainAll(Collection<?> c){
        boolean isRemoved = false;
        final Iterator<Record> itr = iterator();
        while(itr.hasNext()){
            Record record = (Record)itr.next();
            if(!c.contains(record)){
                itr.remove();
                isRemoved = true;
            }
        }
        if(isRemoved){
            modCount++;
        }
        return isRemoved;
    }
    
    /**
     * SẴR[hyсASẴCfbNX^ʂ폜B<p>
     */
    @Override
    public void clear(){
        records.clear();
        if(conditionIndexSearchMap != null){
            conditionIndexSearchMap.clear();
        }
        if(keyIndexSearchMap != null){
            keyIndexSearchMap.clear();
        }
        modCount++;
    }
    
    /**
     * TCYXǧ݂̃TCYɏkB<p>
     * AvP[Vł́ÃIy[VŃTCYŏɂ邱ƂłB <br>
     */
    @SuppressWarnings("unchecked")
    public void trimToSize(){
        Class<?> clazz = null;
        try{
            clazz = Class.forName("java.util.Collections$SynchronizedCollection");
        }catch(ClassNotFoundException e){
            return;
        }
        Object mutex = records;
        try{
            Field field = clazz.getDeclaredField("mutex");
            field.setAccessible(true);
            mutex = field.get(records);
        }catch(IllegalAccessException e){
        }catch(NoSuchFieldException e){
        }catch(SecurityException e){
        }
        ArrayList<Record> list = null;
        try{
            Field field = clazz.getDeclaredField("c");
            field.setAccessible(true);
            list = (ArrayList<Record>)field.get(records);
        }catch(IllegalAccessException e){
            return;
        }catch(NoSuchFieldException e){
            return;
        }catch(SecurityException e){
            return;
        }
        synchronized(mutex){
            list.trimToSize();
        }
    }
    
    /**
     * SẴR[h؂B<p>
     *
     * @return ،ʁBtruȅꍇAؐ
     * @exception PropertyGetException vpeB̎擾Ɏsꍇ
     * @exception PropertyValidateException vpeB̌؎ɗOꍇ
     */
    public boolean validate() throws PropertyGetException, PropertyValidateException{
        for(int i = 0, imax = records.size(); i < imax; i++){
            Record record = records.get(i);
            if(!record.validate()){
                return false;
            }
        }
        return true;
    }
    
    /**
     * R[hXg𕡐B<p>
     *
     * @return R[hXg
     */
    public Object clone(){
        return cloneRecordList();
    }
    
    /**
     * XL[}f[^Ȃ̃R[hXg𕡐B<p>
     *
     * @return ̃R[hXg
     */
    public RecordList cloneSchema(){
        RecordList clone = null;
        try{
            clone = (RecordList)super.clone();
            clone.modCount = 0;
            clone.records = Collections.<Record>synchronizedList(new ArrayList<Record>());;
            if(conditionIndexMap != null
                 && conditionIndexMap.size() != 0){
                clone.conditionIndexMap = Collections.synchronizedMap(new HashMap<String, Expression>());
                clone.conditionIndexMap.putAll(conditionIndexMap);
            }
            if(keyIndexMap != null
                 && keyIndexMap.size() != 0){
                clone.keyIndexMap = Collections.synchronizedMap(new HashMap<String, KeyIndex>());
                clone.keyIndexMap.putAll(keyIndexMap);
            }
            if(partUpdateOrderBy != null){
                clone.partUpdateOrderBy = new int[partUpdateOrderBy.length];
                System.arraycopy(
                    partUpdateOrderBy,
                    0,
                    clone.partUpdateOrderBy,
                    0,
                    partUpdateOrderBy.length
                );
            }
            if(partUpdateIsAsc != null){
                clone.partUpdateIsAsc = new boolean[partUpdateIsAsc.length];
                System.arraycopy(
                    partUpdateIsAsc,
                    0,
                    clone.partUpdateIsAsc,
                    0,
                    partUpdateIsAsc.length
                );
            }
        }catch(CloneNotSupportedException e){
            return null;
        }
        return clone;
    }
    
    /**
     * R[hXg𕡐B<p>
     *
     * @return R[hXg
     */
    public RecordList cloneRecordList(){
        final RecordList recList = cloneSchema();
        if(size() == 0){
            return recList;
        }
        for(int i = 0; i < records.size(); i++){
            final Record rec = getRecord(i).cloneRecord();
            recList.addRecord(rec);
        }
        return recList;
    }
    
    /**
     *  XVɁAw肳ꂽvpeB̗\[gL[ɂă\[g悤ɐݒ肷B<p>
     *
     * @param orderBy \[gL[ƂȂvpeBz
     * @param isAsc \[gꍇtrueB~\[gꍇ́Afalse
     */
    public void setPartUpdateSort(String[] orderBy, boolean[] isAsc){
        final int[] propertyIndexes = new int[orderBy.length];
        for(int i = 0; i < orderBy.length; i++){
            propertyIndexes[i] = recordSchema.getPropertyIndex(orderBy[i]);
            if(propertyIndexes[i] == -1){
                throw new DataSetException("No such property " + orderBy[i]);
            }
        }
        setPartUpdateSort(propertyIndexes, isAsc);
    }
    
    /**
     *  XVɁAw肳ꂽCfbNX̃vpeB\[gL[ɂă\[g悤ɐݒ肷B<p>
     *
     * @param orderBy \[gL[ƂȂvpeBCfbNXz
     * @param isAsc \[gꍇtrueB~\[gꍇ́Afalse
     */
    public void setPartUpdateSort(int[] orderBy, boolean[] isAsc){
        if(orderBy == null || orderBy.length == 0){
            throw new DataSetException("Property index array is empty.");
        }
        if(isAsc != null && orderBy.length != isAsc.length){
            throw new DataSetException("Length of property index array and sort flag array is unmatch.");
        }
        
        final int fieldSize = recordSchema.getPropertySize();
        for(int index : orderBy){
            if(index >= fieldSize){
                throw new DataSetException("No such property " + index);
            }
        }
        
        partUpdateOrderBy = orderBy;
        partUpdateIsAsc = isAsc;
    }
    
    /**
     * XVi[R[h}X^XVR[h𐶐B<p>
     *
     * @param updateType XV^Cv
     * @param containsValue XVꂽRecordR[h}X^XVR[hɊ܂߂ꍇ́Atrue
     * @return R[h}X^XVR[h
     * @exception DataSetException R[h}X^XVR[h̐Ɏsꍇ
     */
    public PartUpdateRecords createPartUpdateRecords(
        PartUpdateRecords records,
        CodeMasterUpdateKey.UpdateType updateType,
        boolean containsValue
    ) throws DataSetException{
        if(records == null){
            records = new PartUpdateRecords();
        }
        for(Record record : this.records){
            CodeMasterUpdateKey key = record.createCodeMasterUpdateKey();
            key.setUpdateType(updateType);
            if(containsValue){
                records.addRecord(key, record);
            }else{
                records.addRecord(key);
            }
        }
        return records;
    }
    
    /**
     * w肳ꂽR[h}X^XVL[ɊY郌R[hi[XV쐬B<p>
     *
     * @param key R[h}X^XVL[
     * @return XVR[h܂񂾕XV
     */
    public PartUpdateRecords fillPartUpdateRecords(CodeMasterUpdateKey key){
        PartUpdateRecords records = new PartUpdateRecords();
        records.addRecord(key);
        return fillPartUpdateRecords(records);
    }
    
    /**
     * w肳ꂽXVɊY郌R[hi[XV쐬B<p>
     *
     * @param records XV
     * @return XVR[h܂񂾕XV
     */
    public PartUpdateRecords fillPartUpdateRecords(PartUpdateRecords records){
        if(records == null || records.size() == 0
             || (!records.containsAdd() && !records.containsUpdate())){
            return records;
        }
        records.setFilledRecord(true);
        Record rec = createRecord();
        final CodeMasterUpdateKey[] keys = records.getKeyArray();
        for(int i = 0; i < keys.length; i++){
            CodeMasterUpdateKey key = (CodeMasterUpdateKey)keys[i];
            final CodeMasterUpdateKey.UpdateType updateType = key.getUpdateType();
            
            records.removeRecord(key);
            
            // pRecordɌL[ݒ肷
            rec.setCodeMasterUpdateKey(key);
            
            // RecordSet̎L[݂̂CodeMasterUpdateKeyɕϊ
            key = rec.createCodeMasterUpdateKey(key);
            key.setUpdateType(updateType);
            
            // 폜̏ꍇ́ACodeMasterUpdateKeyo^
            if(key.isRemove()){
                records.addRecord(key);
                continue;
            }
            
            // ǉ܂͍XVꂽRecord
            final Record searchRec = searchByPrimaryKey(rec);
            records.addRecord(key, searchRec);
        }
        return records;
    }
    
    /**
     * XV荞񂾁AfB[vRs[CX^X𐶐B<p>
     *
     * @param records XV
     * @return XV荞񂾁AfB[vRs[CX^X
     */
    public PartUpdate cloneAndUpdate(PartUpdateRecords records){
        final RecordList newRecList = cloneSchema();
        CodeMasterUpdateKey tmpKey = new CodeMasterUpdateKey();
        CodeMasterUpdateKey key = null;
        for(Record oldRecord : this.records){
            tmpKey = oldRecord.createCodeMasterUpdateKey(tmpKey);
            key = records == null ? null : records.getKey(tmpKey);
            Record newRecord = null;
            if(key == null){
                newRecord = oldRecord.cloneRecord();
            }else{
                switch(key.getUpdateType()){
                case ADD:
                case UPDATE:
                    newRecord = (Record)records.removeRecord(key);
                    break;
                case REMOVE:
                default:
                    records.removeRecord(key);
                    continue;
                }
            }
            if(newRecord != null){
                newRecList.addRecord(newRecord);
            }
        }
        if(records != null && records.size() != 0){
            for(Map.Entry<CodeMasterUpdateKey, Object> entry : records.getRecords().entrySet()){
                switch(entry.getKey().getUpdateType()){
                case ADD:
                case UPDATE:
                    final Record record = (Record)entry.getValue();
                    if(record != null){
                        newRecList.addRecord(record);
                    }
                    break;
                default:
                }
            }
        }
        if(partUpdateOrderBy != null && partUpdateOrderBy.length != 0){
            newRecList.sort(partUpdateOrderBy, partUpdateIsAsc);
        }
        
        return newRecList;
    }
    
    public void writeExternal(ObjectOutput out) throws IOException{
        out.writeObject(name);
        out.writeObject(schema);
        if(conditionIndexMap == null || conditionIndexMap.size() == 0){
            out.writeInt(0);
        }else{
            out.writeInt(conditionIndexMap.size());
            final Iterator<Map.Entry<String, Expression>> entries
                 = conditionIndexMap.entrySet().iterator();
            while(entries.hasNext()){
                final Map.Entry<String, Expression> entry = entries.next();
                out.writeObject(entry.getKey());
                out.writeObject(entry.getValue());
            }
        }
        if(keyIndexMap == null || keyIndexMap.size() == 0){
            out.writeInt(0);
        }else{
            out.writeInt(keyIndexMap.size());
            final Iterator<Map.Entry<String, KeyIndex>> entries
                 = keyIndexMap.entrySet().iterator();
            while(entries.hasNext()){
                final Map.Entry<String, KeyIndex> entry = entries.next();
                out.writeObject(entry.getKey());
                out.writeObject(entry.getValue());
            }
        }
        out.writeObject(partUpdateOrderBy);
        out.writeObject(partUpdateIsAsc);
        out.writeInt(records.size());
        for(int i = 0, imax = records.size(); i < imax; i++){
            Record record = records.get(i);
            for(int j = 0, jmax = record.size(); j < jmax; j++){
                out.writeObject(record.getProperty(j));
            }
        }
    }
    
    public void readExternal(ObjectInput in) throws IOException, ClassNotFoundException{
        name = (String)in.readObject();
        schema = (String)in.readObject();
        if(schema != null){
            recordSchema = RecordSchema.getInstance(schema);
        }
        int size = in.readInt();
        if(size > 0){
            conditionIndexMap = Collections.synchronizedMap(new HashMap<String, Expression>());
            for(int i = 0; i < size; i++){
                conditionIndexMap.put((String)in.readObject(), (Expression)in.readObject());
            }
        }
        if(conditionIndexMap != null){
            conditionIndexSearchMap = Collections.synchronizedMap(new HashMap<String, List<Record>>());
        }
        size = in.readInt();
        if(size > 0){
            keyIndexMap = Collections.synchronizedMap(new HashMap<String, KeyIndex>());
            for(int i = 0; i < size; i++){
                keyIndexMap.put((String)in.readObject(), (KeyIndex)in.readObject());
            }
        }
        if(keyIndexMap != null){
            keyIndexSearchMap = Collections.synchronizedMap(new HashMap<String, Map<Object, List<Record>>>());
        }
        partUpdateOrderBy = (int[])in.readObject();
        partUpdateIsAsc = (boolean[])in.readObject();
        final int recSize = in.readInt();
        records = Collections.<Record>synchronizedList(new ArrayList<Record>());
        if(recordSchema != null){
            final int propSize = recordSchema.getPropertySize();
            for(int i = 0; i < recSize; i++){
                Record record = createRecord();
                for(int j = 0; j < propSize; j++){
                    record.setProperty(j, in.readObject());
                }
                addRecord(record);
            }
        }
    }
    
    protected class RecordIterator implements Iterator<Record>, Serializable{
        
        private static final long serialVersionUID = 200743372396432511L;
        
        protected int cursor = 0;
        protected int lastRet = -1;
        protected int expectedModCount = modCount;
        
        @Override
        public boolean hasNext(){
            return cursor != size();
        }
        
        @Override
        public Record next(){
            checkForComodification();
            try{
                Record next = get(cursor);
                lastRet = cursor++;
                return next;
            }catch(IndexOutOfBoundsException e){
                checkForComodification();
                throw new NoSuchElementException();
            }
        }
        
        @Override
        public void remove(){
            if(lastRet == -1){
                throw new IllegalStateException();
            }
            checkForComodification();
            
            try{
                RecordList.this.remove(lastRet);
                if(lastRet < cursor){
                    cursor--;
                }
                lastRet = -1;
                expectedModCount = modCount;
            }catch(IndexOutOfBoundsException e){
                throw new ConcurrentModificationException();
            }
        }
        
        final void checkForComodification(){
            if(modCount != expectedModCount){
                throw new ConcurrentModificationException();
            }
        }
    }
    
    protected class RecordListIterator extends RecordIterator
     implements ListIterator<Record>{
        
        private static final long serialVersionUID = 1979810413080499078L;
        
        public RecordListIterator(int index){
            cursor = index;
        }
        
        @Override
        public boolean hasPrevious(){
            return cursor != 0;
        }
        
        @Override
        public Record previous(){
            checkForComodification();
            try{
                int i = cursor - 1;
                Record previous = get(i);
                lastRet = cursor = i;
                return previous;
            }catch(IndexOutOfBoundsException e){
                checkForComodification();
                throw new NoSuchElementException();
            }
        }
        
        @Override
        public int nextIndex(){
            return cursor;
        }
        
        @Override
        public int previousIndex(){
            return cursor - 1;
        }
        
        @Override
        public void set(Record o){
            if(lastRet == -1){
                throw new IllegalStateException();
            }
            checkForComodification();
            
            try{
                RecordList.this.set(lastRet, o);
                expectedModCount = modCount;
            }catch(IndexOutOfBoundsException e){
                throw new ConcurrentModificationException();
            }
        }
        
        @Override
        public void add(Record o){
            checkForComodification();
            
            try{
                RecordList.this.add(cursor++, o);
                lastRet = -1;
                expectedModCount = modCount;
            }catch(IndexOutOfBoundsException e){
                throw new ConcurrentModificationException();
            }
        }
    }
    
    protected static class RecordComparator implements Comparator<Record>{
        
        private String[] propNames;
        private boolean[] isAsc;
        
        public RecordComparator(String[] propNames){
            this(propNames, null);
        }
        
        public RecordComparator(String[] propNames, boolean[] isAsc){
            if(propNames == null || propNames.length == 0){
                throw new DataSetException("Property name array is empty.");
            }
            if(isAsc != null && propNames.length != isAsc.length){
                throw new DataSetException("Length of property name array and sort flag array is unmatch.");
            }
            this.propNames = propNames;
            this.isAsc = isAsc;
        }
        
        public RecordComparator(RecordSchema recordSchema, int[] propIndexes){
            this(recordSchema, propIndexes, null);
        }
        
        public RecordComparator(RecordSchema recordSchema, int[] propIndexes, boolean[] isAsc){
            if(recordSchema == null){
                throw new DataSetException("Schema not initalize.");
            }
            if(propIndexes == null || propIndexes.length == 0){
                throw new DataSetException("Property name array is empty.");
            }
            if(isAsc != null && propIndexes.length != isAsc.length){
                throw new DataSetException("Length of column name array and sort flag array is unmatch.");
            }
            propNames = new String[propIndexes.length];
            for(int i = 0; i < propIndexes.length; i++){
                propNames[i] = recordSchema.getPropertyName(propIndexes[i]);
                if(propNames == null){
                    throw new DataSetException("Property not found : " + propIndexes[i]);
                }
            }
            if(propNames == null || propNames.length == 0){
                throw new DataSetException("Property name array is empty.");
            }
            this.isAsc = isAsc;
        }
        
        @SuppressWarnings("unchecked")
        @Override
        public int compare(Record rd1, Record rd2){
            if(rd1 == null && rd2 == null){
                return 0;
            }
            if(rd1 != null && rd2 == null){
                return 1;
            }
            if(rd1 == null && rd2 != null){
                return -1;
            }
            for(int i = 0; i < propNames.length; i++){
                Object val1 = rd1.getProperty(propNames[i]);
                Object val2 = rd2.getProperty(propNames[i]);
                if(val1 != null && val2 == null){
                    return (isAsc == null || isAsc[i]) ? 1 : -1;
                }
                if(val1 == null && val2 != null){
                    return (isAsc == null || isAsc[i]) ? -1 : 1;
                }
                if(val1 != null && val2 != null){
                    int comp = 0;
                    if(val1 instanceof Comparable){
                        comp = ((Comparable<Object>)val1).compareTo(val2);
                    }else{
                        comp = val1.hashCode() - val2.hashCode();
                    }
                    if(comp != 0){
                        return (isAsc == null || isAsc[i]) ? comp : -1 * comp;
                    }
                }
            }
            return 0;
        }
    }
    
    protected static class KeyIndex implements Serializable{
        
        private static final long serialVersionUID = 2501333319162543622L;
        
        protected int[] propertyIndexes;
        
        public KeyIndex(int[] indexes){
            propertyIndexes = indexes;
        }
        
        public Object createKey(Record record){
            ArrayList<Object> key = null;
            for(int propertyIndex : propertyIndexes){
                Object val = null;
                try{
                    val = record.getProperty(propertyIndex);
                }catch(PropertyGetException e){
                    key = null;
                    break;
                }
                if(key == null){
                    key = new ArrayList<Object>();
                }
                key.add(val);
            }
            if(key != null){
                key.trimToSize();
            }
            return key;
        }
    }
}
