////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
// 
// 
// This file is part of the Microsoft .NET Micro Framework Porting Kit Code Samples and is unsupported. 
// Copyright (C) Microsoft Corporation. All rights reserved. Use of this sample source code is subject to 
// the terms of the Microsoft license agreement under which you licensed this sample source code. 
// 
// THIS SAMPLE CODE AND INFORMATION ARE PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER EXPRESSED OR IMPLIED, 
// INCLUDING BUT NOT LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND/OR FITNESS FOR A PARTICULAR PURPOSE.
// 
// 
////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

/*
 *	  Copyright (C) Renesas Technology America,Ltd. 2009  All rights reserved.
 */

#include <tinyhal.h>

#include "scif.h"
#include "../iodefine.h"

#ifdef	PLATFORM_DEPENDENT_TX_USART_BUFFER_SIZE
#define TX_USART_BUFFER_SIZE	PLATFORM_DEPENDENT_TX_USART_BUFFER_SIZE
#else
#define TX_USART_BUFFER_SIZE	512
#endif

#ifdef	PLATFORM_DEPENDENT_RX_USART_BUFFER_SIZE
#define RX_USART_BUFFER_SIZE	PLATFORM_DEPENDENT_RX_USART_BUFFER_SIZE
#else
#define RX_USART_BUFFER_SIZE	512
#endif

extern UINT8 TxBuffer_Com[TX_USART_BUFFER_SIZE * TOTAL_USART_PORT];
extern UINT8 RxBuffer_Com[RX_USART_BUFFER_SIZE * TOTAL_USART_PORT];

#ifdef RX62N_ENABLE_USB
#include "tkdn_usb.h"
#include "usb_hal.h"

/*PID Values*/
#define PID_NAK 	0
#define PID_BUF 	1
#define PID_STALL_1 2
#define PID_STALL_2 3

bool UsbTxBufferEmptyInterruptEnable;
UINT8 UsbRxBuffer[USB_BUFFER_SIZE];
int UsbRxBufferStart;
int UsbRxBufferEnd;
bool UsbRxBufferEmptyInterruptEnable;
#endif

//--//

BOOL CPU_USART_Initialize( int ComPortNum, int BaudRate, int Parity, int DataBits, int StopBits, int FlowValue )
{	
	unsigned int i = 0;
	unsigned int scsmr = 0;

	if(Parity == USART_PARITY_NONE) scsmr = 0;		// 8N1, divide by 1 
	else if (Parity == USART_PARITY_ODD) scsmr = SCIF_SMR_PE | SCIF_SMR_PM;
	else if (Parity == USART_PARITY_EVEN) scsmr = SCIF_SMR_PE;
	else return FALSE;

	if(DataBits == 7) scsmr|= SCIF_SMR_CHR;
	if(StopBits == USART_STOP_BITS_TWO) scsmr|= SCIF_SMR_STOP;

	if( ComPortNum == 0){
		SYSTEM.MSTPCRB.BIT.MSTPB31 = 0;
		PORT2.DDR.BIT.B1 = 0;
		PORT2.ICR.BIT.B1 = 1;

		SCI0.SCR.BYTE = 0;		 // SCR Clear
		SCI0.SMR.BYTE = scsmr;
		SCI0.SEMR.BYTE = SEMR_CALC(BaudRate);
		SCI0.BRR = BRR_CALC(BaudRate);
		// Delay 1 ms
		i = 31250;		// 8 ns/instruction * 4 instructions * loop_count = 1ms = 1000000 ns
		while (i--);	// loop count = 31250

		SCI0.SCR.BYTE = 0x0071;	 // Enable Transmit & Receive
		//SCI0 Transmit interrput and Receive interrupt : level 5
		ICU.IER[0x1A].BIT.IEN7 = 1;
		ICU.IER[0x1B].BIT.IEN0 = 1;
		ICU.IPR[0x81].BIT.IPR = 5;
	}
	else if( ComPortNum == 1 ){
		SYSTEM.MSTPCRB.BIT.MSTPB30 = 0;
		IOPORT.PFFSCI.BIT.SCI1S = 1;
		PORTF.DDR.BIT.B3 = 0;
		PORTF.ICR.BIT.B3 = 1;

		SCI1.SCR.BYTE = 0;		 // SCR Clear
		SCI1.SMR.BYTE = scsmr;
		SCI1.SEMR.BYTE = SEMR_CALC(BaudRate);
		SCI1.BRR = BRR_CALC(BaudRate);
		// Delay 1 ms
		i = 31250;		// 8 ns/instruction * 4 instructions * loop_count = 1ms = 1000000 ns
		while (i--);	// loop count = 31250

		SCI1.SCR.BYTE = 0x0071;	 // Enable Transmit & Receive
		//SCI1 Transmit interrput and Receive interrupt : level 5
		ICU.IER[0x1B].BIT.IEN3 = 1;
		ICU.IER[0x1B].BIT.IEN4 = 1;
		ICU.IPR[0x81].BIT.IPR = 5;
	}
	else if( ComPortNum == 2 ){
#ifndef RX62N_ENABLE_USB
		SCI2.SCR.BYTE = 0;		 // SCR Clear
		SCI2.SMR.BYTE = scsmr;
		SCI2.SEMR.BYTE = SEMR_CALC(BaudRate);
		SCI2.BRR = BRR_CALC(BaudRate);
		// Delay 1 ms
		i = 31250;		// 8 ns/instruction * 4 instructions * loop_count = 1ms = 1000000 ns
		while (i--);	// loop count = 31250

		SCI2.SCR.BYTE = 0x0071;	 // Enable Transmit & Receive

		//SCI2 Transmit interrput and Receive interrupt : level 13 for debugger port
		ICU.IER[0x1B].BIT.IEN7 = 1;
		ICU.IER[0x1C].BIT.IEN0 = 1;
		ICU.IPR[0x82].BIT.IPR = 13;
#else
		TKUSB_Init();
		UsbTxBufferEmptyInterruptEnable = false;
		UsbRxBufferStart = 0;
		UsbRxBufferEnd = 0;
		UsbRxBufferEmptyInterruptEnable = true;
#endif
	}
	else
		return FALSE;

			
	return TRUE;
}

BOOL CPU_USART_Uninitialize( int ComPortNum )
{
	switch(ComPortNum)
	{
	case 0:
		SCI0.SCR.BYTE = 0;		 // SCR Clear
		break;
	case 1:
		SCI1.SCR.BYTE = 0;		 // SCR Clear
		break;
	case 2:
#ifndef RX62N_ENABLE_USB
		SCI2.SCR.BYTE = 0;		 // SCR Clear
#else
		USBHAL_Disable();
#endif
		break;
	}

	return TRUE;
}

BOOL CPU_USART_TxBufferEmpty( int ComPortNum )
{
	switch(ComPortNum)
	{
	case 0:
		return (SCI0.SSR.BIT.TDRE) != 0;
	case 1:
		return (SCI1.SSR.BIT.TDRE) != 0;
	case 2:
#ifndef RX62N_ENABLE_USB
		return (SCI2.SSR.BIT.TDRE) != 0;
#else
		return TKUSB_SendDataCount() < (USB_BUFFER_SIZE - 1);
#endif
	}

	return TRUE;
}

BOOL CPU_USART_TxShiftRegisterEmpty( int ComPortNum )
{
	switch(ComPortNum)
	{
	case 0:
		return (SCI0.SSR.BIT.TEND) != 0;
	case 1:
		return (SCI1.SSR.BIT.TEND) != 0;
	case 2:
#ifndef RX62N_ENABLE_USB
		return (SCI2.SSR.BIT.TEND) != 0;
#else
		return TKUSB_SendDataCount() == 0;
#endif
	}

	return TRUE;
}

void CPU_USART_WriteCharToTxBuffer( int ComPortNum, UINT8 c )
{
	switch( ComPortNum )
	{
	case 0:
		while((SCI0.SSR.BIT.TDRE) == 0x0);
		SCI0.TDR = c;
		break;
		
	case 1:
		while((SCI1.SSR.BIT.TDRE) == 0x0);
		SCI1.TDR = c;
		break;

	case 2:
#ifndef RX62N_ENABLE_USB
		while((SCI2.SSR.BIT.TDRE) == 0x0);
		SCI2.TDR = c;
#else
		{
			GLOBAL_LOCK(irq);

			TKUSB_SendByte(c);
		}
#endif
		break;
	}
}

void CPU_USART_TxBufferEmptyInterruptEnable( int ComPortNum, BOOL Enable )
{
	switch( ComPortNum )
	{
	case 0:
		if(Enable)
			SCI0.SCR.BIT.TIE = 1;
		else
			SCI0.SCR.BIT.TIE = 0;
		break;
	case 1:
		if(Enable)
			SCI1.SCR.BIT.TIE = 1;
		else
			SCI1.SCR.BIT.TIE = 0;
		break;
	case 2:
#ifndef RX62N_ENABLE_USB
		if(Enable)
			SCI2.SCR.BIT.TIE = 1;
		else
			SCI2.SCR.BIT.TIE = 0;
#else
		UsbTxBufferEmptyInterruptEnable = Enable;
		if(Enable){
			while(TKUSB_SendDataCount() < (USB_BUFFER_SIZE - 1))
			{
				char c;

				if(USART_RemoveCharFromTxBuffer( 2, c ))
				{
					CPU_USART_WriteCharToTxBuffer( 2, c );
				}
				else
					break;

				if(!UsbTxBufferEmptyInterruptEnable)
					break;
			}
		}
#endif
		break;
	}
}

BOOL CPU_USART_TxBufferEmptyInterruptState( int ComPortNum )
{
	switch(ComPortNum)
	{
	case 0:
		return 0 != SCI0.SCR.BIT.TIE;
	case 1:
		return 0 != SCI1.SCR.BIT.TIE;
	case 2:
#ifndef RX62N_ENABLE_USB
		return 0 != SCI2.SCR.BIT.TIE;
#else
		return UsbTxBufferEmptyInterruptEnable;
#endif
	}

	return TRUE;
}

void CPU_USART_RxBufferFullInterruptEnable( int ComPortNum, BOOL Enable )
{
	switch(ComPortNum)
	{
	case 0:
		if(Enable)
			SCI0.SCR.BIT.RIE = 1;
		else
			SCI0.SCR.BIT.RIE = 0;
		break;
	case 1:
		if(Enable)
			SCI1.SCR.BIT.RIE = 1;
		else
			SCI1.SCR.BIT.RIE = 0;
		break;
	case 2:
#ifndef RX62N_ENABLE_USB
		if(Enable)
			SCI2.SCR.BIT.RIE = 1;
		else
			SCI2.SCR.BIT.RIE = 0;
#else
		UsbRxBufferEmptyInterruptEnable = Enable;
		if(Enable){
			GLOBAL_LOCK(irq);

			while(UsbRxBufferStart != UsbRxBufferEnd)
			{
				if(!USART_AddCharToRxBuffer(2, UsbRxBuffer[UsbRxBufferStart]))
					break;
				UsbRxBufferStart++;
				UsbRxBufferStart &= (USB_BUFFER_SIZE - 1);

				if(!UsbRxBufferEmptyInterruptEnable)
					break;
			}

			if(UsbRxBufferEmptyInterruptEnable && (UsbRxBufferStart == UsbRxBufferEnd))
			{
				USB0.PIPE1CTR.BIT.PID = PID_BUF;
			}
		}
		else{
			USB0.PIPE1CTR.BIT.PID = PID_NAK;
		}
#endif
		break;
	}
}

BOOL CPU_USART_RxBufferFullInterruptState( int ComPortNum )
{
	switch(ComPortNum)
	{
	case 0 :
		return 0 != SCI0.SCR.BIT.RIE;
	case 1:
		return 0 != SCI1.SCR.BIT.RIE;
	case 2:
#ifndef RX62N_ENABLE_USB
		return 0 != SCI2.SCR.BIT.RIE;
#else
		return UsbRxBufferEmptyInterruptEnable;
#endif
	default:
		return FALSE;
	}
}

void CPU_USART_ProtectPins( int ComPortNum, BOOL On )
{
}

UINT32 CPU_USART_PortsCount()
{
	return TOTAL_USART_PORT;
}

void CPU_USART_GetPins( int ComPortNum, GPIO_PIN& rxPin, GPIO_PIN& txPin,GPIO_PIN& ctsPin, GPIO_PIN& rtsPin )
{	
	rxPin = GPIO_PIN_NONE; 
	txPin = GPIO_PIN_NONE; 
	ctsPin= GPIO_PIN_NONE; 
	rtsPin= GPIO_PIN_NONE; 

	return;
}

BOOL CPU_USART_SupportNonStandardBaudRate ( int ComPortNum )
{
	return FALSE;
}

void CPU_USART_GetBaudrateBoundary( int ComPortNum, UINT32& maxBaudrateHz, UINT32& minBaudrateHz )
{
	maxBaudrateHz = 0;
	minBaudrateHz = 0;
}

BOOL CPU_USART_IsBaudrateSupported( int ComPortNum, UINT32 & BaudrateHz )
{	
	return FALSE;
}

BOOL CPU_USART_TxHandshakeEnabledState( int comPort )
{
	return TRUE;
}

void USART_RxISR( UINT32 port )
{
	char c;

	GLOBAL_LOCK(irq);

	volatile __evenaccess unsigned char* rxReg;
	volatile __evenaccess unsigned char* rxData;

	switch(port)
	{
		case 0:
			rxReg  = &SCI0.SSR.BYTE;
			rxData = &SCI0.RDR;
			break;
			
		case 1: 
			rxReg  = &SCI1.SSR.BYTE;
			rxData = &SCI1.RDR;
			break;

		case 2:
			rxReg  = &SCI2.SSR.BYTE;
			rxData = &SCI2.RDR;
			break;

		default:
			ASSERT(FALSE);
			return;
	}
	
	while((*rxReg) & 0x0002)
	{
		c = *rxData;

		*rxReg &= 0xFFFD; 

		if(!USART_AddCharToRxBuffer(port, c))
		{
			break;
		}
		Events_Set( SYSTEM_EVENT_FLAG_COM_IN );

	}
}

void USART_TxISR( UINT32 port )
{
	char c;
	GLOBAL_LOCK(irq);

	if(USART_RemoveCharFromTxBuffer( port, c ))
	{
		CPU_USART_WriteCharToTxBuffer( port, c );
		Events_Set( SYSTEM_EVENT_FLAG_COM_OUT );
	}
	else
	{
		CPU_USART_TxBufferEmptyInterruptEnable( port, FALSE );
	}
}
#ifdef RX62N_ENABLE_USB
extern "C" void UBSCDC_RxISR( char c )
{
	GLOBAL_LOCK(irq);

	if(!UsbRxBufferEmptyInterruptEnable || (UsbRxBufferStart != UsbRxBufferEnd)
		|| !USART_AddCharToRxBuffer(2, c))
	{
		if(((UsbRxBufferEnd + 1 - UsbRxBufferStart + USB_BUFFER_SIZE) & (USB_BUFFER_SIZE - 1)) == 0)
		{
			UsbRxBufferStart++;
			UsbRxBufferStart &= (USB_BUFFER_SIZE - 1);
		}
		UsbRxBuffer[UsbRxBufferEnd++] = c;
		UsbRxBufferEnd &= (USB_BUFFER_SIZE - 1);
	}
	else{
		Events_Set( SYSTEM_EVENT_FLAG_COM_IN );
	}
}

extern "C" void UBSCDC_TxISR()
{
	Events_Set( SYSTEM_EVENT_FLAG_COM_OUT );

	while(TKUSB_SendDataCount() < (USB_BUFFER_SIZE - 1))
	{
		char c;

		if(USART_RemoveCharFromTxBuffer( 2, c ))
		{
			CPU_USART_WriteCharToTxBuffer( 2, c );
		}
		else
			break;
	}
}
#endif
