/*
 *  @File syscall.c
 *  @Brief system call functions
 *  
 *  @Author      Sodex
 *  @Revision    0.1
 *  @License     suspension
 *  @Date        creae: 2007/07/09  update: 2007/07/09
 *      
 *  Copyright (C) 2007 Sodex
 */

#include <syscall.h>
#include <ihandlers.h>
#include <descriptor.h>
#include <vga.h>
#include <ext3fs.h>
#include <fs.h>
#include <process.h>
#include <execve.h>
#include <io.h>
#include <key.h>
#include <chdir.h>

PRIVATE void sys_exit(int exit);
PRIVATE int sys_open(const char* pathname, int flags, mode_t mode);
PRIVATE int sys_read(int fd, const void* buf, size_t count);
PRIVATE int __stdin_read(int fd, const void* buf, size_t count);
PRIVATE void sys_write(int fd, const void* buf, size_t count);
PRIVATE void sys_close(int fd);
PRIVATE int sys_getdentry();
PRIVATE int sys_getpstat();

PUBLIC void init_syscall()
{
  set_trap_gate(0x80, &asm_syscall);
}

/* Firstly, we implements systemcall as "C" code, but afterwards we'll
 * modify the code which is written by assembler for efficient kernel.
 */
PUBLIC int i80h_syscall(u_int32_t p1, u_int32_t p2, u_int32_t p3,
                        u_int32_t p4, u_int32_t p5, u_int32_t p6,
                        u_int32_t nr_syscall)
{
  int ret = 0;
  switch (nr_syscall) {
  case SYS_CALL_EXIT:
    sys_exit(p1);
    break;

  case SYS_CALL_FORK:
    break;

  case SYS_CALL_READ:
    ret = sys_read(p1, p2, p3);
    break;

  case SYS_CALL_WRITE:
    sys_write(p1, p2, p3);
    break;

  case SYS_CALL_OPEN:
    ret = sys_open(p1, p2, p3);
    break;

  case SYS_CALL_CLOSE:
    sys_close(p1);
    break;

  case SYS_CALL_EXECVE:
    sys_execve(p1, p2, NULL);
    break;

  case SYS_CALL_CHDIR:
    ret = sys_chdir(p1);
    break;

  case SYS_CALL_GETDENTRY:
    ret = sys_getdentry();
    break;

  case SYS_CALL_GETPSTAT:
    ret = sys_getpstat();
    break;
  }

  return ret;
}

PRIVATE void sys_exit(int exit)
{
  struct task_struct* next = dlist_entry(current->run_list.next,
                                         struct task_struct, run_list);
  dlist_remove(&(current->run_list));
  current = next;
  asm __volatile__ (" int $0x20 ");
}

PRIVATE int sys_read(int fd, const void* buf, size_t count)
{
  int ret;
  struct file* fs = current->files->fs_fd[fd];
  if (fs->f_stdioflag == FLAG_STDIN) {
    ret = __stdin_read(fd, buf, count);
  } else {
    ret = ext3_read(fd, buf, count);
  }
  return ret;
}

PRIVATE int __stdin_read(int fd, const void* buf, size_t count)
{
  char *p;
  int total = 0;
  char stdin[KEY_BUF+1];
  memset(stdin, 0, KEY_BUF+1);

  asm("sti");
  while (TRUE) {
    //_kprintf("stdin_read\n");
    if (get_stdin(stdin) != NULL) {
      int getlen = strlen(stdin);
      //_kprintf("not null:%c %s getlen:%x\n", stdin[getlen-1], buf, getlen);
      if (stdin[getlen-1] == KEY_ENTER) {
        //_kputc('\n');
        memcpy(buf + total, stdin, getlen-1);
        total += getlen;
        return total;
      } else if (stdin[getlen-1] == KEY_BACK) {
        if (getlen > 0 && total > 0) {
          //_kputc(KEY_BACK);
          memset(buf + total - 1, 0, 1);
          total--;
        }
      } else {
        //_kputc(stdin[getlen-1]);
        memcpy(buf + total, stdin, getlen);
        total += getlen;
        memset(stdin, 0, KEY_BUF+1);
      }
    }
  }
}

PRIVATE void sys_write(int fd, const void* buf, size_t count)
{
  struct file* fs = current->files->fs_fd[fd];
  if (fs->f_stdioflag == FLAG_STDOUT) {
    int i;
    for (i=0; i<count; i++)
      _kputc(((char*)buf)[i]);
  } else if (fs->f_stdioflag == FLAG_STDERR) {
    int i;
    for (i=0; i<count; i++)
      _kputc(((char*)buf)[i]);
  } else if (fs->f_stdioflag == FLAG_FILE) {
    ext3_write(fd, buf, count);
    _kprintf("FLAG_FILE\n");
  } else {
    _kprintf("sys_write else\n");
    // error
  }
}

PRIVATE int sys_open(const char* pathname, int flags, mode_t mode)
{
  disable_scheduling();

  int fd = ext3_open(pathname, flags, mode);

  enable_scheduling();

  return fd;
}

PRIVATE void sys_close(int fd)
{
  close(fd);
}

PRIVATE int sys_getdentry()
{
  return (int)current->dentry;
}

PRIVATE int sys_getpstat()
{
  return (int)current;
}
