/*
  =============================================================================
  
   Class    : nos.c
  
  =============================================================================
*/
#include "nos.h"

/*
  -----------------------------------------------------------------------------
  
   public : work ram

  -----------------------------------------------------------------------------
*/
#if USE_NOS_DEBUG
#define NOS_DEBUG_SYMBOL
#else
#define NOS_DEBUG_SYMBOL static
#endif

int nos_tick;
NOS_DEBUG_SYMBOL struct Nos nos_sys;

#if USE_NOS_MSG
NOS_DEBUG_SYMBOL struct Nos_Buf nos_mes;
                                /* メッセージ本体 */

NOS_DEBUG_SYMBOL struct Nos_Box nos_box[NUM_MSG_BOX];
                                /* メッセージBOX */
#endif
#if USE_NOS_MEM
NOS_DEBUG_SYMBOL struct Nos_Mem nos_mem;
                                /* メモリ管理 */
#endif
#if USE_NOS_RESRC
NOS_DEBUG_SYMBOL struct Nos_Resrc nos_res[NUM_NOS_RESRC];
                                /* リソース共有 */
#endif
#if USE_NOS_EVENT
NOS_DEBUG_SYMBOL struct Nos_Link nos_event[NUM_NOS_EVENT];
                                /* イベント同期 */
#endif
#if USE_NOS_TIMER
int nos_timer;
NOS_DEBUG_SYMBOL struct Nos_Link nos_time;
                                /* タイマー */
#endif

/*
  -----------------------------------------------------------------------------
  
  Define : 諸々

  -----------------------------------------------------------------------------
*/
#define NOS_PAGE sizeof(struct Nos_Page)
#define NOS_SYS_ID NUM_NOS_TASK

/*
  -----------------------------------------------------------------------------
  
  Macro  : ポインタキャスト

  -----------------------------------------------------------------------------
*/
#define NosLink(l) ((struct Nos_Link *)(l))
#define NosMsg(m) ((struct Nos_Msg *)(m))
#define NosSlot(s) ((struct Nos_Slot *)(s))
#define NosMem(m) ((struct Nos_Page *)(m))

/*
  -----------------------------------------------------------------------------
  
  Macro  : 条件判定Macro
  
  Param  : ビットパターン
         : 条件式

  Return : 真   条件成立  

  -----------------------------------------------------------------------------
*/
#define UseNosDebug(f, c) (USE_NOS_DEBUG & (f) && (c))
#define UseNosMsg(f, c) (USE_NOS_MSG & (f) && (c))

/*
  -----------------------------------------------------------------------------
  
  Macro  : 次に配置するQueueをセット

  Param  : Queue
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
#define NextQueue(q)                                                    \
    nos_sys.queue = (void *)(q);                                        \
    nos_tick = 0

/*
  -----------------------------------------------------------------------------
  
  Macro  : 範囲チェック

  Param  : チェック値
         : 最大値
  
  Return : 真   範囲エラー
 
  -----------------------------------------------------------------------------
*/
#define IsOver(v, l) (USE_NOS_DEBUG & USE_NOS_B0 && (0 > (v) || (v) >= (l)))
                                /* 0 <-> 最大値 */

/*
  -----------------------------------------------------------------------------
  
  Macro  : 指定したスロットの優先順位を実行中スロットの優先順位と比較

  Param  : 比較するスロット
  
  Return : 真   次のタスクをスケジュール
 
  -----------------------------------------------------------------------------
*/
#define NextSched(s) (nos_sys.now->info.pri > ((struct Nos_Link *)s)->info.pri)
                                /* 高優先順位 ? */

/*
  -----------------------------------------------------------------------------
  
  Macro  : Tick値の判定

  Param  : void
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
#define SensTick() ((nos_tick > 0) ? NOS_NONE : NOS_WAIT)
                                /* Tick有効 ? NOS_NOE : NOS_WAIT */

#define SensTick2() ((nos_tick > 0) ? (void *)NOS_NONE : (void *)NOS_WAIT)
                                /* Tick有効 ? NOS_NOE : NOS_WAIT */

/*
  -----------------------------------------------------------------------------
  
  Macro  : リンクアドレスを返す

  Param  : リンク
  
  Return : リンク
 
  -----------------------------------------------------------------------------
*/
#if USE_NOS_DEBUG & USE_NOS_B1
#define GetLink(l) ((l)->next = (l))
                                /* カナリア値もセット */
#else
#define GetLink(l) (l)
#endif

/*
  -----------------------------------------------------------------------------
  
  Macro  : タスクがWait状態か判定

  Param  : void
  
  Return : 真   Queue使用中 (WAIT発生中)
 
  -----------------------------------------------------------------------------
*/
#define IsTaskWait()                                                \
    (USE_NOS_DEBUG & USE_NOS_B2 &&                                  \
     nos_sys.run.info.id != NOS_SYS_ID && nos_sys.queue != &nos_sys.run)
                        /* SUPERタスク実行中ではない && QueueがWait状態 ? */

/*
  -----------------------------------------------------------------------------
  
  Method : ページ相当のバイト数に変換 (端数切り捨て)

  Param  : メモリサイズ
  
  Return : ページサイズ (bytes)
 
  -----------------------------------------------------------------------------
*/
#define GetPage(m) (((m) >> NOS_MEM_ALLOC) << NOS_MEM_ALLOC)
                                /* ブロックサイズに整える */

/*
  -----------------------------------------------------------------------------
  
  Method : LinkデータをQueueに追加

  Param  : 追加先のQueue
         : 追加するLink
  
  Return : 追加したLink
 
  -----------------------------------------------------------------------------
*/
static struct Nos_Link *addLink(struct Nos_Link *aqueue, struct Nos_Link *alink)
{
#if 1
    while (null != (alink->next = aqueue->next) &&
           alink->info.pri >= (alink->next)->info.pri) aqueue = alink->next;
                                /* 終端ではない && */
                                /* 次の優先順位が高いか同じ ? */
#else
    struct Nos_Link *n;         /* Next */

    for (n = aqueue->next;
         null != n && alink->info.pri >= n->info.pri; n = n->next) aqueue = n;
    /* 終端ではない && 次の優先順位が高いか同じ ? */

    alink->next = n;
#endif
    return (aqueue->next = alink);
}
 
#if USE_NOS_MSG & USE_NOS_B0

/*
  -----------------------------------------------------------------------------
  
  Method : メッセージボックスのチェック
          
  Param  : メッセージBox番号
  
  Return : 受信メッセージ数
 
  -----------------------------------------------------------------------------
*/
int sensMsg(int box)
{
    return
        IsOver(box, NUM_MSG_BOX) ? NOS_ERROR : (nos_box + box)->msg.info.len;
}

/*
  -----------------------------------------------------------------------------
  
  Method : メッセージを送信する
          
  Param  : 送信先メッセージBox番号
         : 送信メッセージ
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
void *sendMsg(int box, void *msg)
{
    struct Nos_Box *rb;         /* Recv Box */

    if (IsOver(box, NUM_MSG_BOX) ||
        UseNosDebug(USE_NOS_B1,
                    NosLink(msg) < NosLink(&nos_mes) ||
                    NosLink(msg) >=
                    NosLink(((char *)&nos_mes + sizeof(struct Nos_Buf))) ||
                    NosLink(msg)->next != msg))
        return (void *)NOS_ERROR;
                                /* パラメータチェック || */
                                /* 無効メッセージチェック */
    
    if (USE_NOS_MSG & USE_NOS_B1)
        NosMsg(msg)->send = nos_sys.run.info.id;
                                /* 送信元タスクID */

    rb = nos_box + box;

    rb->msg.info.len++;         /* 送信先メッセージ++ */
    if (UseNosMsg(USE_NOS_B2, NosMsg(msg)->own == box)) {
        /* オーナーに返却 ?  */

        NosMsg(msg)->next = NosLink(rb->msg.next);
        rb->msg.next = msg;
                                /* Queueの先頭に配置 (FILO) */
    } else
        addLink(&rb->msg, msg); /* 優先順位 + FIFO */

    if (rb->task.info.len) {
        /* メッセージ待ちのTaskがある ? */

        struct Nos_Link *wt;    /* Wait Task */

        rb->task.info.len--;
        rb->task.next = (wt = rb->task.next)->next;

        if (NextSched(addLink(&nos_sys.run, wt))) nos_tick = 0;
                                /* メッセージ待ちのタスクを実効Queueへ */
                                /* 次のスケジューリング ? */
    }
    return (void *)SensTick2();
}

/*
  -----------------------------------------------------------------------------
  
  Method : 受信したメッセージを得る
          
  Param  : 受信メッセージBox番号
  
  Return : 受信メッセージ
 
  -----------------------------------------------------------------------------
*/
void *recvMsg(int box)
{
    if (IsOver(box, NUM_MSG_BOX) || IsTaskWait())
        return (void *)NOS_ERROR;
                                /* パラメータチェック || 既に待機状態 ? */

    if (0 < nos_tick) {
        /* 実行可能 ? */

        struct Nos_Box *rb;     /* Recv Box */

        rb = nos_box + box;
        if (rb->msg.info.len) {
            /* メッセージBoxが空ではない ? メッセージを返す */

            struct Nos_Link *rm; /* Recv Message */

            rb->msg.info.len--;
            rb->msg.next = (rm = rb->msg.next)->next;
                                /* 受信Queueから切り離す */
            return GetLink(rm);
        }
        /* メッセージが空 ? */

        if (nos_sys.run.info.id != NOS_SYS_ID) {
            /* SUPERタスク以外からの呼び出し ? */

            rb->task.info.len++;
                                /* メッセージ待ちのタスク++ */
            NextQueue(&rb->task);
        }
    }
    return (void *)NOS_WAIT;
}

/*
  -----------------------------------------------------------------------------
  
  Method : メッセージの初期化
          
  Param  : 初期化メッセージアドレス
         : メッセージボックス番号
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
static void initMsg(void *msg, int box)
{
    struct Nos_Link *ib;        /* Init Bos */

    ib = &(nos_box + box)->msg;
    ib->info.len++;
    
    NosMsg(msg)->pri = NOS_MSG_PRI;
    NosMsg(msg)->own = box;
#if USE_NOS_MSG & USE_NOS_B1
    NosMsg(msg)->send = NOS_SYS_ID;
#endif
    addLink(ib, msg);
}

#include "nos_conf.c"

/*
  -----------------------------------------------------------------------------
  
  Method : Constracture /メッセージボックスの初期化
          
  Param  : void
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
static void initNosBox(void)
{
    struct Nos_Box *ib;         /* Init message Box */
    int i;                      /* Index */

    ib = nos_box;
    for (i = 0; i < NUM_MSG_BOX; i++) {
        /* メッセージBOXの初期化 */

        ib->task.next = null;
        ib->task.info.len = 0;
                                /* メッセージ待ちタスクQueue */

        ib->msg.next = null;
        ib->msg.info.len = 0;
                                /* メッセージQueue */
        ib++;
    }
}
#endif

#if USE_NOS_EVENT & USE_NOS_B0

/*
  -----------------------------------------------------------------------------
  
  Method : イベント待ちの状態を調べる
          
  Param  : イベント番号
  
  Return : イベントステータス
 
  -----------------------------------------------------------------------------
*/
int sensEvent(int num)
{
    return
        IsOver(num, NUM_NOS_EVENT) ? NOS_ERROR : (nos_event + num)->info.stat;
}

/*
  -----------------------------------------------------------------------------
  
  Method : イベントを待つ
          
  Param  : イベント番号
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
int waitEvent(int num)
{
    struct Nos_Link *eb;        /* Event Box */

    if (IsOver(num, NUM_NOS_EVENT) || IsTaskWait())
        return NOS_ERROR;   /* パラメータチェック || 既に待機状態 ? */

    eb = nos_event + num;
    if (NOS_WAIT != eb->info.stat) return NOS_NONE;
                                /* Event発生済み ? 戻る */

    NextQueue(eb);
    return NOS_WAIT;
}

/*
  -----------------------------------------------------------------------------
  
  Method : イベントのセット
          
  Param  : イベント番号
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
int setEvent(int num)
{
    struct Nos_Link *ew, *wt;   /* Event Wait, Wait Task */

    if (IsOver(num, NUM_NOS_EVENT)) return NOS_ERROR;
                                /* パラメータチェック */

    ew = nos_event + num;
    ew->info.stat = nos_sys.run.info.id;
                                /* イベントON = タスクID */

    while (null != (wt = ew->next)) {
        /* Set待ちのTaskがある ? 実行Queueへ */

        ew->next = wt->next;
        if (NextSched(addLink(&nos_sys.run, wt))) nos_tick = 0;
                                /* 高優先順位発生 ? ステータス変更 */
    }
    return SensTick();
}

/*
  -----------------------------------------------------------------------------
  
  Method : イベントのクリア
          
  Param  : イベント番号
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
int clearEvent(int num)
{
    if (IsOver(num, NUM_NOS_EVENT)) return NOS_ERROR;

    (nos_event + num)->info.stat = NOS_WAIT;
    return SensTick();
}

/*
  -----------------------------------------------------------------------------
  
  Method : Constracture / Eventの初期化
          
  Param  : void
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
static void initNosEvent(void)
{
    int i;
    struct Nos_Link *ie;        /* Init Event */

    ie = nos_event;
    for (i = 0; i < USE_NOS_EVENT; i++) {
        /* Eventの初期化 */

        ie->next = null;
        ie->info.stat = NOS_WAIT;
        ie++;
    }
}
#endif

#if USE_NOS_RESRC & USE_NOS_B0
/*
  -----------------------------------------------------------------------------
  
  Method : リソースの状態を調べる
          
  Param  : リソース番号
  
  Return : ロックタスク番号
 
  -----------------------------------------------------------------------------
*/
int sensResrc(int num)
{
    return IsOver(num, NUM_NOS_RESRC) ? NOS_ERROR : (nos_res + num)->lock;
                                /* 未使用ならNOS_WAIT(0)なので、タスクIDを返す */
}

/*
  -----------------------------------------------------------------------------
  
  Method : リソースの所得
          
  Param  : リソース番号
         : 優先順位 (上限度優先プロトコル有効時)
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
#if USE_NOS_RESRC & USE_NOS_B1
#define GET_RESRC int num, int pri
#else
#define GET_RESRC int num
#endif
int getResrc(GET_RESRC)
{
    if (IsOver(num, NUM_NOS_RESRC) || IsTaskWait())
        return NOS_ERROR;       /* リソース番号異常 ? || 既に待機状態 ?*/

    if (0 < nos_tick) {
        /* 実効可能 ? */
    
        struct Nos_Resrc *tr;       /* Target Resrc */

        tr = nos_res + num;
        if (UseNosDebug(USE_NOS_B2, tr->lock == nos_sys.now->info.id))
            return NOS_ERROR;   /* 既に自身が確保している ? */

        if (!tr->lock) {
            /* リソース確保可能 ? */

            tr->lock = nos_sys.run.info.id;
#if USE_NOS_RESRC & USE_NOS_B1
            if (tr->lock != NOS_SYS_ID) {
                /* システムタスク以外 ? */

                tr->pri = nos_sys.now->info.pri;
                if (pri && tr->pri > pri) nos_sys.now->info.pri = pri;
                /* 上限度優先 && より上限度優先 ? 優先順位変更 */
            }
#endif
            return NOS_NONE;
        }
        /* リソースが確保できない ? */

        NextQueue(tr);
    }
    return NOS_WAIT;
}

/*
  -----------------------------------------------------------------------------
  
  Method : リソースの開放
          
  Param  : リソース番号
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
int purgeResrc(int num)
{
    struct Nos_Resrc *tr;       /* Target Resrc */
    struct Nos_Link *wt;        /* Wait Task */

    tr = nos_res + num;
    if (IsOver(num, NUM_NOS_RESRC) ||
        nos_sys.run.info.id != tr->lock) return NOS_ERROR;
                                /* リソース番号異常 ? */
                                /* ロックIDが違う? */

#if USE_NOS_RESRC & USE_NOS_B1
    if (tr->lock != NOS_SYS_ID) nos_sys.now->info.pri = tr->pri;
                                /* システムタスクではない ? 優先度復元 */
#endif
    tr->lock = 0;               /* 空き */
    if (null != (wt = tr->next)) {
        /* Resorce待ちのTask ? */

        tr->next = wt->next;
        if (NextSched(addLink(&nos_sys.run, wt))) nos_tick = 0;
    }
    return SensTick();
}

/*
  -----------------------------------------------------------------------------
  
  Method : Constracture / Resorceの初期化
          
  Param  : void
 
  Return : void
 
  -----------------------------------------------------------------------------
*/
static void initNosResrc(void)
{
    struct Nos_Resrc *ir;       /* Init Resrc */
    int i;                      /* Index */

    ir = nos_res;
    for (i = 0; i < NUM_NOS_RESRC; i++) {
        /* リソースの初期化 */
        
        ir->next = null;
        ir->lock = 0;           /* 空き */
        ir++;
    }
}
#endif

#if USE_NOS_MEM & USE_NOS_B0

/*
  -----------------------------------------------------------------------------
  
  Method : バッファの探査

  Param  : 探査モード   真:空きメモリの合計
         :              偽:確保可能な最大メモリ
  
  Return : 最大空き容量
 
  -----------------------------------------------------------------------------
*/
NOS_MEM_PAGE sensMem(boolean mode)
{
    struct Nos_Page *fb;        /* Free Block */
    NOS_MEM_PAGE m;             /* Max */
    NOS_MEM_PAGE a;             /* All */

    m = NOS_PAGE;
    a = 0;

    for (fb = NosMem(&nos_mem.alloc)->next; null != fb; fb = fb->next) {
        /* 空きブロックの全サーチ */

        a += (fb->size - NOS_PAGE);
                                /* 空き容量計算 */
        if (m < fb->size) m = fb->size;
    }
    return mode ? a : (GetPage(m) - NOS_PAGE);
                                /* 管理領域を除いた容量を返す */
}

/*
  -----------------------------------------------------------------------------
  
  Method : メモリの確保

  Param  : 確保するメモリ量
  
  Return : NOS_WAIT     確保できない
         : 他           確保したメモリ
 
  -----------------------------------------------------------------------------
*/
void *allocMem(NOS_MEM_PAGE size)
{
#if USE_NOS_MEM & USE_NOS_B1
    struct Nos_Page *pb, *ab, *mb;
                                /* Pre Block, Alloc Block, Max Block  */
#else
    struct Nos_Page *pb, *ab;
                                /* Pre Block, Alloc Block */
#endif

    if (IsTaskWait()) return (void *)NOS_ERROR;
                                /* 既に待機状態 ? */

    if (0 >= nos_tick) return (void *)NOS_WAIT;
                                /* タイムアップ ? */

    size = GetPage(size + NOS_PAGE + (1 << NOS_MEM_ALLOC) - 1);
                                /* 要求サイズを必要ページ数相当に変換 */
    pb = &nos_mem.alloc;
#if USE_NOS_MEM & USE_NOS_B1
    mb = null;
    for (ab = pb->next; ab && GetPage(ab->size) != size; ab = ab->next) {
        /* 完全に一致するページを最後までサーチ */

        if (size < ab->size &&
            (null == mb || mb->size < ab->size)) mb = ab;
        /* 確保可能なブロック発見 && */
        /* (最大ブロック未発見 || より大きい空きブロック) ? 最大ブロックを記録 */

        pb = ab;
    }
    if (null == ab) ab = mb;    /* 完全一致ブロックが無い ? 最大空きブロック */
#else
    for (ab = pb->next;
         null != ab && GetPage(ab->size) < size; ab = ab->next) pb = ab;
        /* 確保可能なブロックではない ?  */
#endif

    if (null == ab) {
        /* 確保可能メモリが無い ? */

        nos_sys.now->wait.size = size;
        
        NextQueue(&nos_mem.wait);
        return (void *)NOS_WAIT;
    }

    if (GetPage(ab->size) == size) {
        /* 全体割り当て ? 該当ブロックを外す */

        pb->next = ab->next;
    } else {
        /* 部分割り当て */

        ab->size -= size;       /* ブロックを分割 */

        ab = NosMem((char *)ab + ab->size);
        ab->size = size;        /* ブロックの後半を割り当てる */
    }
    return GetLink(ab) + 1;
}

/*
  -----------------------------------------------------------------------------
  
  Method : メモリ開放／メモリが確保できる可能性のある全タスクを実行キューへ
  
  Param  : 今回操作したメモリブロック
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
static void *reallocTask(struct Nos_Page *mblock)
{
    struct Nos_Link *wt, *rt;   /* Wait Task, Run Task */
    NOS_MEM_PAGE p;             /* Page */

    p = mblock->size;
    wt = &nos_mem.wait;
    for (rt = wt->next; null != rt; rt = wt->next) {
        /* メモリ待ちキューから実行キューへの移動判定 */

        if (NosSlot(rt)->wait.size > p) {
            /* メモリを確保できない ? */
            
            wt = rt;            /* タスクを残す */
            continue;
        }
        /* 今回生成したメモリを確保できる可能性がある ? */

        p -= NosSlot(rt)->wait.size;
                                /* 残りページ数を減らす */

        wt->next = rt->next;
        if (NextSched(addLink(&nos_sys.run, rt))) nos_tick = 0;
                                /* タスクを実行Queueへ */
                                /*  次スケジュール発生 ? */
    }
    return SensTick2();
}

/*
  -----------------------------------------------------------------------------
  
  Method : メモリの開放
  
  Param  : 開放するメモリ
  
  Return : バッファ開放    待機ステータス        
         : バッファ        開放エラー
 
  -----------------------------------------------------------------------------
*/
void *purgeMem(void *pmem)
{
    struct Nos_Page *fb, *pb;    /* Free Block, Purge Block */
    
    if (UseNosDebug(USE_NOS_B1, null == pmem))
        return (void *)NOS_ERROR;
                                /* 使用中メモリではない ? */

    pb = (struct Nos_Page *)pmem - 1;
                                /* 管理情報を所得 */

    if (UseNosDebug(USE_NOS_B1, null == pb || pb->next != pb))
        return (void *)NOS_ERROR;
                                /* 使用中メモリではない ? */

    for (fb = &nos_mem.alloc; null != fb->next && fb->next < pb; fb = fb->next);
                                /* 挿入位置をサーチ */

    if (fb->next ==  NosMem((char *)pb + pb->size)) {
        /* 後ろのメモリと連続 ? 後ろのメモリを結合 */

        pb->size += (fb->next)->size;
        pb->next = (fb->next)->next;
      } else {
        /* 連続していないからそのままセット */
        
        pb->next = fb->next;
    }

    if (NosMem((char *)fb + fb->size) == pb) {
        /* 前のメモリと連続 ? 前のメモリへ結合 */

        fb->size += pb->size;
        fb->next = pb->next;

        pb = fb;                /* 新しくできたブロックの先頭 */
    } else {
        /* 前のメモリと非連続 */

        fb->next = pb;
    }
    return reallocTask(pb);     /* メモリ待ちタスク判定 */
}

/*
  -----------------------------------------------------------------------------
  
  Method : メモリの初期化
  
  Param  : 初期化するメモリ
         : 初期化するサイズ
  
  Return : 待機要求
 
  -----------------------------------------------------------------------------
*/
void *initMem(void *imem, NOS_MEM_PAGE size)
{
    struct Nos_Page *ib;        /* Init Block */

    ib = imem;

    ib->size = size;
    ib->next = ib;
    if (UseNosDebug(USE_NOS_B0, !(ib->size >> NOS_MEM_ALLOC)))
        return (void *)NOS_ERROR;
                                /* 確保できない ? */

    ib++;

    return purgeMem(ib);
}

/*
  -----------------------------------------------------------------------------
  
  Method : Constracture /メモリアロケーターの初期化
          
  Param  : void
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
static void initNosMem(void)
{
    nos_mem.wait.next = null;   /* メモリ確保待機Queue */

    nos_mem.alloc.size = 0;
    nos_mem.alloc.next = null;  /* メモリブロック管理情報 */
}
#endif

#if USE_NOS_TIMER & USE_NOS_B0
/*
  -----------------------------------------------------------------------------
  
  Method : タイマーのセット
          
  Param  : スリープ時間
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
int sleepTask(int time)
{
    if (IsTaskWait()) return NOS_ERROR;
    
    nos_sys.now->wait.pri = nos_sys.now->info.pri;
    nos_sys.now->info.timer = time;
                                /* スリープ中はプライオリティを */
                                /* タイマーとして使う */
    NextQueue(&nos_time);
    return NOS_WAIT;
}

/*
  -----------------------------------------------------------------------------
  
  Method : スリープ待機の解除
          
  Param  : タスクID
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
int wakeupTask(int task)
{
    struct Nos_Slot *st, *wt;   /* Sleep Task, Wakeup Task */

    if (IsOver(task, NUM_NOS_TASK) || UseNosDebug(USE_NOS_B0, task == 0))
        return NOS_ERROR;       /* 無効なタスク番号 ? */

    st = (void *)&nos_time;
    for (wt = (void *)st->next; null != wt; wt = (void *)st->next) {
        /* 最後までサーチする */

        if (wt->id != task) {
            /* 該当タスクではない */

            st = wt;            /* 次へ */
            continue;
        }

        wt->info.pri = wt->wait.pri;
                                /* プライオリティに戻す */
        st->next = wt->next;
        if (NextSched(addLink(&nos_sys.run, NosLink(wt)))) nos_tick = 0;
                                /* 次スケジューリング ? */
    }
    return SensTick();
}

/*
  -----------------------------------------------------------------------------
  
  Method : スリープ中のタスクを起こす
          
  Param  : カレント時間
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
static void wakeupSleep(int time)
{
    struct Nos_Slot *sq, *st;   /* Sleep Queue & Task */

    nos_timer = 0;            /* 処理前にクリア */

    sq = (void *)&nos_time;
    while (null != (st = (void *)sq->next)) {
        /* Queue全体をサーチ */

        if (0 >= (st->info.timer -= time)) {
            /* 時間更新、起床 ? */

            st->info.pri = st->wait.pri;
            sq->next = st->next; 
                                /* プライオリティに戻してQueueから外す */
            if (NextSched(addLink(&nos_sys.run, NosLink(st)))) nos_tick = 0;
                                /* タスクを取り出して実行Queueへ */
            continue;
        }
        sq = st;
    }
}

/*
  -----------------------------------------------------------------------------
  
  Method : Constracture / タイマーの初期化
          
  Param  : void
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
static void initNosTimer(void)
{
    nos_timer = 0;
    nos_time.next = null;
}
#endif

/*
  -----------------------------------------------------------------------------
  
  Method : Taskの実行

  Param  : 実行スレッド
         : 優先順位
  
  Return : タスクID
 
  -----------------------------------------------------------------------------
*/
int runTask(int (*rthread)(void), unsigned int pri)
{
    struct Nos_Slot *nt;        /*New Task */

    if (IsOver(pri, ((unsigned int)-1) >> 1)) return NOS_ERROR;
    
    nt = (void *)nos_sys.free.next;
    if (null == nt) return NOS_WAIT;
                                /* 空きスロットが無い ? */

    nos_sys.free.next = nt->next;
    
    nt->thread = rthread;       /* 初回に呼び出すThread */
    nt->info.pri = pri;         /* 優先順位 */
    addLink(&nos_sys.run, NosLink(nt));
                                /* run queueに追加 */
    return nt->id;
}

/*
  -----------------------------------------------------------------------------
  
  Method : Taskのアボート
          
  Param  : void
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
int purgeTask(void)
{
    if (IsTaskWait()) return NOS_ERROR;
    
    NextQueue(&nos_sys.free);
    return NOS_WAIT;
}

/*
  -----------------------------------------------------------------------------
  
  Method : スケジューラ
          
  Param  : void
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
void runSched(void)
{
    struct Nos_Slot *nr;        /* Now Run */
    int s;                      /* Stat */

    if (UseNosDebug(USE_NOS_B3, nos_sys.run.info.id != NOS_SYS_ID)) return;
                                /* タスク内部からの呼び出し ? */

    nos_tick = NOS_TICK_TIME;   /* TickTimer初期化 */

    nos_sys.queue = &nos_sys.run;
                                /* 終了後に投入するQueueのデフォルト */

    nos_sys.run.next = (nos_sys.now = (void *)nos_sys.run.next)->next;
    nr = nos_sys.now;           /* 実行するTaskを取り出す */

    nos_sys.run.info.id = nr->id;
                                /* タスクIDでロック (run queueをワークに) */

    do {
        /* タスクが待機状態になるまで */
        
        s = nr->thread();       /* 実行Thread呼び出し */
#if USE_NOS_TIMER & USE_NOS_B0
        if (nos_timer) wakeupSleep(nos_timer);
                                /* タイマー待機タスク起床処理 */
#endif
    } while (s && 0 < nos_tick);

    addLink(nos_sys.queue, NosLink(nr));
                                /* Queueに再配置移動 */

    nos_sys.run.info.id = NOS_SYS_ID;
                                /* システムのアンロック */
}    

/*
  -----------------------------------------------------------------------------
  
  Method : 次のThreadをセット
          
  Param  : 実行Thread
  
  Return : 待機ステータス
 
  -----------------------------------------------------------------------------
*/
int nextThread(boolean (*nthread)(void))
{
    if (UseNosDebug(USE_NOS_B3, nos_sys.run.info.id == NOS_SYS_ID))
        return NOS_ERROR;       /* タスク外からの呼び出し ? */

    nos_sys.now->thread = nthread;
                                /* Threadの変更 */

    return SensTick();
}    

/*
  -----------------------------------------------------------------------------
  
  Method : 次のTaskをスケジュール
          
  Param  : void
  
  Return : NOS_WAIT
 
  -----------------------------------------------------------------------------
*/
int nextTask(void)
{
    nos_tick = 0;
    return NOS_WAIT;
}

/*
  -----------------------------------------------------------------------------
  
  Method : Constracture /Taskの初期化
          
  Param  : void
  
  Return : void
 
  -----------------------------------------------------------------------------
*/
static void initNosTask(void)
{
    int i;
    struct Nos_Slot *it;        /* Init Task */

    nos_sys.free.next = null;

    it = nos_sys.slot;            /* free task slot */
    for (i = 0; i < NUM_NOS_TASK - 1; i++) {
        /* 最後尾以外を空きタスクとして登録する */

        it->info.pri = NUM_NOS_TASK >> 1;
        it->id = i + 1;         /* タスクIDは1から */
        addLink(&nos_sys.free, NosLink(it));
        it++;
    }

    /* 最後尾のスロットをシステムに割り当てる */

    it->info.pri = ((unsigned int)-1) >> 1;
#if USE_NOS_DEBUG
    it->id = NOS_SYS_ID;
#endif
    it->thread = nextTask;

    addLink(&nos_sys.run, NosLink(it));
                                /* RunQueueに直接投入 */
}

/*
  -----------------------------------------------------------------------------
  
  Method : 初期化
          
  Param  : void
 
  Return : void
 
  -----------------------------------------------------------------------------
*/
void initNos(void)
{
    nos_sys.run.info.id = NOS_SYS_ID;
                                /* 実行中タスクなし */

    nos_sys.run.next = null;      /* RUN queueの初期化 */
    nos_tick = NOS_TICK_TIME; /* Tick値の初期化 */

    /* Method */
    
    initNosTask();
#if USE_NOS_MSG & USE_NOS_B0
    initNosBox();               /* メッセージBoxの初期化 */
    initNosMsg(&nos_mes);       /* メッセージの初期化 */
#endif

#if USE_NOS_EVENT & USE_NOS_B0
    initNosEvent();             /* Eventの初期化 */
#endif

#if USE_NOS_RESRC & USE_NOS_B0
    initNosResrc();             /* Resorceの初期化 */
#endif

#if USE_NOS_MEM & USE_NOS_B0
    initNosMem();               /* Memoryの初期化 */
#endif

#if USE_NOS_TIMER & USE_NOS_B0
    initNosTimer();             /* Timerの初期化 */
#endif
}
