package PluginManager::Command;
use base qw( Class::Accessor::Fast );

use strict;

use YAML::Tiny;
use PluginManager::Repository;

sub new {
	my $class = shift;
	my $self = shift || {};
	bless $self , $class;

	return $self;
}

sub options {
	();
}

sub process {
	undef;
}

sub usage {
	'';
}

sub repository_root {
	PluginManager::Repository::repository_root(1);
}

sub repository_pool {
	PluginManager::Repository::repository_pool(1);
}

sub instance_for {
	my $class = shift;
	my ($cmd) = @_;

	if ($cmd eq 'make_package') {
		PluginManager::Command::MakePackage->new;
	}
	elsif ($cmd eq 'update_repository') {
		PluginManager::Command::UpdateRepository->new;
	}
	else {
		PluginManager::Command::Usage->new;
	}
}


package PluginManager::Command::Error;

sub new {
	my $class = shift;
	my $self = { @_ };
	bless $self , $class;

	$self->{status} ||= -1;

	return $self;
}


package PluginManager::Command::Usage;
use base qw( PluginManager::Command );

sub usage {
	<<__EOM__;
usage: $0 <subcommand> [options] [args]
__EOM__
}

sub process {
	my $self = shift;
	PluginManager::Command::Error->new(message => $self->usage);
}


package PluginManager::Command::MakePackage;
use base qw( PluginManager::Command );

use File::Basename;
use File::Spec;
use File::Copy::Recursive qw/ rcopy /;
use File::Find;
use File::Path;
use POSIX;
use PluginManager::Util;

sub usage {
	<<__EOM__;
usage: $0 make_package [options] PluginName1 [PluginName2 [PluginName3]]
__EOM__
}

sub options {
	('all');
}

sub translate_value {
	my $self = shift;
	my ($mod, $word) = @_;

	if (ref($word) eq 'HASH') {
		foreach my $key (keys(%$word)) {
			$word->{$key} = $self->translate_value($mod, $word->{$key});
		}
		$word;
	}
	elsif (ref($word)) {
		$word;
	}
	else {
		$mod->translate($word);
	}
}

sub make_control_locale {
	my $self = shift;
	my ($mod, $locale) = @_;

	MT->set_language($locale);

	my %ctrl = ();
	foreach my $key (
		'id', 'name', 'version', 'description',
		'author_name', 'author_link',
		'plugin_link', 'doc_link',
	) {
		$ctrl{$key} = $mod->$key;
	}
	$ctrl{locale} = $locale;

	my $config = File::Spec->catfile($mod->{full_path}, 'mpm', 'config.yaml');
	if (-f $config) {
		my $yaml = YAML::Tiny->read($config);
		my $info = $yaml->[0];
		foreach my $key (
			'category', 'compatiblity', 'license', 'target',
		) {
			$ctrl{$key} = $self->translate_value($mod, $info->{$key})
				if $info->{$key};
		}
	}

	\%ctrl;
}

sub make_control {
	my $self = shift;
	my ($mod) = @_;

	my @ctrls = ();

	my @locales = ();
	if (my $l10n = $mod->l10n_class) {
		eval(" require $l10n ; ");

		foreach my $l (keys(%{ MT->supported_languages })) {
			my $handle = $l10n->get_handle($l);
			if ($handle->language_tag eq $l) {
				push(@locales, $l);
			}
		}
	}
	else {
		@locales = ('en-us');
	}

	foreach my $l (@locales) {
		my $ctrl = $self->make_control_locale($mod, $l);
		$ctrl->{locales} = \@locales;
		push(@ctrls, $ctrl);
	}

	\@ctrls;
}

sub make_dir {
	my $self = shift;
	my ($mod, $ctrls) = @_;

	my $basename = basename($mod->{full_path}, '.pl');
	my $ver = $mod->version;
	my $mpmname = $basename . '-' . $ver;

	my $tmpdir = tempdir( CLEANUP => 1 ); 
	my $destdir = File::Spec->catdir($tmpdir, $mpmname);
	mkdir($destdir);

	$destdir;
}

sub copy_cgi {
	my $self = shift;
	my ($mod, $ctrls, $destdir) = @_;

	my $basename = basename($mod->{full_path});

	rcopy(
		$mod->{full_path},
		File::Spec->catdir($destdir, 'dist', 'plugins', $basename)
	);
}

sub copy_static {
	my $self = shift;
	my ($mod, $ctrls, $destdir) = @_;

	my $app = MT->instance;
	my $static = File::Spec->catdir(
		$app->server_path(), 'mt-static', 'plugins'
	);
	my $basename = basename($mod->{full_path});
	my $plugin_static = File::Spec->catdir($static, $basename);

	if (-e $plugin_static) {
		rcopy(
			$plugin_static,
			File::Spec->catdir($destdir, 'dist', 'mt-static', 'plugins', $basename)
		);
	}
}

sub copy_control {
	my $self = shift;
	my ($mod, $ctrls, $destdir) = @_;
	my $ctrl_dir = File::Spec->catdir(
		$destdir, 'control'
	);
	mkdir($ctrl_dir);

	my $yaml = new YAML::Tiny;
	foreach my $c (@$ctrls) {
		my $name = 'control-' . $c->{locale} . '.yaml';
		$yaml->[0] = $c;
		$yaml->write(File::Spec->catfile($ctrl_dir, $name));
	}

	foreach my $f (glob(File::Spec->catdir($mod->{full_path}, 'mpm' , '*'))) {
		next if not -d $f;
		my $base = basename($f);
		rcopy($f, File::Spec->catdir($destdir, $base));
	}
}

sub package {
	my $self = shift;
	my ($mod, $ctrls, $destdir) = @_;

	my $cwd = getcwd();
	chdir(dirname($destdir));

	require Archive::Tar;
	my $tar = Archive::Tar->new;
	finddepth({
		wanted => sub {
			if ($_ !~ m/(^|\/)(\.svn|CVS|\.git(attributes|ignore)?)(\/|$)/) {
				$tar->add_files($_);
			}

			if (-f $_) {
				unlink($_);
			}
			elsif (-d $_) {
				rmdir($_);
			}
		},
		no_chdir => 1,
	}, basename($destdir));

	chdir($cwd);

	my $pool = $self->repository_pool;

	mkpath($pool);

	my $outfile = File::Spec->catfile($pool, basename($destdir) . '.mpm');
	$tar->write($outfile, 1);

    my $file_mode = (~ oct(MT->config('UploadUmask'))) & oct('666');
	chmod($file_mode, $outfile);

	$outfile;
}

sub available_components {
	PluginManager::Repository::available_components(@_);
}

sub process {
	my $self = shift;
	my $app = MT->instance;
	my ($opts, @names) = @_;

	if ($opts->{all}) {
		@names = $self->available_components;
	}
	elsif (! @names) {
		return PluginManager::Command::Error->new(message => $self->usage);
	}

	foreach my $name (@names) {
		my $mod = $app->component($name);
		if (! $mod) {
			my $msg =
				"Installed plugins:\n"
				. join("\n", $self->available_components)
				. "\n";
			return PluginManager::Command::Error->new(message => $msg);
		}

		my $ctrls = $self->make_control($mod);
		my $destdir = $self->make_dir($mod, $ctrls);
		$self->copy_cgi($mod, $ctrls, $destdir);
		$self->copy_static($mod, $ctrls, $destdir);
		$self->copy_control($mod, $ctrls, $destdir);
		my $file = $self->package($mod, $ctrls, $destdir);

		if (! $opts->{q}) {
			print("$file\n");
		}
	}
}

package PluginManager::Command::UpdateRepository;
use base qw( PluginManager::Command );

use File::Basename;
use File::Spec;
use Digest::MD5  qw(md5 md5_hex md5_base64);

sub usage {
	<<__EOM__;
usage: $0 update_repository [options]
__EOM__
}

sub process {
	my $self = shift;
	my $app = MT->instance;
	my ($opts) = @_;

	my $root = $self->repository_root;
	my $packages = {};

	require Archive::Tar;
	foreach my $mpm (glob(File::Spec->catfile($self->repository_pool, '*.mpm'))) {
		my $basename = basename($mpm, '.mpm');
		(my $relative = $mpm) =~ s/$root//;
		$relative =~ s#^/?#./#;

		my $tar = Archive::Tar->new;
		$tar->read($mpm, 1);

		my $get_info = sub {
			my $file = shift;

			my $yaml_string = $tar->get_content($file);
			my $yaml = YAML::Tiny->read_string($yaml_string);

			my $pkg = $yaml->[0];

			$pkg->{filename} = $relative;
			$pkg->{size} = (stat($mpm))[7];
			$pkg->{md5sum} = md5_hex(do {
				open(my $fh, '<', $mpm); local $/; <$fh>;
			});

			$pkg;
		};

		my $en_us_hash = undef;
		foreach my $l (keys(%{ MT->supported_languages })) {
			my $yaml_file = File::Spec->catfile(
				$basename, 'control', 'control-' . $l . '.yaml'
			);
			if (! $tar->contains_file($yaml_file)) {
				$yaml_file = File::Spec->catfile(
					$basename, 'control', 'control-en-us.yaml'
				);
			}

			my $pkg = undef;
			if ($yaml_file =~ m/en-us\.yaml$/) {
				if (! $en_us_hash) {
					$en_us_hash = $pkg = $get_info->($yaml_file);
				}
				else {
					$pkg = $en_us_hash;
				}
			}
			else {
				$pkg = $get_info->($yaml_file);
			}

			my $version = quotemeta($pkg->{version});
			#(my $sig = $basename) =~ s/-$version//;
			my $sig = $pkg->{id};

			$packages->{$l} ||= {};
			$packages->{$l}->{$sig} ||= {};
			$packages->{$l}->{$sig}->{$pkg->{version}} = $pkg;
		}
	}

    my $file_mode = (~ oct(MT->config('UploadUmask'))) & oct('666');
	foreach my $l (keys(%$packages)) {
		my $file = File::Spec->catfile($root, 'packages-' . $l . '.yaml.gz');

		my $yaml = YAML::Tiny->new;
		$yaml->[0] = $packages->{$l};

		$yaml->write($file);

		my $fh = IO::Zlib->new($file, 'wb9');
		if (defined $fh) {
			print($fh $yaml->write_string);
			$fh->close;
			chmod($file_mode, $file);
		}
	}
}

package PluginManager::Command::MakeDistribution;
use base qw( PluginManager::Command );

use File::Basename;
use File::Spec;
use File::Find;
use Cwd;

sub usage {
	<<__EOM__;
usage: $0 make_distribution [options]
__EOM__
}

sub process {
	my $self = shift;
	#PluginManager::Command::UpdateRepository::process($self, @_);

	my $root = $self->repository_root;

	my $cwd = getcwd();
	chdir($root);

	require Archive::Tar;
	my $tar = Archive::Tar->new;
	finddepth({
		wanted => sub {
			if ($_ =~ m/(packages-(.*)\.yaml\.gz|\.mpm)$/) {
				$tar->add_files($_);
			}
		},
		no_chdir => 1,
	}, '.');

	chdir($cwd);

	my ($sec,$min,$hour,$mday,$mon,$year,$wday,$yday,$isdst) = localtime(time);
	$year += 1900;
	$mon++;

	my $outfile = File::Spec->catfile(
		$root,
		sprintf(
			'%04d%02d%02d%02d%02d.mpd', $year, $mon, $mday, $hour, $min
		)
	);
	$tar->write($outfile, 1);

    my $file_mode = (~ oct(MT->config('UploadUmask'))) & oct('666');
	chmod($file_mode, $outfile);
}

1;
