package PluginManager::App;

use File::Basename;
use POSIX;
use File::Find;
use YAML::Tiny;
use Encode;
use PluginManager::Util;

use strict;

sub init_request {
	my ($plugin, $app) = @_;
	if (my $mode = $app->param('__mode')) {
		if ($mode eq 'pluginmanager') {
			$app->{upgrade_required} = 0;
		}
	}
}

sub is_upgrade_required {
	my $app = MT->instance;
	my $comps = @MT::Components;
	$app->{upgrade_required} = 0;

	my $schema  = $app->config('SchemaVersion');
	my $version = $app->config('MTVersion');
	if (   !$schema
		|| ( $schema < $app->schema_version )
		|| ( ( !$version || ( $version < $app->version_number ) )
			&& $app->config->NotifyUpgrade )
		)
	{
		$app->{upgrade_required} = 1;
	}
	else {
		foreach my $plugin (@MT::Components) {
			if ( $plugin->needs_upgrade ) {
				$app->{upgrade_required} = 1;
				last;
			}
		}
	}

	$app->{upgrade_required};
}

sub pluginmanager {
    my $app = shift;

	if (my $blog_id = $app->param('blog_id')) {
		$app->redirect($app->app_uri . '?blog_id=' . $blog_id);
	}

    my $plugin = MT->component('PluginManager');
	my $op = $app->param('op') || 'package_list';

	$app->config->AltTemplatePath(File::Spec->catfile(
		$plugin->{full_path}, 'tmpl'
	));

	if ($op eq 'setting') {
		&setting($plugin, $app, &default_param);
	}
	elsif ($op eq 'update_setting') {
		&update_setting($plugin, $app, &default_param);
	}
	elsif ($op eq 'ftp_prompt') {
		&ftp_prompt($plugin, $app, &default_param);
	}
	elsif ($op eq 'import') {
		&import($plugin, $app, &default_param);
	}
	elsif ($op eq 'do_import') {
		&do_import($plugin, $app, &default_param);
	}
	elsif ($op eq 'export') {
		&export($plugin, $app, &default_param);
	}
	elsif ($op eq 'do_export') {
		&do_export($plugin, $app, &default_param);
	}
	elsif ($op eq 'repository') {
		&repository($plugin, $app, &default_param);
	}
	elsif ($op eq 'create_repository') {
		&create_repository($plugin, $app, &default_param);
	}
	elsif ($op eq 'package_update') {
		&package_update($plugin, $app, &default_param);
	}
	elsif ($op eq 'upgrade') {
		&upgrade($plugin, $app, &default_param);
	}
	elsif ($op eq 'task') {
		&task($plugin, $app, &default_param);
	}
	else {
    	&package_list($plugin, $app, &default_param);
	}
}

sub default_param {
	my $app = MT->instance;
    my $plugin = MT->component('PluginManager');
	my %param = ();

	my @ignores = (
		'time', 'updated', 'created', 'initialize',
		'imported_package', 'imported_distribution',
		'created_package', 'created_distribution',
	);
    my $q = new CGI('');
	foreach my $p ($app->param) {
		if (! grep({ $_ eq $p } @ignores)) {
			$q->param($p, $app->param($p));
		}
	}
	my $app_uri = $app->app_uri;

	local $CGI::USE_PARAM_SEMICOLONS;
	$CGI::USE_PARAM_SEMICOLONS = 0;

	foreach my $o (
        'package_list', 'setting', 'package_update', 'upgrade',
		'import', 'export', 'repository',
    ) {
		$q->param('op', $o);
		$param{'link_' . $o} = $app_uri . '?' . $q->query_string;
	}

	return \%param;
}

sub packages_merge {
	my ($original, $add) = @_;
	foreach my $p (keys(%$add)) {
		if (exists($original->{$p})) {
			foreach my $v (keys(%{ $add->{$p} })) {
				$original->{$p}->{$v} = $add->{$p}->{$v};
			}
		}
		else {
			$original->{$p} = $add->{$p};
		}
	}

	$original;
}

sub packages_load {
	require Compress::Zlib;
	require YAML::Tiny;

    my ($plugin, $app, $param) = @_;

    my $packages_class = MT->model('pm_packages');
    my $locale = $app->current_language;
    my @pkgs = $packages_class->load({ 'locale' => $locale });

	my $packages = {};
	foreach my $p (@pkgs) {
		my $str = Compress::Zlib::memGunzip($p->packages);
		my $yaml = YAML::Tiny->read_string(decode('utf8', $str));

		&packages_merge($packages, $yaml->[0]);
	}

	$packages;
}

sub writable {
	my $plugin = MT->component('PluginManager');
	my $dir = dirname($plugin->{full_path});
	my $file = File::Spec->catfile($dir, 'test');
	for (my $i = 0;;$i++) {
		last if (! -f $file);
		$file .= $i;
	}

	open(my $fh, '>', $file) or return 0;
	close($fh) or return 0;
	unlink($file) or return 0;

	return 1;
}

sub has_ftp_settings {
	my $plugin = MT->component('PluginManager');
	my $hash = $plugin->get_config_hash;
	$hash->{'ftp_user'} || 0;
}

sub package_list {
    my ($plugin, $app, $param) = @_;

	my $packages = &packages_load(@_);
	my $writable = &writable;

	if (! $app->param('time')) {
		if (
			(! $writable) && (! &has_ftp_settings)
		) {
			return $app->redirect(
				$app->app_uri . '?__mode=pluginmanager&op=setting&initialize=1'
			);
		}

		if (! scalar(%$packages)) {
			return $app->redirect(
				$app->app_uri . '?__mode=pluginmanager&op=package_update&initialize=1'
			);
		}
	}

    my $installed_class = MT->model('pm_installed');
	my @installeds = $installed_class->load;
	foreach my $i (@installeds) {
		my $control = $i->make_control_locale(
			$app->current_language
		);
		$control->{'installed'} = 1;

		$packages->{$i->signature} ||= {};
		$packages->{$i->signature}{$control->{version}} = $control;
	}

	$param->{packages} = $packages;

	$param->{listing_screen} = 1;
	$param->{updated} = $app->param('updated') ? 1 : 0;
	$param->{writable} = $writable;

    $plugin->load_tmpl('pm_package_list.tmpl', $param);
}

sub package_update {
    require LWP::UserAgent;
	require Compress::Zlib;
	require YAML::Tiny;

    my ($plugin, $app, $param) = @_;

    my @repos = $plugin->get_config_value('repositories');
	while(@repos && ref $repos[0]) {
		@repos = @{ $repos[0] };
	}
	@repos = map({ split /\n/ } @repos);

    my $packages_class = MT->model('pm_packages');
    my $locale = $app->current_language;
    $packages_class->remove({ 'locale' => $locale });
    
	my $ua = LWP::UserAgent->new;
	if (my $proxy = MT->config('HTTPProxy')) {
		$ua->proxy('http', $proxy);
	}
	$ua->timeout(10);

    foreach my $r (@repos) {
        $r =~ s/\/$//;
        my $yaml_gz = $r . '/packages-' . $locale . '.yaml.gz';

        my $response = $ua->get($yaml_gz);
        if ($response->is_success) {
			my $str = Compress::Zlib::memGunzip($response->content);
			my $yaml = YAML::Tiny->read_string($str);
			my $packs = $yaml->[0];

			foreach my $p (keys(%$packs)) {
				foreach my $v (keys(%{ $packs->{$p} })) {
					$packs->{$p}->{$v}->{'repository'} = $r;
				}
			}

			$yaml = YAML::Tiny->new;
			$yaml->[0] = $packs;
			$str = Compress::Zlib::memGzip($yaml->write_string, 9);

            my $pkgs = $packages_class->new;
            $pkgs->locale($locale);
            $pkgs->repository($r);
            $pkgs->packages($str);
			$pkgs->save();
        }
        else {
            die $response->status_line;
        }
    }

	my $updated = $app->param('initialize') ? '' : '&updated=1';
    $app->redirect(
        $app->app_uri . '?__mode=pluginmanager&op=package_list' . $updated . '&time=' . time
    );
}

sub execute_script {
	package PluginManager::App::execute_script;
	our $manager = MT->component('PluginManager');
	do($_[0]) if -f $_[0];
	if ($@) {
		$_[1]->($@) if ref $_[1];
		0;
	}
	else {
		1;
	}
}

sub install_distribution {
	require Archive::Tar;

	my ($file, $plugin_base, $handler) = @_;

	my $app = MT->instance;
	my $plugin = MT->component('PluginManager');
	my $temp_dir =  tempdir(undef, CLEANUP => 1);

	## extract
	my $tar = Archive::Tar->new;

	my $cwd = getcwd;
	chdir($temp_dir);
	$tar->read($file);
	$tar->extract;
	chdir($cwd);

	foreach my $f (glob(File::Spec->catfile($temp_dir, 'pool', '*.mpm'))) {
		&install($f);
	}
}

sub rcopy {
	my ($src, $dst) = @_;

	my $app = MT->instance;
	my $plugin = MT->component('PluginManager');

	if (my $password = $app->param('ftp_password')) {
		require Net::FTP;
		require File::Find;

		my $hash = $plugin->get_config_hash;

		my $ftp = Net::FTP->new($hash->{'ftp_host'});
		$ftp->login($hash->{'ftp_user'}, $password);
		$ftp->cwd($hash->{'ftp_directory'});
		$ftp->binary;
		if (-d $src) {
			if ($dst) {
				$ftp->mkdir($dst, 1);
				$ftp->cwd($dst);
			}

			my $cwd = getcwd;
			chdir($src);
			File::Find::find({
				wanted => sub {
					if (-d $_) {
						$ftp->mkdir($_);
					}
					else {
						$ftp->put($_, $_);
					}
					$ftp->site(
						'CHMOD ' . sprintf('%04lo', (stat($_))[2]) . ' ' . $_
					);
				},
				no_chdir => 1,
			}, '.');
			chdir($cwd);
		}
		else {
			$ftp->put($src, $dst);
		}
	}
	else {
		require File::Copy::Recursive;

		File::Copy::Recursive::rcopy(
			$src,
			File::Spec->catdir($app->server_path(), $dst)
		);
	}
}

sub install {
	require Archive::Tar;

	my ($file, $plugin_base, $handler) = @_;

	my $app = MT->instance;
	my $plugin = MT->component('PluginManager');
	$plugin_base ||= dirname($plugin->{full_path});
	my $temp_dir =  tempdir(undef, CLEANUP => 1);

	## extract
	my $tar = Archive::Tar->new;

	my $cwd = getcwd;
	chdir($temp_dir);

	my @extracted = sort(grep($_, map({
		my $name = $_->full_path;
		my $is_dist = $name =~ s#[^/]*(/|\\)dist(/|\\)##;
		if (! $is_dist) {
			'';
		}
		elsif ($name =~ m/^plugins$|^mt-static(.plugins)?$/) {
			'';
		}
		else {
			$name;
		}
	} $tar->read($file, 1))));
	$tar->extract;
	chdir($cwd);

	my ($dir) = grep(-d $_, glob(File::Spec->catfile($temp_dir, '*')));


	# do preinst script
	{
		&execute_script(
			File::Spec->catfile($dir, 'script', 'preinst.pl'), $handler
		) or return;
	}

	## installation process
	# plugins/
	foreach my $f (glob(File::Spec->catfile($dir, 'dist', 'plugins', '*'))) {
		&rcopy($f, File::Spec->catdir(basename($plugin_base), basename($f)));
	}

	# mt-static/plugins
	my $static_base = File::Spec->catdir(
		'mt-static', basename($plugin_base)
	);
	foreach my $f (glob(File::Spec->catfile($dir, 'dist', 'mt-static', 'plugins', '*'))) {
		&rcopy($f, File::Spec->catdir($static_base, basename($f)));
	}

	# top-level script plugins.cgi
	my $server_path = $app->server_path();
	foreach my $f (glob(File::Spec->catfile($dir, 'dist', '*'))) {
		my $base = basename($f);
		if (grep(/^mt-static$|^plugins$/, $base)) {
			next;
		}
		&rcopy($f, $base);
	}

	# do postinst script
	{
		&execute_script(
			File::Spec->catfile($dir, 'script', 'postinst.pl'), $handler
		) or return;
	}

	## save package information
	# script/
	my $script_tgz = '';
	{
		my $cwd = getcwd;
		chdir($dir);

		if (glob(File::Spec->catfile('script', '*'))) {

			require Archive::Tar;
			my $tar = Archive::Tar->new;
			finddepth({
				wanted => sub {
					$tar->add_files($_);
				},
				no_chdir => 1,
			}, 'script');

			$tar->write('script.tar.gz', 1);
			$script_tgz = do{
				open(my $fh, 'script.tar.gz'); local $/; <$fh>
			};
		}

		chdir($cwd);
	}

	my ($signature, $version);
	# control/
	my $control = '';
	{
		my $control_data = {};
		foreach my $ctrl (glob(File::Spec->catfile($dir, 'control', '*'))) {
			my ($locale) = ($ctrl =~ m/control-(.*).yaml$/);
			my $yaml = YAML::Tiny->read($ctrl);
			$control_data->{$locale} = $yaml->[0];

			if (! $signature) {
				$signature = $control_data->{$locale}{'id'};
				$version = $control_data->{$locale}{'version'};
			}
		}

		my $yaml = YAML::Tiny->new;
		$yaml->[0] = $control_data;
		$control = $yaml->write_string;
	}

	# seve
	my $installed_class = MT->model('pm_installed');
	my ($inst) = $installed_class->load(
		{'signature' => $signature}
	);
	$inst ||= $installed_class->new;
	$inst->files(join("\n", @extracted));
	$inst->signature($signature);
	$inst->version($version);
	$inst->control($control);
	$inst->script($script_tgz);
	$inst->save();
}

sub task {
	require File::Spec;

    my ($plugin, $app, $param) = @_;

	my $type = $app->param('type');
	my $index = $app->param('index');
	my $upgrade = $app->session('pluginmanager_upgrade');

	my $task = undef;
	foreach my $set (@{ $upgrade->{tasks} }) {
		if ($set->{type} eq $type) {
			$task = $set->{tasks}->[$index];
		}
	}
	if (! $task) {
		return;
	}

	my $execute_script = sub {
		&execute_script(@_, sub { ($task->{error}) = @_ });
	};

	my $do_install = sub {
		## prepare installation file
		my $file = File::Spec->catfile(
			$upgrade->{temp_dir}, $task->{filename}
		);

		# concatinate downloaded files
		open(my $fh, '>', $file);
		foreach my $f (sort(glob($file . '.*'))) {
			my $cont = do{ open(my $fh, $f); local $/; <$fh> };
			print($fh $cont);
		}
		close($fh);

		&install($file, undef, sub { ($task->{error}) = @_ });
	};

	my $do_remove = sub {
		my $installed_class = MT->model('pm_installed');

		my $installed = $installed_class->load({
			'signature' => $task->{signature},
		});
		if (! $installed) {
			return;
		}

		my $tempdir = tempdir(undef, CLEANUP => 1);
		open(my $fh, '>', File::Spec->catfile($tempdir, 'script.tar.gz'));
		print($fh $installed->script);
		close($fh);

		require Archive::Tar;
		my $tar = Archive::Tar->new;
		my $cwd = getcwd;
		chdir($tempdir);
		$tar->read('script.tar.gz', 1);
		$tar->extract;
		chdir($cwd);

		# do preremove script
		{
			$execute_script->(
				File::Spec->catfile($tempdir, 'script', 'prerm.pl')
			) or return;
		}

		# remove
		my $plugin_base = dirname($plugin->{full_path});
		my $static_base = File::Spec->catdir(
			$app->server_path(), 'mt-static', 'plugins'
		);
		my $server_path = $app->server_path();

		my $ftp = undef;
		if (my $password = $app->param('ftp_password')) {
			require Net::FTP;

			my $hash = $plugin->get_config_hash;

			$ftp = Net::FTP->new($hash->{'ftp_host'});
			$ftp->login($hash->{'ftp_user'}, $password);
			$ftp->cwd($hash->{'ftp_directory'});
			$ftp->binary;

			$plugin_base = basename(dirname($plugin->{full_path}));
			$static_base = File::Spec->catdir(
				'mt-static', $plugin_base
			);
			$server_path = '';
		}

		foreach my $f (reverse(split(/\n/, $installed->files))) {
			if ($f =~ s{^plugins/}{}) {
				$f = File::Spec->catfile($plugin_base, $f);
			}
			elsif ($f =~ s{^mt-static/}{}) {
				$f = File::Spec->catfile($static_base, $f);
			}
			else {
				$f = $server_path ?
					File::Spec->catfile($server_path, $f) : $f;
			}

			if ($ftp) {
				if (-d $f) {
					$ftp->rmdir($f);
				}
				elsif (-f $f) {
					$ftp->delete($f);
				}
			}
			else {
				if (-d $f) {
					rmdir($f);
				}
				elsif (-f $f) {
					unlink($f);
				}
			}
		}

		$installed_class->remove({
			'signature' => $task->{'signature'},
		});

		# do postremove script
		{
			$execute_script->(
				File::Spec->catfile($tempdir, 'script', 'postrm.pl')
			) or return;
		}
	};

	if ($type eq 'download') {
	    require LWP::UserAgent;
		require File::Basename;

		my $index = $app->param('download_index');
		my $bytes = $app->param('bytes');

		my $ua = LWP::UserAgent->new();
		if (my $proxy = MT->config('HTTPProxy')) {
			$ua->proxy('http', $proxy);
		}
		$ua->timeout(10);

		my $request = HTTP::Request->new(
			'GET', $task->{url}, HTTP::Headers->new(
				'Range' => 'bytes=' . $bytes,
			),
		);

		#my $response = $ua->get($task->{url});
		my $response = $ua->request($request);
        if ($response->is_success) {
			my $filename = File::Spec->catfile(
				$upgrade->{temp_dir},
				$task->{filename} . '.' . $index
			);
			open(my $fh, '>', $filename);
			print($fh $response->content);
		}
		else {
			return;
		}
	}
	elsif ($type eq 'install') {
		$do_install->();
	}
	elsif ($type eq 'remove') {
		$do_remove->();
	}
	elsif ($type eq 'install') {
		$do_remove->();
		$do_install->();
	}

	$task->{upgrade_required} = &is_upgrade_required;

	require JSON;
	my $json = JSON::objToJson($task);
	return $json;
}

sub create_task_common {
	my ($app, $linker, $set, $hash, $pkg, $ver) = @_;
	my $app_uri = $app->app_uri;
	my $task = {
		signature => $pkg,
		version => $ver,
	};

	$linker->param('op', 'task');
	$linker->param('type', $set->{'type'});
	$linker->param('index', scalar(@{ $set->{tasks} }));

	local $CGI::USE_PARAM_SEMICOLONS;
	$CGI::USE_PARAM_SEMICOLONS = 0;

	$task->{action} = $app_uri . '?' . $linker->query_string;
	$task->{name} = $hash->{name};

	push(@{ $set->{tasks} }, $task);
	$task;
}

sub upgrade {
    my ($plugin, $app, $param) = @_;

	my $upgrade = {};
	my ($task_download, $task_install, $task_remove, $task_upgrade) = (
		{
			name => $plugin->translate('Download'),
			type => 'download',
			tasks => [],
		},
		{
			name => $plugin->translate('Install'),
			type => 'install',
			tasks => [],
		},
		{
			name => $plugin->translate('Uninstall'),
			type => 'remove',
			tasks => [],
		},
		{
			name => $plugin->translate('Upgrade'),
			type => 'upgrade',
			tasks => [],
		},
	);

    my $linker = new CGI({
		'__mode' => 'pluginmanager'
	});

	my $create_task_download = sub {
		my ($pkg) = @_;
		my $task = &create_task_common($app, $linker, $task_download, @_);

		my $action = $task->{action};
		delete($task->{action});
		$task->{tasks} = [];

		$task->{url} = $pkg->{'repository'} . '/' . $pkg->{'filename'};
		#$task->{filename} = basename($pkg->{'filename'}, '.mpm');
		$task->{filename} = basename($pkg->{'filename'});

		my $size = $pkg->{size};
		my $offset = 0;
		my $unit = int($plugin->get_config_value('download_unit')) || 512;
		my $split_size = $unit * 1024;

		while($offset < $size) {
			my $end = $offset + $split_size - 1;
			if ($end > $size - 1) {
				$end = $size - 1;
			}
			push(@{ $task->{tasks} }, {
				action =>
					$action . '&bytes=' . $offset . '-' . $end .
					'&download_index=' .
					sprintf('%03d', scalar(@{ $task->{tasks} }))
			});
			$offset = $end + 1;
		}

		$task;
	};

	my $create_task_upgrade = sub {
		&create_task_common($app, $linker, $task_upgrade, @_);
	};

	my $create_task_install = sub {
		my ($pkg) = @_;
		my $task = &create_task_common($app, $linker, $task_install, @_);
		#$task->{filename} = basename($pkg->{'filename'}, '.mpm');
		$task->{filename} = basename($pkg->{'filename'});

		$task;
	};

	my $create_task_remove = sub {
		my $installed = shift;
		my $control = $installed->make_control_locale(
			$app->current_language
		);

		my $task = &create_task_common($app, $linker, $task_remove, $control, @_);
	};

    my $installed_class = MT->model('pm_installed');
	my @installeds = $installed_class->load;

	my $availables = &packages_load(@_);

	my @pkgs = $app->param('packages');
	foreach my $p (@pkgs) {
		my $new = $app->param($p);
		if (! $new) {
			my ($installed) = grep({ $_->signature eq $p } @installeds);
			if ($installed) {
				# uninstall
				$create_task_remove->($installed, $p, $new);
				next;
			}
			else {
				# do nothing
				next;
			}
		}
		else {
			my ($inst) = grep({ $_->signature eq $p } @installeds);
			if ($inst && ($inst->version eq $new)) {
				# do nothing
				next;
			}

			my $new_pkg = $availables->{$p}->{$new};

			$create_task_download->($new_pkg, $p, $new);
			if (grep({ $_->signature eq $p } @installeds)) {
				# upgrade
				$create_task_upgrade->($new_pkg, $p, $new);
			}
			else {
				# install
				$create_task_install->($new_pkg, $p, $new);
			}
		}
	}

	$upgrade->{tasks} = [
		(@{ $task_download->{tasks} } ? ($task_download) : ()),
		(@{ $task_install->{tasks} } ? ($task_install) : ()),
		(@{ $task_remove->{tasks} } ? ($task_remove) : ()),
		(@{ $task_upgrade->{tasks} } ? ($task_upgrade) : ()),
	];

	$upgrade->{temp_dir} = tempdir();

	$upgrade->{upgrade_script} = $app->mt_path . $app->config->UpgradeScript;

	$app->session('pluginmanager_upgrade', $upgrade);

	require JSON;
	my $json = JSON::objToJson($upgrade);
	return $json;
}

my @config_scalar_keys = (
	'download_unit', 'ftp_host', 'ftp_user', 'ftp_directory',
);
my @config_array_keys = (
	'repositories',
);

sub ftp_prompt {
    my ($plugin, $app, $param) = @_;
	$param->{'id'} = $app->param('id');
    $plugin->load_tmpl('pm_ftp_prompt.tmpl', $param);
}

sub setting {
    my ($plugin, $app, $param) = @_;

	my $hash = $plugin->get_config_hash;

	foreach my $k (@config_scalar_keys) {
		$param->{$k} = $hash->{$k} if $hash->{$k};
	}

	foreach my $k (@config_array_keys) {
		$param->{$k} = join("\n", @{ $hash->{$k} }) if $hash->{$k};
	}

	$param->{updated} = $app->param('updated') ? 1 : 0;
	$param->{initialize} = $app->param('initialize') ? 1 : 0;

    $plugin->load_tmpl('pm_setting.tmpl', $param);
}

sub update_setting {
    my ($plugin, $app, $param) = @_;

	my $hash = $plugin->get_config_hash;

	foreach my $k (@config_scalar_keys) {
		if (my $v = $app->param($k)) {
			$hash->{$k} = $v;
		}
	}

	foreach my $k (@config_array_keys) {
		if (my $v = $app->param($k)) {
			$hash->{$k} = [ split(/\r\n|\r|\n/, $v) ];
		}
	}

	$plugin->set_config_value($hash);

    $app->redirect(
        $app->app_uri . '?__mode=pluginmanager&op=setting&updated=1&time=' . time
    );
}

sub import {
    my ($plugin, $app, $param) = @_;

	my $writable = &writable;

	if (! $app->param('time')) {
		if (
			(! $writable) && (! &has_ftp_settings)
		) {
			return $app->redirect(
				$app->app_uri . '?__mode=pluginmanager&op=setting&initialize=1'
			);
		}
	}


	$param->{listing_screen} = 1;
	$param->{imported_package} = $app->param('imported_package') ? 1 : 0;
	$param->{imported_distribution} = $app->param('imported_distribution') ? 1 : 0;
	$param->{writable} = $writable;

    $plugin->load_tmpl('pm_import.tmpl', $param);
}

sub do_import {
    my ($plugin, $app, $param) = @_;
	my ($imported, $c);

    require LWP::UserAgent;

	my $ua = LWP::UserAgent->new;
	if (my $proxy = MT->config('HTTPProxy')) {
		$ua->proxy('http', $proxy);
	}
	$ua->timeout(10);

	if (my $fh = $app->param('import_package_file')) {
		my ($temp, $filename) = tempfile();
		while (read($fh, $c, 1024)) {
			print($temp $c);
		}
		close($temp);
		close($fh);

		&install($filename);
		$imported = 'imported_package';
	}

	if (my $url = $app->param('import_package_url')) {
        my $response = $ua->get($url);
        if ($response->is_success) {
			my ($temp, $filename) = tempfile();
			print($temp $response->content);
			close($temp);

			&install($filename);
        }
        else {
            die $response->status_line;
        }
		$imported = 'imported_package';
	}

	if (my $fh = $app->param('import_distribution_file')) {
		my ($temp, $filename) = tempfile();
		while (read($fh, $c, 1024)) {
			print($temp $c);
		}
		close($temp);
		close($fh);

		&install_distribution($filename);
		$imported = 'imported_distribution';
	}

	if (my $url = $app->param('import_distribution_url')) {
        my $response = $ua->get($url);
        if ($response->is_success) {
			my ($temp, $filename) = tempfile();
			print($temp $response->content);
			close($temp);

			&install_distribution($filename);
        }
        else {
            die $response->status_line;
        }
		$imported = 'imported_distribution';
	}

    $app->redirect(
        $app->app_uri . '?__mode=pluginmanager&op=import&' . $imported . '=1&time=' . time
    );
}

sub export {
    my ($plugin, $app, $param) = @_;

	my $cwd = getcwd;

	require PluginManager::Repository;
	my $url = PluginManager::Repository::repository_url();

	my $pool = PluginManager::Repository::repository_pool();
	chdir($pool);
	my @pkgs = map({
		my $hash = {};
		$hash = {
			'filename' => $_,
			'url' => $url . 'pool/' . $_,
		};
		$hash;
	} glob(File::Spec->catfile('*.mpm')));
	chdir($cwd);
	$param->{'packages'} = \@pkgs if @pkgs;

	my $root = PluginManager::Repository::repository_root();
	chdir($root);
	my @dists = map({
		my $hash = {};
		$hash = {
			'filename' => $_,
			'url' => $url . $_,
		};
		$hash;
	} glob(File::Spec->catfile('*.mpd')));
	chdir($cwd);
	$param->{'distributions'} = \@dists if @dists;

	$param->{'available_components'} = [ map({
		my $c = $app->component($_);
		my %hash = ();
		$hash{$_} = $c->$_ for ('id', 'name');
		\%hash;
	} PluginManager::Repository::available_components()) ];

	$param->{listing_screen} = 1;
	$param->{created_package} = $app->param('created_package') ? 1 : 0;
	$param->{created_distribution} = $app->param('created_distribution') ? 1 : 0;

    $plugin->load_tmpl('pm_export.tmpl', $param);
}

sub do_export {
    my ($plugin, $app, $param) = @_;

	my $created = 'created_package';

	my @packages = $app->param('packages');

	require PluginManager::Command;
	my $maker = PluginManager::Command::MakePackage->new;
	$maker->process({'q' => 1}, @packages);

	if ($app->param('distribution')) {
		my $updater = PluginManager::Command::MakeDistribution->new;
		$updater->process();
		my $created = 'created_distribution';
	}

    $app->redirect(
        $app->app_uri . '?__mode=pluginmanager&op=export&' . $created . '=1&time=' . time
    );
}

sub repository {
    my ($plugin, $app, $param) = @_;

	require PluginManager::Repository;
	$param->{'repository_url'} = PluginManager::Repository::repository_url();
	$param->{'available_components'} = [ map({
		my $c = $app->component($_);
		my %hash = ();
		$hash{$_} = $c->$_ for ('id', 'name');
		\%hash;
	} PluginManager::Repository::available_components()) ];

	$param->{listing_screen} = 1;
	$param->{created} = $app->param('created') ? 1 : 0;

    $plugin->load_tmpl('pm_repository.tmpl', $param);
}

sub create_repository {
    my ($plugin, $app, $param) = @_;

	my @packages = $app->param('packages');

	require PluginManager::Command;
	my $maker = PluginManager::Command::MakePackage->new;
	$maker->process({'q' => 1}, @packages);

	my $updater = PluginManager::Command::UpdateRepository->new;
	$updater->process();

    $app->redirect(
        $app->app_uri . '?__mode=pluginmanager&op=repository&created=1&time=' . time
    );
}

1;
