package jp.gr.java_conf.ka_ka_xyz.nyatla.nymmd;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;

import android.graphics.Bitmap;
import android.graphics.BitmapFactory;
import jp.gr.java_conf.ka_ka_xyz.exception.MMDARException;
import jp.gr.java_conf.ka_ka_xyz.exception.TooHugeFileException;
import jp.gr.java_conf.ka_ka_xyz.exception.UnsupportedFileTypeException;
import jp.gr.java_conf.ka_ka_xyz.util.Log;

import jp.nyatla.nymmd.IMmdDataIo;

public class MMDModelInfo implements IMmdDataIo {

	private String vmdId;
	private String pmdId;

	private InputStream vmdIs;
	private InputStream pmdIs;
	
	private static final long MAX_FILE_SIZE = 5000000;
	
	private HashMap<String, Bitmap> textureCache = new HashMap<String, Bitmap>();

	// TODO 複数のテクスチャ
	public MMDModelInfo(String pmdId, String vmdId) {
		this.vmdId = vmdId;
		this.pmdId = pmdId;

		File pmdFile = new File(pmdId);
		
		try {
			pmdIs = new FileInputStream(pmdFile);
		} catch (FileNotFoundException e) {
			throw new IllegalStateException(e);
		}

		File vmdFile = new File(vmdId);
		if(MAX_FILE_SIZE < vmdFile.length()){
			throw new TooHugeFileException(vmdFile.getName() + " is over " + MAX_FILE_SIZE + " byte.");
		}

		try {
			vmdIs = new FileInputStream(vmdFile);
		} catch (FileNotFoundException e) {
			throw new IllegalStateException(e);
		}

	}

	public Bitmap getTextureBitmap(String i_name) {

		Bitmap bitmap = this.textureCache.get(i_name);

		if (bitmap == null) {
			File pmd = new File(pmdId);
			File texturePath = pmd.getParentFile();
			File texture = new File(texturePath, i_name);
			InputStream is;
			try {
				is = new FileInputStream(texture);
				bitmap = normalizeBitmap(BitmapFactory.decodeStream(is));
				this.textureCache.put(i_name, bitmap);
				is.close();
			} catch (FileNotFoundException e) {
				throw new UnsupportedFileTypeException(e);
			} catch (Exception e) {
				throw new MMDARException(e);
			}
		}

		return bitmap;
	}

	private Bitmap normalizeBitmap(Bitmap ori) {

		if (ori == null) {
			return null;
		}
		if (isNormarizedBitmap(ori)) {
			return ori;
		}
		int length = getNormarizedLength(ori.getWidth(), ori.getHeight());
		Log.d("MMDINFO", "length: " + length);
		Bitmap newBitmap = Bitmap.createScaledBitmap(ori, length, length, true);
		return newBitmap;
	}

	private static int getNormarizedLength(int x, int y) {

		int length = Math.max(x, y);
		int rtn = 1;
		final int two = 2;
		for (int pow = 2; pow < 15; pow++) {
			rtn = (int) Math.pow(two, pow);
			if (length < rtn + 1) {
				return rtn;
			}
		}
		return rtn;
	}

	private static boolean isPowerOfTwo(int x) {
		return x > 0 && (x & (x - 1)) == 0;
	}

	private static boolean isNormarizedBitmap(Bitmap bitmap) {
		int width = bitmap.getWidth();
		int height = bitmap.getHeight();

		if (bitmap != null && width == height && isPowerOfTwo(width)) {
			return true;
		}
		return false;
	}

	@Override
	public InputStream getPMDIS() {
		return this.pmdIs;
	}

	@Override
	public InputStream getVMDIS() {
		return this.vmdIs;
	}

	@Override
	public String toString() {
		return "MMDModelInfo [vmdId=" + vmdId + ", pmdId=" + pmdId + "]";
	}

	@Override
	public void finalize() {
		try {
			super.finalize();
			this.pmdIs.close();
			this.vmdIs.close();
			for (String key : textureCache.keySet()) {
				textureCache.get(key).recycle();
			}
		} catch (Throwable e) {
			throw new RuntimeException(e);
		}
	}

}