# -*- coding: utf-8 -*-

module Mint::Generator

  #
  # 二次方程式の問題を生成するジェネレータ
  #
  # == オプション
  # [_answer_type_]
  #   解の種類を指定します。
  #   以下の３種類から１つ指定します。
  #   [_rational_]
  #     整数解を生成します。
  #   [_irrational_]
  #     無理数解を生成します。
  #   [_imaginary_]
  #     虚数解を生成します。
  # [_r_min_]
  #   r の最小値を１以上の整数で指定します。
  # [_r_max_]
  #   r の最大値を１以上の整数で指定します。
  #   素数を指定することは出来ません。
  # [_p_min_]
  #   p の最小値を０以上の整数でしていします。
  # [_p_max_]
  #   p の最大値を１以上の整数で指定します。
  #   素数を指定することは出来ません。
  # [_p_minus_]
  #   真を指定すると、p の値として負の値も生成します。
  # [_q_min_]
  #   q の最小値を０以上の整数で指定します。
  # [_q_max_]
  #   q の最大値を１以上の整数で指定します。
  #   素数を指定することは出来ません。
  #   また平方数の数列に属する値を指定することも出来ません。
  #
  # == r, p, q について
  # r, p, q を用いて 係数 a, b, c を生成します.
  # それぞれ以下の式で求めることができます.
  # - a' = r^2
  # - b' = -2pr
  # - c' = (p^2 - q)
  # - gcd = GCD(a', b', c')
  # とすると
  # - a = a' / gcd
  # - b = b' / gcd
  # - c = c' / gcd
  #
  class QuadraticEquation < Base

    private

    include Utilities

    option :answer_type, :rational
    option :r_min,       1
    option :r_max,       20
    option :p_min,       0
    option :p_max,       12
    option :p_minus,     true
    option :q_min,       0
    option :q_max,       40

    def generate_problem
      expression
    end

    def expression
      r, p ,q = generate_quadeqn_params(options[:answer_type])
      a, b, c = generate_coefficients(r, p, q)
      term1(a) + term2(b) + term3(c)
    end

    def generate_coefficients(r, p, q)
      a = r * r
      b = -2 * p * r
      c = p * p - q
      gcd = a.gcd(b).gcd(c)

      a /= gcd
      b /= gcd
      c /= gcd

      [a, b, c]
    end

    def term1(a)
      case
        when a == 1  then 'x^2'
        when a == -1 then '-x^2'
        when a == 0  then ''
        else;             "#{a} * x^2"
      end
    end

    def term2(b)
      case
        when b == 1  then " + x"
        when b == -1 then " - x"
        when b > 0   then " + #{b} * x"
        when b < 0   then " - #{-b} * x"
        when b == 0  then ''
      end
    end

    def term3(c)
      case
        when c == 0 then ''
        when c > 0  then " + #{c}"
        else;            " - #{-c}"
      end
    end

    def generate_quadeqn_params(type)
      r = generate_r
      p = generate_p
      q = __send__(:"#{type}_q")

      [r, p, q]
    end

    def generate_r
      min = [1, options[:r_min]].max
      create_integer(min, options[:r_max], false)
    end

    def generate_p
      create_integer(options[:p_min], options[:p_max], options[:p_minus])
    end

    def rational_q
      squares(options[:q_min], options[:q_max]).sample
    end

    def irrational_q
      min = [0, options[:q_min]].max
      squares = squares(options[:q_min], options[:q_max])
      create_integer(min, options[:q_max], false) {|qa|
        !squares.include?(qa)
      }
    end

    def imaginary_q
      min = [1, options[:q_min]].max
      -create_integer(min, options[:q_max], false)
    end

    def squares(min, max)
      min, max = [min, max].sort
      (min..max).to_a.map {|i| i * i }
    end
  end
end

