# -*- coding: utf-8 -*-

module Mint

  class ExpressionNode # :nodoc:
    attr_reader :left, :right, :operator
    attr_accessor :minus, :parenthesis
    attr_accessor :parent
    alias parenthesis? parenthesis
    alias minus? minus

    def initialize(left, right)
      @left     = left
      @right    = right
      @operator = :unknown
      left.parent  = self
      right.parent = self

      @minus       = false
      @parenthesis = false
    end
    def to_s
      show to_s_local
    end
    def to_latex
      self.parenthesis = false if self.instance_of?(FractionNode)
      result = show to_latex_local
      self.parenthesis = true if self.instance_of?(FractionNode)
      result
    end
    def naked
      result = self.dup
      result.parenthesis = false
      result
    end
    private
    def to_s_local
      "#{left} #{operator} #{right}"
    end
    def to_latex_local
      "#{left.to_latex} #{operator} #{right.to_latex}"
    end
    def show(string)
      if minus
        return "-(#{string})" if self.parent.instance_of?(FractionNode)
        return "(-(#{string}))"
      end
      return "(#{string})" if parenthesis
      string
    end
  end

  class AdditionNode < ExpressionNode # :nodoc:
    def initialize(left, right)
      super
      @operator = :+
    end
  end

  class SubtractionNode < ExpressionNode # :nodoc:
    def initialize(left, right)
      super
      @operator = :-
    end
    def to_s_local
      "#{left} #{operator} #{right}"
    end
  end

  class MultipleNode < ExpressionNode # :nodoc:
    def initialize(left, right)
      super
      @operator = :*
    end
    def to_latex_local
      _right = right.to_latex
      if _right.match(/\A\(*-?\d/) && !_right.match(/[^(]+[+\-\*\/]|div/)
        return "#{left.to_latex} \\times #{_right}"
      end
      "#{left.to_latex}#{_right}"
    end
  end

  class DivisionNode < ExpressionNode # :nodoc:
    def initialize(left, right)
      super
      @operator = :div
    end
    def to_latex_local
      "#{left.to_latex} \\div #{right.naked.to_latex}"
    end
  end

  class FractionNode < ExpressionNode # :nodoc:
    def initialize(left, right)
      super
      @operator = :/
    end
    def to_latex_local
      result = "\\frac{#{left.naked.to_latex}}{#{right.naked.to_latex}}"
    end
  end

  class LiteralNode # :nodoc:
    attr_accessor :value, :minus
    attr_accessor :parenthesis, :parent
    alias parenthesis? parenthesis
    alias minus? minus

    def initialize(value)
      @value = value

      @minus       = false
      @parenthesis = false
    end
    def to_s
      show to_s_local
    end
    def to_latex
      show to_latex_local
    end
    def naked
      result = self.dup
      result.parenthesis = false
      result
    end
    private
    def to_s_local
      @value.to_s
    end
    def to_latex_local
      @value.to_s
    end
    def show(string)
      if minus
        return "-#{string}" if self.parent.instance_of?(FractionNode)
        return "(-#{string})"
      end
      return "(#{string})"  if parenthesis
      string
    end
  end

  class DecimalNode < LiteralNode # :nodoc:
    attr_accessor :value
    def initialize(value)
      @value = value
    end
    def to_s_local
      round_n(value.to_f, 3).to_s
    end

    private
    def round_n(num, nth)
      result = num * (10 ** nth)
      (result.round() * (10 ** -nth)).to_f
    end
  end

  class FactorialNode < LiteralNode # :nodoc:
    attr_reader :value, :power
    def initialize(value, power)
      super value
      @power = power
    end
    def to_s_local
      _power = filter_minus(power) {|v| v.to_s }
      "#{value}^#{_power}"
    end
    def to_latex_local
      return to_s if value.instance_of?(String)
      _power = filter_minus(power) {|v| v.to_latex }
      "#{value.to_latex}^#{_power}"
    end
    def filter_minus(value)
      if value.instance_of?(LiteralNode) && value.minus?
        return "-#{value.__send__(:to_s_local)}"
      end
      yield value
    end
  end

  class RootNode < LiteralNode # :nodoc:
    attr_accessor :minus
    attr_reader :value
    def to_s_local
      "sqrt(#{value})"
    end
    def to_latex_local
      "\\sqrt{#{value}}"
    end
  end
end

