#
# Cookbook Name:: nagios
# Recipe:: default
#
# Copyright 2013, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

root_cfg_dir = nil
nagios_pkgs = nil
nagios_service = nil
httpd_service = nil
site_cfg_dir = nil

case node[:platform_family]
  when 'debian'
    root_cfg_dir = '/etc/nagios3'
    nagios_pkgs = [
      'nagios3',
      'nagios-plugins',
      'nagios-images',
    ]
    if node['nagios']['web']['AuthType'] == 'Kerberos' then
      nagios_pkgs.push('libapache2-mod-auth-kerb')
    end
    nagios_service = 'nagios3'
    httpd_service = 'apache2'
    httpd_nagios_conf = "#{root_cfg_dir}/apache2.conf"
  when 'rhel'
    root_cfg_dir = '/etc/nagios'
    nagios_pkgs = [
      'nagios',
      'nagios-plugins',
    ]
    nagios_pkgs = nagios_pkgs + \
      %w{load users ping ssh ntp http disk swap procs file_age}.map {|item|
        "nagios-plugins-#{item}"
      }
    if node['nagios']['web']['AuthType'] == 'Kerberos' then
      nagios_pkgs.push('mod_auth_kerb')
    end
    nagios_service = 'nagios'
    httpd_service = 'httpd'
    httpd_nagios_conf = '/etc/httpd/conf.d/nagios.conf'
end

relative_root_cfg_dir = root_cfg_dir.sub(%r{^/}, '')
site_cfg_dir = "#{root_cfg_dir}/#{node['nagios']['cluster_name']}"

nagios_pkgs.each {|pkg|
  resources(:package => pkg) rescue package pkg do
    action :install
  end
}

service nagios_service do
  action [:enable]
  supports :status => true, :restart => true, :reload => true
end

[
  'nagios.cfg',
  'cgi.cfg',
].each {|cfg_file|
  template "/#{relative_root_cfg_dir}/#{cfg_file}" do
    source  "#{relative_root_cfg_dir}/#{cfg_file}"
    owner 'root'
    group 'root'
    mode '0644'
    variables({
      'cfg_dir' => site_cfg_dir
    })
    notifies :restart, "service[#{nagios_service}]"
  end
}

resources(:service => httpd_service) rescue service httpd_service do
  action [:enable]
  supports :status => true, :restart => true, :reload => true
end

template httpd_nagios_conf do
  source httpd_nagios_conf.sub(%r{^/}, '')
  owner 'root'
  group 'root'
  mode '0644'
  notifies :restart, "service[#{httpd_service}]"
end

if node['nagios']['check_external_commands'] == '1' then
  case node[:platform_family]
  when 'debian'
    bash 'modify_permissions_for_check_external_commands' do
      code <<-EOC
        service #{nagios_service} stop
        override='nagios www-data 2710 /var/lib/nagios3/rw'
        dpkg-statoverride --list | grep "$override"
        if [ $? -ne 0 ]; then
          dpkg-statoverride --update --add $override
        fi
        override='nagios nagios 751 /var/lib/nagios3'
        dpkg-statoverride --list | grep "$override"
        if [ $? -ne 0 ]; then
          dpkg-statoverride --update --add $override
        fi
        #service #{nagios_service} start
      EOC
      notifies :restart, "service[#{nagios_service}]"
    end
  when 'rhel'
    bash 'modify_permissions_for_check_external_commands' do
      code <<-EOC
        service #{nagios_service} stop
        chown nagios:apache /var/spool/nagios/cmd
        chmod 2710 /var/spool/nagios/cmd
        chmod  751 /var/spool/nagios
        #service #{nagios_service} start
      EOC
      notifies :restart, "service[#{nagios_service}]"
    end
  end
end

if node['nagios']['check_ganglia_metric']['enabled'] then
  pkg = 'python-setuptools'
  resources(:package => pkg) rescue package pkg do
    action :install
  end

  if node.platform_family?('rhel') then
    bash 'easy_install_upgrade_distribute' do
      code <<-EOC
        easy_install -U distribute
      EOC
    end
  end
  
  easy_install_package 'check_ganglia_metric' do
    action :install
  end

  easy_install_package 'NagAconda' do
    version node['nagios']['NagAconda']['version']
    action :install
  end
end

directory site_cfg_dir do
  owner 'root'
  group 'root'
  mode '0755'
  action :create
  recursive true
end

[
  'hosts.cfg',
  'services.cfg',
  'contacts.cfg',
  'timeperiods.cfg',
  'checkcommands.cfg',
].each {|cfg_file|
  template "#{site_cfg_dir}/#{cfg_file}" do
    source "etc/nagios/CLUSTER_NAME/#{cfg_file}"
    owner 'root'
    group 'root'
    mode '0644'
    notifies :restart, "service[#{nagios_service}]"
  end
}

