#
# Cookbook Name:: hdp
# Recipe:: yum_repo
#
# Copyright 2013, whitestar
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

def get_repo_urls(hdp_version, node_platform, node_platform_version)
  platform = nil
  case node_platform
    when 'centos', 'rhel'
      platform = "centos#{node_platform_version.split('.')[0]}"
    when 'suse'
      platform = 'suse11'
  end

  hdp_repo_ctx = nil
  ambari_repo_ctx = nil
  case hdp_version
    when '2.0.6.0'  # GA (hadoop-2.2.0)
      hdp_repo_ctx = "HDP/#{platform}/2.x/updates/#{hdp_version}"
      ambari_repo_ctx = nil
    when '2.0.5.0'  # Beta (hadoop-2.1.0)
      hdp_repo_ctx = (node_platform == 'suse') ? nil : "HDP/#{platform}/2.x/updates/#{hdp_version}"
      ambari_repo_ctx = nil
    when '2.0.4.0'  # Community Preview (hadoop-2.1.0)
      hdp_repo_ctx = (node_platform == 'suse') ? nil : "HDP/#{platform}/2.x/GA"
      ambari_repo_ctx = nil
    when '2.0.0.2'  # Alpha (hadoop-2.0.3-alpha)
      hdp_repo_ctx = (node_platform == 'suse') ? nil : "HDP-#{hdp_version}/repos/#{platform}"
      ambari_repo_ctx = nil
    when '2.0.0.1'  # Alpha (hadoop-2.0.2-alpha)
      hdp_repo_ctx = (node_platform == 'suse') ? nil : "HDP-#{hdp_version}/repos/#{platform}"
      ambari_repo_ctx = nil
    when '1.3.2'  # (hadoop-1.2.0)
      hdp_repo_ctx = "HDP/#{platform}/1.x/GA"
      ambari_repo_ctx = "ambari/#{platform}/1.x/updates/1.2.5.17"
    when '1.3.1'  # (hadoop-1.2.0)
      hdp_repo_ctx = "HDP/#{platform}/1.x/GA/1.3.0.0"
      ambari_repo_ctx = "ambari/#{platform}/1.x/updates/1.2.4.9"
    when '1.3.0'
      hdp_repo_ctx = "HDP/#{platform}/1.x/GA/1.3.0.0"
      ambari_repo_ctx = "ambari/#{platform}/1.x/updates/1.2.3.7"
    when '1.2.4', '1.2.3.1', '1.2.3'  # (hadoop-1.1.2)
      normalized_platform = (node_platform == 'suse') ? 'centos5' : platform
      hdp_repo_ctx = "HDP/#{normalized_platform}/1.x/updates/1.2.1"
      ambari_repo_ctx = "ambari/#{normalized_platform}/1.x/updates/1.2.2.4"
    when '1.2.2', '1.2.1'
      hdp_repo_ctx = "HDP-1.2.0/repos/#{platform}"
      ambari_repo_ctx = "ambari/#{platform}/1.x/GA"
    when '1.2.0'
      hdp_repo_ctx = "HDP-1.2.0/repos/#{platform}"
      ambari_repo_ctx = nil
    else
      Chef::Application.fatal!("Non supported version: #{hdp_version}")
  end

  download_site_url = node['hdp']['archive_url']

  return [
    (hdp_repo_ctx.nil?) ? nil : "#{download_site_url}/#{hdp_repo_ctx}/hdp.repo",
    (ambari_repo_ctx.nil?) ? nil : "#{download_site_url}/#{ambari_repo_ctx}/ambari.repo"
  ]
end


platform = nil
platform_version = nil

if node['hdp']['yum_repo']['only_for_mirroring'] then
  platform = node['hdp']['yum_repo']['mirroring']['platform']
  platform_version = node['hdp']['yum_repo']['mirroring']['platform_version']
else
  unless node[:kernel][:machine] == 'x86_64' then
    Chef::Application.fatal!("Non supported architecture: #{node[:kernel][:machine]}")
  end
  platform = node[:platform]
  platform_version = node[:platform_version]
end

if node['hdp']['yum_repo']['only_for_mirroring'] \
  || node[:platform_family] == 'rhel' then
  hdp_repo_file = nil
  ambari_repo_file = nil

  case node[:platform]
  when 'centos', 'rhel'
    hdp_repo_file = '/etc/yum.repos.d/hdp.repo'
    ambari_repo_file = '/etc/yum.repos.d/ambari.repo'
  when 'suse'
    hdp_repo_file = '/etc/zypp/repos.d/hdp.repo'
    ambari_repo_file = '/etc/zypp/repos.d/ambari.repo'
  when 'debian'  # for mirroring only.
    pkg = 'yum'
    resources(:package => pkg) rescue package pkg do
      action :install
    end
    hdp_repo_file = '/etc/yum/repos.d/hdp.repo'
    ambari_repo_file = '/etc/yum/repos.d/ambari.repo'
  end

  hdp_version = node['hdp']['version']
  hdp_repo_url, ambari_repo_url \
    = get_repo_urls(hdp_version, platform, platform_version)

  hdp_repo_add_command \
    = (hdp_repo_url.nil?) ? '' : "wget -nv #{hdp_repo_url} -O #{hdp_repo_file}"
  ambari_repo_add_command \
    = (ambari_repo_url.nil?) ? '' : "wget -nv #{ambari_repo_url} -O #{ambari_repo_file}"

  pkg = 'wget'
  resources(:package => pkg) rescue package pkg do
    action :nothing
  end.run_action(:install)  # at compile-time

  if node['hdp']['yum_repo']['update'] then
    [
      hdp_repo_file,
      ambari_repo_file
    ].each {|repo_file|
      file repo_file do
        action :nothing
      end.run_action(:delete)  # at compile-time
    }
  end

  bash "add_hdp_yum_repositories-#{hdp_version}" do
    code <<-EOC
      #{hdp_repo_add_command}
      #{ambari_repo_add_command}
    EOC
    creates hdp_repo_file unless hdp_repo_url.nil?
    creates ambari_repo_file unless ambari_repo_url.nil?
    action :nothing
  end.run_action(:run)  # at compile-time
end
