#ifndef QT_IO_DEVICES_HPP_
#define QT_IO_DEVICES_HPP_
/*
 * qio_devices.hpp
 *
 *  Created on: 13 янв. 2020 г.
 *      Author: alexrayne <alexraynepe196@gmail.com>
  ------------------------------------------------------------------------
    Copyright (c) alexrayne

   All rights reserved.
   Redistribution and use in source and binary forms, with or without
   modification, are permitted provided that the following conditions are met:
   - Redistributions of source code must retain the above copyright
     notice, this list of conditions and the following disclaimer.
   - Redistributions in binary form must reproduce the above copyright
     notice, this list of conditions and the following disclaimer in the
     documentation and/or other materials provided with the distribution.
   - Neither the name of ARM nor the names of its contributors may be used
     to endorse or promote products derived from this software without
     specific prior written permission.
   *
   THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
   AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
   IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
   ARE DISCLAIMED. IN NO EVENT SHALL COPYRIGHT HOLDERS AND CONTRIBUTORS BE
   LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
   CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
   SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
   INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
   CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
   ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
   POSSIBILITY OF SUCH DAMAGE. *
  ------------------------------------------------------------------------
 * this is a Qt StdIO devices
 */

#include <QObject>
#include "stdio_hal.hpp"




/* QConsoleIODevice provide StdIO_Device via Qt signaling.
 * it transfers io() in/out to signals
 *      on_input(QByteArray)/signal_output(QByteArray)
 *
 * Usage:
    io_console = new QConsoleIODevice();
    connect(&ui_console, &Console::getData, io_console, &QConsoleIODevice::on_input );
    connect(io_console, &QConsoleIODevice::signal_output, &ui_console, &Console::putData );

    ui_console.setEnabled(true);

    _shell = QShell::newShell( io_console, "testshell" );
    connect(_shell, &QShell::finished, this, &TestShell::on_buttonBox_accepted );
    shell().take_own( std::move(*io_console) );
    shell().start();
 *
 * */

#include <QQueue>
#include <QSemaphore>
#include "lib/cli/stdio_hal.hpp"
#include "lib/cli/hal_ringbuffer.h"

class QConsoleIODeviceBase:public QObject
                    , public StdIO_Device
{
    Q_OBJECT

public:
    typedef StdIO_Device inherited_io;
    QConsoleIODeviceBase(void* buf, unsigned bufsize, QObject *parent = nullptr);

    //QBuffer& in() {return cin ;}

public:// slots:
    void on_input(const QByteArray &data);

signals:
    void signal_output(const QByteArray data);

public: //StdIO_Device
    virtual int get_char() override;
    virtual int get_wait(unsigned to = 0) override;

public:     //StdOUT_Device
    //* блокирующая печать
    virtual int putChar(int ch) override;
    //*  неблокирующая печать
    virtual int postData ( const void* str, unsigned len) override;

public:     //HAL_OUTBlock_Device
    //* блокирующая печать
    virtual int puts( const char* str) override;
    //*  \return - длинна отправленного участка
    virtual int putData ( const void* str, unsigned len) override;

    //*  ожидание доступности печати
    //*  \return - количество байт возможных для неблокирующей печати
    virtual int put_wait(unsigned to = 0) override { (void)to; return (~0u>>1);}
    //*  почти тоже put_wait, ждет полного опустошения
    virtual int put_flush(unsigned to = 0) override { (void)to; return 0;}
    //*  очищает буфер, прерывая текущую отправку
    virtual int put_drop() override {return 0;}

    //* монополизация вывода (puts, putData предпочтительно использую ее )
    //* \arg onoff - захват/освобождение
    //* \return    - состояние захвачн ли вывод
    virtual bool put_access(bool onoff, unsigned to = toInfinite) override
    {
        (void)onoff; (void)to;
        return true;
    }

protected:
    QSemaphore avail;
    // хороший буффер без конкуренции писатель/читатель. примитивы QT - гуано
    HAL_RingBuffer cin;
};


template<unsigned bufsize = 64>
class QConsoleIODevice : public QConsoleIODeviceBase {
public:
    QConsoleIODevice(QObject *parent = nullptr)
        :QConsoleIODeviceBase(cin_buf, bufsize, parent)
    {}

protected:
    uint8_t     cin_buf[bufsize];
};



//----------------------------------------------------------------------------------

//#include <QtNetwork>
#include <QTcpSocket>

class QSockIODevice : public QTcpSocket
                    , public StdIO_Device
                    , public HAL_INBlock_Device
{
    Q_OBJECT
public:
    typedef QTcpSocket inherited;
    typedef qintptr sockid_t;

    QSockIODevice(QObject *parent = nullptr);
    QSockIODevice(sockid_t id, QObject *parent = nullptr);
    virtual ~QSockIODevice();

    bool init(sockid_t x);

public: //StdIO_Device
    virtual int get_char();
    virtual int get_wait(unsigned to = 0);

public: //HAL_INBlock_Device
    // блокирующий прием
    //*  \return - длинна полученного участка
    virtual int getData ( void* dst, unsigned len);
    //*  ожидание amount доступных символов
    virtual int get_waitfor(unsigned amount, unsigned to = 0);

public:     //StdOUT_Device
    //* блокирующая печать
    virtual int putChar(int ch);
    //*  неблокирующая печать
    virtual int postData ( const void* str, unsigned len);

public:     //HAL_OUTBlock_Device
    //* блокирующая печать
    virtual int puts( const char* str);
    //*  \return - длинна отправленного участка
    virtual int putData ( const void* str, unsigned len);

    //*  ожидание доступности печати
    //*  \return - количество байт возможных для неблокирующей печати
    virtual int put_wait(unsigned to = 0);
    //*  почти тоже put_wait, ждет полного опустошения
    virtual int put_flush(unsigned to = 0);
    //*  очищает буфер, прерывая текущую отправку
    virtual int put_drop();

    //* монополизация вывода (puts, putData предпочтительно использую ее )
    //* \arg onoff - захват/освобождение
    //* \return    - состояние захвачн ли вывод
    virtual bool put_access(bool onoff, unsigned to = toInfinite){
        (void)onoff; (void)to;
        return true;
    }
};



#endif /* LINUX_SOCKIO_DEVICE_HPP_ */
