/*
 * r_i2c_masterx.c
 *
 *  Created on: 9/04/2021
 *      Author: alexraynepe196@gmail.com
 * ----------------------------------------------------------------------
 * драйвер устройства i2c с внутренней адресацией
 */

#include "r_i2c_masterx.h"
#include <stdbool.h>


//< handle i2s transfer by I2C_status() invoke
#define I2C_ISR_POLLING 0
//< handle i2s transfer by i2c ISR events
#define I2C_ISR_ACTIVE  1

#ifndef I2C_ISR_STYLE
#define I2C_ISR_STYLE   I2C_ISR_POLLING
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#if 1
#include <assert.h>
#define ASSERT(...) assert(__VA_ARGS__)
#else
#define ASSERT(...)
#endif


//#include <trace.h>
//#include <tracelog.h>
#if (0)
#define I2C_puts(s)     debug_puts(s)
#define I2C_putchar(c)  debug_putchar(0, c)
#define I2C_printf(...) debug_printf(__VA_ARGS__)
#elif (0)
#define I2C_puts(s)     puts(s)
#define I2C_putchar(c)  putchar(c)
#define I2C_printf(...) printf(__VA_ARGS__)
#else
#define I2C_puts(s)
#define I2C_putchar(c)
#define I2C_printf(...)
#endif


///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///
#include <trace_probes.h>
#ifndef trace_i2c_start
//* это пробы-сигналы тестовых пинов, для визуализации обмена на осцилографе
trace_need(i2c_start)
trace_need(i2c_work)
trace_need(i2c_irq)
#endif
#ifndef trace_i2c_adr
trace_need(i2c_adr)
#endif
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///
struct i2c_descriptor;
typedef struct i2c_descriptor i2c_descriptor;


enum i2c_io_state{
      i2cio_IDLE
    , i2cio_ADDR
    , i2cio_XSLA
    , i2cio_XRECV
    , i2cio_XSEND
    , i2cio_BREAK
};
typedef enum i2c_io_state i2c_io_state;


enum i2c_ioDModes{
      i2co_IDLE       = 0
    , i2co_XMIT       = 2
    , i2co_DRECV      = i2co_XRECV | i2co_XMIT
    , i2co_DSEND      = i2co_XSEND | i2co_XMIT
};


struct i2c_descriptor
{
    I2C_basic       base;

    unsigned char   file_sla;   //< stdio::read/write sla
    unsigned char   sla;
    uint32_t        addr;

    //автомат I2C_xfer
    i2c_io_state    state;
    i2c_io_option   mode;
    int             err;

    const unsigned char*    adata;
    unsigned                alen;
    unsigned char*          xdata;
    unsigned                xlen;

    I2C_on_event    ev;
    void*           ev_ctx;
};



///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///


#if 1
// system have single i2c
#include "r_iic_master.h"

i2c_descriptor      I2C_port;

#define SELFARG
#define SELF        (I2C_port)

#define API(method) (g_i2c_master_on_iic.method)

#else
// system use multiple i2c
#define SELFARG     i2c_descriptor* self,
#define SELF        (*self)

#define API(method) (*api->method)

#endif


#ifndef CODE_ISR
#ifdef __HOTFUNC
#define CODE_ISR    __HOTFUNC
#else
#define CODE_ISR
#endif
#endif

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////
///

void I2C_OnEvent(i2c_master_callback_args_t *);

static
void i2c_reset(i2c_descriptor* io){
    io->state   = i2cio_IDLE;
    io->err     = FSP_SUCCESS;
};

i2cHandle I2C_init_master(/*I2C_REGS_TYPE*/ const void* io){ //, i2c_descriptor* handle
    SELF.base.port = (i2c_master_instance_t*)io;
    //i2c_master_api_t*   api = SELF.base.port->p_api;
    i2c_master_ctrl_t*  port= SELF.base.port->p_ctrl;

    SELF.ev = NULL;

    I2C_Status ok;
    i2c_reset(&SELF);
    ok = API(open)(port, SELF.base.port->p_cfg);
    ASSERT(ok == FSP_SUCCESS);
    ok = API(callbackSet)(port, I2C_OnEvent, &SELF, NULL);
    ASSERT(ok == FSP_SUCCESS);
    (void)ok;
    return &(SELF.base);
}

I2C_Status I2C_close_master(i2cHandle h){
    if (I2C_status(h) != (int)I2C_BUSY)
        return I2C_OK;

    return I2C_abort(h);
}


void I2C_assign_handler(i2cHandle h, I2C_on_event ev, void* ctx){
    i2c_descriptor*     self = (i2c_descriptor*)(h->port);
    (void) self;
    i2c_descriptor*     io  = &SELF;

    io->ev      = ev;
    io->ev_ctx  = ctx;
}

static inline
void I2C_notify(i2c_descriptor*     self, i2c_master_event_t ev){
    if ( SELF.ev != NULL)
        (SELF.ev)(&(SELF.base), ev, SELF.ev_ctx);
}

static
void I2C_finale(i2c_descriptor*     self, i2c_master_event_t ev){
    I2C_notify(self, ev);
    SELF.state = i2cio_IDLE;
}



I2C_Status I2C_xfer_start(i2cHandle h, unsigned char SLA);

static
void I2C_data_assign(void* data, unsigned dlen){
    i2c_descriptor*     io  = &SELF;

    io->xlen     = dlen;
    if (io->xlen > 0){
        ASSERT(data != NULL);
        io->xdata    = (unsigned char*)data;
    }
    else
        io->xdata    = NULL;
}

/** операции и2ц с выделеным адресом. адресная часть - передается как нормальные данные
 * */
I2C_Status I2C_xfer(i2cHandle h, unsigned char SLA
          , const void* addr, unsigned alen
          , void* data, unsigned dlen
          , i2c_io_option mode
         )
{

    //i2c_descriptor*     self = (i2c_descriptor*)(h->port);
    i2c_descriptor*     io  = &SELF;
    const i2c_master_api_t*   api = io->base.port->p_api;
    //i2c_master_ctrl_t*  port= io->base.port->p_ctrl;
    (void)api;

    if ( UNLIKELY(io->state > i2cio_IDLE) )
        return I2C_BUSY;

    io->alen     = alen;
    if (alen > 0){
        ASSERT(addr != NULL);
        io->adata    = (const unsigned char*)addr;
    }
    else
        io->adata    = NULL;

    I2C_data_assign(data, dlen);
    io->mode     = mode;

    return I2C_xfer_start(h, SLA);
}



/// @brief this operations continues on last SLA, with no adress - device provide adress on it`s own
I2C_Status I2C_xfer_next(i2cHandle h, void* data, unsigned dlen, i2c_io_option mode ){
    //i2c_descriptor*     self = (i2c_descriptor*)(h->port);
    i2c_descriptor*     io  = &SELF;
    const i2c_master_api_t*   api = io->base.port->p_api;
    //const i2c_master_ctrl_t*  port= io->base.port->p_ctrl;
    (void)api;

    if ( UNLIKELY(io->state > i2cio_IDLE) )
        return I2C_BUSY;

    io->adata    = NULL;
    //io->alen     = 0;
    io->mode     = mode;
    I2C_data_assign(data, dlen);

    return I2C_xfer_start(h, io->sla);
}



I2C_Status I2C_xfer2(i2cHandle h, uint32_t SLA_mode, uint32_t addr , void* data, unsigned dlen ){
    //i2c_descriptor*     self = (i2c_descriptor*)(h->port);
    i2c_descriptor*     io  = &SELF;
    const i2c_master_api_t*   api = io->base.port->p_api;
    //i2c_master_ctrl_t*  port= io->base.port->p_ctrl;
    (void)api;

    if ( UNLIKELY(io->state > i2cio_IDLE) )
        return I2C_BUSY;

    io->addr     = addr;
    io->adata    = (const unsigned char*)&(io->addr);
    uint8_t SLA  = SLA_mode >> 8;
    io->mode     = SLA_mode & 0xff;
    io->alen     = 1+ ((SLA_mode & i2co_ADDR)>> i2co_ADDR_Pos);
    I2C_data_assign(data, dlen);

    return I2C_xfer_start(h, SLA);
}

/// @brief this operations continues on last SLA
I2C_Status I2C_xfer_continue(i2cHandle h, uint32_t SLA_mode, uint32_t addr ,void* data, unsigned dlen ){
    //i2c_descriptor*     self = (i2c_descriptor*)(h->port);
    i2c_descriptor*     io  = &SELF;
    const i2c_master_api_t*   api = io->base.port->p_api;
    //const i2c_master_ctrl_t*  port= io->base.port->p_ctrl;
    (void)api;

    if ( UNLIKELY(io->state > i2cio_IDLE) )
        return I2C_BUSY;

    io->addr     = addr;
    io->adata    = (const unsigned char*)&(io->addr);
    //io->alen     = 1+ ((SLA_mode & i2co_ADDR)>> i2co_ADDR_Pos);
    io->mode     = SLA_mode & 0xff;
    I2C_data_assign(data, dlen);

    return I2C_xfer_start(h, io->sla);
}




I2C_Status I2C_xfer_start(i2cHandle h, unsigned char SLA)
{
    (void)h;
    I2C_Status res = I2C_OK;
    //i2c_descriptor*     self = (i2c_descriptor*)(h->port);
    i2c_descriptor*     io  = &SELF;
    const i2c_master_api_t*   api = io->base.port->p_api;
    i2c_master_ctrl_t*        port= io->base.port->p_ctrl;
    (void)api;

    if ( UNLIKELY(io->state > i2cio_IDLE) )
        return I2C_BUSY;

    trace_i2c_start_on();
    trace_i2c_adr_off();
    I2C_putchar('<');

    i2c_master_xfer_mode_t restart = 0;
    if (io->mode & i2co_LockBus)
        restart = I2C_MASTER_RESTART;

    // TODO: provide support 10bit SLA
    res = API(slaveAddressSet)(port, SLA, I2C_MASTER_ADDR_MODE_7BIT);
    io->sla = SLA;
    SLA = SLA <<1;

    io->err = 0;
    if ((io->mode & i2co_XSEND) != 0) {
        //только отсылаем data
        SLA &= ~1;
        if (io->adata != NULL) {
            //отсылаем адрес
            io->state    = i2cio_ADDR;
            if (io->xlen > 0)
                restart = I2C_MASTER_BUS_KEEP;
            trace_i2c_adr_on();
            res = API(write)(port, io->adata, io->alen, restart );
        }
        else if (io->xdata != NULL) {
            io->state    = i2cio_XSEND;
            res = API(write)(port, io->xdata, io->xlen, restart );
        }
    }
    else {
        if (io->adata != NULL) {
            SLA &= ~1;
            //отсылаем адрес
            io->state    = i2cio_ADDR;
            if (io->xlen > 0)
                restart = I2C_MASTER_RESTART;
            trace_i2c_adr_on();
            res = API(write)(port, io->adata, io->alen, restart );
        }
        else if (io->xdata != NULL) {
            //только читаем data
            SLA |= 1;
            io->state    = i2cio_XRECV;
            res = API(read)(port, io->xdata, io->xlen, restart );
        }
    }

    I2C_printf("@%02x", SLA);
    trace_i2c_start_off();

    if (res != FSP_SUCCESS) {
        I2C_putchar('#');
        io->err = res;
        if (io->state == i2cio_ADDR) {
            res = I2C_NOSLA;
        }
        io->state = i2cio_IDLE;
        I2C_finale(io, I2C_MASTER_EVENT_ABORTED);
    } // if (res != FSP_SUCCESS)
    else
        trace_i2c_work_on();

    I2C_putchar('>');
    return res;
}


CODE_ISR
I2C_Status I2C_ISR_xfer(i2c_descriptor* self){

    i2c_descriptor*     io  = &SELF;
    const i2c_master_api_t*   api = io->base.port->p_api;
    i2c_master_ctrl_t*  port= io->base.port->p_ctrl;
    (void)self; (void)api;

    i2c_io_state        state = io->state;
    int res;
    i2c_master_xfer_mode_t restart = 0;

    trace_i2c_adr_off();

    I2C_putchar('^');
    I2C_putchar( (char)('0'+(unsigned)(state)) );

    switch (state){
        case i2cio_IDLE: return I2C_OK;

        case i2cio_XSLA: {

            if (io->mode & i2co_LockBus)
                restart = I2C_MASTER_RESTART;

            ASSERT(io->xdata != NULL);

            if ((io->mode & i2co_XSEND) != 0){
                io->state = i2cio_XSEND;
                // TODO: move it i2C_status() ?
                // last operation have sent SLA
                res = API(write)(port, io->xdata, io->xlen, restart | I2C_MASTER_CONTINUE );

                I2C_printf("$>%02x", io->xlen);
            }
            else{
                io->state = i2cio_XRECV;
                res = API(read)(port, io->xdata, io->xlen, restart );

                I2C_printf("$<%02x", io->xlen);
            }
            io->err = res;

            if (res == FSP_SUCCESS)
                res = I2C_BUSY;
            else {
                trace_i2c_work_off();
                I2C_finale(io, I2C_MASTER_EVENT_ABORTED);
            }

            return res;
        }

        //case i2cio_ADDR:
        //case i2cio_XRECV:
        //case i2cio_XSEND:
        default:
            return I2C_BUSY;
    }; // switch (state)

}



CODE_ISR
void I2C_OnEvent(i2c_master_callback_args_t* ev)
{
    trace_i2c_irq_twist();

    //i2c_descriptor*     self= (i2c_descriptor*)ev->p_context;
    i2c_descriptor*     io  = &SELF;
    const i2c_master_api_t*   api = io->base.port->p_api;
    //i2c_master_ctrl_t*  port= io->base.port->p_ctrl;
    (void)api;

    i2c_io_state        state = io->state;
    I2C_putchar( (char)('0'+(unsigned)(state)) );

    i2c_master_event_t event = ev->event;

    if (event == I2C_MASTER_EVENT_ABORTED){
        I2C_putchar( '!' );
        io->err = I2C_ABORT;
    }
    else
    switch (state){

    case i2cio_ADDR:
        if (UNLIKELY(ev->event != I2C_MASTER_EVENT_TX_COMPLETE)){
            io->err = I2C_NOSLA;
            break;
        }

        if ( (((unsigned)(io->xdata)) == 0) || (io->xlen == 0) ) {
            io->err = I2C_OK;
            break;
        }
        else {
            io->state = i2cio_XSLA;
#if I2C_ISR_STYLE == I2C_ISR_ACTIVE
            I2C_ISR_xfer(io);
            return;
#else
            I2C_notify(io, I2C_MASTER_EVENT_BUSY);
            return;
#endif
        }
        // fallthrough

    case i2cio_XRECV:
    case i2cio_XSEND:
        trace_i2c_work_off();
        io->err = I2C_OK;
        break;

    case i2cio_IDLE:
    default:
        return;
    }; // switch (state)

    I2C_finale(io, ev->event);
}

I2C_Status I2C_abort(i2cHandle h)
{
    i2c_descriptor*     self = (i2c_descriptor*)(h->port);
    i2c_descriptor*     io  = &SELF;
    const i2c_master_api_t*   api = io->base.port->p_api;
    i2c_master_ctrl_t*  port= io->base.port->p_ctrl;
    (void)self;(void)api;

    int ok = API(abort)(port);
    (void)ok;
    trace_i2c_work_off();
    io->err = FSP_ERR_ABORTED;
    I2C_finale(io, I2C_MASTER_EVENT_ABORTED);

    return I2C_OK;
}

I2C_Status I2C_status(i2cHandle h){
    i2c_descriptor*     self = (i2c_descriptor*)(h->port);
    i2c_descriptor*     io  = &SELF;
    //const i2c_master_api_t*   api = io->base.port->p_api;
    //i2c_master_ctrl_t*  port= io->base.port->p_ctrl;
    (void)self;

    if (io->state == i2cio_IDLE)
        return io->err;

    I2C_putchar('?');
    I2C_putchar( (char)('0'+(unsigned)(io->state)) );

    return I2C_ISR_xfer(io);
}


#ifdef __cplusplus

/** это врап линуховых файловых операций на И2Цпорту.
 * они используют собственный SLA и не пересекаются с I2C_write/read
 * */
int read(i2cHandle port, void* data, int len){
    i2c_descriptor& io = I2C_port;
    I2C_Status res = I2C_xfer(port, io.file_sla, NULL, 0, data, len, i2co_XRECV);
    if (res == I2C_OK)
        return len;
    else if (res == I2C_NOSLA)
        return 0;
    else
        return len - io.xlen;
}

int write(i2cHandle port, const void* data, int len){
    i2c_descriptor& io = I2C_port;
    I2C_Status res = I2C_xfer(port, io.file_sla, NULL, 0, (void*)data, len, i2co_XSEND);
    if (res == I2C_OK)
        return len;
    else if (res == I2C_NOSLA)
        return 0;
    else
        return len - io.xlen;
}

int ioctl(i2cHandle port, I2C_ioctl_code code, int data){
    i2c_descriptor& io = I2C_port;
    io.file_sla = data;
    return 0;
}

#endif // __cplusplus
