/*
 * Copyright 2006 Mask@ Project.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.maskat.ide.wizards;

import org.eclipse.core.resources.IContainer;
import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.IResource;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.Path;
import org.eclipse.jface.dialogs.IDialogPage;
import org.eclipse.jface.viewers.ILabelProvider;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.viewers.ITreeContentProvider;
import org.eclipse.jface.wizard.WizardPage;
import org.eclipse.swt.SWT;
import org.eclipse.swt.events.ModifyEvent;
import org.eclipse.swt.events.ModifyListener;
import org.eclipse.swt.events.SelectionAdapter;
import org.eclipse.swt.events.SelectionEvent;
import org.eclipse.swt.layout.GridData;
import org.eclipse.swt.layout.GridLayout;
import org.eclipse.swt.widgets.Button;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.Label;
import org.eclipse.swt.widgets.Text;
import org.eclipse.ui.dialogs.ContainerSelectionDialog;
import org.eclipse.ui.dialogs.ElementTreeSelectionDialog;
import org.eclipse.ui.model.WorkbenchContentProvider;
import org.eclipse.ui.model.WorkbenchLabelProvider;
import org.eclipse.ui.views.navigator.ResourceSorter;

/**
 * The "New" wizard page allows setting the container for the new file as well
 * as the file name. The page will only accept file name without the extension
 * OR with the extension that matches the expected one (html).
 */

public class MaskatHtmlContainerNewWizardPage extends WizardPage {
	private Text containerText;

	private Text fileText;

	private Text titleText, layoutFileText, eventFileText, maskatFWPathText;

	private Text widthText, heightText;

	private ISelection selection;

	/**
	 * Constructor for SampleNewWizardPage.
	 * 
	 * @param pageName
	 */
	public MaskatHtmlContainerNewWizardPage(ISelection selection) {
		super("wizardPage");
		setTitle("Maskat Container HTML");
		setDescription("This wizard creates a container html file for maskat framework.");
		this.selection = selection;
	}

	/**
	 * @see IDialogPage#createControl(Composite)
	 */
	public void createControl(Composite parent) {
		ModifyListener textModifyListener = new ModifyListener() {
			public void modifyText(ModifyEvent e) {
				dialogChanged();
			}
		};
		Composite container = new Composite(parent, SWT.NULL);
		GridLayout layout = new GridLayout();
		container.setLayout(layout);
		layout.numColumns = 5;
		layout.verticalSpacing = 9;
		Label label = new Label(container, SWT.NULL);
		label.setText("&Container:");

		containerText = new Text(container, SWT.BORDER | SWT.SINGLE);
		GridData gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 3;
		containerText.setLayoutData(gd);
		containerText.addModifyListener(textModifyListener);

		Button button = new Button(container, SWT.PUSH);
		button.setText("Browse...");
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleBrowse();
			}
		});
		label = new Label(container, SWT.NULL);
		label.setText("&File name:");

		fileText = new Text(container, SWT.BORDER | SWT.SINGLE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 4;
		fileText.setLayoutData(gd);
		fileText.addModifyListener(textModifyListener);

		label = new Label(container, SWT.NULL);
		label.setText("&Title:");

		titleText = new Text(container, SWT.BORDER | SWT.SINGLE);
		titleText.setText("Maskat Application");
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 4;
		titleText.setLayoutData(gd);
		titleText.addModifyListener(textModifyListener);

		label = new Label(container, SWT.NULL);
		label.setText("&Maskat FW Path:");

		maskatFWPathText = new Text(container, SWT.BORDER | SWT.SINGLE);
		maskatFWPathText.setText("maskatFW");
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 4;
		maskatFWPathText.setLayoutData(gd);
		maskatFWPathText.addModifyListener(textModifyListener);

		label = new Label(container, SWT.NULL);
		label.setText("&Layout File:");

		layoutFileText = new Text(container, SWT.BORDER | SWT.SINGLE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 3;
		layoutFileText.setLayoutData(gd);
		layoutFileText.addModifyListener(textModifyListener);

		button = new Button(container, SWT.PUSH);
		button.setText("Browse...");
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleBrowseLayout();
			}
		});

		label = new Label(container, SWT.NULL);
		label.setText("&Event File:");

		eventFileText = new Text(container, SWT.BORDER | SWT.SINGLE);
		gd = new GridData(GridData.FILL_HORIZONTAL);
		gd.horizontalSpan = 3;
		eventFileText.setLayoutData(gd);
		eventFileText.addModifyListener(textModifyListener);

		button = new Button(container, SWT.PUSH);
		button.setText("Browse...");
		button.addSelectionListener(new SelectionAdapter() {
			public void widgetSelected(SelectionEvent e) {
				handleBrowseEvent();
			}
		});

		label = new Label(container, SWT.NULL);
		label.setText("Div &width:");

		widthText = new Text(container, SWT.BORDER | SWT.SINGLE);
		widthText.setText("800");
		gd = new GridData(GridData.FILL_HORIZONTAL);
		widthText.setLayoutData(gd);
		widthText.addModifyListener(textModifyListener);

		label = new Label(container, SWT.NULL);
		label.setText("Div &height:");

		heightText = new Text(container, SWT.BORDER | SWT.SINGLE);
		heightText.setText("600");
		gd = new GridData(GridData.FILL_HORIZONTAL);
		heightText.setLayoutData(gd);
		heightText.addModifyListener(textModifyListener);

		initialize();
		dialogChanged();
		setControl(container);
	}

	protected void handleBrowseLayout() {
		String value = handleBrowserResource();
		if (value != null)
			layoutFileText.setText(value);
	}

	protected void handleBrowseEvent() {
		String value = handleBrowserResource();
		if (value != null)
			eventFileText.setText(value);
	}

	protected String handleBrowserResource() {
		ILabelProvider lp = new WorkbenchLabelProvider();
		ITreeContentProvider cp = new WorkbenchContentProvider();

		ElementTreeSelectionDialog dialog = new ElementTreeSelectionDialog(this
				.getShell(), lp, cp);
		dialog.setTitle("Select file");
		dialog.setMessage("Select a file from the list:");
		dialog.addFilter(new FileExtensionFilter("xml"));
		if (container != null && container.getProject() != null)
			dialog.setInput(container.getProject());
		else
			dialog.setInput(ResourcesPlugin.getWorkspace().getRoot());
		dialog.setSorter(new ResourceSorter(ResourceSorter.NAME));

		if (dialog.open() == ContainerSelectionDialog.OK) {
			Object[] result = dialog.getResult();
			if (result.length == 1) {
				this.getDialogSettings();
				return ((IFile) result[0]).getName();
			}
		}
		return null;
	}

	private IContainer container = null;

	/**
	 * Tests if the current workbench selection is a suitable container to use.
	 */

	private void initialize() {
		if (selection != null && selection.isEmpty() == false
				&& selection instanceof IStructuredSelection) {
			IStructuredSelection ssel = (IStructuredSelection) selection;
			if (ssel.size() > 1)
				return;
			Object obj = ssel.getFirstElement();
			if (obj instanceof IResource) {
				if (obj instanceof IContainer)
					container = (IContainer) obj;
				else
					container = ((IResource) obj).getParent();
				containerText.setText(container.getFullPath().toString());
			}
		}
		fileText.setText("container.html");
	}

	/**
	 * Uses the standard container selection dialog to choose the new value for
	 * the container field.
	 */

	private void handleBrowse() {
		ContainerSelectionDialog dialog = new ContainerSelectionDialog(getShell(),
				ResourcesPlugin.getWorkspace().getRoot(), true,
				"Select new file container");
		if (dialog.open() == ContainerSelectionDialog.OK) {
			Object[] result = dialog.getResult();
			if (result.length == 1) {
				containerText.setText(((Path) result[0]).toString());
			}
		}
	}

	/**
	 * Ensures that both text fields are set.
	 */

	private void dialogChanged() {
		IResource container = ResourcesPlugin.getWorkspace().getRoot().findMember(
				new Path(getContainerName()));
		String fileName = getFileName();

		if (getContainerName().length() == 0) {
			updateStatus("File container must be specified");
			return;
		}
		if (container == null
				|| (container.getType() & (IResource.PROJECT | IResource.FOLDER)) == 0) {
			updateStatus("File container must exist");
			return;
		}
		if (!container.isAccessible()) {
			updateStatus("Project must be writable");
			return;
		}
		if (fileName.length() == 0) {
			updateStatus("File name must be specified");
			return;
		}
		if (fileName.replace('\\', '/').indexOf('/', 1) > 0) {
			updateStatus("File name must be valid");
			return;
		}
		int dotLoc = fileName.lastIndexOf('.');
		if (dotLoc != -1) {
			String ext = fileName.substring(dotLoc + 1);
			if (ext.equalsIgnoreCase("html") == false) {
				updateStatus("File extension must be \"html\"");
				return;
			}
		}
		if (layoutFileText.getText().length() == 0) {
			updateStatus("Layout File Name must be specified");
			return;
		}
		if (eventFileText.getText().length() == 0) {
			updateStatus("Event File Name must be specified");
			return;
		}
		if (!isPositiveInteger(widthText.getText())) {
			updateStatus("Div width must be a positive integer.");
			return;
		}
		if (!isPositiveInteger(heightText.getText())) {
			updateStatus("Div height must be a positive integer.");
			return;
		}

		updateStatus(null);
	}

	private boolean isPositiveInteger(String value) {
		try {
			int intVal = Integer.parseInt(value);
			if (intVal > 0) {
				return true;
			}
		} catch (NumberFormatException e) {
			return false;
		}
		return false;
	}

	private void updateStatus(String message) {
		setErrorMessage(message);
		setPageComplete(message == null);
	}

	public String getContainerName() {
		return containerText.getText();
	}

	public String getFileName() {
		return fileText.getText();
	}

	public String getHTMLTitle() {
		return titleText.getText();
	}

	public String getLayoutFileName() {
		return layoutFileText.getText();
	}

	public String getEventFileName() {
		return eventFileText.getText();
	}

	public String getFWPath() {
		return maskatFWPathText.getText();
	}

	public int getDivWidth() {
		return Integer.parseInt(widthText.getText());
	}

	public int getDivHeight() {
		return Integer.parseInt(heightText.getText());
	}
}