/*!
 * @file GLObject.h
 * @brief 3D Object
 */

#ifndef __GLOBJECT_H__
#define __GLOBJECT_H__

#ifdef _MSC_VER
#include <windows.h>
#include <tchar.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <GL/gl.h>
#include <GL/glu.h>
#include <math.h>

#include <iostream>
#include <vector>

#include "HomogeneousTransformation.h"

using namespace std;

/*!
 * @brief Calc outer product
 * @param vec0 input vector 0 (3D)
 * @param vec1 input vector 1 (3D)
 * @param result vec0 x vec1 (3D)
 */
void OuterProduct(float *vec0,float *vec1,float *result);

/*!
 * @brief Calc mean of vectors
 * @param vlist input vectors (3D x n)
 * @param length length of vectors (n)
 * @param result mean of vectors (3D)
 */
void VectorMean(float *vlist,int length,float *result);


/*!
 * @class GLObject
 * @brief Prototype of 3D Object
 */
class GLObject{
 private:

 public:
  //! @brief Constructor
  GLObject(){
    transx=transy=transz=rotx=roty=rotz=0.0;
    type="GLObject";
  }
  //! @brief Destructor
  ~GLObject(){}

  float transx,transy,transz;
  float rotx,roty,rotz;
  string type;

  /*!
   * @brief translate object
   * @param tx x
   * @param ty y
   * @param tz z
   */
  void setTranslate(float tx,float ty,float tz);
  /*!
   * @brief rotate object
   * @param rx x [rad]
   * @param ry y [rad]
   * @param rz z [rad]
   */
  void setRotate(float rx,float ry,float rz);

  /*!
   * @brief get translation
   * @param vec (x y z)
   */
  void getTranslate(float *vec){
    vec[0]=transx;
    vec[1]=transy;
    vec[2]=transz;
  }
  /*!
   * @brief get translation x
   * @return x
   */
  float getTranslateX(){return transx;}
  /*!
   * @brief get translation y
   * @return y
   */
  float getTranslateY(){return transy;}
  /*!
   * @brief get translation z
   * @return z
   */
  float getTranslateZ(){return transz;}

  /*!
   * @brief get rotation
   * @param vec (x y z) [rad]
   */
  void getRotate(float *vec){
    vec[0]=rotx;
    vec[1]=roty;
    vec[2]=rotz;
  }
  /*!
   * @brief get rotation x
   * @return x [rad]
   */
  float getRotateX(){return rotx;}
  /*!
   * @brief get rotation y
   * @return y [rad]
   */
  float getRotateY(){return roty;}
  /*!
   * @brief get rotation z
   * @return z [rad]
   */
  float getRotateZ(){return rotz;}

  //! @brief translate object
  void translate();
  //! @brief rotate object
  void rotate();

  /*!
   * @brief get type of object
   * @return Object type
   */
  string &getType(){return type;}

  //! @brief reset object
  virtual void reset(){}
  //! @brief draw object
  virtual void glDraw(){}
};



class GLDummyObject : public GLObject{
 private:
  float *mcolor;
  int count;
  int listid;

 public:
  GLDummyObject();
  ~GLDummyObject();
  
  void glDraw();
  void reset();
};

//! @brief 3D point cloud
class GLFeaturePoints : public GLObject{
 private:
  vector<float> points3d;
  int count;
  float *color;
  float pointsize;
  vector<float> colorlist;
  bool use_color_list;
 public:
  GLFeaturePoints();
  ~GLFeaturePoints();
  void glDraw();
  void reset(){}
  
  void setPoints(int num,float *pts);
  void setPoints(vector<float> &pts);
  int getPointsNum(){return count;}
  void getPoints(float *pts){
    memcpy(pts,&points3d[0],count*3*sizeof(float));
  }

  void useColorList(bool isu){use_color_list=isu;}
  void setColorList(vector<float> &cls){
    colorlist.assign(cls.begin(),cls.end());
  }
  void getColorList(vector<float> &cls){
    cls.assign(colorlist.begin(),colorlist.end());
  }
};

//! @brief 2D grid
class GLGrid : public GLObject{
 private:
  int gridnum;
  float gridsize;
  float *color;
  float linewidth;
 public:
  GLGrid();
  ~GLGrid();
  void glDraw();
  void reset(){}
  
  void setGrid(int num,float size);

  void setLineWidth(float ww){linewidth=ww;}
  void setColor(float rr,float gg,float bb){
    color[0]=rr;
    color[1]=gg;
    color[2]=bb;
  }
};


class GLCoordinate : public GLObject{
 private:
  float asize;
  float linewidth;
  float red[3],blue[3],green[3];

 public:
  GLCoordinate();
  ~GLCoordinate();

  void glDraw();
  void reset(){}
  void setArrowSize(float as){asize=as;}
  float getArrowSize(){return asize;}
  void setLineWidth(float lw){linewidth=lw;}
  float getLineWidth(){return linewidth;}
};


/*
class GLPolygon : public GLObject{
 private:
  vector<float> verts;
  vector<float> norms;
  float *color;
  bool isTexture;
  vector<unsigned char> tex_img;
  vector<float> texco;
  unsigned int texi;
  unsigned int listid;

  float rotangle,rotomega;

 public:
  GLPolygon();
  ~GLPolygon();

  void setColor(float *col){memcpy(color,col,sizeof(float)*4);}
  void setVertices(vector<float> &vv){verts.assign(vv.begin(),vv.end());}
  void setTexCoords(vector<float> &tt){texco.assign(tt.begin(),tt.end());}
  void setNormals(vector<float> &nn){norms.assign(nn.begin(),nn.end());}
  void resetNormals(){norms.resize(0);}
  void setTexture(int width,int height,unsigned char depth,unsigned char *imgbuf);
  void resetTexture(){isTexture=false;}

  void setGLList();
  void glDraw();

  void setOmega(float omg){rotomega=omg;}
  void reset(){rotangle=0.0;}
};
// primitives
void setRectanglePolygon(GLPolygon *glpoly,float width,float height);
void setCirclePolygon(GLPolygon *glpoly,float ir,float or);
*/


///////////////////////////////////////
/*!
 * @class GLMaterial
 * @brief  material data (not extention of GLObject)
 */
class GLMaterial{
 private:
  unsigned int texi;
  bool use_texture;
  vector <float> mtlcolor;
  vector <unsigned char> imagebuf;

 public:
  GLMaterial();
  ~GLMaterial();
  /*!
   * @brief set material color in RGBA color space
   * @param color 4D RGBA color vector (0.0 - 1.0)
   */
  void setMaterialColor(float *color);
  /*!
   * @brief set texture image
   */
  void setTexture(int width,int height,unsigned char depth,unsigned char *imgbuf);
  /*!
   * @brief substitute texture image
   */
  void subTexture(int xoff,int yoff,int width,int height,unsigned char depth,unsigned char *imgbuf);
  /*!
   * @brief reset texture image
   */
  void resetTexture(){use_texture=false;}
  bool isTexture(){return use_texture;}

  void glBeginMaterial();
  void glEndMaterial();
};

//////////////////////////////////
/*!
 * @class GLPolygon
 * @brief Polygon object
 */
class GLPolygon : public GLObject{
 private:
  vector<float> vertices;
  vector<float> tvertices;
  vector<float> normals;
  vector<float> vnormals;
  vector<int> faces;
  vector<int> tfaces;
  int vlength;
  int flength;
  int mclength;
  unsigned int listid;

  vector<GLMaterial *> materials;

  unsigned char normtype;

  float handco;

  bool uselist;

  bool isvisible;

 public:
  GLPolygon();
  ~GLPolygon();

  //! @brief set vertices (3xn)
  void setVertices(float *verts,int length);
  void setVertices(vector<float> &verts);

  //! @brief set face by vertex indices (3xn), when index[0] < 0 then index[2] means material index
  void setFaces(int *fcs,int length);
  void setFaces(vector<int> &fcs);

  //! @brief set GLMaterial list
  void setMaterialList(GLMaterial **mtls,int length);
  void setMaterialList(vector<GLMaterial *> &mtls);
  //! @brief destruct all GLMaterial in list
  void destructMaterialList();

  //! @brief recreate material list and set GLMaterial at first of list
  void setMaterial(GLMaterial *mtl);
  //! @brief add GLMaterial to list
  void addMaterial(GLMaterial *mtl);

  void setNormals(float *norms);
  void setNormals(vector<float> &norms);
  void calcNormals();

  void setVertexNormals(float *vnorms);
  void setVertexNormals(vector<float> &vnorms);
  void calcVertexNormals();

  void setTexVertices(vector<float> &tv){tvertices.assign(tv.begin(),tv.end());}
  void setTexFaces(vector<int> &tf){tfaces.assign(tf.begin(),tf.end());}

  void drawFaces();
  void setGLList();
  void useList(bool isuse){uselist=isuse;}

  void setNormalType(unsigned char type){normtype=type;}

  void setLeftHandCoordinate(){handco=-1.0;}
  void setRightHandCoordinate(){handco=1.0;}

  //! @brief set visibility of object
  void setVisible(bool is){isvisible=is;}
  //! @brief get visibility of object
  bool isVisible(){return isvisible;}

  void glDraw();
  void reset(){}
};

class GLObjObject : public GLObject{
 private:
  GLPolygon *glpoly;
  vector<GLMaterial *> materials;
  vector<float> mtlcolors;
  int mclength;

 public:
  GLObjObject();
  ~GLObjObject();

  GLPolygon *getGLPolygon(){return glpoly;}

  void setMaterialColors(float *mtlc,int length);
  void setMaterialColors(vector<float> &mtlc);
  void createGLMaterials();

  void setGLList(){glpoly->setGLList();}
  void setNormalType(unsigned char type){glpoly->setNormalType(type);}

  void glDraw();
  void reset(){}
};


// for pmd

enum bone_transform_type{
  BONE_LOCAL,
  BONE_WORLD
};

struct bone_sequence{
  unsigned int index;
  float tv[3];
  float quat[4];
};

class GLBone{
 private:
  float init_tf[16];
  float current_tf[16];
  float world_tf[16];
  float current_tv[3];
  float current_quat[4];
  bone_transform_type type;

  GLBone *parrent_bone;
  
  string name;

  // for motion
  vector<bone_sequence> sequence;
  vector<bone_sequence> iksequence;

 public:
  GLBone();
  ~GLBone();

  void setName(string nm){name=nm;}
  string getName(){return name;}

  void setParrent(GLBone *par){parrent_bone=par;}

  float *getInitTransformation(){return init_tf;}
  float *getCurrentTransformation(){return current_tf;}
  float *getWorldTransformation(){return world_tf;}
  float *getCurrentQuaternion(){return current_quat;}

  void setInitPos(float *tv);
  void setCurrentTrans(float *tv);
  void setCurrentQuaternion(float *quat);

  void calcWorldTransformation();

  void setTransformType(bone_transform_type tp){type=tp;}

  void clearSequence();
  void clearIKSequence();
  void addSequence(unsigned int index,float *tv,float *quat);
  void addIKSequence(unsigned int index);
  void setReturnSequence(unsigned int index);
  void selectSequence(unsigned int index);
  void selectIKSequence(unsigned int index);

};

/*!
 * @class GLPMDObject
 * @brief MikuMikuDance object
 */
class GLPMDObject : public GLObject{
 private:
  GLPolygon *glpoly;
  vector<GLMaterial *> materials;

  vector<int> bones_parent;
  vector<float> bones_headpos;
  vector<string> bones_names;
  vector<GLBone *> bones;

  vector<float> init_vertices;
  vector<float> current_vertices;
  vector<float> init_normals;
  vector<float> current_normals;
  vector<int> vertex_bones;
  vector<float> vertex_bones_weight;

  vector<int> ik_bones;
  vector<int> ik_target_bones;
  vector < vector < int > > ik_children;
  vector<float> ik_weights;

  vector<char> updated;

  float scale;
  int count;

  bool isanime;


 public:
  GLPMDObject();
  ~GLPMDObject();


  GLPolygon *getGLPolygon(){return glpoly;}
  void setGLList(){glpoly->setGLList();}

  void setInitVertices(vector<float> &verts){
    init_vertices.assign(verts.begin(),verts.end());
    glpoly->setVertices(init_vertices);
    current_vertices.resize(init_vertices.size());
  }
  void setInitNormals(vector<float> &norms){
    init_normals.assign(norms.begin(),norms.end());
    glpoly->setVertexNormals(init_normals);
    current_normals.resize(init_normals.size());
  }
  void setVertexWeights(vector<int> &vert_b,vector<float> &vert_w){
    vertex_bones.assign(vert_b.begin(),vert_b.end());
    vertex_bones_weight.assign(vert_w.begin(),vert_w.end());
  }

  void createMaterials(int count);
  GLMaterial *getMaterial(int idx){return materials[idx];}
  void sendMaterials(){glpoly->setMaterialList(materials);}

  void setBones(vector<int> &parent,vector<float> &headpos,vector<string> &names);
  void setBonesTransformType(bone_transform_type type);
  GLBone *getBone(int idx){return bones[idx];}

  string getBoneName(int idx){return bones_names[idx];}

  void drawBones();

  void setScale(float sc){scale=sc;}
  float getScale(){return scale;}

  void resetPose(){
    glpoly->setVertices(init_vertices);
    glpoly->setVertexNormals(init_normals);
    glpoly->setGLList();
    for(int i=0;i<updated.size();i++)
      updated[i]=0;
  }
  void setBonePose(string bname,float *tvec,float *quat);
  void addBoneSequence(string bname,unsigned int index,float *tvec,float *quat);
  void addBoneIKSequence(unsigned int index);
  void setReturnBoneSequence(unsigned int index);

  void clearBoneSequence();

  void calcCurrentVertices();


  void setIKBones(vector<int> &bns){
    ik_bones.assign(bns.begin(),bns.end());
  }
  void setIKTargetBones(vector<int> &bns){
    ik_target_bones.assign(bns.begin(),bns.end());
  }
  void setIKChildrens(vector < vector <int> > &chs){
    ik_children.resize(chs.size());
    for(int i=0;i<ik_children.size();i++)
      ik_children[i].assign(chs[i].begin(),chs[i].end());
  }
  void setIKWeights(vector<float> &wts){
    ik_weights.assign(wts.begin(),wts.end());
  }

  vector<int> &getIKBones(){return ik_bones;}
  vector<int> &getIKTargetBones(){return ik_target_bones;}
  vector < vector <int> > &getIKChildrens(){return ik_children;}
  vector<float> &getIKWeights(){return ik_weights;}
  vector<char> &getUpdated(){return updated;}

  void glDraw();
  void reset();

  void animate();
  void deanimate();
  void selectSequence(unsigned int index,bool ik=true);
  void selectIKSequence(unsigned int index);
};

/*!
 * @class GLMQOObject
 * @brief Metasequoia object
 */

class GLMQOObject : public GLObject{
 private:
  vector<GLPolygon *> polygons;
  vector<GLMaterial *> materials;

 public:
  GLMQOObject();
  ~GLMQOObject();

  void createMaterials(int num);
  GLMaterial *getMaterial(int idx){return materials[idx];}

  void createPolygon(){polygons.push_back(new GLPolygon());}
  GLPolygon *getPolygon(int idx){return polygons[idx];}
  int nPolygons(){return polygons.size();}

  void setMaterialsToPolygon(int idx);

  void glDraw();
  void reset(){}
};


//! @brief box object
class GLBox : public GLPolygon{
 private:
  GLMaterial *color_mtl;

 public:
  GLBox(float x,float y,float z);
  ~GLBox();

  void setSize(float x,float y, float z);
  void setColor(float r,float g,float b,float a);
  void setColor(float *color);
};

//! @brief voxel object
class GLVoxel : public GLObject{
 private:
  GLBox *voxel_object;
  vector<float> map;
  int width,height;
  float vx,vy,vz;

 public:
  GLVoxel();
  GLVoxel(float x, float y, float z);
  ~GLVoxel();

  void setVoxelSize(float x, float y, float z);
  void setMap(int w,int h,vector<float> &m);

  void glDraw();
  void reset(){}
};


#endif
