/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "data.h"
#include "graphicDevice.h"
#include "texture.h"

#if GRAPH2D_LOG_DETAIL_LEVEL
#include <libmana.h>
#endif

#if defined(TARGET_IPHONE)
#include "platform/iOS/texture_impl.h"
#elif defined(TARGET_WINDOWS)
#include "platform/windows/texture_impl.h"
#endif

namespace Graph2D
{
	GLuint Texture::lastBindTextureName = 0;

	Texture::Texture()
		: instance(NULL)
		, compressed(false)
	{
	}

	Texture::Texture(const std::string& filename, const bool compressed)
		: instance(NULL)
		, compressed(false)
	{
		load(filename, compressed);
	}

	Texture::~Texture()
	{
		unload();
	}

	void Texture::serialize(mana_stream* stream) const
	{
		if(instance)
		{
			mana_stream_push_unsigned_char(stream, compressed ? 1 : 0);
			mana_stream_push_unsigned_char(stream, 1);
			mana_stream_push_string(stream, (char*)filename.c_str());
		}
		else
		{
			mana_stream_push_unsigned_char(stream, 0);
		}

	}

	void Texture::deserialize(mana_stream* stream)
	{
		unload();

		compressed = mana_stream_pop_unsigned_char(stream) ? true : false;

		if(mana_stream_pop_unsigned_char(stream))
		{
			load(mana_stream_get_string_pointer(stream), false);

			mana_steram_seek(stream, mana_stream_get_string_length(stream) + 1);
		}
	}

	bool Texture::load(const std::string& filename, const bool compressed)
	{
		unload();

#if GRAPH2D_LOG_DETAIL_LEVEL >= 1
		MANA_TRACE("Texture::load: %s ", filename.c_str());
#endif
		instance = Implementation::Texture::load(filename, compressed);
		if(instance != NULL)
		{
			this->filename = filename;
			this->compressed = compressed;

			bind();
			GRAPH2D_CALL_GL(glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE));
			GRAPH2D_CALL_GL(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT));
			GRAPH2D_CALL_GL(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT));
			GRAPH2D_CALL_GL(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST));
			GRAPH2D_CALL_GL(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST));

#if GRAPH2D_LOG_DETAIL_LEVEL >= 1
			const GLuint name = Implementation::Texture::getName(instance, compressed);
			MANA_TRACE("#%d ... OK\n", name);
#endif
			return true;
		}
		else
		{
#if GRAPH2D_LOG_DETAIL_LEVEL >= 1
			MANA_TRACE("... NG\n");
#endif
			return false;
		}
	}

	void Texture::unload()
	{
		if(instance)
		{
#if GRAPH2D_LOG_DETAIL_LEVEL >= 1
			const GLuint name = Implementation::Texture::getName(instance, compressed);
			MANA_TRACE("Texture::unload: %s #%d ... ", filename.c_str(), name);
#endif

			if(lastBindTextureName == Implementation::Texture::getName(instance, compressed))
				lastBindTextureName = 0;
			Implementation::Texture::unload(instance, compressed);
			instance = NULL;
		}

		filename.clear();
	}

	void Texture::bind()
	{
		const GLuint name = Implementation::Texture::getName(instance, compressed);
		if(lastBindTextureName != name)
		{
			GraphicDevice::flush();

			glBindTexture(GL_TEXTURE_2D, name);
			lastBindTextureName = name;
		}
	}

	void Texture::unbind()
	{
		if(lastBindTextureName != 0)
		{
			GraphicDevice::flush();

			lastBindTextureName = 0;
		}
	}

	const std::string& Texture::getFileName() const
	{
		return filename;
	}

	Vector2 Texture::getSize() const
	{
		return Implementation::Texture::getSize(instance, compressed);
	}

	Vector2 Texture::getTextureSize() const
	{
		return Implementation::Texture::getTextureSize(instance, compressed);
	}

	Vector2 Texture::getMaxSize() const
	{
		return Implementation::Texture::getMaxSize(instance, compressed);
	}
}
