/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "textWindow.h"
#include "text.h"
#include <float.h>

namespace Graph2D
{
	TextWindow::TextWindow()
		: timer(0)
		, wait(FLT_MAX)
	{
		type = TYPE_TEXT_WINDOW;

		allocate(1);

		text = new Graph2D::Text();
		set(0, text);
	}

	TextWindow::~TextWindow()
	{
		text->release();
	}

	void TextWindow::clear()
	{
		text->clear();
	}

	bool TextWindow::empty() const
	{
		return text->empty();
	}

	void TextWindow::setFontName(const std::string& fontName)
	{
		text->setFontName(fontName);
	}

	void TextWindow::setFontSize(const float fontSize)
	{
		text->setFontSize(fontSize);
	}

	const Color& TextWindow::getTextColor() const
	{
		return text->getTextColor();
	}

	void TextWindow::setTextColor(const Color& color)
	{
		text->setTextColor(color);
	}

	void TextWindow::setTextColor(const float r, const float g, const float b, const float a)
	{
		setTextColor(Color(r, g, b, a));
	}
	
	TextAlignmentH TextWindow::getTextAlignmentH() const
	{
		return text->getTextAlignmentH();
	}

	void TextWindow::setTextAlignmentH(const TextAlignmentH textAlignment)
	{
		text->setTextAlignmentH(textAlignment);
	}

	TextAlignmentV TextWindow::getTextAlignmentV() const
	{
		return text->getTextAlignmentV();
	}

	void TextWindow::setTextAlignmentV(const TextAlignmentV textAlignment)
	{
		return text->setTextAlignmentV(textAlignment);
	}

	const std::string& TextWindow::getString() const
	{
		return text->getString();
	}

	void TextWindow::setString(const std::string& text)
	{
		this->text->setString(text);

		if(closed())
			open();
		timer = 0;
	}

	void TextWindow::format(const char* format, ...)
	{
		va_list ap;
		va_start(ap, format);
		vformat(format, ap);
		va_end(ap);
	}

	void TextWindow::vformat(const char* format, va_list ap)
	{
		text->vformat(format, ap);

		if(closed())
			open();
		timer = 0;
	}

	void TextWindow::replace(const char* key, const char* word)
	{
		text->replace(key, word);
	}

	float TextWindow::getAutoCloseTimer() const
	{
		return wait;
	}

	void TextWindow::setAutoCloseTimer(const float wait)
	{
		this->wait = wait;
	}

	void TextWindow::disableAutoClose()
	{
		wait = FLT_MAX;
	}

	bool TextWindow::finished() const
	{
		return wait <= timer;
	}

	Text& TextWindow::getText()
	{
		return *text;
	}

	const Text& TextWindow::getText() const
	{
		return getText();
	}

#if defined(DEBUG) || defined(_DEBUG) || defined(_DEBUG_) || !defined(NDEBUG)
	void TextWindow::open()
	{
		super::open();
	}

	void TextWindow::close()
	{
		super::close();
	}
#endif
	void TextWindow::onUpdate(const UpdateInfomation& updateInfomation)
	{
		if(isAnimationStoped() && !closed())
		{
			timer += updateInfomation.deltaTime;
			if(wait < timer)
			{
				close();
			}
		}
		super::onUpdate(updateInfomation);
	}

#if defined(DEBUG) || defined(_DEBUG) || defined(_DEBUG_) || !defined(NDEBUG)
	void TextWindow::onDraw(const DrawRect& drawRect)
	{
		super::onDraw(drawRect);
	}
#endif

	void TextWindow::onSizeChanged()
	{
		text->setPosition(0, 0);
		text->setSize(getClientSize());
	}
}
