/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 * 
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 * 
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 * 
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 * 
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 * 
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#if !defined(___GRAPH2D_CURVE_H___)
#define ___GRAPH2D_CURVE_H___

#include "array.h"
#include "color.h"
#include "vector2.h"
#include "../../mana/library/mana_stream.h"

namespace Graph2D
{
	typedef enum InterpolationType
	{
		INTERPOLATION_CONST,		// 萔
		INTERPOLATION_LINER,		// `
		INTERPOLATION_HERMITE,		// G~[gXvC
		INTERPOLATION_CATMULL,		// Catmull-RomXvC
		INTERPOLATION_SLERP,		// ʐ`
		INTERPOLATION_SQUAD,		// ʂR
		INTERPOLATION_COUNT
	}InterpolationType;

	////////////////////////////////////////////////////////////////////////////////
	class CurveKey
	{
	protected:
		InterpolationType interpolationType;
		float second;

	public:
		CurveKey(const InterpolationType interpolationType, const float second)
		{
			setInterpolationType(interpolationType);
			setTime(second);
		}

	public:
		void set(const CurveKey& curveKey)
		{
			setInterpolationType(curveKey.getInterpolationType());
			setTime(curveKey.getTime());
		}

		InterpolationType getInterpolationType() const
		{
			return interpolationType;
		}

		void setInterpolationType(const InterpolationType interpolationType)
		{
			this->interpolationType = interpolationType;
		}

		float getTime() const
		{
			return second;
		}

		void setTime(float second)
		{
			this->second = second;
		}

		float getTimeSpan(const CurveKey& curveKey)
		{
			return curveKey.second - second;
		}
	};

	////////////////////////////////////////////////////////////////////////////////
	class CurveKey2 : public CurveKey
	{
	protected:
		Vector2 vector;

	public:
		CurveKey2(const InterpolationType interpolationType, const float second, const Vector2& vector) : CurveKey(interpolationType, second)
		{
			setVector(vector);
		}

	public:
		void set(const CurveKey2& curveKey)
		{
			CurveKey::set(static_cast<const CurveKey&>(curveKey));
			setVector(curveKey.getVector());
		}

		const Vector2& getVector() const
		{
			return vector;
		}

		void setVector(const Vector2& vector)
		{
			this->vector = vector;
		}
	};

	////////////////////////////////////////////////////////////////////////////////
	class CurveKey4 : public CurveKey
	{
	protected:
		Color color;

	public:
		CurveKey4(const InterpolationType interpolationType, const float second, const Color& color) : CurveKey(interpolationType, second)
		{
			setColor(color);
		}

	public:
		void set(const CurveKey4& curveKey)
		{
			CurveKey::set(static_cast<const CurveKey&>(curveKey));
			setColor(curveKey.getColor());
		}

		const Color& getColor() const
		{
			return color;
		}

		void setColor(const Color& color)
		{
			this->color = color;
		}
	};

	////////////////////////////////////////////////////////////////////////////////
	class Curve
	{
	protected:
		float currentTime;

	public:
		Curve();

	public:
		void rewind();
		void update(const float deltaTime);
		bool finish();

		virtual float getLength() = 0;
	};

	////////////////////////////////////////////////////////////////////////////////
	class Curve2 : public Curve
	{
	protected:
		Array<CurveKey2*> curveKeys;
		bool normalized;

	public:
		Curve2();
		virtual ~Curve2();

	public:
		void release();

		CurveKey2* getKeyFrame(const int index);
		void addKeyFrame(CurveKey2* curveKey);

		float getLength();
		int getKeyCount() const;
		Vector2 getValue();

	protected:
		void normalize();
	};

	////////////////////////////////////////////////////////////////////////////////
	class Curve4 : public Curve
	{
	protected:
		Array<CurveKey4*> curveKeys;
		bool normalized;

	public:
		Curve4();
		virtual ~Curve4();

	public:
		void release();

		CurveKey4* getKeyFrame(const int index);
		void addKeyFrame(CurveKey4* curveKey);

		float getLength();
		int getKeyCount() const;
		Color getValue();

	protected:
		void normalize();
	};
}

#endif
