/*
 * graph2D
 * Copyright (c) 2009 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

#include "button.h"
#include "text.h"

#define BUTTON_TEXT_SIZE (13)

namespace Graph2D
{

	Button::Button()
		: textAlignmentH(TEXT_ALIGNMENT_CENTER)
		, textAlignmentV(TEXT_ALIGNMENT_MIDDLE)
		, dirty(true)
	{
		initialize();
	}

	Button::Button(const char* text)
		: textAlignmentH(TEXT_ALIGNMENT_CENTER)
		, textAlignmentV(TEXT_ALIGNMENT_MIDDLE)
		, dirty(true)
	{
		initialize();
		label->setString(text);
	}

	Button::Button(const std::string& text)
		: textAlignmentH(TEXT_ALIGNMENT_CENTER)
		, textAlignmentV(TEXT_ALIGNMENT_MIDDLE)
		, dirty(true)
	{
		initialize();
		label->setString(text);
	}

	Button::~Button()
	{
		label->release();
		tag->release();
	}

	void Button::initialize()
	{
		type = TYPE_BUTTON;

		//allocate(2);

		label = new Text(BUTTON_TEXT_SIZE);
		//set(0, label);

		tag = new Text(BUTTON_TEXT_SIZE);
		//set(1, tag);

		setFrameStyle(FRAME_STYLE_NORMAL);
		setColor(0.5f, 0.5f, 0.5f, 0.9f, 0.0f);
		setSize(BUTTON_TEXT_SIZE + BUTTON_MARGIN_SIZE_X, BUTTON_TEXT_SIZE + BUTTON_MARGIN_SIZE_Y);
	}

	void Button::setSize(const Vector2& size)
	{
		super::setSize(size);

		dirty = true;
	}

	void Button::adjustSize()
	{
		label->adjustSize();
		label->setPosition(BUTTON_MARGIN_X, BUTTON_MARGIN_Y);
		super::setSize(label->getSize() + Vector2(BUTTON_MARGIN_SIZE_X, BUTTON_MARGIN_SIZE_Y));

		dirty = true;
	}

	const std::string& Button::getString() const
	{
		return label->getString();
	}

	void Button::setString(const std::string& text)
	{
		label->setString(text);

		dirty = true;
	}

	void Button::format(const char* fmt, ...)
	{
		va_list ap;
		va_start(ap, fmt);
		label->vformat(fmt, ap);
		va_end(ap);

		dirty = true;
	}

	void Button::replace(const char* key, const char* word)
	{
		label->replace(key, word);

		dirty = true;
	}

	void Button::setFontSize(const float fontSize)
	{
		label->setFontSize(fontSize);
		tag->setFontSize(fontSize);

		dirty = true;
	}

	//! 水平方向のテキストアライメントを取得
	TextAlignmentH Button::getTextAlignmentH() const
	{
		return static_cast<TextAlignmentH>(textAlignmentH);
	}

	//! 水平方向のテキストアライメントを設定
	void Button::setTextAlignmentH(const TextAlignmentH alignment)
	{
		textAlignmentH = static_cast<unsigned char>(alignment);

		dirty = true;
	}

	//! 垂直方向のテキストアライメントを取得
	TextAlignmentV Button::getTextAlignmentV() const
	{
		return static_cast<TextAlignmentV>(textAlignmentV);
	}

	//! 垂直方向のテキストアライメントを設定
	void Button::setTextAlignmentV(const TextAlignmentV alignment)
	{
		textAlignmentV = static_cast<unsigned char>(alignment);

		dirty = true;
	}

	void Button::setTextColor(const Color& color)
	{
		label->setTextColor(color);
		tag->setTextColor(color);
	}

	void Button::onSerialize(mana_stream* stream) const
	{
		super::onSerialize(stream);
		mana_stream_mark(stream);
	}

	const std::string& Button::getTagString() const
	{
		return tag->getString();
	}

	void Button::setTagString(const std::string& text)
	{
		tag->setString(text);

		dirty = true;
	}

	void Button::formatTag(const char* fmt, ...)
	{
		va_list ap;
		va_start(ap, fmt);
		tag->vformat(fmt, ap);
		va_end(ap);

		dirty = true;
	}

	void Button::onDeserialize(mana_stream* stream)
	{
		super::onDeserialize(stream);
		mana_stream_check(stream);
	}

	void Button::onUpdate(const UpdateInfomation& updateInfomation)
	{
		if(dirty)
		{
			Vector2 newPosition;

			label->adjustSize();
			tag->adjustSize();

			switch(textAlignmentH)
			{
			case TEXT_ALIGNMENT_LEFT:
				newPosition.x = getLeftFrameSize();
				break;
			case TEXT_ALIGNMENT_CENTER:
				newPosition.x = size.x * 0.5f - label->getSize().x * 0.5f;
				break;
			case TEXT_ALIGNMENT_RIGHT:
				newPosition.x = size.x - label->getSize().x - BUTTON_MARGIN_SIZE_X;
				break;
			}

			switch(textAlignmentV)
			{
			case TEXT_ALIGNMENT_TOP:
				newPosition.y = BUTTON_MARGIN_SIZE_Y;
				break;
			case TEXT_ALIGNMENT_MIDDLE:
				newPosition.y = size.y * 0.5f - label->getSize().y * 0.5f;
				break;
			case TEXT_ALIGNMENT_BOTTOM:
				newPosition.y = size.y - label->getSize().x - BUTTON_MARGIN_SIZE_Y;
				break;
			}

			label->setPosition(newPosition);

			if(!tag->getString().empty())
			{
				newPosition.x = size.x - tag->getSize().x - BUTTON_MARGIN_SIZE_X;
				tag->setPosition(newPosition);
			}

			dirty = false;
		}

		label->onUpdate(updateInfomation);
		tag->onUpdate(updateInfomation);

		super::onUpdate(updateInfomation);
	}

	void Button::onDraw(const DrawRect& drawRect)
	{
		super::onDraw(drawRect);

		if(getVisible() == false || getOwnerDraw() == true)
			return;

		Color baseColor = getDrawColor(drawRect);
		if(baseColor.a <= 0.0f)
			return;
#if 0
		const Vector2 positionOffset(getLeftFrameSize(), 0);
		const Vector2 sizeOffset(getLeftFrameSize() + getRightFrameSize(), 0);
#else
		const Vector2 positionOffset(0, 0);
		const Vector2 sizeOffset(0, 0);
#endif
		if(!label->empty())
		{
			DrawRect newDrawRect;
			newDrawRect.position = drawRect.getDrawLeftTopPosition() + (label->getPosition() + positionOffset) * drawRect.getScale();
			newDrawRect.size = label->getSize() - sizeOffset;
			newDrawRect.scale = label->getScale() * drawRect.getScale();
			newDrawRect.updateClippingInfomation(drawRect);
			newDrawRect.color *= color;
			label->onDraw(newDrawRect);
		}

		if(!tag->empty())
		{
			DrawRect newDrawRect;
			newDrawRect.position = drawRect.getDrawLeftTopPosition() + (tag->getPosition() + positionOffset) * drawRect.getScale();
			newDrawRect.size = tag->getSize() - sizeOffset;
			newDrawRect.scale = tag->getScale() * drawRect.getScale();
			newDrawRect.updateClippingInfomation(drawRect);
			newDrawRect.color *= color;
			tag->onDraw(newDrawRect);
		}
	}

// 押した時のコンポーネントの移動量を外部から設定できるように変更

	bool Button::onTouchesBegan(const Vector2& localPosition)
	{
#if 0
		label->setPosition(label->getPosition() + Vector2(BUTTON_MARGIN_X, BUTTON_MARGIN_Y));

		tag->setPosition(tag->getPosition() + Vector2(BUTTON_MARGIN_X, BUTTON_MARGIN_Y));
#endif
		return super::onTouchesBegan(localPosition);
	}

	bool Button::onTouchesEnded(const Vector2& localPosition)
	{
#if 0
		label->setPosition(label->getPosition() - Vector2(BUTTON_MARGIN_X, BUTTON_MARGIN_Y));

		tag->setPosition(tag->getPosition() - Vector2(BUTTON_MARGIN_X, BUTTON_MARGIN_Y));
#endif
		return super::onTouchesEnded(localPosition);
	}

	bool Button::onTouchesCancelled(const Vector2& localPosition)
	{
#if 0
		label->setPosition(label->getPosition() - Vector2(BUTTON_MARGIN_X, BUTTON_MARGIN_Y));

		tag->setPosition(tag->getPosition() - Vector2(BUTTON_MARGIN_X, BUTTON_MARGIN_Y));
#endif
		return super::onTouchesCancelled(localPosition);
	}

	void Button::onSizeChanged()
	{
		dirty = true;

		label->onSizeChanged();

		tag->onSizeChanged();

		super::onSizeChanged();
	}

	bool Button::compare(const Button& other) const
	{
		if(!super::compare(other))
			return false;

		if(label->compare(*other.label) == false)
			return false;

		if(tag->compare(*other.tag) == false)
			return false;

		return true;
	}
}
