/*
 * graph2D
 * Copyright (c) 2011 Shun Moriya <shun126@users.sourceforge.jp>
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event will the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Permission is granted to anyone to use this software for any purpose,
 * including commercial applications, and to alter it and redistribute it
 * freely, subject to the following restrictions:
 *
 *  1. The origin of this software must not be misrepresented; you must not
 *     claim that you wrote the original software. If you use this software
 *     in a product, an acknowledgment in the product documentation would be
 *     appreciated but is not required.
 *
 *  2. Altered source versions must be plainly marked as such, and must not be
 *     misrepresented as being the original software.
 *
 *  3. This notice may not be removed or altered from any source
 *     distribution.
 */

//#define NO_FILL				// defineすると、メモリを定数で埋めない
//#define NO_CHECK			// defineすると、メモリブロックを検証しない
//#define RANDOM_FILL		// defineすると、メモリを乱数で埋めます。
//#define ENABLE_COMPACTION	// メモリコンパクションも対応する

#include "allocator.h"
#include <assert.h>
#include <stdio.h>
#include <string.h>

#define MEMORY_LARGE_BLOCK_SIZE			(1024)			//!< 大きいメモリブロックと判断する閾値
#define MEMORY_MALLOC_FILL_NUMBER		0xCC			//!< メモリを確保したときにフィルする番号
#define MEMORY_FREE_FILL_NUMBER			0xDE			//!< メモリを開放したときにフィルする番号

#define MEMORY_BLOCK_FLAG_CACHE			0x01			//!< キャッシュメモリブロック
#define MEMORY_BLOCK_FLAG_LOCK_CACHE	0x02			//!< キャッシュロック中
#define MEMORY_BLOCK_FLAG_INITIALIZED	0x04			//!< 初期化済み
#define MEMORY_BLOCK_FLAG_MARKER_UNUSE	0x80			//!< マーカ(未使用)

#ifdef RANDOM_FILL
#undef	NO_FILL
#endif

#if defined(NDEBUG) || defined(NO_CHECK)
#define CHECK()
#else
#define CHECK()	check();
#endif

#if defined(NDEBUG)
#define NO_FILL
#define NO_CHECK
#undef RANDOM_FILL
#endif

namespace Graph2D
{
	////////////////////////////////////////////////////////////////////////////////////////////////////
#if !defined(TARGET_WINDOWS)
	const int Allocator::MEMORY_ALIGN_SIZE;			//!< 標準アライメント
#endif

	////////////////////////////////////////////////////////////////////////////////////////////////////
#if	defined(RANDOM_FILL)
	static void memset_rand(void* pointer, int size)
	{
		for(int i = 0; i < size; i++)
		{
			static unsigned long next = 1;
			next = next * 1103515245L + 12345;
			reinterpret_cast<unsigned char*>(pointer)[i] = (next >> 16);
		}
	}
#endif

	////////////////////////////////////////////////////////////////////////////////////////////////////
	void Allocator::MemoryBlock::insertPreviousUseList(Allocator::MemoryBlock* block)
	{
		this->previous->next = block;
		block->previous = this->previous;
		block->next = this;
		this->previous = block;
	}

	void Allocator::MemoryBlock::insertNextUseList(Allocator::MemoryBlock* block)
	{
		this->next->previous = block;
		block->previous = this;
		block->next = this->next;
		this->next = block;
	}

	void Allocator::MemoryBlock::removeUseList()
	{
		previous->next = next;
		next->previous = previous;
	}

	void Allocator::MemoryBlock::removeFreeList()
	{
		previousFree->nextFree = nextFree;
		nextFree->previousFree = previousFree;
	}

	void Allocator::MemoryBlock::removeCacheList()
	{
		previousCache->nextCache = nextCache;
		nextCache->previousCache = previousCache;
	}

	void Allocator::MemoryBlock::mergePreviousFreeList()
	{
		assert(previous);
		assert(size < 0);
		previous->identification = identification;
		previous->reffrenceCount = reffrenceCount;
		previous->flag = flag;
		previous->size = -(previous->size - size);
#if !defined(NDEBUG)
		previous->comment = comment;
#endif
		previous->removeFreeList();
		removeUseList();
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	void Allocator::insertFreeList(MemoryBlock* block)
	{
		MemoryBlock* pointer = block->previous;
		while(pointer->size <= 0)
		{
			pointer = pointer->previous;
		}
		if(pointer == block)
		{
			block_expansion.previousFree->nextFree = block;
			block->previousFree = block_expansion.previousFree;
			block->nextFree = &block_expansion;
			block_expansion.previousFree = block;
		}
		else
		{
			pointer->nextFree->previousFree = block;
			block->previousFree = pointer;
			block->nextFree = pointer->nextFree;
			pointer->nextFree = block;
		}
	}

	void Allocator::insertCacheList(Allocator::MemoryBlock* block)
	{
		block_expansion.previousCache->nextCache = block;
		block->previousCache = block_expansion.previousCache;
		block->nextCache = &block_expansion;
		block_expansion.previousCache = block;
	}

	int Allocator::getAlignedSize(int size)
	{
		return (size + (MEMORY_ALIGN_SIZE - 1)) & -MEMORY_ALIGN_SIZE;
	}

	int Allocator::getHeaderSize()
	{
		return getAlignedSize(sizeof(MemoryBlock));
	}

	Allocator::MemoryBlock* Allocator::toHeader(void* block)
	{
		return reinterpret_cast<MemoryBlock*>(reinterpret_cast<int>(block) - getHeaderSize());
	}

	void* Allocator::toData(Allocator::MemoryBlock* block)
	{
		return reinterpret_cast<void*>(reinterpret_cast<int>(block) + getHeaderSize());
	}

	Allocator::Allocator()
		: block_head_address(NULL)
#if !defined(NDEBUG)
		, block_tail_address(NULL)
#endif
		, size(0)
		, number_of_blocks(1)
	{
	}

	Allocator::~Allocator()
	{
		finalize();
	}

	/*!
	 * @brief	ヒープメモリブロックの作成と初期化を行います
	 * @param	buffer		先頭ヒープメモリアドレス
	 * @param	size		ヒープメモリサイズ
	 */
	void Allocator::initialize(void* buffer, int size)
	{
		union
		{
			MemoryBlock* bh;
			void* p;
			int l;
		}a0, a1;

		a0.p = buffer;
		a1.l = getAlignedSize(a0.l);

		// ヒープのサイズを計算して記録します
		this->size = (size - (a1.l - a0.l)) & (-MEMORY_ALIGN_SIZE);

		// ヒープの先頭サイズを記録します
		block_head_address = a1.bh;

		// 拡張情報の初期化
		memset(a1.bh, 0, sizeof(MemoryBlock));

#if !defined(NDEBUG)
		// ヒープの最終アドレスを記録
		block_tail_address = reinterpret_cast<MemoryBlock*>(reinterpret_cast<char*>(a1.p) + this->size);
#endif

		number_of_blocks = 1;

		a1.bh->previous = a1.bh;
		a1.bh->next = a1.bh;
		a1.bh->size = this->size;

#if !defined(NDEBUG) && !defined(NOFILL)
		{
#if	defined(RANDOM_FILL)
			memset_rand(toData(a1.bh), this->size - getHeaderSize);
#else
			memset(toData(a1.bh), MEMORY_FREE_FILL_NUMBER, this->size - getHeaderSize());
#endif
			printf("create: from %p to %p, size of header %d\n", block_head_address, block_tail_address, getHeaderSize());
		}
#endif

		// キャッシュメモリリストの初期化
		block_expansion.previousCache = &block_expansion;
		block_expansion.nextCache = &block_expansion;

		// フリーメモリリストの初期化
		block_expansion.previousFree = a1.bh;
		block_expansion.nextFree = a1.bh;
		a1.bh->previousFree = &block_expansion;
		a1.bh->nextFree = &block_expansion;

		// セマフォ作成
		//sem_init(&semaphore, 0, 0);
	}

	void Allocator::finalize()
	{
		// セマフォ解放
		//sem_destroy(&semaphore);
	}

	void Allocator::lock()
	{
		//sem_wait(&semaphore);
	}

	void Allocator::unlock()
	{
		//sem_post(&semaphore);
	}

	/*!
	 * @brief	メモリブロックを併合します（セマフォなし）
	 * @param	block		メモリブロックポインタ
	 */
	Allocator::MemoryBlock* Allocator::_merge(Allocator::MemoryBlock* block)
	{
		Allocator::MemoryBlock* side;

		// 未割り当てメモリブロックか調べる
		assert(block->size > 0);

		// 開放するメモリブロックと隣接しているなら併合する
		side = block->next;
		if((side->size > 0) &&(((char*)block + block->size) == (char*)side))
		{
			block->size += side->size;
			side->removeFreeList();
			side->removeUseList();
			number_of_blocks--;
		}

		// 開放するメモリブロックと隣接しているなら併合する
		side = block->previous;
		if((side->size > 0) &&(((char*)block - side->size) == (char*)side))
		{
			side->size += block->size;
			block->removeFreeList();
			block->removeUseList();
			number_of_blocks--;

			block = side;
		}

		return block;
	}

	/*!
	 * @brief	メモリブロックを解放します（セマフォなし）
	 * @param	pointer		ポインタ
	 */
	Allocator::MemoryBlock* Allocator::_free(void* pointer)
	{
		MemoryBlock* block;

#if !defined(NDEBUG) 
		{
#if	!defined(NOFILL)
			int allocatedSize = -toHeader(pointer)->size - getHeaderSize();
			if(allocatedSize > 0)
			{
#if	defined(RANDOM_FILL)
				memset_rand(pointer, allocatedSize);
#else
				memset(pointer, MEMORY_FREE_FILL_NUMBER, allocatedSize);
#endif
			}
#endif
			block = toHeader(pointer);
			block->comment = NULL;
		}
#endif

		block = toHeader(pointer);
		block->size = -block->size;
		insertFreeList(block);

		return _merge(block);
	}

	/*!
	 * @brief	メモリブロックを確保します(ファーストマッチ・セマフォなし）
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
	void* Allocator::_malloc_low(int size)
	{
		int remainSize;

		Allocator::MemoryBlock* block = block_expansion.nextFree;

		size = getAlignedSize(size + getHeaderSize());
		for(;;)
		{
			remainSize = block->size - size;
			if(remainSize == 0)
			{
				block->removeFreeList();
				block->identification = 0;
				block->flag = 0;
				block->reffrenceCount = 1;
				block->size = -block->size;

				return toData(block);
			}

			if(remainSize > 0)
				break;

			block = block->nextFree;
			if(block == &block_expansion)
				return NULL;
		}
		if(remainSize <= getHeaderSize())
		{
			// 余り領域でヘッダを作れない場合、ブロックを丸ごと使用する
			block->removeFreeList();
			block->identification = 0;
			block->flag = 0;
			block->reffrenceCount = 1;
			block->size = -block->size;

			return toData(block);
		}
		else
		{
			MemoryBlock* splitBlock = (MemoryBlock*)((char*)block + size);

			block->removeFreeList();
			block->identification = 0;
			block->flag = 0;
			block->reffrenceCount = 1;
			block->size = -size;

			splitBlock->identification = 0;
			splitBlock->flag = 0;
			splitBlock->reffrenceCount = 1;
			splitBlock->size = remainSize;				// 空きブロックサイズ設定
			splitBlock->previous = block;			// リンクポインタ更新
			block->next->previous = splitBlock;	// リンクポインタ更新
			splitBlock->next = block->next;		// リンクポインタ更新
			block->next = splitBlock;				// リンクポインタ更新

			//insertFreeList(splitBlock);

			{
				MemoryBlock* previous = splitBlock->previous;
				while(previous->size <= 0)
					previous = previous->previous;
				if(previous == splitBlock)
				{
					block_expansion.previousFree->nextFree = splitBlock;
					splitBlock->previousFree = block_expansion.previousFree;
					splitBlock->nextFree = &block_expansion;
					block_expansion.previousFree = splitBlock;
				}
				else
				{
					previous->nextFree->previousFree = splitBlock;
					splitBlock->previousFree = previous;
					splitBlock->nextFree = previous->nextFree;
					previous->nextFree = splitBlock;
				}
			}

			number_of_blocks++;

			return toData(block);
		}
	}

	/*!
	 * @brief	メモリブロックを確保します（ファーストマッチ・セマフォなし）
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
	void* Allocator::_malloc_high(int size)
	{
		int remainSize;

		Allocator::MemoryBlock* block = block_expansion.previousFree;

		size = getAlignedSize(size + getHeaderSize());
		for(;;)
		{
			remainSize = block->size - size;
			if(remainSize == 0)
			{
				block->removeFreeList();
				block->identification = 0;
				block->flag = 0;
				block->reffrenceCount = 1;
				block->size = -block->size;

				return toData(block);
			}

			if(remainSize > 0)
				break;

			block = block->previousFree;
			if(block == &block_expansion)
				return NULL;
		}
		if(remainSize <= getHeaderSize())
		{
			block->removeFreeList();
			block->identification = 0;
			block->flag = 0;
			block->reffrenceCount = 1;
			block->size = -block->size;

			return toData(block);
		}
		else
		{
			MemoryBlock* splitBlock = (MemoryBlock*)((char*)block + remainSize);
			splitBlock->identification = 0;
			splitBlock->flag = 0;
			splitBlock->reffrenceCount = 1;
			splitBlock->size = -size;
			splitBlock->previous = block;
			block->size = remainSize;
			block->next->previous = splitBlock;
			splitBlock->next = block->next;
			block->next = splitBlock;

			number_of_blocks++;

			return toData(splitBlock);
		}
	}

	/*!
	 * @brief	メモリブロックを確保します（セマフォなし）
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
	void* Allocator::_malloc(int size)
	{
		/*
		 * 【注意】
		 * _compact_move_cache_blockはhandleまたは未参照cacheを低位メモリへ移動させます。
		 * 低位メモリへは長寿命データを配置するようにしてください。
		 */
		void* pointer = size > MEMORY_LARGE_BLOCK_SIZE ? _malloc_low(size) : _malloc_high(size);

#if !defined(NDEBUG)
		if(pointer)
		{
			MemoryBlock* block = toHeader(pointer);
#if	!defined(NOFILL)
#if	defined(RANDOM_FILL)
			memset_rand(pointer, size);
#else
			memset(pointer, MEMORY_MALLOC_FILL_NUMBER, size);
#endif
#endif
			assert(block->size < 0);

			block->previousFree = 0;
			block->nextFree = 0;
			block->comment = NULL;
		}
#endif

		return pointer;
	}

	/*!
	 * @brief	アドレスを調整したメモリブロックを確保します（セマフォなし）
	 * @param	alignment		アドレス調整サイズ
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
	void* Allocator::_memalign(int alignment, int size)
	{
		if(alignment <= MEMORY_ALIGN_SIZE)
		{
			return _malloc(size);
		}
		else
		{
			int pointer = (int)_malloc(size + getHeaderSize() + alignment);
			if(pointer & (alignment - 1))
			{
				int aligned = ((pointer + getHeaderSize() + (alignment - 1)) & -alignment);
				MemoryBlock* ba = toHeader((void*)pointer);
				MemoryBlock* bb = toHeader((void*)aligned);
				int ba_size = (int)bb - (int)ba;	// aブロックの新しいサイズ
				int bb_size = -ba->size - ba_size;	// bブロックのサイズ

				assert(aligned - pointer >= getHeaderSize());

				bb->identification = 0;
				bb->flag = 0;
				bb->reffrenceCount = 1;
				bb->previous = ba;
				bb->next = ba->next;
				bb->size = -bb_size;

				ba->next->previous = bb;
				ba->next = bb;
				ba->size = -ba_size;

				_free(toData(ba));
				pointer = (int)toData(bb);

				number_of_blocks++;
			}
			return (void*)pointer;
		}
	}

	/*!
	 * @brief	メモリブロックを縮小します（セマフォなし）
	 * @param	block	メモリブロック
	 * @param	size			メモリブロックサイズ
	 */
	void Allocator::_reduce(Allocator::MemoryBlock* block, int size)
	{
		MemoryBlock* splitBlock = (MemoryBlock*)((char*)block + size);
		int difference = -block->size - size;

		if((block->next > block) &&(block->next->size > 0))
		{
			// 次ブロックが連続アドレスでかつ、空きの場合併合する
			MemoryBlock nextBlock;
			memcpy(&nextBlock, block->next, sizeof(MemoryBlock));

			block->next->removeFreeList();

			block->next = splitBlock;
			block->size = -size;

			splitBlock->identification = 0;
			splitBlock->flag = 0;
			splitBlock->reffrenceCount = 1;
			splitBlock->size = difference + nextBlock.size;
			splitBlock->next = nextBlock.next;
			splitBlock->next->previous = splitBlock;
			splitBlock->previous = block;
			insertFreeList(splitBlock);
		}
		else if(difference > getHeaderSize())
		{
			// 空きスペースにヘッダを作れるなら作る
			block->size = -(int)size;
			splitBlock->identification = 0;
			splitBlock->flag = 0;
			splitBlock->reffrenceCount = 1;
			splitBlock->next = block->next;
			splitBlock->next->previous = splitBlock;
			splitBlock->previous = block;
			block->next = splitBlock;
			splitBlock->size = difference;
			insertFreeList(splitBlock);
			number_of_blocks++;
		}
	}

	/*!
	 * @brief	メモリブロックを拡張します（セマフォなし）
	 * @param	block	メモリブロック
	 * @param	size			メモリブロックサイズ
	 * @param	difference		差分（必ず正数）
	 * @return	メモリブロック
	 */
	void* Allocator::_expand(Allocator::MemoryBlock* block, int size, int difference)
	{
		int previousBlockSize;

		// case 1 次のブロックに十分な空きがある
		int nextBlockSize = ((block->next > block) &&(block->next->size > 0)) ? block->next->size : 0;
		if(nextBlockSize >= difference)
		{
			Allocator::MemoryBlock* next = block->next;

			if((nextBlockSize - difference) <= getHeaderSize())
			{
				// case 1-1 次のブロックのサイズがヘッダサイズ以下になるので次のブロックを併合します
				next = block->next;
				assert(next->size > 0);
				block->size -= next->size;
				next->removeFreeList();
				next->removeUseList();
				number_of_blocks--;
			}
			else
			{
				// case 1-2 次のブロックのサイズがヘッダサイズ以上になるので次のブロックを分割します
				Allocator::MemoryBlock* splitBlock;
				Allocator::MemoryBlock* splitNextBlock;
				Allocator::MemoryBlock* splitPreviousBlock;
				int splitSize;

				next->removeFreeList();

				splitBlock = (Allocator::MemoryBlock*)((char*)block + size);
				splitNextBlock = next->next;
				splitPreviousBlock = block;
				splitSize = (-block->size + next->size) - size;

				block->next = splitBlock;
				block->size = -size;

				splitBlock->identification = 0;
				splitBlock->flag = 0;
				splitBlock->reffrenceCount = 1;
				splitBlock->next = splitNextBlock;
				splitBlock->previous = splitPreviousBlock;
				splitBlock->size = splitSize;

				splitNextBlock->previous = splitBlock;

				insertFreeList(splitBlock);
			}

			return toData(block);
		}

		// case 2  前のブロックに十分な空きがある
		previousBlockSize = ((block->previous < block) && (block->previous->size > 0)) ? block->previous->size : 0;
		if(previousBlockSize >= difference)
		{
			Allocator::MemoryBlock* previous = block->previous;

			if((previousBlockSize - difference) <= getHeaderSize())
			{
				// case 2-1 前のブロックのサイズがヘッダサイズ以下になるので、前のブロックと併合します
				block->mergePreviousFreeList();
				assert((-block->size) - getHeaderSize() >= 0);
				memcpy(toData(previous), toData(block), (-block->size) - getHeaderSize());
				number_of_blocks--;
			}
			else
			{
				// case 2-2 前のブロックのサイズがヘッダサイズ以上になるので、ブロックを分割します
				block->mergePreviousFreeList();
				memcpy(toData(previous), toData(block), (-block->size) - getHeaderSize());
				number_of_blocks--;

				Allocator::MemoryBlock* splitBlock = (MemoryBlock*)((char*)previous + size);
				splitBlock->identification = 0;
				splitBlock->flag = 0;
				splitBlock->reffrenceCount = 1;
				splitBlock->next = previous->next;
				splitBlock->previous = previous;
				splitBlock->size = -previous->size - size;

				previous->next->previous = splitBlock;

				previous->next = splitBlock;
				previous->size = -(int)size;

				insertFreeList(splitBlock);
				number_of_blocks++;
			}

			return toData(previous);
		}

		// case 3  前後のブロックを併合可能

		if((previousBlockSize + nextBlockSize) >= difference)
		{
			Allocator::MemoryBlock* previous = block->previous;

			// blockを開放
			block->size = -block->size;
			insertFreeList(block);

			// 前後のブロックと併合
			_merge(block);

			previous->removeFreeList();
			previous->identification = 0;
			previous->flag = 0;
			previous->reffrenceCount = 1;

			memcpy(toData(previous), toData(block), previous->size - getHeaderSize());
			previous->size = -previous->size;
			_reduce(previous, size);

			return toData(previous);
		}
		else
		// case 4 新たにメモリを確保
		{
			void* newBlock = _malloc(size - getHeaderSize());
			if(newBlock)
			{
				memcpy(newBlock, toData(block), (-block->size) - getHeaderSize());
				_free(toData(block));
			}
			return newBlock;
		}
	}

	/*!
	 * @brief	メモリブロックを再確保します（セマフォなし）
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
	void* Allocator::_realloc(void* pointer, int size)
	{
		if(pointer == NULL)
		{
			return _malloc(size);
		}

		if((pointer != NULL) && (size == 0))
		{
			_free(pointer);
			return NULL;
		}

		Allocator::MemoryBlock* block = toHeader(pointer);

		assert(block->size < 0);

		size = getAlignedSize(size + getHeaderSize());

		int difference = size + block->size;
		if(difference == 0)
		{
			return pointer;
		}
		else if(difference < 0)
		{
			_reduce(block, size);
			return pointer;
		}
		else
		{
			return _expand(block, size, difference);
		}
	}

	/*!
	 * @brief	アドレスを調整したメモリブロックを確保します（セマフォなし）
	 * @param	alignment		アドレス調整サイズ
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
#if !defined(NDEBUG)
	void* Allocator::memalign(int aligment, int size, const char* comment)
#else
	void* Allocator::memalign(int aligment, int size)
#endif
	{
		void* pointer;

		assert(aligment >= 0);
		assert(size >= 0);

		CHECK();
		lock();

		for(;;)
		{
			pointer = _memalign(aligment, size);
			if(pointer != NULL)
			{
#if !defined(NDEBUG)
				toHeader(pointer)->comment = comment;
#endif
				break;
			}

#if defined(ENABLE_COMPACTION)
			if((!_compact(size)) && (!_cache_sweep(size)))
				break;
#else
			if(!cache_sweep(size))
				break;
#endif
		}

		unlock();
		CHECK();

		return pointer;
	}

	/*!
	 * @brief	メモリブロックを確保します
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
#if !defined(NDEBUG)
	void* Allocator::malloc(int size, const char* comment)
#else
	void* Allocator::malloc(int size)
#endif
	{
		void* pointer;

		assert(size >= 0);

		CHECK();
		lock();

		for(;;)
		{
			pointer = _malloc(size);
			if(pointer != NULL)
			{
#if !defined(NDEBUG)
				toHeader(pointer)->comment = comment;
#endif
				break;
			}

#if defined(ENABLE_COMPACTION)
			if((!_compact(size)) && (!_cache_sweep(size)))
				break;
#else
			if(!cache_sweep(size))
				break;
#endif
		}

		unlock();
		CHECK();

		return pointer;
	}

	/*!
	 * @brief	メモリブロックを確保し、初期化します
	 * @param	count		バッファ個数
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
#if !defined(NDEBUG)
	void* Allocator::calloc(int count, int size, const char* comment)
#else
	void* Allocator::calloc(int count, int size)
#endif
	{
		void* pointer;

		assert(count >= 0);
		assert(size >= 0);

		CHECK();
		lock();

		for(;;)
		{
			pointer = _malloc(count * size);
			if(pointer != NULL)
			{
				memset(pointer, 0, count * size);
#if !defined(NDEBUG)
				toHeader(pointer)->comment = comment;
#endif
				break;
			}

#if defined(ENABLE_COMPACTION)
			if((!_compact(size)) && (!_cache_sweep(size)))
				break;
#else
			if(!cache_sweep(size))
				break;
#endif
		}

		unlock();
		CHECK();

		return pointer;
	}

	/*!
	 * @brief	メモリブロックを再確保します
	 * @param	pointer		バッファアドレス
	 * @param	size		バッファサイズ
	 * @return	メモリブロック
	 */
#if !defined(NDEBUG)
	void* Allocator::realloc(void* pointer, int size, const char* comment)
#else
	void* Allocator::realloc(void* pointer, int size)
#endif
	{
		void* new_pointer;

		assert(size >= 0);

		CHECK();
		lock();

		for(;;)
		{
			new_pointer = _realloc(pointer, size);
			if(new_pointer != NULL)
			{
#if !defined(NDEBUG)
				toHeader(new_pointer)->comment = comment;
#endif
				break;
			}

#if defined(ENABLE_COMPACTION)
			if((!_compact(size)) && (!_cache_sweep(size)))
				break;
#else
			if(!cache_sweep(size))
				break;
#endif
		}

		unlock();
		CHECK();

		return new_pointer;
	}

	/*****************************************************************************/
	/* memory cache */
	/*****************************************************************************/
	/*!
	 * @brief	ハンドル領域と参照無しのキャッシュ領域のコンパクションを実行する。
	 * @note	まだ、ポインタを飛び越えたメモリブロックの移動が出来ない。
	 *
	 *	+-----------------------+
	 *	| free                  |
	 *	+-----------------------+
	 *	| handle又は未参照cache | メモリがこの状態の場合ブロックを前に移動する
	 *	+-----------------------+
	 */
	void Allocator::_compact_move_cache_block(MemoryBlock* block)
	{
		MemoryBlock* next = block->next;
		if(! (next->flag & MEMORY_BLOCK_FLAG_LOCK_CACHE))
		{
			int freeAreaSize = block->size;
			MemoryBlock* savedPreviousCacheBlock = NULL;
			MemoryBlock* savedNextCacheBlock = NULL;

			// キャッシュブロックならば前後のブロックのポインタを保存
			if(next->flag & MEMORY_BLOCK_FLAG_CACHE)
			{
				savedPreviousCacheBlock = next->previousCache;
				savedNextCacheBlock = next->nextCache;
			}

			// このfreeブロックをフリーリストから削除
			block->removeFreeList();

			// メモリブロックを移動します
			next->previous = block->previous;
			memcpy(block, next, -next->size);

			// メモリブロックを分割して、使用中の状態に変更します
			next = ((MemoryBlock *)((int)(block) - block->size));
			next->identification = 0;
			next->flag = 0;
			next->reffrenceCount = 1;
			next->previous = block;
			next->next = block->next;
			next->size = -freeAreaSize;

#if !defined(NDEBUG)
			// メモリブロックを移動した証として内容を0xEEに変更
			memset(toData(next), 0xEE, -next->size - getHeaderSize());
#endif

			// 分割したメモリブロックのリンク情報を修正します
			block->next->previous = next;
			block->next = next;

			// freeブロックを併合します
			_free(toData(next));

			// このキャッシュブロックをキャッシュリストに追加します
			if(next->flag & MEMORY_BLOCK_FLAG_CACHE)
			{
				savedPreviousCacheBlock->nextCache = block;
				savedNextCacheBlock->previousCache = block;
			}

			// 最終チェック
			assert(block->previousCache == savedPreviousCacheBlock);
			assert(block->nextCache == savedNextCacheBlock);
			assert(block->previous->next == block);
			assert(block->next == next);
			assert(next->previous == block);
			assert(next->next->previous == next);
#if !defined(NDEBUG)
			if(next->flag & MEMORY_BLOCK_FLAG_CACHE)
			{
				assert(block == block->previousCache->nextCache);
				assert(block == block->nextCache->previousCache);
			}
#endif
		}
	}

	/*!
	 * @brief	ハンドル領域と参照無しのキャッシュ領域のコンパクションを実行する。
	 * @note	まだ、ポインタを飛び越えたメモリブロックの移動が出来ない。
	 */
	bool Allocator::_compact(int size)
	{
		MemoryBlock* block = block_head_address;

		do{
			if(block->size > 0 && block->next->size < 0)
			{
				/*
				 * +---------------------+
				 * | free                |　メモリがこの状態の場合、handleを前に移動する
				 * +---------------------+
				 * | cache (reference=0) |
				 * +---------------------+
				 */
				_compact_move_cache_block(block);

				if(size >= block->next->size)
					return true;
			}

			block = block->next;
		}while(block->next != block_head_address);

		return false;
	}

	/*!
	 * @brief	参照無しのキャッシュ領域の軽いコンパクションを実行する。
	 * @param	blocks		最大で解放するブロック数
	 */
	int Allocator::_compact_easy(int blocks)
	{
		MemoryBlock* cacheBlock = block_expansion.nextCache;
		int count = 0;

		while(cacheBlock && cacheBlock != &block_expansion && blocks > 0)
		{
			MemoryBlock* block = cacheBlock->previous;
			MemoryBlock* next = cacheBlock;
			MemoryBlock* nextCache = cacheBlock->nextCache;

			if(block->size > 0 && next->size < 0)
			{
				if(next->identification != 0 && next->reffrenceCount <= 0)
				{
					_compact_move_cache_block(block);
					CHECK();
					blocks--;
					count++;
				}
			}

			cacheBlock = nextCache;
		}

		return count;
	}

	/*!
	 * @brief	ハンドル領域のコンパクションを実行する
	 * @note	まだ、ポインタを飛び越えたメモリブロックの移動が出来ない。
	 */
	bool Allocator::compact(int size)
	{
		lock();
		const bool result = _compact(size);
		unlock();
		return result;
	}

	/*!
	 * @brief	キャッシュ領域の軽いコンパクションを実行する
	 * @param	blocks	最大で移動するブロック数
	 */
	void Allocator::compact_easy(int blocks)
	{
		lock();
		_compact_easy(blocks);
		unlock();
	}

	/*!
	 * @brief	未参照のメモリブロックを解放します（セマフォなし）
	 * @param	必要なデータサイズ
	 */
	bool Allocator::_cache_sweep(int size)
	{
		MemoryBlock* block = block_expansion.nextCache;

		while(block != &block_expansion)
		{
			int is_locked_handle = block->flag & MEMORY_BLOCK_FLAG_LOCK_CACHE;
			if(!is_locked_handle && block->size < 0 && block->reffrenceCount == 0)
			{
				MemoryBlock* nextCache = block->nextCache;

				// キャッシュリストから削除
				block->removeCacheList();
				block->previousCache = 0;
				block->nextCache = 0;

				// 隣接している空きブロックのポインタが返る
				block = _free(toData(block));

				// 十分な連続空き領域があるか？
				if(block->size >= size + getHeaderSize())
					return true;

				block = nextCache;
			}
			else
			{
				block = block->nextCache;
			}
		}

		return false;
	}

	/*!
	 * @brief	未参照のメモリブロックを解放します（セマフォあり）
	 * @param	解放データサイズ
	 */
	bool Allocator::cache_sweep(int size)
	{
		lock();
		const bool result = _cache_sweep(size);
		unlock();
		return result;
	}

	/*!
	 * @brief	全ての未参照のメモリブロックを解放します（セマフォなし）
	 * @param	解放データサイズ
	 */
	int Allocator::_cache_sweep_all()
	{
		int count = 0;
		MemoryBlock* block = block_expansion.nextCache;

		while(block != &block_expansion)
		{
			int is_locked_handle = block->flag & MEMORY_BLOCK_FLAG_LOCK_CACHE;
			if(block->size < 0 && block->reffrenceCount == 0 && !is_locked_handle)
			{
				MemoryBlock* next = block->nextCache;
				block->removeCacheList();
				block->previousCache = 0;
				block->nextCache = 0;

				block = _free(toData(block));
				count++;
				block = next;
			}
			else
			{
				block = block->nextCache;
			}
		}

		return count;
	}

	/*!
	 * @brief	すべての未参照のメモリブロックを解放します（セマフォあり）
	 * @param	解放データサイズ
	 */
	void Allocator::cache_sweep_all()
	{
		lock();
		_cache_sweep_all();
		unlock();
	}

	/*!
	 * @brief	IDが一致するメモリブロックを検索します
	 * @param	identification		ID
	 * @return	メモリポインタ
	 */
	void* Allocator::_cache_assign(unsigned short identification)
	{
		MemoryBlock* block = block_expansion.nextCache;

		while(block != &block_expansion)
		{
			if(block->identification == identification)
			{
				block->reffrenceCount++;
				block->removeCacheList();
				insertCacheList(block);
				assert(block->reffrenceCount > 0);
				return toData(block);
			}
			block = block->nextCache;
		}
		return 0;
	}

	/*!
	 * @brief	IDからメモリブロックを割り当てます
	 * @note	割り当てたメモリブロックの参照数が増えるので、使用後は必ずcache_freeして下さい。
	 * @param	identification	任意のデータID
	 * @return	割り当てたメモリ領域のアドレス
	 */
	void* Allocator::cache_assign(unsigned short identification)
	{
		void* block;

		lock();
		block = _cache_assign(identification);
		unlock();

		return block;
	}

	/*!
	 * @brief	メモリを割り当てる
	 * @note	既に同じIDが割り当ててある場合、割り当て済みのポインタを返します。
	 * @param	size	確保するメモリサイズ（バイト単位）
	 * @param	identification	任意のデータID
	 * @return	割り当てたメモリ領域のアドレス
	 */
	void* Allocator::cache_malloc(unsigned short identification, int size)
	{
		lock();

		void* pointer = _cache_assign(identification);
		if(pointer == NULL)
		{
			for(;;)
			{
				// キャッシュは隣接しやすいように低位に置く
				if((pointer = _malloc_low(size)) != NULL)
				{
					MemoryBlock* block = toHeader(pointer);
					block->identification = identification;
					block->flag = MEMORY_BLOCK_FLAG_CACHE;
#if !defined(NDEBUG)
					block->comment = NULL;
#endif
					insertCacheList(block);
					break;
				}

#if defined(ENABLE_COMPACTION)
				if((!_compact(size)) && (!_cache_sweep(size)))
					break;
#else
				if(!_cache_sweep(size))
					break;
#endif
			}
		}

		unlock();

		return pointer;
	}

	void* Allocator::_memalign_low(int alignment, int size)
	{
		if(alignment <= MEMORY_ALIGN_SIZE)
		{
			return _malloc_low(size);
		}
		else
		{
			int pointer = (int)_malloc_low(size + getHeaderSize() + alignment);
			if(pointer & (alignment - 1))
			{
				int aligned = ((pointer + getHeaderSize() +(alignment - 1)) & -alignment);
				{
					MemoryBlock* ba = toHeader((void*)pointer);
					MemoryBlock* bb = toHeader((void*)aligned);
					int ba_size =(int)bb -(int)ba;		// aブロックの新しいサイズ
					int bb_size = -ba->size - ba_size;	// bブロックのサイズ

					assert(aligned - pointer >= getHeaderSize());

					bb->identification = 0;
					bb->flag = 0;
					bb->reffrenceCount = 1;
					bb->previous = ba;
					bb->next = ba->next;
					bb->size = -bb_size;

					ba->next->previous = bb;
					ba->next = bb;
					ba->size = -ba_size;

					_free(toData(ba));
					pointer =(int)toData(bb);

					++number_of_blocks;
				}
			}
			return (void*)pointer;
		}
	}

	/*!
	 * @brief	アドレスを調整したメモリブロックを確保します
	 * @note	既に同じIDが割り当ててある場合、割り当て済みのポインタを返します。
	 * @param	alignment	アドレス調整サイズ
	 * @param	size	確保するメモリサイズ（バイト単位）
	 * @param	identification	任意のデータID
	 * @return	割り当てたメモリ領域のアドレス
	 */
	void* Allocator::cache_memalign(unsigned short identification, int alignment, int size)
	{
		lock();

		void* pointer = _cache_assign(identification);
		if(pointer == NULL)
		{
			for(;;)
			{
				// キャッシュは隣接しやすいように低位に置く
				if((pointer = _memalign_low(alignment, size)) != NULL)
				{
					MemoryBlock* block = toHeader(pointer);
					block->identification = identification;
					block->flag = MEMORY_BLOCK_FLAG_CACHE;
#if !defined(NDEBUG)
					block->comment = NULL;
#endif
					insertCacheList(block);
					break;
				}

#if defined(ENABLE_COMPACTION)
				if((!_compact(size)) && (!_cache_sweep(size)))
					break;
#else
				if(!_cache_sweep(size))
					break;
#endif
			}
		}

		unlock();

		return pointer;
	}

	/*!
	 * @brief	データが初期化されたかしらべます
	 * @note	初期終了はUnlockメソッドで宣言されます
	 * @param	pointer	メモリ領域のアドレス
	 */
	int Allocator::cache_is_initialized(void* pointer)
	{
		int initialized;

		if(pointer)
		{
			lock();
			initialized = toHeader(pointer)->flag & MEMORY_BLOCK_FLAG_INITIALIZED;
			CHECK();
			unlock();
		}
		else
		{
			initialized = 0;
		}

		return initialized;
	}

	/*!
	 * @brief	データを固定します
	 * @note	lockされたデータは参照回数がゼロ以下になっても解放されません。
	 * @param	pointer	メモリ領域のアドレス
	 */
	void Allocator::cache_lock(void* pointer)
	{
		if(pointer)
		{
			lock();
			toHeader(pointer)->flag |= MEMORY_BLOCK_FLAG_LOCK_CACHE;
			CHECK();
			unlock();
		}
	}

	/*!
	 * @brief	データを固定解除します
	 * @note	データの参照回数がゼロ以下になっていた場合はcache_freeが呼ばれデータが解放されます。
	 * @param	pointer	メモリ領域のアドレス
	 */
	void Allocator::cache_unlock(void* pointer)
	{
		if(pointer)
		{
			lock();
			MemoryBlock* block = toHeader(pointer);
			block->flag |= MEMORY_BLOCK_FLAG_INITIALIZED;
			block->flag &= ~MEMORY_BLOCK_FLAG_LOCK_CACHE;
			CHECK();
			unlock();
		}
	}

	/*!
	 * @brief	メモリの情報を取得します
	 * @param	total_size	総メモリ量
	 * @param	use_size		参照ありメモリ量
	 * @param	free_size	参照なしメモリ量
	 * @return	ブロック数
	 */
	unsigned int Allocator::cache_get_infomation(unsigned int& total_size, unsigned int& use_size, unsigned int& free_size)
	{
		MemoryBlock* block;
		unsigned int blockCount;

		total_size = 0;
		use_size = 0;
		free_size = 0;
		blockCount = 0;

		lock();
		block = block_head_address;

		do{
			blockCount++;
			if(block->size < 0)
			{
				if(block->reffrenceCount > 0)
				{
					use_size -= block->size;
				}else{
					free_size -= block->size;
				}
				total_size -= block->size;
			}
			else
			{
				total_size += block->size;
			}
			block = block->next;
		}while(block != block_head_address);

		CHECK();
		unlock();

		return blockCount;
	}

	/*!
	 * @brief	メモリブロックを解放します
	 * @param	pointer		ポインタ
	 */
	void Allocator::free(void* pointer)
	{
		if(pointer)
		{
			CHECK();
			lock();

			MemoryBlock* block = toHeader(pointer);
			assert(block->size < 0);

			if(block->flag & MEMORY_BLOCK_FLAG_CACHE)
			{
				assert(block->previousCache != 0);
				assert(block->nextCache != 0);
				if(block->reffrenceCount > 0)
				{
					--block->reffrenceCount;
				}
			}
			else
			{
				_free(pointer);
			}

			unlock();
			CHECK();
		}
	}

	/*!
	 * @brief	確保されているメモリサイズを取得します
	 * @param	pointer		バッファアドレス
	 * @return	確保されているメモリサイズ
	 */
	unsigned int Allocator::get_size(void* pointer)
	{
		unsigned int size = 0;
		if(pointer)
		{
			Allocator::MemoryBlock* block;
			lock();
			block = toHeader(pointer);
			size = (-block->size) - getHeaderSize();
			unlock();
		}
		CHECK();
		return size;
	}

	/*!
	 * @brief	メモリブロック数を取得します
	 * @return	メモリブロックの数
	 */
	unsigned int Allocator::get_block_count(void)
	{
		return number_of_blocks;
	}

	/*!
	 * @brief	メモリのサイズを取得します
	 * @return	メモリのサイズ
	 */
	unsigned int Allocator::get_total_size(void)
	{
		return size;
	}

	/*!
	 * @brief	最も大きいメモリブロックのサイズを取得します（セマフォなし）
	 * @return	最も大きいメモリブロックのサイズ
	 */
	int Allocator::_get_available_size()
	{
		int max = 0;

		MemoryBlock* block = block_head_address;
		do{
			if(block->size > max)
			{
				max = block->size;
			}
			block = block->next;
		}while(block != block_head_address);

		return max - getHeaderSize();
	}

	/*!
	 * @brief	最も大きいメモリブロックのサイズを取得します
	 * @return	最も大きいメモリブロックのサイズ
	 */
	unsigned int Allocator::get_available_size()
	{
		CHECK();
		lock();
		unsigned int size = _get_available_size();
		unlock();
		CHECK();
		return size;
	}

	/*!
	 * @brief	総空き容量を取得します（セマフォなし）
	 * @return	総空き容量
	 */
	int Allocator::_get_free_size()
	{
		MemoryBlock* block = block_head_address;
		int totalSize = 0;
		do{
			if(block->size > 0)
			{
				totalSize += block->size;
			}
			block = block->next;
		}while(block != block_head_address);
		return totalSize;
	}

	/*!
	 * @brief	総空き容量を取得します
	 * @return	総空き容量
	 */
	unsigned int Allocator::get_free_size()
	{
		unsigned int totalSize;
		CHECK();
		lock();
		totalSize = _get_free_size();
		unlock();
		CHECK();
		return totalSize;
	}

	/*!
	 * @brief	総使用量を取得します（セマフォなし）
	 * @return	総使用量
	 */
	int Allocator::_get_used_size()
	{
		MemoryBlock* block = block_head_address;
		int totalSize = 0;
		do{
			if(block->size < 0)
			{
				totalSize += -block->size;
			}
			block = block->next;
		}while(block != block_head_address);

		return totalSize;
	}

	/*!
	 * @brief	総使用量を取得します
	 * @return	総使用量
	 */
	unsigned int Allocator::get_used_size()
	{
		unsigned int totalSize;
		CHECK();
		lock();
		totalSize = _get_used_size();
		unlock();
		CHECK();
		return totalSize;
	}

	/*!
	 * @brief	参照回数を取得します
	 * @param	pointer	メモリ領域のアドレス
	 * @return	参照回数
	 */
	unsigned char Allocator::get_reffrence_count(void* pointer)
	{
		unsigned char reffrenceCount;
		lock();
		reffrenceCount = toHeader(pointer)->reffrenceCount;
		CHECK();
		unlock();
		return reffrenceCount;
	}


#if !defined(NDEBUG)
	/*!
	 * @brief	メモリ管理構造に異常がないか検査する
	 */
	void Allocator::check(void)
	{
		MemoryBlock* block;
		MemoryBlock* head;
		MemoryBlock* tail;
		int blocks;

		lock();

		// 変数初期化
		blocks = number_of_blocks;
		head = block_head_address;
		tail = block_tail_address;

#define INVALID(p) ((p < head) || (p >= tail) || (((int) p) & (MEMORY_ALIGN_SIZE - 1)))

		// 使用領域のリンクを調べる
		{
			block = block_head_address;
			do{
				// メモリブロックがヒープの範囲に入っており、アライメントからずれていないか調べる
				assert(!INVALID(block));

				// 次のメモリブロックがヒープの範囲に入っており、アライメントからずれていないか調べる
				MemoryBlock* next = block->next;
				assert(!INVALID(next));

				// 次のメモリブロックの前のメモリブロックへのポインタが自分のアドレスか調べる
				assert(next->previous == block);

				// サイズを足すと次のメモリブロックのアドレスになるか調べる
				if(next != block_head_address)
				{
					int size = block->size;
					if(size < 0)
						size = -size;
					assert(reinterpret_cast<MemoryBlock*>(reinterpret_cast<char*>(block) + size) == next);
				}

				// デバック用に未使用フラグを下げる
				block->flag &= ~MEMORY_BLOCK_FLAG_MARKER_UNUSE;

				// 使用中のブロックの場合、空きブロックリンクがNULLか調べる
				if(block->size > 0)
				{
					assert(block->previousFree);
					assert(block->nextFree);
				}

				// 次のメモリブロックへ
				block = next;

				// メモリブロックの総数確認用
				blocks--;
			} while(block != block_head_address);
		}

		// キャッシュ領域のリンクを調べる
		{
			block = block_expansion.nextCache;
			while(block != &block_expansion)
			{
				// メモリブロックがヒープの範囲に入っており、アライメントからずれていないか調べる
				assert(!INVALID(block));

				// 次のメモリブロックがヒープの範囲に入っており、アライメントからずれていないか調べる
				MemoryBlock* next = block->nextCache;
				if(next != &block_expansion)
				{
					assert(!INVALID(next));
				}

				// 次のメモリブロックの前のメモリブロックへのポインタが自分のアドレスか調べる
				assert(next->previousCache == block);

				// 次のメモリブロックへ
				block = next;
			}
		}

		// 空き領域のリンクを調べる
		{
			block = block_expansion.nextFree;
			while(block != &block_expansion)
			{
				// メモリブロックがヒープの範囲に入っており、アライメントからずれていないか調べる
				assert(!INVALID(block));

				// 次のメモリブロックの前のメモリブロックへのポインタが自分のアドレスか調べる
				MemoryBlock* next = block->nextFree;
				assert(next->previousFree == block);
			
				// デバック用に未使用フラグを上げる
				block->flag |= MEMORY_BLOCK_FLAG_MARKER_UNUSE;

				// 空きブロックリンクがNULLか調べる
				assert(block->size > 0);
				assert(block->previousFree);
				assert(block->nextFree);

				// 次のメモリブロックへ
				block = next;
			}
		}

		// マークした領域が本当に開き領域か調べる
		for(block = block_head_address; block != block_head_address; block = block->next)
		{
			// 空きブロックにマークが付いているか調べる
			if(block->size > 0)
			{
				assert(block->flag & MEMORY_BLOCK_FLAG_MARKER_UNUSE);
			}
		}
#undef INVALID

		// リンクの結果とブロック数が一致しているか調べる
		assert(blocks == 0);

		unlock();
	}

	/*!
	 * @brief	メモリブロックのコメントを設定する（デバッグ用）
	 */
	void Allocator::set_message(void* pointer, const char* message)
	{
		CHECK();
		lock();
		toHeader(pointer)->comment = message;
		unlock();
		CHECK();
	}

#endif
}
