/**
 * LiveML - LiveML is screen(graphic) controling library using XML.
 *
 * LGPL License
 * Copyright (C) 2010 Nothan
 * http://github.com/nothan/liveml/
 * All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Nothan
 * private@nothan.xrea.jp
 *
 * Tsuioku Denrai
 * http://tsuioku-denrai.xrea.jp/
 */

#include "livemlparser.h"

namespace lmlTag
{
  LML_TAGFUNC(liveml)
  {
    return XML_TAGRTN_LEVEL;
  }

  LML_TAGFUNC(actor)
  {
    LiveMLParser &p = (LiveMLParser&)parser;
    ActorParser *actor = p.actorContainer.add((XMLElement*)p.getCurrentTag());
    p.setCurrentActor(actor);
    actor->eventContainer.resize(p.eventTypeContainer.size());
    
    p.unionId = p.unionContainer.NOTFOUND;
    p.unionContainer.clear();

    return XML_TAGRTN_LEVEL;
  }

  LML_TAGFUNC(runion)
  {
    LiveMLParser &p = (LiveMLParser&)parser;
    p.unionId = p.unionContainer.add(param.getString(0));

    return XML_TAGRTN_NULL;
  }

  LML_TAGFUNC(runion_close)
  {
    LiveMLParser &p = (LiveMLParser&)parser;
    p.unionId = p.unionContainer.NOTFOUND;

    return XML_TAGRTN_NULL;
  }

  LML_TAGFUNC(event)
  {
    LiveMLParser &p = (LiveMLParser&)parser;
    TagType *rt = ((XMLElement*)p.getParentTag())->type;
    EventContainer *ec = NULL;

    size_t et = param.getUInteger(0);

    if (!p.eventTypeContainer.has(et))
    {
      return XML_TAGRTN_SKIP;
    }

    ec = &p.getCurrentActor()->eventContainer;
    if (ec == NULL)
    {
      return XML_TAGRTN_SKIP;
    }

    DCODE(printf("added event[%d]\n", param.getUInteger(0));)
    ec->get(et)->tag = (XMLElement*)p.getCurrentTag();

    return XML_TAGRTN_LEVEL;
  }

  LML_TAGFUNC(action)
  {
    LiveMLParser &p = (LiveMLParser&)parser;
    TagType *rt = ((XMLElement*)p.getParentTag())->type;
    ActionContainer *ac = NULL;

    ac = &p.getCurrentActor()->actionContainer;
    if (rt->getName() == "liveml")
    {
      ac = &p.actionContainer;
    }
    if (ac == NULL)
    {
      return XML_TAGRTN_SKIP;
    }

    DCODE(printf("added action[%s]\n", param.getString(0));)

    ActionParser *ap = ac->add((XMLElement*)p.getCurrentTag());
    ap->unionId = p.unionId;

    return XML_TAGRTN_LEVEL;
  }

  LML_TAGFUNC(include)
  {
    LiveMLParser &p = (LiveMLParser&)parser;
    XMLNode *t = p.getCurrentTag();
    p.setCurrentTag(NULL);
    bool result = p.loadFile(param.getString(0));
    p.setCurrentTag(t);

    return XML_TAGRTN_NULL;
  }
}

// parameter types
PARAMFUNC(LiveMLParser::paramText)
{
//  const char *encode = text_encode((char*)value, variableEncoder, (void*)&((LiveMLParser&)parser).variableContainer);

  return new XMLText(((XMLElement*)parser.getCurrentTag())->child);
}

PARAMFUNC(LiveMLParser::paramNumeric)
{
  XMLNode* node = ((XMLElement*)parser.getCurrentTag())->child;
  string tmp;
  if (node != NULL)
  {
    char num[12];
    while (node)
    {
      if (node->isText()) tmp += ((XMLTextNode*)node)->text;
      else
      {
        sprintf(num, "%d", node);
        tmp += "e(";
        tmp += num;
        tmp += ")";
      }
      node = node->next;
    }
    value = tmp.c_str();
  }

  calcu_encode_data encode = calcu_encode((char*)value, NULL);

  return new XMLNumeric(encode.first);
}

PARAMFUNC(LiveMLParser::paramVar)
{
  // name to id
  variable_size id = ((LiveMLParser&)parser).variableEncoder(value, (void*)&((LiveMLParser&)parser).variableContainer);

  return new XMLVar(id);
}

PARAMFUNC(LiveMLParser::paramActor)
{
  size_t id = ((LiveMLParser&)parser).actorTypeContainer.getId(value);

  return new XMLSizeT(id);
}

PARAMFUNC(LiveMLParser::paramAction)
{
  ActionContainer *ac = &((LiveMLParser&)parser).getCurrentActor()->actionContainer;
  size_t id = ac->getId(value);

  return new XMLSizeT(id);
}

PARAMFUNC(LiveMLParser::paramEvent)
{
  size_t id = ((LiveMLParser&)parser).eventTypeContainer.getId(value);

  return new XMLSizeT(id);
}

int LiveMLParser::calcuEncode(char **input, char **output, void *work)
{
  LiveMLParser &parser = *(LiveMLParser*)work;
  calcu_encode_data result;
  char *current = *output;
  char *text = *input;

  /* variable */
  if (*text == '$')
  {
    variable_size *v = (variable_size*)(current + 1);
    char *name = ++text;
    char save;

    while (
      (*text >= 'a' && *text <= 'z') ||
      (*text >= 'A' && *text <= 'Z') ||
      *text == '.' || *text == '_') text++;
    save = *text;
    *text = '\0';

    /* output */
    *current = CALCU_VARIABLE;
    *v = parser.variableEncoder((const char*)name, (void*)&((LiveMLParser&)parser).variableContainer);
    *text = save;
    current += 1 + sizeof(variable_size);

    *output = current;
    *input = text;

    return true;
  }

  /* element */
  if (text[0] == 'e' && text[1]  == '(')
  {
    *current++ = CALCU_ELEMENT;
    text += 2;

    XMLElement **el = (XMLElement**)current;
    *el = (XMLElement*)str_to_int(text);
    current += sizeof(XMLElement*);

    for (text += 2; *text != ')'; text++);
    text++;

    *output = current;
    *input = text;
  }

  return true;
}

// encode variables
variable_size LiveMLParser::variableEncoder(const char *name, void *work)
{
  VariableContainer &variableContainer = *(VariableContainer*)work;
  variable_size scope = VAR_SCOPE_LOCAL;

  DCODE(printf("LiveMLParser::variableEncoder(%s)\n", name);)

  if (name[1] == '.')
  {
    switch (*name)
    {
    case 'm': // member
      scope = VAR_SCOPE_MEMBER;
      break;
    case 'g': // global
      scope = VAR_SCOPE_GLOBAL;
      break;
    case 'p': // parent
      scope = VAR_SCOPE_PARENT;
      break;
    }
    name += 2;
  }

  variable_size id = VAR_GLOBAL_ID(scope, variableContainer.add(scope, name));

  DCODE(printf("'%s' is %d\n", name, id);)

  return id;
}

size_t LiveMLParser::addEventType(const char *name)
{
  eventTypeContainer.add(name);

  return eventTypeContainer.getId(name);
}

LiveMLParser::LiveMLParser(void) :
PARAM_TEXT(registerParameterType(paramText)),
PARAM_NUMERIC(registerParameterType(paramNumeric)),
PARAM_VAR(registerParameterType(paramVar)),
PARAM_ACTOR(registerParameterType(paramActor)),
PARAM_ACTION(registerParameterType(paramAction)),
PARAM_EVENT(registerParameterType(paramEvent))
{
  TagType *reg = _tagTypeContainer.add("liveml", NULL);
  reg->setFunction(lmlTag::liveml);

  ParameterOption *po;

  // actor
  reg = addTagType("actor", "liveml");
  reg->setFunction(lmlTag::actor);
  reg->addParameter(0, PARAM_STRING, "name");
  reg->addParameter(1, PARAM_ACTOR, "type");

  // event
  reg = addTagType("event", "actor");
  reg->setFunction(lmlTag::event);
  reg->addParameter(0, PARAM_EVENT, "type");
  reg->addParameter(1, PARAM_NUMERIC, "case");

  // union
  reg = addTagType("union", "actor");
  reg->setFunction(lmlTag::runion, lmlTag::runion_close);
  reg->addParameter(0, PARAM_STRING, "name");

  // action
  reg = addTagType("action", "liveml actor");
  reg->setFunction(lmlTag::action);
  reg->addParameter(0, PARAM_STRING, "name");

  // endAction
  reg = addTagType("endAction");
  reg->addParameter(0, PARAM_STRING);

  // print
  reg = addTagType("print");
  reg->addParameter(0, PARAM_TEXT);

  // active
  reg = addTagType("active");
  reg->addParameter(0, PARAM_TEXT);
  reg->addParameter(1, PARAM_VAR, "var");

  // activeAction
  reg = addTagType("activeAction");
  reg->addParameter(0, PARAM_TEXT);

  // callAction
  reg = addTagType("callAction");
  reg->addParameter(0, PARAM_TEXT);

  // callEvent
  reg = addTagType("callEvent");
  reg->addParameter(0, PARAM_EVENT);

  // include
  reg = addTagType("include");
  reg->setFunction(lmlTag::include);
  reg->addParameter(0, PARAM_STRING);

  // repeat
  reg = addTagType("repeat");
  reg->addParameter(0, PARAM_NUMERIC, "count");

  // count
  reg = addTagType("count");

  // continue
  reg = addTagType("continue");

  // break
  reg = addTagType("break");

  // if
  reg = addTagType("if");
  reg->addParameter(0, PARAM_NUMERIC, "case");

  // wait
  reg = addTagType("wait");
  reg->addParameter(0, PARAM_NUMERIC);

  // exit
  reg = addTagType("exit");

  // var
  reg = addTagType("var");
  reg->addParameter(0, PARAM_VAR, "name");
  reg->addParameter(1, PARAM_NUMERIC);

  // focus
  reg = addTagType("focus");
  reg->addParameter(0, PARAM_NUMERIC, "target");

  // int
  reg = addTagType("int");
  reg->addParameter(0, PARAM_NUMERIC);

  registerEventType();
  registerActorType();

  calcu_encode_callback(calcuEncode);
  calcu_encode_workarea((void*)this);
}

void LiveMLParser::registerEventType(void)
{
  eventTypeContainer.add("init");
  eventTypeContainer.add("release");
}

void LiveMLParser::registerActorType(void)
{
  actorTypeContainer.add("general");
  actorTypeContainer.add("director");
}
