/** * LiveML - LiveML is screen(graphic) controling library using XML.
 *
 * LGPL License
 * Copyright (C) 2010 Nothan
 * http://github.com/nothan/liveml/
 * All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Nothan
 * private@nothan.xrea.jp
 *
 * Tsuioku Denrai
 * http://tsuioku-denrai.xrea.jp/
 */

#ifndef __LIVEMLRUNNER_H__
#define __LIVEMLRUNNER_H__

#include "livemlparser.h"
#include "list.hpp"
#include "bool_list.hpp"

#define LML_CMDFUNC(name) LML_CMDRTN name(LMLParameter &param)

typedef int LML_CMDRTN;
enum
{
  LML_CMDRTN_NULL = 0x00,
  LML_CMDRTN_STOP = 0x01,
  LML_CMDRTN_NEXT = 0x02,
  LML_CMDRTN_REACTION = 0x04,
  LML_CMDRTN_CHILD = 0x08
};

class LiveMLRunner;

struct NumericVariable
{
  variable_size id;
  fixed_float value;
};

class LMLVariable
{
  fixed_float* getNumeric(variable_size id)
  {
    DCODE(printf("LMLVariable::getNumeric(%d)\n", id);)
    id = VAR_LOCAL_ID(id);
    NumericVariable *v = numericList.front();

    while (v)
    {
      if (v->id == id)
      {
        numericList.chain_front(v);
        break;
      }
      v = list_next(v);
    }

    if (v == NULL)
    {
      DCODE(printf("added variable %d\n", id);)
      v = numericList.push_front();
	    v->id = id;
    }

    return &v->value;
  }

public:
  list<NumericVariable> numericList;

  void clear(void)
  {
    numericList.clear(true);
  }

  void setFixedFloat(variable_size id, fixed_float v)
  {
    DCODE(printf("LMLVariable::setFixedFloat(%d) = %d\n", id, v);)
    *getNumeric(id) = v;
  }

  void setString(variable_size id, const char *v)
  {
  }

  fixed_float getFixedFloat(variable_size id)
  {
    DCODE(printf("LMLVariable::getFixedFloat(%d) = %d\n", id, *getNumeric(id));)
    return *getNumeric(id);
  }

  const char* getString(variable_size id)
  {
    return "";
  }
};

template <class T>
class ActiveStack
{
public:
  list<T> list;

  T* current(void)
  {
    return list.back();
  }

  T* front(void)
  {
    return list.front();
  }

  T* add(void)
  {
    return list.push_back(false);
  }

  void close(T *item)
  {
    item->clear();
    list.remove(item);
  }

  void close()
  {
    close(current());
  }

  void clear(void)
  {
    T *item = front();
    while (item != NULL)
    {
       item->clear();
       item = list_next(item);
    }
    list.clear(true);
  }

  bool has(void)
  {
    return current() != NULL;
  }
};

class ResumeTagStack : public ActiveStack<XMLElement*>
{
public:
  void close(void)
  {
    list.remove(current());
  }

  void clear(void)
  {
    list.clear(true);
  }
};

class ChildAction
{
public:
  ActionParser *action;
  XMLElement *tag;
  ResumeTagStack resumeTagStack;
  // local variable
  LMLVariable variable;

  void clear(void)
  {
    resumeTagStack.clear();
    variable.clear();
  }
};

class ChildActionStack : public ActiveStack<ChildAction>
{
};

class ActiveAction
{
public:
  ChildActionStack childActionStack;
  unsigned int waitCounter;
  ::list<unsigned int> repeatCounter;

  void clear(void)
  {
    childActionStack.clear();
    waitCounter = 0;
    repeatCounter.clear(true);
  }

  void close(void)
  {
    childActionStack.close();
  }

  bool has(void)
  {
    return childActionStack.has();
  }

  ChildAction* current(void)
  {
    return childActionStack.current();
  }

  ChildAction* front(void)
  {
     return childActionStack.front();
  }

  ChildAction* add()
  {
    return childActionStack.add();
  }

  bool add(ActionParser *action)
  {
    if (action == NULL) return false;

    ChildAction *ab = add();
    ab->action = action;
    ab->tag = action->tag;

    return true;
  }
};

class ActiveActionStack : public ActiveStack<ActiveAction>
{
public:
  void clear(void)
  {
    ActiveStack<ActiveAction>::clear();
  }

  ActiveAction* add(ActionParser *action)
  {
    ActiveAction *aa = NULL;
    
    if (action->unionId != UNameContainer<UNameItem>::NOTFOUND)
    {
      aa = current();
      while (aa != NULL)
      {
        if (aa->front()->action->unionId == action->unionId)
        {
          if (aa->front()->action == action) return aa;
          aa->clear();
          break;
        }
        aa = list_prev(aa);
      }
    }
    if (aa == NULL)
    {
      aa = ActiveStack<ActiveAction>::add();
    }
    aa->add(action);

    DCODE(printf("added action[address: %x]\n", aa));

    return aa;
  }
};

class ActiveEventStack : public ActiveStack<ActiveAction>
{
public:
  ActiveAction* add(ActionParser *action)
  {
    ActiveAction *aa = ActiveStack<ActiveAction>::add();
    aa->add(action);

    DCODE(printf("added action[address: %x]\n", aa));

    return aa;
  }
};

class LMLActorObject
{
public:
  ActiveActionStack activeActionStack;
  ActiveEventStack activeEventStack;
  TagAct *parser;
  // member variable
  LMLVariable variable;

  LMLActorObject(void)
  {
  }

  ~LMLActorObject()
  {
    release();
  }

  void release(void)
  {
    parser = NULL;
    variable.clear();
    activeActionStack.clear();
    activeEventStack.clear();
  }

  void setVariableAllocator(list_allocator<NumericVariable> *alloc)
  {
    variable.numericList.allocator(alloc);
  }

  bool addAction(ActionParser *action)
  {
    DCODE(printf("LMLActorObject::setAction(%x)\n", action);)
    if (action == NULL) return false;

    activeActionStack.add(action);

    return true;
  }
  bool addAction(action_type type)
  {
    DCODE(printf("LMLActorObject::setAction(%d)\n", type);)
    ActionParser *action = parser->actionContainer[type];
    if (action->tag == NULL) return false;

    addAction(action);
    return true;
  }
  bool addAction(const char *name)
  {
    DCODE(printf("LMLActorObject::setActionByName(%s)\n", name);)
    return addAction(parser->actionContainer[name]);
  }

  bool setEvent(EventParser *event)
  {
    DCODE(printf("LMLActorObject::setEvent(%x)\n", event);)
    if (event == NULL) return false;

    activeEventStack.add((ActionParser*)event);

    return true;
  }

  bool setEvent(event_type type)
  {
    DCODE(printf("LMLActorObject::setEvent(%d)\n", type);)
    EventParser *event = parser->eventContainer[type];
	  if (event->tag == NULL) return false;

    return setEvent(event);
  }

  bool setEvent(const char *name)
  {
    DCODE(printf("LMLActorObject::setActionByName(%s)\n", name);)
    return setEvent(parser->eventContainer[name]);
  }

  void endAction(const char *name)
  {
    ActiveAction *aa = activeActionStack.current();
    while (aa)
    {
      if (!strcmp(aa->front()->action->getName(), name))
      {
        activeActionStack.close(aa);
        return;
      }
      aa = list_prev(aa);
    }
  }
};

class LMLParameter
{
  friend class LiveMLRunner;
  Parameter *param;
public:
  LiveMLRunner *runner;
  LMLActorObject *obj;
  ActiveAction *action;
  bool close;

  LMLParameter(Parameter *param, LiveMLRunner *runner, LMLActorObject *obj, ActiveAction *action)
  {
    this->param = param;
    this->runner = runner;
    this->obj = obj;
    this->action = action;
    this->close = false;
  }

  LMLVariable* getVariable(variable_size);

  bool has(size_t);
  XMLParameter* get(size_t);
};

class LiveMLRunner
{
  struct FuncPtr
  {
    LML_CMDRTN (*func)(LMLParameter&);
  };

  list<LMLActorObject> _actorList;
  vector<FuncPtr> _commandFuncs;

  list_allocator<ChildAction> _childActionAllocator;
  list_allocator<ActiveAction> _actionAllocator;
  list_allocator<unsigned int> _repeatAllocator;
  list_allocator<NumericVariable> _numericVariableAllocator;
  list_allocator<XMLElement*> _tagPtrAllocator;

  void setObjectAllocators(LMLActorObject&);

public:
  LiveMLParser *parser;
  // global variable
  LMLVariable variable;

  LiveMLRunner(LiveMLParser*);
  virtual ~LiveMLRunner();

  bool registerCommand(
    const char*,
    LML_CMDRTN (*)(LMLParameter&)
  );

  void setMaxActors(size_t);
  LMLActorObject* addActor(const char*);
  void dropActor(LMLActorObject&);

  LML_CMDRTN runCommand(LMLParameter&, XMLElement*);
  LML_CMDRTN runChildAction(LMLParameter&);
  LML_CMDRTN runAction(LMLParameter&);
  void callEvent(LMLActorObject&, event_type);
  void callEvent(LMLActorObject&, const char*);
  void callEvent(event_type);
  void callEvent(EventType*);
  bool runActiveActions(LMLActorObject&);
  bool runObject(LMLActorObject&);
  bool run(void);
  LMLActorObject* getActorObject(size_t);
  size_t getActorObjectId(LMLActorObject&);

  virtual LMLActorObject* createActor(size_t);

  static fixed_float calcuDecode(const char**, void*);
  static const char* variableStringDecoder(variable_size, void*);
};

#endif // __LIVEMLRUNNER_H__
