/**
 * LiveML - LiveML is screen(graphic) controling library using XML.
 *
 * LGPL License
 * Copyright (C) 2010 Nothan
 * http://github.com/nothan/directorml/
 * All rights reserved.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public
 *  License along with this library; if not, write to the Free
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * Nothan
 * private@nothan.xrea.jp
 *
 * Tsuioku Denrai
 * http://tsuioku-denrai.xrea.jp/
 */

#ifndef __LIVEMLPARSER_H__
#define __LIVEMLPARSER_H__

#include "xmlparser_tinyxml.h"
#include "calcu.h"
#include "text.h"

typedef size_t action_type;
typedef size_t event_type;

// variable scope
enum
{
  VAR_SCOPE_LOCAL = 0,
  VAR_SCOPE_MEMBER,
  VAR_SCOPE_PARENT,
  VAR_SCOPE_GLOBAL,
  VAR_SCOPE_MAX
};
#define VAR_NOTFOUND 0x70000000
#define VAR_GLOBAL_ID(scope, id) ((0x00100000 << scope) | id)
#define VAR_LOCAL_ID(id) (0x000FFFFF & id)
#define VAR_SCOPE_TYPE(id) ((0x00F00000 & id) >> 21)

template <class T>
class NameTagContainer : public NameContainer<T>
{
public:
  T* add(XMLElement *t)
  {
    T &i = *NameContainer<T>::add();
    i.tag = t;

    return &i;
  }

  size_t getId(const char *name)
  {
    for (tag_type i = 0; i < NameContainer<T>::_list.size(); i++)
    {
      if (!strcmp(NameContainer<T>::_list[i].getName(), name)) return i;
    }

    return NameContainer<T>::NOTFOUND;
  }
};

class ActionParser
{
public:
  XMLElement *tag;
  size_t unionId;

  ActionParser()
  {
    tag = NULL;
  }

  const char* getName(void)
  {
    const char* name = tag->param.getString(0);
    DCODE(printf("%s = ActionParser::getName()\n", name);)

    return tag->param.getString(0);
  }
};

class EventParser : public ActionParser
{
};

class ActionContainer : public NameTagContainer<ActionParser>
{
};

class EventContainer : public NameTagContainer<EventParser>
{
};

class TagAct : public ActionParser
{
public:
  EventContainer eventContainer;
  ActionContainer actionContainer;
};

class ActorParser : public TagAct
{
};

class ActorContainer : public NameTagContainer<ActorParser>
{
};

class EventType : public UNameItem
{
};

class EventTypeContainer : public UNameContainer<EventType>
{
};

class ActorTypeContainer : public UNameContainer<UNameItem>
{
};

class VariableContainer
{
  vector<string> _list[VAR_SCOPE_MAX];
public:
  string* operator[](variable_size id)
  {
    return &_list[VAR_SCOPE_TYPE(id)][VAR_LOCAL_ID(id)];
  }

  variable_size getId(int scope, const char *name)
  {
    variable_size id = 0, size = _list[scope].size();
    for (; id < size; id++)
    {
      if (_list[scope][id] == name) return id;
    }

    return VAR_NOTFOUND;
  }

  variable_size add(int scope, const char *name)
  {
    variable_size id = getId(scope, name);
    if (id == VAR_NOTFOUND)
    {
      _list[scope].push_back(name);
      id = _list[scope].size() - 1;
    }

    return id;
  }
};

class XMLNumeric : public XMLParameter
{
public:
  const char *encode;

  XMLNumeric(const char *encode) : XMLParameter()
  {
    this->encode = encode;
  }

  ~XMLNumeric()
  {
    free((void*)encode);
  }

  unsigned int getUInteger(void)
  {
    return (unsigned int)getInteger();
  }

  int getInteger(void)
  {
    return fixed_float_to_int(getFixedFloat());
  }

  float getFloat(void)
  {
    return fixed_float_to_float(getFixedFloat());
  }

  fixed_float getFixedFloat(void)
  {
    calcu_decode_data result = calcu_decode(encode);
    return result.value;
  }
};

class XMLVar : public XMLParameter
{
public:
  variable_size id;

  XMLVar(variable_size id) : XMLParameter()
  {
    this->id = id;
  }
};

class XMLText : public XMLParameter
{
public:
  XMLNode *node;

  XMLText(XMLNode *node) : XMLParameter()
  {
    this->node = node;
  }

  const char* getString(void)
  {
    return ((XMLTextNode*)node)->text;
  }
};

class LiveMLParser : public XMLParserTinyXML
{
  ActorParser *_currentActor;
public:
  const param_type PARAM_TEXT;
  const param_type PARAM_NUMERIC;
  const param_type PARAM_VAR;
  const param_type PARAM_ACTOR;
  const param_type PARAM_ACTION;
  const param_type PARAM_EVENT;

  EventTypeContainer eventTypeContainer;
  ActorTypeContainer actorTypeContainer;
  ActorContainer actorContainer;
  ActionContainer actionContainer;
  VariableContainer variableContainer;

  UNameContainer<UNameItem> unionContainer;
  size_t unionId;

  LiveMLParser(void);

  TagType* addTagType(const char *name, const char *parent = "action event")
  {
    return XMLParserTinyXML::addTagType(name, parent);
  }
  void setCurrentActor(ActorParser* t) { _currentActor = t; }
  ActorParser* getCurrentActor(void) { return _currentActor; }
  size_t addEventType(const char*);
  variable_size getVariableId();

  static int calcuEncode(char **, char**, void*);
  static variable_size variableEncoder(const char*, void*);
protected:
  void registerEventType(void);
  void registerActorType(void);

  static PARAMFUNC(paramText);
  static PARAMFUNC(paramNumeric);
  static PARAMFUNC(paramVar);
  static PARAMFUNC(paramActor);
  static PARAMFUNC(paramAction);
  static PARAMFUNC(paramEvent);
};

#endif // __LIVEMLPARSER_H__
