/**
 * list - The doubly-linked list.
 *
 * MIT License
 * Copyright (C) 2010 Nothan
 * http://github.com/nothan/c-utils/
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Nothan
 * private@nothan.xrea.jp
 *
 * Tsuioku Denrai
 * http://tsuioku-denrai.xrea.jp/
 */

#ifndef __LIST_HPP__
#define __LIST_HPP__

#include <stdlib.h>

template <class T>
struct list_item
{
  T data;
  list_item<T> *next;
  list_item<T> *prev;
};

template <class T>
class list_allocator
{
  size_t _size;
  list_item<T> *_pool;
  list_item<T> *_free;
public:
  list_allocator(size_t size = 0)
  {
    _size = 0;
    resize(size);
    clear();
  }

  ~list_allocator()
  {
    resize(0);
  }

  T* operator[](size_t id)
  {
    return (T*)(_pool + id);
  }

  size_t id(T* i)
  {
    return ((char*)i - (char*)_pool) / sizeof(list_item<T>);
  }

  void clear(void)
  {
    _free = NULL;
    for (size_t i = 0; i < _size; i++) remove(_pool + i);
  }

  void resize(size_t size)
  {
    if (_size) free(_pool);

    if (size)
    {
      _pool = (list_item<T>*)malloc(sizeof(list_item<T>) * size);
    }
    else _pool = _free = NULL;

    _size = size;
    clear();
  }

  list_item<T>* add(bool init = true)
  {
    list_item<T>* item = _free;

    if (_free == NULL) return NULL;

    _free = _free->next;

    item->next = NULL;
    item->prev = NULL;
    if (init) item = new(item) list_item<T>;

    return item;
  }

  void remove(list_item<T> *li)
  {
    if (li == _free) throw "list_allocator::remove() > free address removed";

    if (_free) li->next = _free;
    else li->next = NULL;
    _free = li;
  }

  list_item<T>* get_free(void)
  {
    return _free;
  }

  void set_free(list_item<T> *li)
  {
    _free = li;
  }

  list_item<T>* pool(void)
  {
    return _pool;
  }

  size_t size(void)
  {
    return _size;
  }
};

template <class T>
class list
{
  bool _free_alloc;
  list_allocator<T> *_alloc;
  list_item<T> *_active_front;
  list_item<T> *_active_back;

public:

  list(size_t size = 0)
  {
    _free_alloc = true;
    _alloc = new list_allocator<T>;
    resize(size);
    clear();
  }

  ~list()
  {
    if (_free_alloc) delete _alloc;
  }

  void allocator(list_allocator<T> *a, bool is_free = false)
  {
    if (_free_alloc) delete _alloc;
    _alloc = a;
    _free_alloc = is_free;
  }

  list_allocator<T>* allocator(void)
  {
    return _alloc;
  }

  void clear(bool soft = false)
  {
    if (soft)
    {
      if (_active_front)
      {
        _active_back->next = _alloc->get_free();
        _alloc->set_free(_active_front);
      }
    }
    else
    {
      _alloc->clear();
    }
    _active_front = NULL;
    _active_back = NULL;
  }

  void resize(size_t size)
  {
    _alloc->resize(size);
    _active_front = NULL;
    _active_back = NULL;
  }

  void chain_next(list_item<T> *a, list_item<T> *b)
  {
    if (a == b) return;

    unchain((T*)b);
    if (a->next == NULL) _active_back = b;
    else a->next->prev = b;
    b->next = a->next;
    b->prev = a;
    a->next = b;
  }

  void chain_prev(list_item<T> *a, list_item<T> *b)
  {
    if (a == b) return;

    unchain((T*)b);
    if (a->prev == NULL) _active_front = b;
    else a->prev->next = b;
    b->prev = a->prev;
    b->next = a;
    a->prev = b;
  }

  void chain_front(T *i)
  {
    if (_active_front == NULL)
    {
      unchain(i);
      _active_front = (list_item<T>*)i;
      _active_back = _active_front;
      return;
    }
    chain_prev(_active_front, (list_item<T>*)i);
  }

  void chain_back(T *i)
  {
    if (_active_back == NULL)
    {
      unchain(i);
      _active_front = (list_item<T>*)i;
      _active_back = _active_front;
      return;
    }
    chain_next(_active_back, (list_item<T>*)i);
  }

  T* push_front(bool init = true)
  {
    T *item = (T*)_alloc->add(init);
    if (!item) return NULL;
    chain_front(item);

    return item;
  }

  T* push_back(bool init = true)
  {
    T *item = (T*)_alloc->add(init);
    if (!item) return NULL;
    chain_back(item);

    return item;
  }

  void unchain(T *i)
  {
    list_item<T>* li = (list_item<T>*)i;
    if (li == _active_front) _active_front = li->next;
    if (li == _active_back) _active_back = li->prev;
    if (li->next) li->next->prev = li->prev;
    if (li->prev) li->prev->next = li->next;
    li->prev = li->next = NULL;
  }

  void remove(T *i)
  {
    unchain(i);
    _alloc->remove((list_item<T>*)i);
  }

  T* front(void)
  {
    return (T*)_active_front;
  }

  T* back(void)
  {
    return (T*)_active_back;
  }
};

template <class T>
T* list_prev(T *i)
{
  return (T*)((list_item<T>*)i)->prev;
}

template <class T>
T* list_next(T *i)
{
  return (T*)((list_item<T>*)i)->next;
}

#endif
