/**
 * calcu - original calculation.
 *
 * MIT License
 * Copyright (C) 2010 Nothan
 * http://github.com/nothan/c-utils/
 * All rights reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copiGes or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIAGBILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 * Nothan
 * private@nothan.xrea.jp
 *
 * Tsuioku Denrai
 * http://tsuioku-denrai.xrea.jp/
 */

#include "calcu.h"
#include <malloc.h>

static fixed_float (*decodeCallbackFunc)(const char**, void*);
static int (*encodeCallbackFunc)(char**, char**, void*);

static void *decodeWorkarea;
static void *encodeWorkarea;

void calcu_decode_callback(fixed_float (*func)(const char**, void*))
{
  decodeCallbackFunc = func;
}

void calcu_encode_callback(int (*func)(char**, char**, void*))
{
  encodeCallbackFunc = func;
}

void calcu_decode_workarea(void *work)
{
  decodeWorkarea = work;
}

void calcu_encode_workarea(void *work)
{
  encodeWorkarea = work;
}

calcu_decode_data calcu_decode(const char *data)
{
  fixed_float v1;
  fixed_float v2;
  stack_data stack;
  calcu_decode_data result;
  const char *current_data;

  stack_init(&stack, sizeof(fixed_float));
  stack_resize(&stack, 5);

  while (*data != CALCU_EOC)
  {
    switch (*data++)
    {
    case CALCU_NUMERIC:
      v1 = *(fixed_float*)data;
      data += sizeof(fixed_float);
      stack_push(&stack, &v1);
      continue;
    case '+':
      stack_pop(&stack, &v2);
      stack_pop(&stack, &v1);
      v1 += v2;
      stack_push(&stack, &v1);
      continue;
    case '-':
      stack_pop(&stack, &v2);
      stack_pop(&stack, &v1);
      v1 -= v2;
      stack_push(&stack, &v1);
      continue;
    case '%':
      stack_pop(&stack, &v2);
      stack_pop(&stack, &v1);
      v1 %= v2;
      stack_push(&stack, &v1);
      continue;
    case '*':
      stack_pop(&stack, &v2);
      stack_pop(&stack, &v1);
      v1 = fixed_float_multi(v1, v2);
      stack_push(&stack, &v1);
      continue;
    case '/':
      stack_pop(&stack, &v2);
      stack_pop(&stack, &v1);
      v1 = fixed_float_div(v1, v2);
      stack_push(&stack, &v1);
      continue;
    case '&':
      stack_pop(&stack, &v2);
      stack_pop(&stack, &v1);
      v1 = v1 & v2;
      stack_push(&stack, &v1);
      continue;
    case '|':
      stack_pop(&stack, &v2);
      stack_pop(&stack, &v1);
      v1 = v1 | v2;
      stack_push(&stack, &v1);
      continue;
    default:
      data--;
    }

    if (decodeCallbackFunc)
    {
      current_data = data;
      v1 = decodeCallbackFunc(&data, decodeWorkarea);
      if (current_data == data) break;

      stack_push(&stack, &v1);
    }
  }

  stack_pop(&stack, &result.value);
  stack_free(&stack);

  result.ptr = data;

  return result;
}

int operator_priority(char op)
{
  int priority = 0;
  switch (op)
  {
  case '/': priority++;
  case '*': priority++;
  case '%': priority++;
  case '+': priority++;
  case '-': priority++;
  case '&': priority++;
  case '|': priority++;
  case '\0': priority++;
  }

  return priority;
}

typedef struct
{
  int priority;
  char operator;
} calcu_operator;

calcu_encode_data calcu_encode(char* text, char *output)
{
  calcu_encode_data result;
  char *current, *current_text;
  int priority, plus_priority = 0;
  stack_data stack_optr;
  calcu_operator operator;

  if (!output) output = malloc(1024);
  current = output;

  stack_init(&stack_optr, sizeof(calcu_operator));
  stack_resize(&stack_optr, 20);

  while (1)
  {
    while (*text == ' ') text++;

    /* operator */
    if (priority = operator_priority(*text))
    {
      priority += plus_priority;
      while (!stack_pop(&stack_optr, &operator))
      {
        if (priority > operator.priority)
        {
          stack_push(&stack_optr, &operator);
          break;
        }
        /* output */
        *current = operator.operator;
        current++;
      }
      if (*text == '\0') break;
      operator.operator = *text++;
      operator.priority = priority;
      stack_push(&stack_optr, &operator);
      continue;
    }

    /* user function */
    current_text = text;
    if (encodeCallbackFunc)
    {
      if (!encodeCallbackFunc(&text, &current, encodeWorkarea)) break;
    }
    if (current_text != text) continue;

    /* numeric */
    if (*text >= '0' && *text <= '9')
    {
      *current++ = CALCU_NUMERIC;
      *(fixed_float*)current = str_to_fixed_float(text);
      while ((*text >= '0' && *text <= '9') || *text == '.') text++;

      current += sizeof(fixed_float);
      continue;
    }

    if (*text == '(')
    {
      plus_priority += 10;
      text++;
      continue;
    }

    if (*text == ')')
    {
      plus_priority -= 10;
      if (plus_priority < 0) break;
      text++;
      continue;
    }

    break;
  }

  stack_free(&stack_optr);
  *current = CALCU_EOC;

  result.text = text;
  result.first = output;
  result.last = current;

  return result;
}
